<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class ADVAIMMA_Admin {

	public function __construct() {
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
		add_action( 'add_meta_boxes', array( $this, 'register_meta_boxes' ) );
		add_action( 'save_post', array( $this, 'save_meta' ), 10, 2 );
	}

	public function enqueue_admin_assets( $hook ) {
		global $typenow;
		if ( 'advaimma_image_map' !== $typenow ) {
			return;
		}

		if ( function_exists( 'wp_enqueue_media' ) ) {
			wp_enqueue_media();
		}

		wp_enqueue_script( 'jquery-ui-draggable' );

		wp_enqueue_script(
			'advaimma-admin',
			ADVAIMMA_URL . 'assets/js/admin.js',
			array( 'jquery' ),
			ADVAIMMA_VERSION,
			true
		);

		wp_enqueue_style(
			'advaimma-admin-style',
			ADVAIMMA_URL . 'assets/css/admin.css',
			array(),
			ADVAIMMA_VERSION
		);

		wp_localize_script(
			'advaimma-admin',
			'advaimmaAdmin',
			array(
				'nonce'   => wp_create_nonce( 'advaimma_save' ),
				'strings' => array(
					'addPoint' => esc_html__( 'Add Point', 'advance-image-mapping' ),
					'remove'   => esc_html__( 'Remove', 'advance-image-mapping' ),
				),
			)
		);
	}

	public function register_meta_boxes() {
		add_meta_box(
			'advaimma_meta',
			esc_html__( 'Hotspot Builder', 'advance-image-mapping' ),
			array( $this, 'meta_box_callback' ),
			'advaimma_image_map',
			'normal',
			'high'
		);

		add_meta_box(
			'advaimma_design_meta',
			esc_html__( 'Design Settings', 'advance-image-mapping' ),
			array( $this, 'design_meta_box_callback' ),
			'advaimma_image_map',
			'side',
			'default'
		);
	}

	public function meta_box_callback( $post ) {
		$image_id = get_post_meta( $post->ID, 'advaimma_image_id', true );
		$spots    = get_post_meta( $post->ID, 'advaimma_spots', true );

		if ( ! is_array( $spots ) ) {
			$spots = array();
		}

		wp_nonce_field( 'advaimma_meta_nonce', 'advaimma_meta_nonce' );

		$image_url = $image_id ? wp_get_attachment_image_url( intval( $image_id ), 'full' ) : '';
		?>
		<div class="advaimma-meta-header">
			<strong><?php echo esc_html__( 'Shortcode', 'advance-image-mapping' ); ?>:</strong>
			<div class="advaimma-shortcode">[advaimma_map id="<?php echo esc_attr( $post->ID ); ?>"]</div>
		</div>

		<div class="advaimma-field">
			<button type="button" class="button" id="advaimma-select-image">
				<?php echo esc_html__( 'Upload/Select Image', 'advance-image-mapping' ); ?>
			</button>
			<span class="advaimma-image-note">
				<?php echo $image_url ? esc_html( wp_basename( $image_url ) ) : esc_html__( 'No image selected', 'advance-image-mapping' ); ?>
			</span>
			<input type="hidden" id="advaimma-image-id" name="advaimma_image_id" value="<?php echo esc_attr( $image_id ); ?>">
		</div>

		<div id="advaimma-canvas-wrapper" style="<?php echo $image_url ? '' : 'display:none;'; ?>">
			<div class="note-head"><?php echo esc_html__( 'Click on the image to add points or drag existing markers.', 'advance-image-mapping' ); ?></div>
			<div id="advaimma-canvas" style="<?php echo $image_url ? 'background-image:url(' . esc_url( $image_url ) . ');' : ''; ?>">
				<?php
				if ( ! empty( $spots ) ) :
					foreach ( $spots as $i => $s ) :
						$x          = isset( $s['x'] ) ? floatval( $s['x'] ) : 50;
						$y          = isset( $s['y'] ) ? floatval( $s['y'] ) : 50;
						$icon_image = isset( $s['icon_image'] ) ? esc_url( $s['icon_image'] ) : '';
						?>
						<div class="advaimma-spot" data-index="<?php echo esc_attr( $i ); ?>" style="left:<?php echo esc_attr( $x ); ?>%; top:<?php echo esc_attr( $y ); ?>%;">
							<?php if ( $icon_image ) : ?>
								<img src="<?php echo esc_url( $icon_image ); ?>" alt="" />
							<?php else : ?>
								<span class="advaimma-number"><?php echo esc_html( $i + 1 ); ?></span>
							<?php endif; ?>
						</div>
					<?php endforeach;
				endif;
				?>
			</div>
		</div>

		<div id="advaimma-spots-list">
			<ul>
				<?php if ( ! empty( $spots ) ) :
					foreach ( $spots as $i => $s ) :
						$title      = isset( $s['title'] ) ? $s['title'] : '';
						$content    = isset( $s['content'] ) ? $s['content'] : '';
						$icon       = isset( $s['icon'] ) ? $s['icon'] : 'plus';
						$icon_image = isset( $s['icon_image'] ) ? $s['icon_image'] : '';
						$trigger    = ( isset( $s['trigger'] ) && in_array( $s['trigger'], array( 'hover', 'click' ), true ) ) ? $s['trigger'] : 'hover';
						$x          = isset( $s['x'] ) ? floatval( $s['x'] ) : 50;
						$y          = isset( $s['y'] ) ? floatval( $s['y'] ) : 50;
						?>
						<li class="advaimma-spot-item" data-index="<?php echo esc_attr( $i ); ?>">
							<h4><?php echo esc_html( sprintf( __( 'Point %d', 'advance-image-mapping' ), intval( $i + 1 ) ) ); ?></h4>

							<label><?php esc_html_e( 'Title', 'advance-image-mapping' ); ?>
								<input type="text" name="advaimma_spots[<?php echo esc_attr( $i ); ?>][title]" value="<?php echo esc_attr( $title ); ?>">
							</label>

							<label><?php esc_html_e( 'Content', 'advance-image-mapping' ); ?>
								<textarea name="advaimma_spots[<?php echo esc_attr( $i ); ?>][content]" rows="3"><?php echo esc_textarea( $content ); ?></textarea>
							</label>

							<label>
								<?php esc_html_e( 'Icon (Dashicon slug)', 'advance-image-mapping' ); ?>
								<a href="https://developer.wordpress.org/resource/dashicons/#iconlist" target="_blank"><?php esc_html_e( 'Browse', 'advance-image-mapping' ); ?></a>
								<input type="text" name="advaimma_spots[<?php echo esc_attr( $i ); ?>][icon]" value="<?php echo esc_attr( $icon ); ?>">
							</label>

							<label class="upload-custom-icon">
								<p><?php esc_html_e( 'Or upload custom icon', 'advance-image-mapping' ); ?></p>
								<input type="hidden" class="advaimma-icon-image" name="advaimma_spots[<?php echo esc_attr( $i ); ?>][icon_image]" value="<?php echo esc_attr( $icon_image ); ?>">
								<button type="button" class="button advaimma-upload-icon"><?php esc_html_e( 'Upload Icon', 'advance-image-mapping' ); ?></button>
								<span class="advaimma-icon-preview">
									<?php
									if ( $icon_image ) {
										echo '<img src="' . esc_url( $icon_image ) . '" alt="">';
										echo '<span class="advaimma-remove-icon">×</span>';
									}
									?>
								</span>
							</label>

							<label><?php esc_html_e( 'Trigger', 'advance-image-mapping' ); ?>
								<select name="advaimma_spots[<?php echo esc_attr( $i ); ?>][trigger]">
									<option value="hover" <?php selected( $trigger, 'hover' ); ?>><?php esc_html_e( 'Hover', 'advance-image-mapping' ); ?></option>
									<option value="click" <?php selected( $trigger, 'click' ); ?>><?php esc_html_e( 'Click', 'advance-image-mapping' ); ?></option>
								</select>
							</label>

							<input type="hidden" class="coord-x" name="advaimma_spots[<?php echo esc_attr( $i ); ?>][x]" value="<?php echo esc_attr( $x ); ?>">
							<input type="hidden" class="coord-y" name="advaimma_spots[<?php echo esc_attr( $i ); ?>][y]" value="<?php echo esc_attr( $y ); ?>">

							<p><button type="button" class="button advaimma-remove"><?php esc_html_e( 'Remove', 'advance-image-mapping' ); ?></button></p>
						</li>
					<?php endforeach;
				endif; ?>
			</ul>
		</div>
		<?php
	}

	// 🎨 Design meta box
	public function design_meta_box_callback( $post ) {
		$defaults = array(
			'title_color'     => '#ffffff',
			'content_color'   => '#ffffff',
			'tip_bg_color'    => '#333333',
			'point_bg_color'  => '#000000',
			'number_color'    => '#ffffff',
		);

		$colors = wp_parse_args( get_post_meta( $post->ID, 'advaimma_colors', true ), $defaults );

		wp_nonce_field( 'advaimma_design_nonce', 'advaimma_design_nonce' );
		?>
		<p><label>Title Color:<br><input type="color" name="advaimma_colors[title_color]" value="<?php echo esc_attr( $colors['title_color'] ); ?>"></label></p>
		<p><label>Content Color:<br><input type="color" name="advaimma_colors[content_color]" value="<?php echo esc_attr( $colors['content_color'] ); ?>"></label></p>
		<p><label>Tooltip Background:<br><input type="color" name="advaimma_colors[tip_bg_color]" value="<?php echo esc_attr( $colors['tip_bg_color'] ); ?>"></label></p>
		<p><label>Point Background:<br><input type="color" name="advaimma_colors[point_bg_color]" value="<?php echo esc_attr( $colors['point_bg_color'] ); ?>"></label></p>
		<p><label>Point Number Color:<br><input type="color" name="advaimma_colors[number_color]" value="<?php echo esc_attr( $colors['number_color'] ); ?>"></label></p>
		<?php
	}

	public function save_meta( $post_id, $post ) {
		if ( ! isset( $post->post_type ) || 'advaimma_image_map' !== $post->post_type ) {
			return;
		}

		if ( ! empty( $_POST['advaimma_colors'] ) && isset( $_POST['advaimma_design_nonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['advaimma_design_nonce'] ) ), 'advaimma_design_nonce' ) ) {
			$colors = array_map( 'sanitize_hex_color', wp_unslash( $_POST['advaimma_colors'] ) );
			update_post_meta( $post_id, 'advaimma_colors', $colors );
		}

		// Save existing hotspot data (no change)
		if ( empty( $_POST['advaimma_meta_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['advaimma_meta_nonce'] ) ), 'advaimma_meta_nonce' ) ) {
			return;
		}
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}
		if ( ! current_user_can( 'edit_post', $post_id ) ) {
			return;
		}

		if ( isset( $_POST['advaimma_image_id'] ) ) {
			$image_id = intval( wp_unslash( $_POST['advaimma_image_id'] ) );
			if ( $image_id > 0 ) {
				update_post_meta( $post_id, 'advaimma_image_id', $image_id );
			} else {
				delete_post_meta( $post_id, 'advaimma_image_id' );
			}
		}

		$raw_spots = isset( $_POST['advaimma_spots'] ) ? wp_unslash( $_POST['advaimma_spots'] ) : array();
		if ( ! is_array( $raw_spots ) ) {
			return;
		}

		$spots = array();
		foreach ( $raw_spots as $s ) {
			if ( empty( $s['title'] ) && empty( $s['content'] ) ) {
				continue;
			}
			$spots[] = array(
				'title'      => sanitize_text_field( $s['title'] ),
				'content'    => wp_kses_post( $s['content'] ),
				'icon'       => sanitize_text_field( $s['icon'] ),
				'icon_image' => esc_url_raw( $s['icon_image'] ),
				'trigger'    => in_array( $s['trigger'], array( 'hover', 'click' ), true ) ? $s['trigger'] : 'hover',
				'x'          => floatval( $s['x'] ),
				'y'          => floatval( $s['y'] ),
			);
		}

		update_post_meta( $post_id, 'advaimma_spots', $spots );
	}
}
