(function ($) {
	'use strict';

	const $canvas = $('#advaimma-canvas');
	const $canvasWrapper = $('#advaimma-canvas-wrapper');
	const $list = $('#advaimma-spots-list ul');
	let index = $list.find('li').length || 0;
	let markerDragged = false;

	// Hide wrapper if no image set
	if (!$('#advaimma-image-id').val()) {
		$canvasWrapper.hide();
	}

	// Refresh spot numbering
	function refreshSpotNumbers() {
		$list.find('li.advaimma-spot-item').each(function (i) {
			const idx = $(this).data('index');
			$(this).find('h4').text('Point ' + (i + 1));
			$canvas.find('.advaimma-spot[data-index="' + idx + '"]').text(i + 1);
		});
	}

	// Add a new spot
	function addSpot(x, y, data = {}) {
		const i = index++;
		const title = data.title || '';
		const content = data.content || '';
		const icon = data.icon || 'plus';
		const icon_image = data.icon_image || '';
		const trigger = data.trigger || 'hover';
		const xval = (data.x !== undefined) ? data.x : x;
		const yval = (data.y !== undefined) ? data.y : y;

		// Marker
		const $marker = $('<div>', {
			class: 'advaimma-spot',
			'data-index': i,
			text: i + 1,
			css: { left: xval + '%', top: yval + '%' },
			attr: { tabindex: 0, role: 'button', 'aria-label': 'Point ' + (i + 1) }
		});
		$canvas.append($marker);

		// Escape title/content safely
		const safeTitle = $('<div>').text(title).html();
		const safeContent = $('<div>').text(content).html();

		// Spot item markup
		const item = `
		<li class="advaimma-spot-item" data-index="${i}">
			<h4>Point ${i + 1}</h4>
			<label>Title <input type="text" name="advaimma_spots[${i}][title]" value="${safeTitle}"></label>
			<label>Content <textarea name="advaimma_spots[${i}][content]" rows="3">${safeContent}</textarea></label>
			<label>Icon <input type="text" class="advaimma-icon-slug" name="advaimma_spots[${i}][icon]" value="${icon}"></label>
			<label class="upload-custom-icon">Or upload custom icon
				<input type="hidden" class="advaimma-icon-image" name="advaimma_spots[${i}][icon_image]" value="${icon_image}">
				<button type="button" class="button advaimma-upload-icon">Upload Icon</button>
				<span class="advaimma-icon-preview">
					${icon_image ? '<img src="' + icon_image + '" alt="" style="height:28px;margin-left:8px;border-radius:3px"><span class="advaimma-remove-icon">×</span>' : ''}
				</span>
			</label>
			<label>Trigger <select name="advaimma_spots[${i}][trigger]">
				<option value="hover" ${trigger === 'hover' ? 'selected' : ''}>Hover</option>
				<option value="click" ${trigger === 'click' ? 'selected' : ''}>Click</option>
			</select></label>
			<input type="hidden" class="coord-x" name="advaimma_spots[${i}][x]" value="${xval}">
			<input type="hidden" class="coord-y" name="advaimma_spots[${i}][y]" value="${yval}">
			<p><button type="button" class="button advaimma-remove">Remove</button></p>
		</li>`;
		$list.append(item);
		refreshSpotNumbers();
	}

	// Set initial index
	$list.find('li.advaimma-spot-item').each(function () {
		const idx = parseInt($(this).data('index'), 10);
		if (!isNaN(idx)) index = Math.max(index, idx + 1);
	});

	// Select main image
	$(document).on('click.advaimmaAdmin', '#advaimma-select-image', function (e) {
		e.preventDefault();
		const frame = wp.media({
			title: 'Select Image',
			button: { text: 'Use this image' },
			multiple: false,
			library: { type: 'image' }
		});

		frame.on('select', function () {
			const att = frame.state().get('selection').first().toJSON();
			if (att && att.id) {
				$('#advaimma-image-id').val(att.id);
				$('#advaimma-canvas').css('background-image', 'url(' + att.url + ')');
				$('.advaimma-image-note').text(att.filename || att.url);
				$canvasWrapper.fadeIn();
			}
		});
		frame.open();
	});

	// Add spot manually
	$(document).on('click.advaimmaAdmin', '#advaimma-add-spot', function () {
		addSpot(50, 50);
	});

	// Click canvas to add spot
	$canvas.on('click.advaimmaAdmin', function (e) {
		if (markerDragged) {
			markerDragged = false;
			return;
		}
		if ($(e.target).closest('.advaimma-spot').length) return;
		const rect = this.getBoundingClientRect();
		if (!rect.width || !rect.height) return;

		const x = ((e.clientX - rect.left) / rect.width * 100).toFixed(2);
		const y = ((e.clientY - rect.top) / rect.height * 100).toFixed(2);
		addSpot(x, y);
	});

	// Drag markers
	$canvas.on('mousedown.advaimmaAdmin', '.advaimma-spot', function (e) {
		e.preventDefault();
		const $m = $(this);
		const rect = $canvas[0].getBoundingClientRect();
		const startX = e.pageX;
		const startY = e.pageY;
		const startLeftPx = parseFloat($m.css('left')) || 0;
		const startTopPx = parseFloat($m.css('top')) || 0;
		const startLeftPct = (startLeftPx / rect.width) * 100;
		const startTopPct = (startTopPx / rect.height) * 100;

		$(document)
			.on('mousemove.advaimmaAdmin', function (ev) {
				const dxPct = (ev.pageX - startX) / rect.width * 100;
				const dyPct = (ev.pageY - startY) / rect.height * 100;
				const nx = Math.min(100, Math.max(0, startLeftPct + dxPct));
				const ny = Math.min(100, Math.max(0, startTopPct + dyPct));
				$m.css({ left: nx + '%', top: ny + '%' });

				const idx = $m.data('index');
				const $item = $list.find('.advaimma-spot-item[data-index="' + idx + '"]');
				$item.find('.coord-x').val(nx.toFixed(2));
				$item.find('.coord-y').val(ny.toFixed(2));

				markerDragged = true;
			})
			.on('mouseup.advaimmaAdmin', function () {
				$(document).off('.advaimmaAdmin');
			});
	});

	// Remove spot
	$list.on('click.advaimmaAdmin', '.advaimma-remove', function () {
		const $item = $(this).closest('.advaimma-spot-item');
		const idx = $item.data('index');
		$item.remove();
		$canvas.find('.advaimma-spot[data-index="' + idx + '"]').remove();
		refreshSpotNumbers();
	});

	// Upload icon
	$list.on('click.advaimmaAdmin', '.advaimma-upload-icon', function (e) {
		e.preventDefault();
		const $item = $(this).closest('.advaimma-spot-item');
		const frame = wp.media({ title: 'Select Icon', multiple: false, library: { type: 'image' } });

		frame.on('select', function () {
			const att = frame.state().get('selection').first().toJSON();
			if (att && att.url) {
				$item.find('.advaimma-icon-image').val(att.url);
				$item.find('.advaimma-icon-preview').html(
					`<img src="${att.url}" alt="" style="height:28px;margin-left:8px;border-radius:3px">
					 <span class="advaimma-remove-icon">×</span>`
				);
			}
		});
		frame.open();
	});

	// Remove uploaded icon
	$list.on('click.advaimmaAdmin', '.advaimma-remove-icon', function (e) {
		e.preventDefault();
		const $item = $(this).closest('.advaimma-spot-item');
		$item.find('.advaimma-icon-image').val('');
		$item.find('.advaimma-icon-preview').html('');
	});

	// Initial numbering
	refreshSpotNumbers();

	// === Enable WP Color Picker for new color options ===
	if ($('.advaimma-color-picker').length) {
		$('.advaimma-color-picker').wpColorPicker();
	}

})(jQuery);
