<?php
/**
 * Plugin Name: Admin Maintenance Message
 * Description: Zeigt eine konfigurierbare Wartungsleiste im WP-Admin (rollenbasiert) und optional einen Frontend-Wartungsmodus mit wählbarer Landingpage (Template-Datei, WP-Seite oder Custom HTML).
 * Version: 1.0.2
 * Author: Andreas Grzybowski (codekeks.de)
 * License: GPLv2 or later
 * Requires at least: 6.0
 * Requires PHP: 7.4
 * Text Domain: admin-maintance-msg
 * Domain Path: /languages
 */

if (!defined('ABSPATH')) exit;

final class AMMSG_Admin_Maintenance {
    const OPTION_OLD = 'ck_admin_maintenance_opts'; // Migration
    const OPTION     = 'ammsg_options';
    const HANDLE_CSS = 'ammsg-admin-css';
    const PAGE_SLUG  = 'ammsg-admin-maintanance';
    const VERSION    = '1.0.2';

    public function __construct() {
        register_activation_hook(__FILE__, [$this, 'on_activate']);

        // Settings + Seite
        add_action('admin_menu',  [$this, 'add_settings_page']);
        add_action('admin_init',  [$this, 'register_settings']);

        // Admin-Notice
        add_action('admin_notices',         [$this, 'maybe_show_notice']);
        add_action('network_admin_notices', [$this, 'maybe_show_notice']);

        // Admin-Assets
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);

        // Frontend-Wartung
        add_action('template_redirect',     [$this, 'maybe_frontend_maintenance'], 0);

        // Link in Pluginliste
        add_filter('plugin_action_links_' . plugin_basename(__FILE__), function($links){
            if (current_user_can('manage_options')) {
                $url = admin_url('options-general.php?page=' . self::PAGE_SLUG);
                $links[] = '<a href="'.esc_url($url).'">'.esc_html__('Settings', 'admin-maintance-msg').'</a>';
            }
            return $links;
        });
    }

    /* ==================== Defaults: Single Source of Truth ==================== */
    private static function defaults(): array {
        return [
            // Admin Notice (bestehend)
            'enabled'          => 0,
            'from'             => '',
            'to'               => '',
            'message'          => '',

            // NEU: Rollensteuerung für Notice
            'roles'            => [],

            // Frontend Wartungsmodus
            // fe_mode: 'page' | 'custom' | 'template'
            'fe_enabled'       => 0,
            'fe_bypass_logged' => 1,
            'fe_mode'          => 'page',
            'fe_page_id'       => 0,
            'fe_html'          => '',
            'fe_template'      => '', // z. B. 'default.html'
        ];
    }

    /* ==================== Lifecycle ==================== */
    public function on_activate(): void {
        if (!get_option(self::OPTION)) {
            $defaults = self::defaults();
            $old = get_option(self::OPTION_OLD);
            if (is_array($old)) {
                add_option(self::OPTION, wp_parse_args($old, $defaults));
                delete_option(self::OPTION_OLD);
            } else {
                add_option(self::OPTION, $defaults);
            }
        }
    }

    /* ==================== Helpers (Options, Dates, Weekday, Window) ==================== */
    private function get_opts(): array {
        $opts = get_option(self::OPTION, []);
        return wp_parse_args($opts, self::defaults());
    }

    private function parse_local_dt($dt): ?DateTime {
        if (!$dt) return null;
        $tz = function_exists('wp_timezone') ? wp_timezone() : new DateTimeZone('UTC');
        $norm = str_replace(' ', 'T', $dt);
        $d = DateTime::createFromFormat('Y-m-d\TH:i', $norm, $tz);
        if ($d instanceof DateTime) return $d;
        try { return new DateTime($dt, $tz); } catch (Exception $e) { return null; }
    }

    private function weekday_de_short(DateTimeInterface $dt): string {
        $map = ['So','Mo','Di','Mi','Do','Fr','Sa'];
        return $map[(int)$dt->format('w')];
    }
    private function weekday_de_long(DateTimeInterface $dt): string {
        $map = ['Sonntag','Montag','Dienstag','Mittwoch','Donnerstag','Freitag','Samstag'];
        return $map[(int)$dt->format('w')];
    }

    private function format_window_text(?DateTime $from, ?DateTime $to): string {
        if (!$from) return '';
        $tzabbr = $from->format('T');
        if ($to && $from->format('Y-m-d') !== $to->format('Y-m-d')) {
            return sprintf(
                '%s, %s, %s – %s, %s, %s (%s)',
                $this->weekday_de_short($from), $from->format('d.m.Y'), $from->format('H:i'),
                $this->weekday_de_short($to),   $to->format('d.m.Y'),   $to->format('H:i'),
                $tzabbr
            );
        }
        return sprintf(
            '%s, %s, %s–%s (%s)',
            $this->weekday_de_long($from), $from->format('d.m.Y'),
            $from->format('H:i'), $to ? $to->format('H:i') : $from->format('H:i'),
            $tzabbr
        );
    }

    /* ==================== Admin Assets ==================== */
    public function enqueue_admin_assets(): void {
        $css_url = plugin_dir_url(__FILE__) . 'assets/admin.css';
        wp_register_style(self::HANDLE_CSS, $css_url, [], self::VERSION, 'all');
        wp_enqueue_style(self::HANDLE_CSS);
    }

    /* ==================== Settings Page ==================== */
    public function add_settings_page(): void {
        add_options_page(
            __('Wartungshinweis', 'admin-maintance-msg'),
             __('Wartungshinweis', 'admin-maintance-msg'),
            'manage_options',
            self::PAGE_SLUG,
            [$this, 'render_settings_page']
        );
    }

    public function register_settings(): void {
        register_setting(self::OPTION, self::OPTION, [
            'type'              => 'array',
            'sanitize_callback' => [$this, 'sanitize_opts'],
            'default'           => self::defaults(),
        ]);

        add_settings_section('ammsg_section_main', '', '__return_null', self::OPTION);

        // Toggle Notice
        add_settings_field('enabled', __('Wartung aktiv (Admin-Notice)', 'admin-maintance-msg'), function() {
            $o = $this->get_opts(); ?>
            <label>
                <input type="checkbox" name="<?php echo esc_attr(self::OPTION); ?>[enabled]" value="1" <?php checked($o['enabled'], 1); ?> />
                <?php esc_html_e('Hinweis im Admin anzeigen (unabhängig vom Zeitfenster).', 'admin-maintance-msg'); ?>
            </label>
        <?php }, self::OPTION, 'ammsg_section_main');

        // Von / Bis
        add_settings_field('from', __('Von', 'admin-maintance-msg'), function() {
            $o = $this->get_opts(); ?>
            <input type="datetime-local" name="<?php echo esc_attr(self::OPTION); ?>[from]" value="<?php echo esc_attr($o['from']); ?>" />
            <p class="description"><?php esc_html_e('Optional. Zeitzone laut Server/WordPress.', 'admin-maintance-msg'); ?></p>
        <?php }, self::OPTION, 'ammsg_section_main');

        add_settings_field('to', __('Bis', 'admin-maintance-msg'), function() {
            $o = $this->get_opts(); ?>
            <input type="datetime-local" name="<?php echo esc_attr(self::OPTION); ?>[to]" value="<?php echo esc_attr($o['to']); ?>" />
        <?php }, self::OPTION, 'ammsg_section_main');

        // Nachricht
        add_settings_field('message', __('Nachricht (Admin-Notice)', 'admin-maintance-msg'), function() {
            $o = $this->get_opts(); ?>
            <textarea name="<?php echo esc_attr(self::OPTION); ?>[message]" rows="4" style="width:100%;max-width:700px;"><?php echo esc_textarea($o['message']); ?></textarea>
            <p class="description"><?php esc_html_e('Freitext. Platzhalter: {WEEKDAY}, {DATE}, {FROM}, {TO}, {TZ}, {DATETIME_FROM}, {DATETIME_TO}, {WEEKDAY_TO}, {DATE_TO}.', 'admin-maintance-msg'); ?></p>
        <?php }, self::OPTION, 'ammsg_section_main');

        // Rollensteuerung (Notice)
        add_settings_field('roles', __('Sichtbar für Rollen (Notice)', 'admin-maintance-msg'), function () {
            $o = $this->get_opts();
            global $wp_roles;
            $roles = is_object($wp_roles) ? $wp_roles->roles : [];
            ?>
            <select name="<?php echo esc_attr(self::OPTION); ?>[roles][]" multiple size="6" style="min-width:260px">
                <?php foreach ($roles as $key => $data): ?>
                    <option value="<?php echo esc_attr($key); ?>" <?php selected(in_array($key, (array)$o['roles'], true)); ?>>
                        <?php echo esc_html(translate_user_role($data['name'])); ?>
                    </option>
                <?php endforeach; ?>
            </select>
            <p class="description"><?php esc_html_e('Leer lassen = Hinweis für alle Rollen anzeigen.', 'admin-maintance-msg'); ?></p>
            <?php
        }, self::OPTION, 'ammsg_section_main');

        // Frontend: Toggle
        add_settings_field('fe_enabled', __('Frontend-Maintenance aktiv', 'admin-maintance-msg'), function () {
            $o = $this->get_opts(); ?>
            <label><input type="checkbox" name="<?php echo esc_attr(self::OPTION); ?>[fe_enabled]" value="1" <?php checked($o['fe_enabled'], 1); ?> /> <?php esc_html_e('Wartungsmodus im Frontend aktivieren (oder automatisch im Zeitfenster).', 'admin-maintance-msg'); ?></label>
            <p class="description"><?php esc_html_e('Bei aktivem Wartungsmodus wird die reguläre Seite durch eine Landingpage ersetzt.', 'admin-maintance-msg'); ?></p>
            <?php
        }, self::OPTION, 'ammsg_section_main');

        // Frontend: Bypass
        add_settings_field('fe_bypass_logged', __('Bypass für eingeloggte Nutzer', 'admin-maintance-msg'), function () {
            $o = $this->get_opts(); ?>
            <label><input type="checkbox" name="<?php echo esc_attr(self::OPTION); ?>[fe_bypass_logged]" value="1" <?php checked($o['fe_bypass_logged'], 1); ?> /> <?php esc_html_e('Eingeloggte Nutzer dürfen die normale Seite sehen.', 'admin-maintance-msg'); ?></label>
            <?php
        }, self::OPTION, 'ammsg_section_main');

        // Frontend: Modus
        add_settings_field('fe_mode', __('Landingpage-Modus', 'admin-maintance-msg'), function () {
            $o = $this->get_opts();
            $mode = $o['fe_mode'] ?: 'page'; ?>
            <label><input type="radio" name="<?php echo esc_attr(self::OPTION); ?>[fe_mode]" value="page" <?php checked($mode, 'page'); ?> /> <?php esc_html_e('WordPress-Seite verwenden', 'admin-maintance-msg'); ?></label><br>
            <label><input type="radio" name="<?php echo esc_attr(self::OPTION); ?>[fe_mode]" value="custom" <?php checked($mode, 'custom'); ?> /> <?php esc_html_e('Eigener HTML-Block (Textfeld unten)', 'admin-maintance-msg'); ?></label><br>
            <label><input type="radio" name="<?php echo esc_attr(self::OPTION); ?>[fe_mode]" value="template" <?php checked($mode, 'template'); ?> /> <?php esc_html_e('Template-Datei aus assets verwenden', 'admin-maintance-msg'); ?></label>
            <?php
        }, self::OPTION, 'ammsg_section_main');

        // Frontend: Seite
		add_settings_field(
			'fe_page_id',
			__('Seite für Maintenance', 'admin-maintance-msg'),
			function () {
				$o = $this->get_opts();

				// Core-Funktion soll HTML zurückgeben, nicht direkt echo'n
				$select = wp_dropdown_pages([
					'name'              => self::OPTION . '[fe_page_id]',
					'echo'              => 0, // <— wichtig: kein direktes Echo
					'show_option_none'  => __('— Keine Seite ausgewählt —', 'admin-maintance-msg'),
					'option_none_value' => 0,
					'selected'          => (int) $o['fe_page_id'],
				]);

				// Sicher ausgeben: Core-HTML darf nicht esc_html()’ed werden
				// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Safe Core HTML from wp_dropdown_pages()
				echo $select;

				echo '<p class="description">' .
					esc_html__('Wird genutzt, wenn „WordPress-Seite“ gewählt ist.', 'admin-maintance-msg') .
				'</p>';
			},
			self::OPTION,
			'ammsg_section_main'
		);

        // Frontend: Template
        add_settings_field('fe_template', __('Template-Datei (assets)', 'admin-maintance-msg'), function () {
            $o = $this->get_opts();
            $templates = $this->available_templates(); ?>
            <select name="<?php echo esc_attr(self::OPTION); ?>[fe_template]">
                <option value=""><?php esc_html_e('— Bitte wählen —', 'admin-maintance-msg'); ?></option>
                <?php foreach ($templates as $file => $label): ?>
                    <option value="<?php echo esc_attr($file); ?>" <?php selected($o['fe_template'], $file); ?>>
                        <?php echo esc_html($label); ?>
                    </option>
                <?php endforeach; ?>
            </select>
            <p class="description"><?php esc_html_e('Templates liegen unter assets/maintenance/templates/*.html (z. B. default.html).', 'admin-maintance-msg'); ?></p>
            <?php
        }, self::OPTION, 'ammsg_section_main');

        // Frontend: Custom HTML
        add_settings_field('fe_html', __('Custom HTML (Landingpage)', 'admin-maintance-msg'), function () {
            $o = $this->get_opts();
            ?>
            <textarea name="<?php echo esc_attr(self::OPTION); ?>[fe_html]" rows="10" style="width:100%;max-width:900px"><?php echo esc_textarea($o['fe_html']); ?></textarea>
            <p class="description"><?php esc_html_e('Eigener HTML-Block (z. B. mit eigenem CSS aus deinem Theme/Plugin).', 'admin-maintance-msg'); ?></p>
            <?php
        }, self::OPTION, 'ammsg_section_main');
    }

    public function sanitize_opts($input): array {
        $out = $this->get_opts();

        // Notice
        $out['enabled'] = !empty($input['enabled']) ? 1 : 0;
        $out['from']    = isset($input['from']) ? sanitize_text_field($input['from']) : '';
        $out['to']      = isset($input['to'])   ? sanitize_text_field($input['to'])   : '';
        $out['message'] = isset($input['message']) ? wp_kses_post($input['message']) : '';

        foreach (['from','to'] as $k) {
            if ($out[$k] && !$this->parse_local_dt($out[$k])) {
                $out[$k] = '';
            }
        }

        // Rollen
        $out['roles'] = isset($input['roles']) && is_array($input['roles'])
            ? array_values(array_map('sanitize_text_field', $input['roles']))
            : [];

        // Frontend-Maintenance
        $out['fe_enabled']       = !empty($input['fe_enabled']) ? 1 : 0;
        $out['fe_bypass_logged'] = !empty($input['fe_bypass_logged']) ? 1 : 0;

        $mode = isset($input['fe_mode']) ? sanitize_text_field($input['fe_mode']) : 'page';
        $out['fe_mode'] = in_array($mode, ['page','custom','template'], true) ? $mode : 'page';

        $out['fe_page_id'] = isset($input['fe_page_id']) ? (int)$input['fe_page_id'] : 0;

        // Custom HTML – erlaubte Tags (erweiterbar)
        $allowed = [
            'a'      => ['href'=>[], 'title'=>[], 'target'=>[], 'rel'=>[]],
            'p'      => [], 'div'=>['class'=>[], 'id'=>[]], 'span'=>['class'=>[], 'id'=>[]],
            'h1'=>[], 'h2'=>[], 'h3'=>[], 'h4'=>[], 'h5'=>[], 'h6'=>[],
            'ul'=>[], 'ol'=>[], 'li'=>[],
            'strong'=>[], 'em'=>[], 'br'=>[], 'hr'=>[],
            'img'=>['src'=>[], 'alt'=>[], 'width'=>[], 'height'=>[], 'loading'=>[]],
        ];
        $out['fe_html'] = isset($input['fe_html']) ? wp_kses($input['fe_html'], $allowed) : '';

        // Template-Datei
        $templates = array_keys($this->available_templates());
        $tpl = isset($input['fe_template']) ? sanitize_text_field($input['fe_template']) : '';
        $out['fe_template'] = in_array($tpl, $templates, true) ? $tpl : '';

        return $out;
    }

    private function available_templates(): array {
        $dir = plugin_dir_path(__FILE__) . 'assets/maintenance/templates/';
        if (!is_dir($dir)) return [];
        $files = glob($dir . '*.html');
        $out = [];
        foreach ($files as $f) {
            $base = basename($f);
            $out[$base] = $base;
        }
        return $out;
    }

public function render_settings_page(): void {
    if (!current_user_can('manage_options')) return;

    // Erfolgs-Notice sauber vor Hero ausgeben
  

    // Pfade zu (optionalen) Assets
    $hero_url  = plugins_url('assets/hero.png', __FILE__);
    $promo_url = plugins_url('assets/codekeks.png', __FILE__);

    ?>
    <div class="wrap ammsg-admin-wrap">

        <h1 class="screen-reader-text">
            <?php esc_html_e('Wartungshinweis', 'admin-maintance-msg'); ?>
        </h1>

        <?php
        // Erfolgsmeldung sauber an *dieser* Stelle ausgeben
        if ( isset($_GET['settings-updated']) && $_GET['settings-updated'] ) {
            add_settings_error(
                'ammsg_notices',
                'ammsg_saved',
                __('Die Einstellungen wurden gespeichert.', 'admin-maintance-msg'),
                'updated'
            );
        }
        echo '<div class="ammsg-notices">';
        settings_errors('ammsg_notices', false, true);
        echo '</div>';

        ?>
        <!-- Hero (mit sicherer Hintergrund-Grafik, wenn vorhanden) -->
        <div class="ammsg-hero" style="background-image:url('<?php echo esc_url($hero_url); ?>')">
            <div class="ammsg-hero-inner">
                <div>
                    <span class="ammsg-badge"><?php esc_html_e('Wartungsmodus', 'admin-maintance-msg'); ?></span>
                    <h1><?php esc_html_e('Wartungshinweis', 'admin-maintance-msg'); ?></h1>
                    <p><?php esc_html_e('Zeige eine Admin-Bannermeldung und optional eine elegante Frontend-Landingpage.', 'admin-maintance-msg'); ?></p>
                </div>
            </div>
        </div>

        <div class="ammsg-grid">
            <!-- Hauptspalte -->
            <div>
                <div class="ammsg-card">
                    <form method="post" action="options.php">
                        <?php
                            settings_fields(self::OPTION);
                            do_settings_sections(self::OPTION);
                            submit_button(esc_html__('Speichern', 'admin-maintance-msg'));
                        ?>
                    </form>
                    <p class="ammsg-muted" style="margin-top:10px;">
                        <?php esc_html_e('Hinweis erscheint für ausgewählte Rollen auf allen Admin-Seiten. Frontend-Wartung ersetzt die Seite im gewählten Modus (Seite / Template / Custom HTML).', 'admin-maintance-msg'); ?>
                    </p>
                </div>
            </div>

            <!-- Sidebar -->
            <aside class="ammsg-sidebar">
                <div class="ammsg-card promo">
                    <?php // Bild nur ausgeben, wenn es existiert ?>
                    <img src="<?php echo esc_url($promo_url); ?>"
                         onerror="this.style.display='none'"
                         alt="<?php echo esc_attr__('Info', 'admin-maintance-msg'); ?>" width="200" height="200" />
                    <h3 style="margin:.25rem 0 .5rem">codekeks.de — schlicht. schnell. klar. </h3>
                    <p><?php esc_html_e('Alles im Blick: Versionen, Updates & Infos für deine WP-Installation.', 'admin-maintance-msg'); ?></p>
                    <p><a class="button button-primary" href="https://codekeks.de/" target="_blank" rel="noopener">
                        <?php esc_html_e('Mehr erfahren', 'admin-maintance-msg'); ?> →
                    </a></p>
                </div>

                <div class="ammsg-card" style="margin-top:16px">
                    <h3><?php esc_html_e('Support & Links', 'admin-maintance-msg'); ?></h3>
                    <ul style="margin:.6em 0 0 1em; list-style:disc">
                        <li><a href="https://wordpress.org/plugins/admin-maintenance-message/" target="_blank" rel="noopener">WordPress.org</a></li>
                        <li><a href="https://github.com/antman313" target="_blank" rel="noopener">GitHub</a></li>
                        <li><a href="mailto:info@codekeks.de">info@codekeks.de</a></li>
                    </ul>
                </div>
            </aside>
        </div>

        <p class="ammsg-muted" style="margin-top:16px">
            <?php
            printf(
                /* translators: 1: actually date for copyright as date format, 2: developers URL address */
                __('&copy; %1$s by <a href="%2$s" target="_blank" rel="noopener">codekeks.de</a> · Andreas Grzybowski', 'admin-maintance-msg'),
                esc_html( gmdate('Y') ),
                esc_url('https://codekeks.de')
            );
            ?>
        </p>
    </div>
    <?php
}

    /* ==================== Admin-Notice (mit Rollenfilter) ==================== */
    public function maybe_show_notice(): void {
        $o = $this->get_opts();

        // Rollenfilter
        if (!empty($o['roles'])) {
            $user  = wp_get_current_user();
            $allow = false;
            foreach ((array)$o['roles'] as $r) {
                if (in_array($r, (array)$user->roles, true)) { $allow = true; break; }
            }
            if (!$allow) return;
        }

        $enabled = !empty($o['enabled']);
        $from = $this->parse_local_dt($o['from']);
        $to   = $this->parse_local_dt($o['to']);

        $within = false;
        if ($from && $to) {
            $now = new DateTime('now', $from->getTimezone());
            $within = ($now >= $from && $now <= $to);
        }
        if (!$enabled && !$within) return;

        $window_text = $this->format_window_text($from, $to);

        $repl = [];
        if ($from) {
            $repl['{WEEKDAY}']       = $this->weekday_de_long($from);
            $repl['{DATE}']          = $from->format('d.m.Y');
            $repl['{FROM}']          = $from->format('H:i');
            $repl['{DATETIME_FROM}'] = $from->format('d.m.Y H:i');
            $repl['{TZ}']            = $from->format('T');
        }
        if ($to) {
            $repl['{WEEKDAY_TO}']   = $this->weekday_de_long($to);
            $repl['{DATE_TO}']      = $to->format('d.m.Y');
            $repl['{TO}']           = $to->format('H:i');
            $repl['{DATETIME_TO}']  = $to->format('d.m.Y H:i');
        }

        $raw = isset($o['message']) ? trim($o['message']) : '';

        if ($raw === '' && $from && $to) {
            $message = sprintf(
			    /* translators: 1: date (dd.mm.yyyy), 2: start time (HH:MM), 3: end time (HH:MM) */
                __('Wir führen am %1$s zwischen %2$s und %3$s planmäßige Wartungsarbeiten durch. Innerhalb dieses Zeitraums steht die Website nicht zur Verfügung.', 'admin-maintance-msg'),
                $from->format('d.m.Y'),
                $from->format('H:i'),
                $to->format('H:i')
            );
            if ($window_text) {
                $message .= ' – ' . $window_text;
            }
        } else {
            $message = $raw !== '' ? $raw : __('Geplante Wartungsarbeiten.', 'admin-maintance-msg');
            if ($repl) {
                $message = strtr($message, $repl);
            }
            if ($window_text && !preg_match('/\{(WEEKDAY|DATE|FROM|TO|TZ|DATETIME_FROM|DATETIME_TO|WEEKDAY_TO|DATE_TO)\}/', $raw)) {
                $message .= ' – ' . $window_text;
            }
        }
        ?>
        <div class="ammsg-notice" role="alert" aria-live="polite">
            <span class="ammsg-icon" aria-hidden="true">!!!</span>
            <span class="ammsg-text"><?php echo esc_html($message); ?></span>
        </div>
        <?php
    }

    /* ==================== Frontend-Maintenance ==================== */
    public function maybe_frontend_maintenance(): void {
        if (is_admin()) return;

        $o = $this->get_opts();

        // Aktiv, wenn manuell enabled ODER wenn jetzt im Zeitfenster
        $active = !empty($o['fe_enabled']);

        $from = $this->parse_local_dt($o['from'] ?? '');
        $to   = $this->parse_local_dt($o['to'] ?? '');
        if (!$active && $from && $to) {
            $now = new DateTime('now', $from->getTimezone());
            $active = ($now >= $from && $now <= $to);
        }
        if (!$active) return;

        // Bypässe
        if (defined('REST_REQUEST') && REST_REQUEST) return;
        if (wp_doing_ajax()) return;
        if (wp_doing_cron()) return;
        if (!empty($o['fe_bypass_logged']) && is_user_logged_in()) return;

        // SEO / Header
        status_header(503);
        header('Retry-After: 3600');
        add_action('wp_head', function(){
            echo '<meta name="robots" content="noindex, nofollow" />' . "\n";
        });

        $site_name   = get_bloginfo('name');
        $window_text = $this->format_window_text($from, $to);

        // Admin-Text als Basis nutzen (klingt konsistent)
        $message = '';
        $raw = isset($o['message']) ? trim($o['message']) : '';
        if ($raw === '' && $from && $to) {
            $message = sprintf(
				/* translators: 1: date (dd.mm.yyyy), 2: start time (HH:MM), 3: end time (HH:MM) */
                __('Wir führen am %1$s zwischen %2$s und %3$s planmäßige Wartungsarbeiten durch. Innerhalb dieses Zeitraums steht die Website nicht zur Verfügung.', 'admin-maintance-msg'),
                $from->format('d.m.Y'),
                $from->format('H:i'),
                $to->format('H:i')
            );
            if ($window_text) {
                $message .= ' – ' . $window_text;
            }
        } else {
            $message = $raw !== '' ? $raw : __('Wir verbessern gerade die Website. In Kürze sind wir wieder für Sie da.', 'admin-maintance-msg');
            if ($from) {
                $message = strtr($message, [
                    '{WEEKDAY}' => $this->weekday_de_long($from),
                    '{DATE}'    => $from->format('d.m.Y'),
                    '{FROM}'    => $from->format('H:i'),
                    '{TZ}'      => $from->format('T'),
                    '{DATETIME_FROM}' => $from->format('d.m.Y H:i'),
                ]);
            }
            if ($to) {
                $message = strtr($message, [
                    '{WEEKDAY_TO}' => $this->weekday_de_long($to),
                    '{DATE_TO}'    => $to->format('d.m.Y'),
                    '{TO}'         => $to->format('H:i'),
                    '{DATETIME_TO}'=> $to->format('d.m.Y H:i'),
                ]);
            }
            if ($window_text && !preg_match('/\{(WEEKDAY|DATE|FROM|TO|TZ|DATETIME_FROM|DATETIME_TO|WEEKDAY_TO|DATE_TO)\}/', $raw)) {
                $message .= ' – ' . $window_text;
            }
        }

        // Kontext zusammenbauen
        // Kontext zusammenbauen (UNESCAPED Rohwerte)
        $ctx = [
            /* translators: maintance at saved date format, see http://php.net/date */
            'title'         => sprintf(__('Wartungsmodus – %s', 'admin-maintance-msg'), $site_name),
            'message'       => wp_strip_all_tags($message, true),
            'site_name'     => $site_name,
            'window_text'   => $window_text,
            'from'          => $from ? $from->format('H:i') : '',
            'to'            => $to   ? $to->format('H:i')   : '',
            'date'          => $from ? $from->format('d.m.Y') : '',
            'date_to'       => $to   ? $to->format('d.m.Y')   : '',
            'tz'            => $from ? $from->format('T')     : '',
            'datetime_from' => $from ? $from->format('d.m.Y H:i') : '',
            'datetime_to'   => $to   ? $to->format('d.m.Y H:i')   : '',
            'asset_url'     => plugin_dir_url(__FILE__) . 'assets/maintenance/',
        ];

        // Werte für Text-Kontext escapen und als Platzhalter-Replacements vorbereiten
        $repl = $this->build_replacements($ctx);

        $mode = $o['fe_mode'] ?: 'page';

        // 1) Template-Datei aus assets
        if ($mode === 'template' && !empty($o['fe_template'])) {
            $this->render_template_file($o['fe_template'], $repl);
            exit;
        }

        // 2) WordPress-Seite
        if ($mode === 'page' && (int)$o['fe_page_id'] > 0) {
            $post = get_post((int)$o['fe_page_id']);
            if ($post && $post->post_status === 'publish') {
                $title   = apply_filters('the_title', $post->post_title, $post->ID);
                $content = apply_filters('the_content', $post->post_content);
                echo '<!DOCTYPE html><html lang="'.esc_attr(get_locale()).'"><head><meta charset="'.esc_attr(get_bloginfo('charset')).'"><meta name="viewport" content="width=device-width, initial-scale=1"><title>'.esc_html($title ?: $site_name).'</title></head><body>';
                echo $content; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
                echo '</body></html>';
                exit;
            }
        }

        // 3) Custom HTML (bereits via wp_kses gefiltert)
        if ($mode === 'custom' && trim((string)$o['fe_html']) !== '') {
            echo '<!DOCTYPE html><html lang="'.esc_attr(get_locale()).'"><head><meta charset="'.esc_attr(get_bloginfo('charset')).'"><meta name="viewport" content="width=device-width, initial-scale=1"><title>'.esc_html($ctx['title']).'</title></head><body>';
            echo $o['fe_html']; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
            echo '</body></html>';
            exit;
        }

        // 4) Fallback: Standard-Template
        $this->render_template_file('default.html', $repl);
        exit;
    }

    private function build_replacements(array $ctx): array {
        return [
            '{{TITLE}}'         => isset($ctx['title'])         ? esc_html($ctx['title'])         : '',
            '{{MESSAGE}}'       => isset($ctx['message'])       ? esc_html($ctx['message'])       : '',
            '{{SITE_NAME}}'     => isset($ctx['site_name'])     ? esc_html($ctx['site_name'])     : esc_html(get_bloginfo('name')),
            '{{WINDOW}}'        => isset($ctx['window_text'])   ? esc_html($ctx['window_text'])   : '',
            '{{FROM}}'          => isset($ctx['from'])          ? esc_html($ctx['from'])          : '',
            '{{TO}}'            => isset($ctx['to'])            ? esc_html($ctx['to'])            : '',
            '{{DATE}}'          => isset($ctx['date'])          ? esc_html($ctx['date'])          : '',
            '{{DATE_TO}}'       => isset($ctx['date_to'])       ? esc_html($ctx['date_to'])       : '',
            '{{TZ}}'            => isset($ctx['tz'])            ? esc_html($ctx['tz'])            : '',
            '{{DATETIME_FROM}}' => isset($ctx['datetime_from']) ? esc_html($ctx['datetime_from']) : '',
            '{{DATETIME_TO}}'   => isset($ctx['datetime_to'])   ? esc_html($ctx['datetime_to'])   : '',
            '{{ASSET_URL}}'     => isset($ctx['asset_url'])     ? esc_url($ctx['asset_url'])      : esc_url(plugin_dir_url(__FILE__) . 'assets/maintenance/'),
        ];
    }

    private function render_template_file(string $template_file, array $repl): void {
        $base_dir = plugin_dir_path(__FILE__) . 'assets/maintenance/templates/';
        $base_real = realpath($base_dir);
        $real = $base_dir . $template_file;

        // Sicherheit: nur im Templates-Verzeichnis erlauben
        if (!$base_real || !is_string($template_file) || strpos(realpath($real) ?: '', $base_real) !== 0 || !is_readable($real)) {
            // einfacher Fallback
            $title = $repl['{{TITLE}}'] ?? 'Maintenance';
            $msg   = $repl['{{MESSAGE}}'] ?? '';
            $win   = $repl['{{WINDOW}}'] ?? '';
            $charset = esc_attr(get_bloginfo('charset'));
            echo '<!DOCTYPE html><html><head><meta charset="'.esc_html($charset).'"><meta name="viewport" content="width=device-width, initial-scale=1"><title>'.esc_html($title).'</title></head><body>';
            echo '<h1>'.esc_html($title).'</h1>';
            if ($msg) echo '<p>'.esc_html($msg).'</p>';
            if ($win) echo '<p>'.esc_html($win).'</p>';
            echo '</body></html>';
            return;
        }

        $html = file_get_contents($real);
        if ($html === false) {
            $charset = esc_attr(get_bloginfo('charset'));
            echo '<!DOCTYPE html><html><head><meta charset="'.esc_html($charset).'"><title>Maintenance</title></head><body><h1>Maintenance</h1></body></html>';
            return;
        }

        $out = strtr($html, $repl);
        echo $out; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
    }
}

new AMMSG_Admin_Maintenance();