<?php
if (! defined('ABSPATH')) exit; // Exit if accessed directly
if (!class_exists('WP_List_Table')) {
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

class ADDWWOMU_Addresses_Table extends WP_List_Table
{

    public function __construct()
    {
        parent::__construct([
            'singular' => 'customer_address',
            'plural'   => 'addwwomu_customer_addresses',
            'ajax'     => false,
        ]);
    }

    public function get_columns()
    {
        return [
            'cb'             => '<input type="checkbox" />',
            'id'             => 'ID',
            'user_id'        => 'User ID',
            'first_name'     => 'First Name',
            'last_name'      => 'Last Name',
            'address_type'   => 'Address Type',
            'address_label'  => 'Address Label',
            'company'        => 'Company',
            'country'        => 'Country',
            'address_1'      => 'Address 1',
            'address_2'      => 'Address 2',
            'city'           => 'City',
            'state'          => 'State',
            'postcode'       => 'Postcode',
            'phone'          => 'Phone',
            'email'          => 'Email',
            'is_default'     => 'Default',
        ];
    }

    public function get_sortable_columns()
    {
        return [
            'id'         => ['id', true],
            'first_name' => ['first_name', false],
            'last_name'  => ['last_name', false],
        ];
    }

    public function get_bulk_actions()
    {
        return [
            'delete' => __('Delete', 'addweb-woo-multi-address'),
        ];
    }

    public function column_cb($item)
    {
        return sprintf('<input type="checkbox" name="bulk-delete[]" value="%d" />', absint($item['id']));
    }

    public function column_default($item, $column_name)
    {
        return isset($item[$column_name]) ? esc_html($item[$column_name]) : '-';
    }

    public function column_id($item)
    {
        // Generate nonce for delete action
        $delete_nonce = wp_create_nonce('delete_customer_address');
        $page = isset($_REQUEST['page']) ? sanitize_text_field(wp_unslash($_REQUEST['page'])) : ''; // phpcs:ignore

        // Prepare the title and actions
        $title = '<strong>' . esc_html($item['id']) . '</strong>';
        $actions = [
            'delete' => sprintf(
                '<a href="%s">%s</a>',
                esc_url(add_query_arg([
                    'page'             => $page,
                    'action'           => 'delete',
                    'customer_address' => absint($item['id']),
                    '_wpnonce'         => $delete_nonce,
                ])),
                __('Delete', 'addweb-woo-multi-address')
            ),
        ];

        // Return the title and actions
        return $title . $this->row_actions($actions);
    }


    public function prepare_items()
    {
        global $wpdb;
        $table_name = $wpdb->prefix . 'addwwomu_customer_addresses';
        $per_page = 10;
        $current_page = $this->get_pagenum();

        $search = '';
        $where = 'WHERE 1=1';
        $params = [];

        // Search Handling (Nonce check)
        if (
            !empty($_REQUEST['s']) &&
            !empty($_REQUEST['_wpnonce']) &&
            wp_verify_nonce(sanitize_text_field(wp_unslash($_REQUEST['_wpnonce'])), 'search_customer_address')
        ) {
            $search = sanitize_text_field(wp_unslash($_REQUEST['s']));
            $like = '%' . $wpdb->esc_like($search) . '%';
            $where .= " AND (first_name LIKE %s OR last_name LIKE %s OR email LIKE %s OR phone LIKE %s)";
            $params = array_fill(0, 4, $like);
        }

        $orderby_whitelist = ['id', 'first_name', 'last_name'];
        $orderby_raw = isset($_GET['orderby']) ? sanitize_text_field(wp_unslash($_GET['orderby'])) : '';
        $order_raw = isset($_GET['order']) ? sanitize_text_field(wp_unslash($_GET['order'])) : '';

        // Ensure the values are from the whitelist and are sanitized
        $orderby = in_array($orderby_raw, $orderby_whitelist, true) ? esc_sql($orderby_raw) : 'id';
        $order = strtoupper($order_raw) === 'DESC' ? 'DESC' : 'ASC';
        // Total items
        // phpcs:ignore
        $total_sql = "SELECT COUNT(*) FROM $table_name $where";
        $total_query = !empty($params)
            ? $wpdb->prepare($total_sql, ...$params)   // phpcs:ignore
            : $total_sql;
        $total_items = (int) $wpdb->get_var($total_query);  // phpcs:ignore

        $offset = ($current_page - 1) * $per_page;

        // Add LIMIT and OFFSET
        $params[] = $per_page;
        $params[] = $offset;

        // Data query
        $data_sql = "SELECT * FROM $table_name $where ORDER BY $orderby $order LIMIT %d OFFSET %d";
        $data_query = !empty($params)
            ? $wpdb->prepare($data_sql, ...$params)   // phpcs:ignore
            : $data_sql;
        $this->items = $wpdb->get_results($data_query, ARRAY_A);   // phpcs:ignore

        $columns = $this->get_columns();
        $hidden = [];
        $sortable = $this->get_sortable_columns();
        $this->_column_headers = [$columns, $hidden, $sortable];

        $this->set_pagination_args([
            'total_items' => $total_items,
            'per_page'    => $per_page,
            'total_pages' => ceil($total_items / $per_page),
        ]);
    }
}
// Check if action is delete
if (isset($_GET['action']) && $_GET['action'] === 'delete') {
    // Verify nonce
    if (
        isset($_GET['_wpnonce']) &&
        wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'delete_customer_address')
    ) {
        $id = isset($_GET['customer_address']) ? absint($_GET['customer_address']) : 0;

        if ($id > 0) {
            global $wpdb;
            $table_name = $wpdb->prefix . 'addwwomu_customer_addresses';

            // Perform deletion from the database
            $deleted = $wpdb->delete($table_name, ['id' => $id], ['%d']); // phpcs:ignore

            if ($deleted !== false) {
                // Clear cache for this record if applicable
                wp_cache_delete('customer_address_' . $id, 'addwwomu_customer_addresses');
            }

            // Redirect after successful deletion
            wp_safe_redirect(remove_query_arg(['action', 'customer_address', '_wpnonce']));
            exit;
        }
    } else {
        // If nonce is invalid, show a security warning
        wp_die(esc_html(__('Security check failed', 'addweb-woo-multi-address')));
    }
}
