<?php

/**
 * Database Utility Class
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    Activity_Monitor_Pro
 * @subpackage Activity_Monitor_Pro/includes/utils
 */

/**
 * Database Utility Class.
 *
 * Helper methods for database interactions.
 *
 * @package    Activity_Monitor_Pro
 * @subpackage Activity_Monitor_Pro/includes/utils
 * @author     Your Name <email@example.com>
 */
class Activity_Monitor_Pro_Database
{

    /**
     * Insert a log entry.
     *
     * @param array $data Log data.
     * @return int|false The inserted log ID or false on failure.
     */
    public static function insert_log($data)
    {
        global $wpdb;
        $table_name = $wpdb->prefix . 'activitymon_logs';

        // Explicitly map data to format to ensure order matches
        $insert_data = array();
        $format = array();

        // Map fields
        $fields = array(
            'user_id' => '%d',
            'user_name' => '%s',
            'user_email' => '%s',
            'action_type' => '%s',
            'object_type' => '%s',
            'object_id' => '%d',
            'object_name' => '%s',
            'object_subtype' => '%s',
            'description' => '%s',
            'severity' => '%s',
            'ip_address' => '%s',
            'user_agent' => '%s',
            'is_undoable' => '%d',
            'undo_status' => '%s',
            'created_at' => '%s',
            'session_id' => '%s',
            'metadata' => '%s',
        );

        foreach ($fields as $field => $type) {
            if (isset($data[$field])) {
                $insert_data[$field] = $data[$field];
                $format[] = $type;
            }
        }

        // Ensure created_at is set if missing
        if (!isset($insert_data['created_at'])) {
            $insert_data['created_at'] = current_time('mysql');
            $format[] = '%s';
        }

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table insert is necessary for activity logging
        $result = $wpdb->insert($table_name, $insert_data, $format);

        if ($result) {
            do_action('activity_monitor_log_inserted', $wpdb->insert_id, $insert_data);
            return $wpdb->insert_id;
        }

        return false;
    }

    /**
     * Get logs with filtering.
     *
     * @param array $args Filter arguments.
     * @return array List of logs.
     */
    public static function get_logs($args = array())
    {
        global $wpdb;
        $table_name = $wpdb->prefix . 'activitymon_logs';

        $where = array('1=1');
        $query_args = array();

        // Search
        if (!empty($args['search'])) {
            $search = '%' . $wpdb->esc_like($args['search']) . '%';
            $where[] = "(description LIKE %s OR object_name LIKE %s OR user_name LIKE %s)";
            $query_args[] = $search;
            $query_args[] = $search;
            $query_args[] = $search;
        }

        // User ID
        if (!empty($args['user_id'])) {
            $where[] = "user_id = %d";
            $query_args[] = intval($args['user_id']);
        }

        // Object Type
        if (!empty($args['object_type'])) {
            $where[] = "object_type = %s";
            $query_args[] = $args['object_type'];
        }

        // Severity
        if (!empty($args['severity'])) {
            $where[] = "severity = %s";
            $query_args[] = $args['severity'];
        }

        // Limit & Offset
        $limit = isset($args['limit']) ? intval($args['limit']) : 50;
        $offset = isset($args['offset']) ? intval($args['offset']) : 0;

        $sql = "SELECT * FROM $table_name WHERE " . implode(' AND ', $where) . " ORDER BY created_at DESC LIMIT %d OFFSET %d";
        $query_args[] = $limit;
        $query_args[] = $offset;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Dynamic query built from safe components, $table_name from $wpdb->prefix
        return $wpdb->get_results($wpdb->prepare($sql, $query_args), ARRAY_A);
    }

    /**
     * Delete logs older than X days.
     *
     * @param int $days Number of days to retain logs.
     * @return int|false Number of rows affected or false on error.
     */
    public static function delete_old_logs($days = 30)
    {
        global $wpdb;
        $table_name = $wpdb->prefix . 'activitymon_logs';

        $date_limit = gmdate('Y-m-d H:i:s', strtotime("-$days days"));

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name safely derived from $wpdb->prefix
        return $wpdb->query($wpdb->prepare("DELETE FROM $table_name WHERE created_at < %s", $date_limit));
    }

    /**
     * Insert undo data.
     *
     * @param int $log_id The log ID.
     * @param array $undo_data The undo data.
     * @return int|false The inserted ID or false on failure.
     */
    public static function insert_undo_data($log_id, $undo_data)
    {
        global $wpdb;
        $table_name = $wpdb->prefix . 'activitymon_undo_data';

        $data = array(
            'log_id' => $log_id,
            'undo_params' => json_encode($undo_data),
            'created_at' => current_time('mysql'),
        );

        $format = array(
            '%d', // log_id
            '%s', // undo_params
            '%s', // created_at
        );

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Custom table insert for undo data
        $result = $wpdb->insert($table_name, $data, $format);

        if ($result) {
            return $wpdb->insert_id;
        }

        return false;
    }
    /**
     * Update undo status.
     *
     * @param int $log_id The log ID.
     * @param string $status The new status ('pending', 'undone').
     * @return int|false Number of rows affected or false on error.
     */
    public static function update_undo_status($log_id, $status)
    {
        global $wpdb;
        $table_name = $wpdb->prefix . 'activitymon_logs';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom table update, no caching needed
        return $wpdb->update(
            $table_name,
            array('undo_status' => $status),
            array('id' => $log_id),
            array('%s'),
            array('%d')
        );
    }
    /**
     * Get total logs count.
     *
     * @return int Total number of logs.
     */
    public static function get_total_logs_count($args = array())
    {
        global $wpdb;
        $table_name = $wpdb->prefix . 'activitymon_logs';

        $where = array('1=1');
        $query_args = array();

        // Search
        if (!empty($args['search'])) {
            $search = '%' . $wpdb->esc_like($args['search']) . '%';
            $where[] = "(description LIKE %s OR object_name LIKE %s OR user_name LIKE %s)";
            $query_args[] = $search;
            $query_args[] = $search;
            $query_args[] = $search;
        }

        // User ID
        if (!empty($args['user_id'])) {
            $where[] = "user_id = %d";
            $query_args[] = intval($args['user_id']);
        }

        // Object Type
        if (!empty($args['object_type'])) {
            $where[] = "object_type = %s";
            $query_args[] = $args['object_type'];
        }

        // Severity
        if (!empty($args['severity'])) {
            $where[] = "severity = %s";
            $query_args[] = $args['severity'];
        }

        $sql = "SELECT COUNT(*) FROM $table_name WHERE " . implode(' AND ', $where);

        if (!empty($query_args)) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Dynamic query built from safe components, $table_name from $wpdb->prefix
            return (int) $wpdb->get_var($wpdb->prepare($sql, $query_args));
        } else {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.NotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Simple count query, $table_name from $wpdb->prefix
            return (int) $wpdb->get_var($sql);
        }
    }
}
