<?php

/**
 * Post Undo Handler
 *
 * @package    Activity_Monitor_Pro
 * @subpackage Activity_Monitor_Pro/includes/undo/handlers
 */

class Activity_Monitor_Post_Undo_Handler
{

    /**
     * Perform undo operation for posts.
     *
     * @param object $log Log object.
     * @param array $undo_data Data needed to restore the state.
     * @return bool True on success, false on failure.
     */
    public function undo($log, $undo_data)
    {
        $post_id = $log->object_id;

        if (!$post_id) {
            return false;
        }

        // Check if post exists (it might have been permanently deleted)
        // If action was deletion, we might need to recreate it, but for MVP we assume it's in trash or we are reverting an update.
        // If action was 'deleted' (trashed), we restore from trash.

        if ($log->action_type === 'post_deleted') {
            return wp_untrash_post($post_id);
        }

        // For updates and status changes, we restore the previous state
        $update_actions = array('post_updated', 'post_published', 'post_trashed', 'post_restored');

        if (in_array($log->action_type, $update_actions)) {
            $post_arr = array(
                'ID' => $post_id,
            );

            // Check if we have nested structure (new/prev) or flat structure (legacy)
            $data_source = isset($undo_data['prev']) ? $undo_data['prev'] : $undo_data;

            if (isset($data_source['post_title'])) {
                $post_arr['post_title'] = $data_source['post_title'];
            }
            if (isset($data_source['post_content'])) {
                $post_arr['post_content'] = $data_source['post_content'];
            }
            if (isset($data_source['post_status'])) {
                $post_arr['post_status'] = $data_source['post_status'];
            }

            // Update the post
            $result = wp_update_post($post_arr, true);

            if (is_wp_error($result)) {
                return false;
            }
            return true;
        }

        return false;
    }

    /**
     * Perform redo operation for posts.
     *
     * @param object $log Log object.
     * @param array $undo_data Data needed to restore the state.
     * @return bool True on success, false on failure.
     */
    public function redo($log, $undo_data)
    {
        $post_id = $log->object_id;

        if (!$post_id) {
            return false;
        }

        // For updates and status changes, we restore the new state
        $update_actions = array('post_updated', 'post_published', 'post_trashed', 'post_restored');

        if (in_array($log->action_type, $update_actions)) {
            // We need the 'new' state for redo
            if (!isset($undo_data['new'])) {
                return false;
            }

            $data_source = $undo_data['new'];

            $post_arr = array(
                'ID' => $post_id,
            );

            if (isset($data_source['post_title'])) {
                $post_arr['post_title'] = $data_source['post_title'];
            }
            if (isset($data_source['post_content'])) {
                $post_arr['post_content'] = $data_source['post_content'];
            }
            if (isset($data_source['post_status'])) {
                $post_arr['post_status'] = $data_source['post_status'];
            }

            // Update the post
            $result = wp_update_post($post_arr, true);

            if (is_wp_error($result)) {
                return false;
            }
            return true;
        }

        // Handle trash/untrash logic for redo if needed
        if ($log->action_type === 'post_deleted') {
            // If it was deleted (trashed), redo means trashing it again
            return wp_trash_post($post_id);
        }

        return false;
    }
}
