<?php
// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Undo Manager
 *
 * @package    Activity_Monitor_Pro
 * @subpackage Activity_Monitor_Pro/includes/undo
 */

class Activity_Monitor_Undo_Manager
{

    /**
     * The loader that's responsible for maintaining and registering all hooks that power
     * the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      Activity_Monitor_Pro_Loader    $loader    Maintains and registers all hooks for the plugin.
     */
    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     */
    public function __construct()
    {
        // Hooks are registered in Activity_Monitor_Pro class
    }

    /**
     * Handle AJAX undo request.
     */
    public function ajax_handle_undo()
    {
        check_ajax_referer('activity_monitor_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }

        $log_id = isset($_POST['log_id']) ? intval($_POST['log_id']) : 0;

        if (!$log_id) {
            wp_send_json_error('Invalid log ID');
        }

        // Get log details
        global $wpdb;
        $log_table = $wpdb->prefix . 'activitymon_logs';
        $undo_table = $wpdb->prefix . 'activitymon_undo_data';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name safely derived from $wpdb->prefix
        $log = $wpdb->get_row($wpdb->prepare("SELECT * FROM $log_table WHERE id = %d", $log_id));

        if (!$log) {
            wp_send_json_error('Log entry not found');
        }

        // Check if this activity is undoable
        if (!$log->is_undoable) {
            wp_send_json_error('This activity cannot be undone');
        }

        // Get undo data
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name safely derived from $wpdb->prefix
        $undo_data_row = $wpdb->get_row($wpdb->prepare("SELECT undo_params FROM $undo_table WHERE log_id = %d", $log_id));

        if (!$undo_data_row || empty($undo_data_row->undo_params)) {
            wp_send_json_error('No undo data available for this activity');
        }

        $undo_data = json_decode($undo_data_row->undo_params, true);

        // Free version: Only allow undo for posts and pages
        $allowed_types = array('post', 'page');
        if (!in_array($log->object_type, $allowed_types)) {
            wp_send_json_error('Undo for ' . $log->object_type . ' is not available in this version');
        }

        // Dispatch to handler
        $result = false;
        switch ($log->object_type) {
            case 'post':
            case 'page':
                require_once plugin_dir_path(__FILE__) . 'handlers/class-post-undo-handler.php';
                $handler = new Activity_Monitor_Post_Undo_Handler();
                $result = $handler->undo($log, $undo_data);
                break;
            default:
                wp_send_json_error('No undo handler available for object type: ' . $log->object_type);
        }

        if ($result) {
            require_once plugin_dir_path(dirname(__FILE__)) . 'utils/class-database.php';
            Activity_Monitor_Pro_Database::update_undo_status($log_id, 'undone');
            wp_send_json_success('Activity undone successfully');
        } else {
            wp_send_json_error('Failed to undo activity. The undo handler returned false.');
        }
    }

    /**
     * Handle AJAX redo request.
     */
    public function ajax_handle_redo()
    {
        check_ajax_referer('activity_monitor_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }

        $log_id = isset($_POST['log_id']) ? intval($_POST['log_id']) : 0;

        if (!$log_id) {
            wp_send_json_error('Invalid log ID');
        }

        global $wpdb;
        $log_table = $wpdb->prefix . 'activitymon_logs';
        $undo_table = $wpdb->prefix . 'activitymon_undo_data';

        // Get log entry
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name safely derived from $wpdb->prefix
        $log = $wpdb->get_row($wpdb->prepare("SELECT * FROM $log_table WHERE id = %d", $log_id));

        if (!$log) {
            wp_send_json_error('Log entry not found');
        }

        if (!$log->is_undoable) {
            wp_send_json_error('This activity cannot be redone');
        }

        // Get undo data
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name safely derived from $wpdb->prefix
        $undo_data_row = $wpdb->get_row($wpdb->prepare("SELECT undo_params FROM $undo_table WHERE log_id = %d", $log_id));

        if (!$undo_data_row || empty($undo_data_row->undo_params)) {
            wp_send_json_error('No redo data available for this activity');
        }

        $undo_data = json_decode($undo_data_row->undo_params, true);

        // Free version: Only allow redo for posts and pages
        $allowed_types = array('post', 'page');
        if (!in_array($log->object_type, $allowed_types)) {
            wp_send_json_error('Redo for ' . $log->object_type . ' is not available in this version');
        }

        // Dispatch to handler
        $result = false;
        switch ($log->object_type) {
            case 'post':
            case 'page':
                require_once plugin_dir_path(__FILE__) . 'handlers/class-post-undo-handler.php';
                $handler = new Activity_Monitor_Post_Undo_Handler();
                $result = $handler->redo($log, $undo_data);
                break;
            default:
                wp_send_json_error('No handler for this object type');
                break;
        }

        if ($result) {
            require_once plugin_dir_path(dirname(__FILE__)) . 'utils/class-database.php';
            Activity_Monitor_Pro_Database::update_undo_status($log_id, 'pending');
            wp_send_json_success('Activity redone successfully');
        } else {
            wp_send_json_error('Failed to redo activity');
        }
    }
}
