<?php
// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Report Scheduler Class
 *
 * @package    Activity_Monitor_Pro
 * @subpackage Activity_Monitor_Pro/includes/reports
 */

class Activity_Monitor_Report_Scheduler
{

    /**
     * The loader that's responsible for maintaining and registering all hooks that power
     * the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      Activity_Monitor_Pro_Loader    $loader    Maintains and registers all hooks for the plugin.
     */
    protected $loader;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param      Activity_Monitor_Pro_Loader    $loader    The loader that's responsible for maintaining and registering all hooks that power the plugin.
     */
    public function __construct($loader)
    {
        $this->loader = $loader;
        $this->register_hooks();
    }

    /**
     * Register hooks.
     */
    private function register_hooks()
    {
        $this->loader->add_action('activity_monitor_send_reports', $this, 'send_reports');

        // Schedule event if not already scheduled
        if (!wp_next_scheduled('activity_monitor_send_reports')) {
            wp_schedule_event(time(), 'daily', 'activity_monitor_send_reports');
        }
    }

    /**
     * Send reports.
     * Free version: Weekly reports only.
     */
    public function send_reports()
    {
        // Check if reports are enabled
        $reports_enabled = get_option('activity_monitor_enable_reports', false);

        if (!$reports_enabled) {
            return;
        }

        $admin_email = get_option('admin_email');

        // Free version: Weekly reports only (sent on Mondays)
        if (gmdate('w') != 1) { // Not Monday
            return;
        }

        $this->generate_and_send_report($admin_email, 'weekly');
    }

    /**
     * Generate and send report.
     */
    private function generate_and_send_report($email, $type)
    {
        require_once plugin_dir_path(dirname(dirname(__FILE__))) . 'utils/class-database.php';

        // Get logs for the last 24 hours
        $args = array(
            'limit' => 50, // Top 50 actions
            // In a real implementation, we'd filter by date range here
        );

        $logs = Activity_Monitor_Pro_Database::get_logs($args);
        $count = Activity_Monitor_Pro_Database::get_total_logs_count();

        $subject = sprintf('Activity Monitor Report - %s', gmdate('Y-m-d'));

        $message = "Activity Monitor Pro - Daily Report\n\n";
        $message .= "Total Activities: $count\n\n";
        $message .= "Recent Critical Actions:\n";

        foreach ($logs as $log) {
            if ($log['severity'] === 'critical') {
                $message .= sprintf("- [%s] %s: %s\n", $log['created_at'], $log['user_name'], $log['description']);
            }
        }

        $message .= "\nLogin to your dashboard to view full details.";

        wp_mail($email, $subject, $message);
    }
}
