<?php
// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

/**
 * User Activity Logger
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    Activity_Monitor_Pro
 * @subpackage Activity_Monitor_Pro/includes/loggers
 */

/**
 * User Activity Logger.
 *
 * Handles logging of user-related activities.
 *
 * @package    Activity_Monitor_Pro
 * @subpackage Activity_Monitor_Pro/includes/loggers
 * @author     Your Name <email@example.com>
 */
class Activity_User_Logger implements Activity_Logger_Interface
{

    /**
     * Register hooks for the logger.
     */
    public function register_hooks()
    {
        add_action('wp_login', array($this, 'log_user_login'), 10, 2);
        add_action('wp_logout', array($this, 'log_user_logout'));
        add_action('wp_login_failed', array($this, 'log_login_failed'));
        add_action('user_register', array($this, 'log_user_register'));
        add_action('delete_user', array($this, 'log_user_delete'));
    }

    /**
     * Log user login.
     *
     * @param string $user_login User login.
     * @param object $user       User object.
     */
    public function log_user_login($user_login, $user)
    {
        $this->log_activity(array(
            'action_type' => 'user_login',
            'object_type' => 'user',
            'object_id' => $user->ID,
            'object_name' => $user_login,
            'description' => sprintf('User "%s" logged in.', $user_login),
            'severity' => 'info',
        ));
    }

    /**
     * Log user logout.
     *
     * @param int $user_id User ID.
     */
    public function log_user_logout($user_id)
    {
        $user = get_userdata($user_id);
        if (!$user) {
            return;
        }

        $this->log_activity(array(
            'action_type' => 'user_logout',
            'object_type' => 'user',
            'object_id' => $user_id,
            'object_name' => $user->user_login,
            'description' => sprintf('User "%s" logged out.', $user->user_login),
            'severity' => 'info',
        ));
    }

    /**
     * Log failed login.
     *
     * @param string $username Username.
     */
    public function log_login_failed($username)
    {
        $this->log_activity(array(
            'action_type' => 'login_failed',
            'object_type' => 'user',
            'object_id' => 0,
            'object_name' => $username,
            'description' => sprintf('Failed login attempt for username "%s".', $username),
            'severity' => 'warning',
        ));
    }

    /**
     * Log user registration.
     *
     * @param int $user_id User ID.
     */
    public function log_user_register($user_id)
    {
        $user = get_userdata($user_id);
        $this->log_activity(array(
            'action_type' => 'user_created',
            'object_type' => 'user',
            'object_id' => $user_id,
            'object_name' => $user->user_login,
            'description' => sprintf('New user "%s" registered.', $user->user_login),
            'severity' => 'info',
        ));
    }

    /**
     * Log user deletion.
     *
     * @param int $user_id User ID.
     */
    public function log_user_delete($user_id)
    {
        $user = get_userdata($user_id);
        if (!$user) {
            return;
        }

        $this->log_activity(array(
            'action_type' => 'user_deleted',
            'object_type' => 'user',
            'object_id' => $user_id,
            'object_name' => $user->user_login,
            'description' => sprintf('User "%s" deleted.', $user->user_login),
            'severity' => 'warning',
        ));
    }

    /**
     * Log the activity.
     *
     * @param array $args Activity arguments.
     */
    public function log_activity($args)
    {
        Activity_Logger_Manager::log($args);
    }

    /**
     * Check if the activity can be undone.
     *
     * @return bool
     */
    public function can_undo()
    {
        return false; // User actions mostly not undoable easily yet
    }

    /**
     * Prepare data for undo.
     *
     * @param array $args Activity arguments.
     * @return array
     */
    public function prepare_undo_data($args)
    {
        return array();
    }
}
