<?php
// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Theme Activity Logger
 *
 * @package    Activity_Monitor_Pro
 * @subpackage Activity_Monitor_Pro/includes/loggers
 */
class Activity_Theme_Logger implements Activity_Logger_Interface
{

    /**
     * Register hooks for the logger.
     */
    public function register_hooks()
    {
        // Theme Switch
        add_action('switch_theme', array($this, 'log_theme_switch'), 10, 3);

        // Theme Deleted
        add_action('delete_theme', array($this, 'log_theme_delete'), 10, 1);

        // Theme Installed/Updated (handled via upgrader hooks usually, but we can try specific actions if available)
        // For now, upgrader_process_complete is a good place for updates/installs
        add_action('upgrader_process_complete', array($this, 'log_theme_install_update'), 10, 2);

        // Theme File Editor
        add_action('wp_edit_theme_plugin_file', array($this, 'log_theme_file_edit'), 10, 2);
    }

    /**
     * Log Theme Switch
     *
     * @param string $new_name
     * @param WP_Theme $new_theme
     * @param WP_Theme $old_theme
     */
    public function log_theme_switch($new_name, $new_theme, $old_theme)
    {
        Activity_Logger_Manager::log(array(
            'action_type' => 'activate',
            'object_type' => 'theme',
            'object_id' => 0,
            'object_name' => $new_name,
            /* translators: 1: Old theme name, 2: New theme name */
            'description' => sprintf(__('Theme switched from "%1$s" to "%2$s"', 'activity-monitor-pro'), $old_theme->get('Name'), $new_theme->get('Name')),
            'is_undoable' => 1,
            'undo_data' => array(
                'action' => 'switch_theme',
                'theme_stylesheet' => $old_theme->get_stylesheet()
            )
        ));
    }

    /**
     * Log Theme Delete
     *
     * @param string $stylesheet
     */
    public function log_theme_delete($stylesheet)
    {
        Activity_Logger_Manager::log(array(
            'action_type' => 'delete',
            'object_type' => 'theme',
            'object_id' => 0,
            'object_name' => $stylesheet,
            /* translators: %s: Theme name */
            'description' => sprintf(__('Theme "%s" deleted', 'activity-monitor-pro'), $stylesheet),
        ));
    }

    /**
     * Log Theme Install/Update
     * 
     * @param WP_Upgrader $upgrader
     * @param array $hook_extra
     */
    public function log_theme_install_update($upgrader, $hook_extra)
    {
        if (!isset($hook_extra['type']) || $hook_extra['type'] !== 'theme') {
            return;
        }

        $action = $hook_extra['action']; // 'install' or 'update'

        if ($action === 'install') {
            Activity_Logger_Manager::log(array(
                'action_type' => 'install',
                'object_type' => 'theme',
                'object_id' => 0,
                'object_name' => 'Theme', // Hard to get exact name here without parsing result
                'description' => __('New theme installed', 'activity-monitor-pro'),
            ));
        } elseif ($action === 'update') {
            $themes = isset($hook_extra['themes']) ? $hook_extra['themes'] : array();
            foreach ($themes as $theme) {
                Activity_Logger_Manager::log(array(
                    'action_type' => 'update',
                    'object_type' => 'theme',
                    'object_id' => 0,
                    'object_name' => $theme,
                    /* translators: %s: Theme name */
                    'description' => sprintf(__('Theme "%s" updated', 'activity-monitor-pro'), $theme),
                ));
            }
        }
    }

    /**
     * Log Theme File Edit
     *
     * @param string $file
     * @param string $theme
     */
    public function log_theme_file_edit($file, $theme)
    {
        Activity_Logger_Manager::log(array(
            'action_type' => 'update',
            'object_type' => 'theme_file',
            'object_id' => 0,
            'object_name' => $file,
            /* translators: %s: Theme file path */
            'description' => sprintf(__('Theme file "%s" edited', 'activity-monitor-pro'), $file),
        ));
    }

    /**
     * Log the activity.
     *
     * @param array $args Activity arguments.
     */
    public function log_activity($args)
    {
        Activity_Logger_Manager::log($args);
    }

    /**
     * Check if the activity can be undone.
     *
     * @return bool
     */
    public function can_undo()
    {
        return true;
    }

    /**
     * Prepare data for undo.
     *
     * @param array $args Activity arguments.
     * @return array
     */
    public function prepare_undo_data($args)
    {
        return isset($args['undo_data']) ? $args['undo_data'] : array();
    }
}
