<?php
// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Taxonomy Activity Logger
 *
 * @package    Activity_Monitor_Pro
 * @subpackage Activity_Monitor_Pro/includes/loggers
 */
class Activity_Taxonomy_Logger implements Activity_Logger_Interface
{

    /**
     * Register hooks for the logger.
     */
    public function register_hooks()
    {
        add_action('created_term', array($this, 'log_term_creation'), 10, 3);
        add_action('edited_term', array($this, 'log_term_edit'), 10, 3);
        add_action('delete_term', array($this, 'log_term_deletion'), 10, 4);
    }

    /**
     * Log Term Creation
     *
     * @param int $term_id
     * @param int $tt_id
     * @param string $taxonomy
     */
    public function log_term_creation($term_id, $tt_id, $taxonomy)
    {
        $term = get_term($term_id, $taxonomy);
        if (is_wp_error($term)) {
            return;
        }

        Activity_Logger_Manager::log(array(
            'action_type' => 'create',
            'object_type' => 'taxonomy',
            'object_id' => $term_id,
            'object_name' => $term->name,
            /* translators: 1: Taxonomy type (e.g., category, tag), 2: Term name */
            'description' => sprintf(__('New %1$s "%2$s" created', 'activity-monitor-pro'), $taxonomy, $term->name),
            'is_undoable' => 1,
            'undo_data' => array(
                'action' => 'delete_term',
                'term_id' => $term_id,
                'taxonomy' => $taxonomy
            )
        ));
    }

    /**
     * Log Term Edit
     *
     * @param int $term_id
     * @param int $tt_id
     * @param string $taxonomy
     */
    public function log_term_edit($term_id, $tt_id, $taxonomy)
    {
        $term = get_term($term_id, $taxonomy);
        if (is_wp_error($term)) {
            return;
        }

        // Ideally we would check what changed, but for now we log the edit
        Activity_Logger_Manager::log(array(
            'action_type' => 'update',
            'object_type' => 'taxonomy',
            'object_id' => $term_id,
            'object_name' => $term->name,
            /* translators: 1: Taxonomy type (e.g., Category, Tag), 2: Term name */
            'description' => sprintf(__('%1$s "%2$s" updated', 'activity-monitor-pro'), ucfirst($taxonomy), $term->name),
            // Undo for edit is harder without storing previous state. 
            // We can store the full object if we want to support full restore.
            'is_undoable' => 0,
        ));
    }

    /**
     * Log Term Deletion
     *
     * @param int $term_id
     * @param int $tt_id
     * @param string $taxonomy
     * @param object $deleted_term
     */
    public function log_term_deletion($term_id, $tt_id, $taxonomy, $deleted_term)
    {
        Activity_Logger_Manager::log(array(
            'action_type' => 'delete',
            'object_type' => 'taxonomy',
            'object_id' => $term_id,
            'object_name' => $deleted_term->name,
            /* translators: 1: Taxonomy type (e.g., Category, Tag), 2: Term name */
            'description' => sprintf(__('%1$s "%2$s" deleted', 'activity-monitor-pro'), ucfirst($taxonomy), $deleted_term->name),
            'is_undoable' => 1,
            'undo_data' => array(
                'action' => 'restore_term',
                'term_name' => $deleted_term->name,
                'taxonomy' => $taxonomy,
                'slug' => $deleted_term->slug,
                'description' => $deleted_term->description,
                'parent' => $deleted_term->parent
            )
        ));
    }

    /**
     * Log the activity.
     *
     * @param array $args Activity arguments.
     */
    public function log_activity($args)
    {
        Activity_Logger_Manager::log($args);
    }

    /**
     * Check if the activity can be undone.
     *
     * @return bool
     */
    public function can_undo()
    {
        return true;
    }

    /**
     * Prepare data for undo.
     *
     * @param array $args Activity arguments.
     * @return array
     */
    public function prepare_undo_data($args)
    {
        return isset($args['undo_data']) ? $args['undo_data'] : array();
    }
}
