<?php
// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

/**
 * System Activity Logger
 *
 * @package    Activity_Monitor_Pro
 * @subpackage Activity_Monitor_Pro/includes/loggers
 */
class Activity_System_Logger implements Activity_Logger_Interface
{

    /**
     * Register hooks for the logger.
     */
    public function register_hooks()
    {
        // Core Updates
        add_action('_core_updated_successfully', array($this, 'log_core_update'));

        // Option Updates
        add_action('updated_option', array($this, 'log_option_update'), 10, 3);
        add_action('added_option', array($this, 'log_option_add'), 10, 2);
        add_action('deleted_option', array($this, 'log_option_delete'), 10, 1);

        // Settings Errors (often indicates settings saved)
        // add_action('settings_errors', array($this, 'log_settings_change')); 
    }

    /**
     * Log WordPress Core Update
     *
     * @param string $new_version
     */
    public function log_core_update($new_version)
    {
        Activity_Logger_Manager::log(array(
            'action_type' => 'update',
            'object_type' => 'system',
            'object_id' => 0,
            'object_name' => 'WordPress Core',
            /* translators: %s: WordPress version number */
            'description' => sprintf(__('WordPress updated to version %s', 'activity-monitor-pro'), $new_version),
        ));
    }

    /**
     * Log Option Update
     *
     * @param string $option
     * @param mixed $old_value
     * @param mixed $value
     */
    public function log_option_update($option, $old_value, $value)
    {
        // Ignore transient, cron, and site_transient options
        if (strpos($option, '_transient_') === 0 || strpos($option, '_site_transient_') === 0 || $option === 'cron') {
            return;
        }

        // Ignore internal WP options that change frequently
        $ignored_options = array('rewrite_rules', 'recently_edited', 'auto_updater.lock');
        if (in_array($option, $ignored_options)) {
            return;
        }

        if ($old_value === $value) {
            return;
        }

        Activity_Logger_Manager::log(array(
            'action_type' => 'update',
            'object_type' => 'option',
            'object_id' => 0,
            'object_name' => $option,
            /* translators: %s: Option name */
            'description' => sprintf(__('Option "%s" updated', 'activity-monitor-pro'), $option),
            'is_undoable' => 1,
            'undo_data' => array(
                'action' => 'update_option',
                'option' => $option,
                'old_value' => $old_value,
                'new_value' => $value
            )
        ));
    }

    /**
     * Log Option Add
     *
     * @param string $option
     * @param mixed $value
     */
    public function log_option_add($option, $value)
    {
        if (strpos($option, '_transient_') === 0 || strpos($option, '_site_transient_') === 0 || $option === 'cron') {
            return;
        }

        Activity_Logger_Manager::log(array(
            'action_type' => 'create',
            'object_type' => 'option',
            'object_id' => 0,
            'object_name' => $option,
            /* translators: %s: Option name */
            'description' => sprintf(__('Option "%s" added', 'activity-monitor-pro'), $option),
            'is_undoable' => 1,
            'undo_data' => array(
                'action' => 'delete_option', // Undo of add is delete
                'option' => $option,
                'new_value' => $value // Needed for Redo (which is Add)
            )
        ));
    }

    /**
     * Log Option Delete
     *
     * @param string $option
     */
    public function log_option_delete($option)
    {
        if (strpos($option, '_transient_') === 0 || strpos($option, '_site_transient_') === 0 || $option === 'cron') {
            return;
        }

        // We might need to capture the old value before delete if we want to undo it, 
        // but deleted_option hook fires after. 
        // For now, we just log the deletion. Undo might be limited here without pre-delete hook.

        Activity_Logger_Manager::log(array(
            'action_type' => 'delete',
            'object_type' => 'option',
            'object_id' => 0,
            'object_name' => $option,
            /* translators: %s: Option name */
            'description' => sprintf(__('Option "%s" deleted', 'activity-monitor-pro'), $option),
        ));
    }

    /**
     * Log the activity.
     *
     * @param array $args Activity arguments.
     */
    public function log_activity($args)
    {
        Activity_Logger_Manager::log($args);
    }

    /**
     * Check if the activity can be undone.
     *
     * @return bool
     */
    public function can_undo()
    {
        return true;
    }

    /**
     * Prepare data for undo.
     *
     * @param array $args Activity arguments.
     * @return array
     */
    public function prepare_undo_data($args)
    {
        return isset($args['undo_data']) ? $args['undo_data'] : array();
    }
}
