<?php
// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Post Activity Logger
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    Activity_Monitor_Pro
 * @subpackage Activity_Monitor_Pro/includes/loggers
 */

/**
 * Post Activity Logger.
 *
 * Handles logging of post and page activities.
 *
 * @package    Activity_Monitor_Pro
 * @subpackage Activity_Monitor_Pro/includes/loggers
 * @author     Your Name <email@example.com>
 */
class Activity_Post_Logger implements Activity_Logger_Interface
{

    /**
     * Register hooks for the logger.
     */
    /**
     * Register hooks for the logger.
     */
    public function register_hooks()
    {
        add_action('post_updated', array($this, 'log_post_update'), 10, 3);
        add_action('before_delete_post', array($this, 'log_post_deletion'));
    }

    /**
     * Log post updates and status changes.
     *
     * @param int $post_id Post ID.
     * @param WP_Post $post_after Post object after the update.
     * @param WP_Post $post_before Post object before the update.
     */
    public function log_post_update($post_id, $post_after, $post_before)
    {
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        // Ignore revisions and auto-drafts
        if ('revision' === $post_after->post_type || 'auto-draft' === $post_after->post_status) {
            return;
        }

        // Check for changes
        $changes = array();
        if ($post_after->post_title !== $post_before->post_title) {
            $changes[] = 'title';
        }
        if ($post_after->post_content !== $post_before->post_content) {
            $changes[] = 'content';
        }
        if ($post_after->post_status !== $post_before->post_status) {
            $changes[] = 'status';
        }

        if (empty($changes)) {
            return;
        }

        // Prepare Undo Data
        $undo_data = array(
            'prev' => array(
                'post_title' => $post_before->post_title,
                'post_content' => $post_before->post_content,
                'post_status' => $post_before->post_status,
            ),
            'new' => array(
                'post_title' => $post_after->post_title,
                'post_content' => $post_after->post_content,
                'post_status' => $post_after->post_status,
            ),
        );

        // Get post type label
        $post_type_obj = get_post_type_object($post_after->post_type);
        $post_type_label = $post_type_obj ? $post_type_obj->labels->singular_name : ucfirst($post_after->post_type);

        $action_type = 'post_updated';
        $description = sprintf('%s "%s" updated.', $post_type_label, $post_after->post_title);
        $severity = 'info';

        // Refine description based on changes
        if (in_array('status', $changes)) {
            if ('publish' === $post_after->post_status && 'publish' !== $post_before->post_status) {
                $action_type = 'post_published';
                $description = sprintf('%s "%s" published.', $post_type_label, $post_after->post_title);
            } elseif ('trash' === $post_after->post_status) {
                $action_type = 'post_trashed';
                $description = sprintf('%s "%s" moved to trash.', $post_type_label, $post_after->post_title);
                $severity = 'warning';
            } elseif ('trash' === $post_before->post_status && 'publish' === $post_after->post_status) {
                $action_type = 'post_restored';
                $description = sprintf('%s "%s" restored from trash.', $post_type_label, $post_after->post_title);
            }
        }

        $this->log_activity(array(
            'action_type' => $action_type,
            'object_type' => $post_after->post_type, // Use actual post type (post, page, etc.)
            'object_id' => $post_id,
            'object_name' => $post_after->post_title,
            'object_subtype' => $post_after->post_type,
            'description' => $description,
            'severity' => $severity,
            'is_undoable' => 1,
            'undo_data' => $undo_data,
        ));
    }

    /**
     * Log post deletion.
     *
     * @param int $post_id Post ID.
     */
    public function log_post_deletion($post_id)
    {
        $post = get_post($post_id);

        // Ignore revisions and auto-drafts
        if (!$post || 'revision' === $post->post_type || 'auto-draft' === $post->post_status) {
            return;
        }

        // Get post type label
        $post_type_obj = get_post_type_object($post->post_type);
        $post_type_label = $post_type_obj ? $post_type_obj->labels->singular_name : ucfirst($post->post_type);

        $this->log_activity(array(
            'action_type' => 'post_deleted',
            'object_type' => $post->post_type, // Use actual post type
            'object_id' => $post_id,
            'object_name' => $post->post_title,
            'object_subtype' => $post->post_type,
            'description' => sprintf('%s "%s" permanently deleted.', $post_type_label, $post->post_title),
            'severity' => 'warning',
        ));
    }

    /**
     * Log the activity.
     *
     * @param array $args Activity arguments.
     */
    public function log_activity($args)
    {
        Activity_Logger_Manager::log($args);
    }

    /**
     * Check if the activity can be undone.
     *
     * @return bool
     */
    public function can_undo()
    {
        return true;
    }

    /**
     * Prepare data for undo.
     *
     * @param array $args Activity arguments.
     * @return array
     */
    public function prepare_undo_data($args)
    {
        // Placeholder for undo data preparation
        return array();
    }
}
