<?php
// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Plugin Activity Logger
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    Activity_Monitor_Pro
 * @subpackage Activity_Monitor_Pro/includes/loggers
 */

/**
 * Plugin Activity Logger.
 *
 * Handles logging of plugin-related activities.
 *
 * @package    Activity_Monitor_Pro
 * @subpackage Activity_Monitor_Pro/includes/loggers
 * @author     Your Name <email@example.com>
 */
class Activity_Plugin_Logger implements Activity_Logger_Interface
{

    /**
     * Register hooks for the logger.
     */
    public function register_hooks()
    {
        add_action('activated_plugin', array($this, 'log_plugin_activation'));
        add_action('deactivated_plugin', array($this, 'log_plugin_deactivation'));
        // add_action( 'upgrader_process_complete', array( $this, 'log_plugin_update' ), 10, 2 ); // Complex to implement correctly in MVP
    }

    /**
     * Log plugin activation.
     *
     * @param string $plugin Plugin path.
     */
    public function log_plugin_activation($plugin)
    {
        if (!function_exists('get_plugin_data')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $plugin_file = wp_normalize_path(WP_PLUGIN_DIR . '/' . $plugin);
        $plugin_data = get_plugin_data($plugin_file);
        $name = $plugin_data['Name'];

        $this->log_activity(array(
            'action_type' => 'plugin_activated',
            'object_type' => 'plugin',
            'object_id' => 0, // Plugins don't have numeric IDs
            'object_name' => $name,
            'object_subtype' => $plugin,
            'description' => sprintf('Plugin "%s" activated.', $name),
            'severity' => 'info',
        ));
    }

    /**
     * Log plugin deactivation.
     *
     * @param string $plugin Plugin path.
     */
    public function log_plugin_deactivation($plugin)
    {
        if (!function_exists('get_plugin_data')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $plugin_file = wp_normalize_path(WP_PLUGIN_DIR . '/' . $plugin);
        $plugin_data = get_plugin_data($plugin_file);
        $name = $plugin_data['Name'];

        $this->log_activity(array(
            'action_type' => 'plugin_deactivated',
            'object_type' => 'plugin',
            'object_id' => 0,
            'object_name' => $name,
            'object_subtype' => $plugin,
            'description' => sprintf('Plugin "%s" deactivated.', $name),
            'severity' => 'info',
        ));
    }

    /**
     * Log the activity.
     *
     * @param array $args Activity arguments.
     */
    public function log_activity($args)
    {
        Activity_Logger_Manager::log($args);
    }

    /**
     * Check if the activity can be undone.
     *
     * @return bool
     */
    public function can_undo()
    {
        return true;
    }

    /**
     * Prepare data for undo.
     *
     * @param array $args Activity arguments.
     * @return array
     */
    public function prepare_undo_data($args)
    {
        return array();
    }
}
