<?php
// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Media Activity Logger
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    Activity_Monitor_Pro
 * @subpackage Activity_Monitor_Pro/includes/loggers
 */

/**
 * Media Activity Logger.
 *
 * Handles logging of media/attachment activities.
 *
 * @package    Activity_Monitor_Pro
 * @subpackage Activity_Monitor_Pro/includes/loggers
 * @author     Your Name <email@example.com>
 */
class Activity_Media_Logger implements Activity_Logger_Interface
{

    /**
     * Register hooks for the logger.
     */
    public function register_hooks()
    {
        add_action('add_attachment', array($this, 'log_attachment_add'));
        add_action('edit_attachment', array($this, 'log_attachment_edit'));
        add_action('delete_attachment', array($this, 'log_attachment_delete'));
    }

    /**
     * Log attachment upload.
     *
     * @param int $post_id Attachment ID.
     */
    public function log_attachment_add($post_id)
    {
        $post = get_post($post_id);

        $this->log_activity(array(
            'action_type' => 'media_uploaded',
            'object_type' => 'media',
            'object_id' => $post_id,
            'object_name' => $post->post_title,
            'object_subtype' => $post->post_mime_type,
            'description' => sprintf('Media "%s" uploaded.', $post->post_title),
            'severity' => 'info',
        ));
    }

    /**
     * Log attachment edit.
     *
     * @param int $post_id Attachment ID.
     */
    public function log_attachment_edit($post_id)
    {
        $post = get_post($post_id);

        $this->log_activity(array(
            'action_type' => 'media_updated',
            'object_type' => 'media',
            'object_id' => $post_id,
            'object_name' => $post->post_title,
            'object_subtype' => $post->post_mime_type,
            'description' => sprintf('Media "%s" updated.', $post->post_title),
            'severity' => 'info',
        ));
    }

    /**
     * Log attachment deletion.
     *
     * @param int $post_id Attachment ID.
     */
    public function log_attachment_delete($post_id)
    {
        $post = get_post($post_id);

        $this->log_activity(array(
            'action_type' => 'media_deleted',
            'object_type' => 'media',
            'object_id' => $post_id,
            'object_name' => $post->post_title,
            'object_subtype' => $post->post_mime_type,
            'description' => sprintf('Media "%s" deleted.', $post->post_title),
            'severity' => 'warning',
            'is_undoable' => 0, // Deleting media deletes the file, hard to undo without backup
        ));
    }

    /**
     * Log the activity.
     *
     * @param array $args Activity arguments.
     */
    public function log_activity($args)
    {
        Activity_Logger_Manager::log($args);
    }

    /**
     * Check if the activity can be undone.
     *
     * @return bool
     */
    public function can_undo()
    {
        return false;
    }

    /**
     * Prepare data for undo.
     *
     * @param array $args Activity arguments.
     * @return array
     */
    public function prepare_undo_data($args)
    {
        return array();
    }
}
