<?php
// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Manager for activity loggers
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    Activity_Monitor_Pro
 * @subpackage Activity_Monitor_Pro/includes/loggers
 */

/**
 * Manager for activity loggers.
 *
 * Responsible for initializing and managing all activity loggers.
 *
 * @package    Activity_Monitor_Pro
 * @subpackage Activity_Monitor_Pro/includes/loggers
 * @author     Your Name <email@example.com>
 */
class Activity_Logger_Manager
{

    /**
     * The array of registered loggers.
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $loggers    The array of registered loggers.
     */
    private $loggers = array();

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     */
    public function __construct()
    {
        $this->load_loggers();
    }

    /**
     * Load and register all loggers.
     *
     * @since    1.0.0
     */
    private function load_loggers()
    {
        // Load logger files
        require_once plugin_dir_path(__FILE__) . 'class-post-logger.php';
        require_once plugin_dir_path(__FILE__) . 'class-user-logger.php';
        require_once plugin_dir_path(__FILE__) . 'class-plugin-logger.php';
        require_once plugin_dir_path(__FILE__) . 'class-media-logger.php';
        require_once plugin_dir_path(__FILE__) . 'class-system-logger.php';
        require_once plugin_dir_path(__FILE__) . 'class-theme-logger.php';
        require_once plugin_dir_path(__FILE__) . 'class-taxonomy-logger.php';

        // Instantiate and register loggers
        $this->register_logger(new Activity_Post_Logger());
        $this->register_logger(new Activity_User_Logger());
        $this->register_logger(new Activity_Plugin_Logger());
        $this->register_logger(new Activity_Media_Logger());
        $this->register_logger(new Activity_System_Logger());
        $this->register_logger(new Activity_Theme_Logger());
        $this->register_logger(new Activity_Taxonomy_Logger());
    }

    /**
     * Register a single logger.
     *
     * @since    1.0.0
     * @param object $logger The logger instance to register.
     */
    public function register_logger($logger)
    {
        if ($logger instanceof Activity_Logger_Interface) {
            $this->loggers[] = $logger;
            $logger->register_hooks();
        }
    }

    /**
     * Central logging method.
     *
     * @since    1.0.0
     * @param array $args Activity arguments.
     */
    public static function log($args)
    {
        // Validate required fields
        if (empty($args['action_type']) || empty($args['object_type'])) {
            return;
        }

        // Free version: Only log basic activity types
        $allowed_types = array('post', 'page', 'media', 'user', 'attachment');
        if (!in_array($args['object_type'], $allowed_types)) {
            return;
        }

        // Add default values
        $defaults = array(
            'user_id' => get_current_user_id(),
            'user_name' => wp_get_current_user()->display_name,
            'user_email' => wp_get_current_user()->user_email,
            'ip_address' => isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : '',
            'user_agent' => isset($_SERVER['HTTP_USER_AGENT']) ? sanitize_text_field(wp_unslash($_SERVER['HTTP_USER_AGENT'])) : '',
            'created_at' => current_time('mysql'),
            'severity' => 'info',
            'is_undoable' => 0,
            'undo_status' => 'not_applicable',
        );

        $data = wp_parse_args($args, $defaults);

        // Extract undo_data if present
        $undo_data = isset($data['undo_data']) ? $data['undo_data'] : null;
        unset($data['undo_data']);

        // Save to database
        require_once plugin_dir_path(dirname(__FILE__)) . 'utils/class-database.php';
        $log_id = Activity_Monitor_Pro_Database::insert_log($data);

        if ($log_id && !empty($undo_data)) {
            Activity_Monitor_Pro_Database::insert_undo_data($log_id, $undo_data);
        }
    }
}
