<?php
// Exit if accessed directly.
if (!defined('ABSPATH')) {
    exit;
}

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    Activity_Monitor_Pro
 * @subpackage Activity_Monitor_Pro/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * @since      1.0.0
 * @package    Activity_Monitor_Pro
 * @subpackage Activity_Monitor_Pro/includes
 * @author     Your Name <email@example.com>
 */
class Activity_Monitor_Pro
{

    /**
     * The loader that's responsible for maintaining and registering all hooks that power
     * the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      Activity_Monitor_Pro_Loader    $loader    Maintains and registers all hooks for the plugin.
     */
    protected $loader;

    /**
     * The unique identifier of this plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $plugin_name    The string used to uniquely identify this plugin.
     */
    protected $plugin_name;

    /**
     * The current version of the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $version    The current version of the plugin.
     */
    protected $version;

    /**
     * Define the core functionality of the plugin.
     *
     * Set the plugin name and the plugin version that can be used throughout the plugin.
     * Load the dependencies, define the locale, and set the hooks for the admin area and
     * the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function __construct()
    {
        if (defined('ACTIVITY_MONITOR_PRO_VERSION')) {
            $this->version = ACTIVITY_MONITOR_PRO_VERSION;
        } else {
            $this->version = '1.0.0';
        }
        $this->plugin_name = 'activity-monitor-pro';

        $this->load_dependencies();
        $this->set_locale();
        $this->define_admin_hooks();
        $this->define_public_hooks();

    }

    /**
     * Load the required dependencies for this plugin.
     *
     * Include the following files that make up the plugin:
     *
     * - Activity_Monitor_Pro_Loader. Orchestrates the hooks of the plugin.
     * - Activity_Monitor_Pro_i18n. Defines internationalization functionality.
     * - Activity_Monitor_Pro_Admin. Defines all hooks for the admin area.
     * - Activity_Monitor_Pro_Public. Defines all hooks for the public side of the site.
     *
     * @since    1.0.0
     * @access   private
     */
    private function load_dependencies()
    {

        /**
         * The class responsible for orchestrating the actions and filters of the
         * core plugin.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-loader.php';

        /**
         * The class responsible for defining internationalization functionality
         * of the plugin.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-i18n.php';

        /**
         * The class responsible for defining all actions that occur in the admin area.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/admin/class-admin.php';
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/admin/class-menu.php';

        /**
         * The class responsible for managing activity loggers.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/loggers/class-logger-interface.php';
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/loggers/class-logger-manager.php';

        /**
         * The class responsible for database interactions.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/utils/class-database.php';

        $this->loader = new Activity_Monitor_Pro_Loader();
        new Activity_Logger_Manager(); // Initialize loggers

    }

    /**
     * Define the locale for this plugin for internationalization.
     *
     * Uses the Activity_Monitor_Pro_i18n class in order to set the domain and to register the hook
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function set_locale()
    {
        // Internationalization is handled automatically by WordPress.org
    }

    /**
     * Register all of the hooks related to the admin area functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_admin_hooks()
    {

        $plugin_admin = new Activity_Monitor_Pro_Admin($this->get_plugin_name(), $this->get_version());
        $plugin_menu = new Activity_Monitor_Pro_Menu($this->get_plugin_name(), $this->get_version());

        $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_styles');
        $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts');
        $this->loader->add_action('admin_menu', $plugin_menu, 'add_plugin_admin_menu');

        // AJAX Hooks
        $this->loader->add_action('wp_ajax_activity_monitor_get_latest_logs', $plugin_admin, 'ajax_get_latest_logs');
        $this->loader->add_action('wp_ajax_activity_monitor_get_logs', $plugin_admin, 'ajax_get_logs');
        $this->loader->add_action('wp_ajax_activity_monitor_export_csv', $plugin_admin, 'ajax_export_csv');
        $this->loader->add_action('wp_ajax_activity_monitor_clear_logs', $plugin_admin, 'ajax_clear_logs');

        // Undo Manager
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/undo/class-undo-manager.php';
        $undo_manager = new Activity_Monitor_Undo_Manager();
        $this->loader->add_action('wp_ajax_activity_monitor_undo', $undo_manager, 'ajax_handle_undo');
        $this->loader->add_action('wp_ajax_activity_monitor_redo', $undo_manager, 'ajax_handle_redo');

        // Reports (weekly reports for free version)
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/reports/class-report-scheduler.php';
        new Activity_Monitor_Report_Scheduler($this->loader);

        // Cron Hooks
        $this->loader->add_action('activity_monitor_daily_cleanup', $this, 'cleanup_old_logs');
        if (!wp_next_scheduled('activity_monitor_daily_cleanup')) {
            wp_schedule_event(time(), 'daily', 'activity_monitor_daily_cleanup');
        }



    }

    /**
     * Cleanup old logs.
     * Free version: 30 days retention.
     */
    public function cleanup_old_logs()
    {
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/utils/class-database.php';

        // Free version: 30 days retention
        $days = 30;
        Activity_Monitor_Pro_Database::delete_old_logs($days);
    }

    /**
     * Register all of the hooks related to the public-facing functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_public_hooks()
    {
        // No public hooks for now, but placeholder for future
    }

    /**
     * Run the loader to execute all of the hooks with WordPress.
     *
     * @since    1.0.0
     */
    public function run()
    {
        $this->loader->run();
    }

    /**
     * The name of the plugin used to uniquely identify it within the context of
     * WordPress and to define internationalization functionality.
     *
     * @since     1.0.0
     * @return    string    The name of the plugin.
     */
    public function get_plugin_name()
    {
        return $this->plugin_name;
    }

    /**
     * The reference to the class that orchestrates the hooks with the plugin.
     *
     * @since     1.0.0
     * @return    Activity_Monitor_Pro_Loader    Orchestrates the hooks of the plugin.
     */
    public function get_loader()
    {
        return $this->loader;
    }

    /**
     * Retrieve the version number of the plugin.
     *
     * @since     1.0.0
     * @return    string    The version number of the plugin.
     */
    public function get_version()
    {
        return $this->version;
    }

}
