<?php

/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://x.com/wolexfeelcool
 * @since      1.0.0
 *
 * @package    Activity_Monitor_Pro
 * @subpackage Activity_Monitor_Pro/admin
 */

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the admin-specific stylesheet and JavaScript.
 *
 * @package    Activity_Monitor_Pro
 * @subpackage Activity_Monitor_Pro/admin
 * @author     Wole Akinloye <lovabuildagency@gmail.com>
 */
class Activity_Monitor_Pro_Admin
{

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param      string    $plugin_name       The name of this plugin.
     * @param      string    $version    The version of this plugin.
     */
    public function __construct($plugin_name, $version)
    {

        $this->plugin_name = $plugin_name;
        $this->version = $version;

    }

    /**
     * Register the stylesheets for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_styles()
    {

        /**
         * This function is provided for demonstration purposes only.
         *
         * An instance of this class should be passed to the run() function
         * defined in Activity_Monitor_Pro_Loader as all of the hooks are defined
         * in that particular class.
         *
         * The Activity_Monitor_Pro_Loader will then create the relationship
         * between the defined hooks and the functions defined in this
         * class.
         */

        wp_enqueue_style($this->plugin_name, ACTIVITY_MONITOR_PRO_URL . 'admin/css/activity-monitor-pro-admin.css', array(), $this->version, 'all');

    }

    /**
     * Register the JavaScript for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_scripts()
    {

        /**
         * This function is provided for demonstration purposes only.
         *
         * An instance of this class should be passed to the run() function
         * defined in Activity_Monitor_Pro_Loader as all of the hooks are defined
         * in that particular class.
         *
         * The Activity_Monitor_Pro_Loader will then create the relationship
         * between the defined hooks and the functions defined in this
         * class.
         */

        // Enqueue main admin script globally
        wp_enqueue_script($this->plugin_name, ACTIVITY_MONITOR_PRO_URL . 'admin/js/activity-monitor-pro-admin.js', array('jquery'), $this->version, false);

        // Localize script for AJAX
        wp_localize_script($this->plugin_name, 'activity_monitor_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('activity_monitor_nonce'),
        ));

        // Get current screen
        $screen = get_current_screen();

        // Enqueue dashboard-specific script
        if ($screen && $screen->id === 'toplevel_page_activity-monitor-pro') {
            wp_enqueue_script(
                $this->plugin_name . '-dashboard',
                ACTIVITY_MONITOR_PRO_URL . 'admin/js/dashboard.js',
                array('jquery', $this->plugin_name),
                $this->version,
                false
            );
        }

        // Enqueue activity log-specific script on activity log page
        // Screen ID varies by WordPress version, so check broadly
        if ($screen && strpos($screen->id, 'activity-monitor') !== false && strpos($screen->id, '-log') !== false) {
            wp_enqueue_script(
                $this->plugin_name . '-activity-log',
                ACTIVITY_MONITOR_PRO_URL . 'admin/js/activity-log.js',
                array('jquery', $this->plugin_name),
                $this->version,
                false
            );
        }

    }

    /**
     * AJAX handler to get latest logs.
     */
    public function ajax_get_latest_logs()
    {
        check_ajax_referer('activity_monitor_nonce', 'nonce');

        require_once plugin_dir_path(dirname(__FILE__)) . 'utils/class-database.php';
        $recent_logs = Activity_Monitor_Pro_Database::get_logs(array('limit' => 5));

        if (empty($recent_logs)) {
            wp_send_json_error('No logs found');
        }

        ob_start();
        ?>
        <ul class="activity-list">
            <?php foreach ($recent_logs as $log): ?>
                <li class="amp-activity-item">
                    <div class="amp-avatar">
                        <?php echo esc_html(strtoupper(substr($log['user_name'], 0, 1))); ?>
                    </div>
                    <div class="amp-content">
                        <div class="amp-user-name">
                            <?php echo esc_html($log['user_name']); ?>
                            <span class="amp-severity-indicator severity-<?php echo esc_attr($log['severity']); ?>"
                                title="<?php echo esc_attr($log['severity']); ?>"></span>
                        </div>
                        <span class="amp-action-desc">
                            <?php echo esc_html($log['description']); ?>
                        </span>
                        <span class="amp-time">
                            <?php echo esc_html(human_time_diff(strtotime($log['created_at']), current_time('timestamp')) . ' ago'); ?>
                        </span>
                    </div>
                    <div class="amp-actions">
                        <?php if ($log['is_undoable']): ?>
                            <?php if ($log['undo_status'] === 'undone'): ?>
                                <button class="amp-btn amp-btn-sm amp-btn-outline amp-redo-btn"
                                    data-id="<?php echo esc_attr($log['id']); ?>">
                                    <span class="dashicons dashicons-redo"></span> Redo
                                </button>
                            <?php else: ?>
                                <button class="amp-btn amp-btn-sm amp-btn-outline amp-undo-btn"
                                    data-id="<?php echo esc_attr($log['id']); ?>">
                                    <span class="dashicons dashicons-undo"></span> Undo
                                </button>
                            <?php endif; ?>
                        <?php endif; ?>
                    </div>
                </li>
            <?php endforeach; ?>
        </ul>
        <?php
        $html = ob_get_clean();

        wp_send_json_success(array('html' => $html));
    }

    /**
     * Handle AJAX request to get full logs with filtering.
     */
    public function ajax_get_logs()
    {
        check_ajax_referer('activity_monitor_nonce', 'nonce');

        $page = isset($_POST['page']) ? intval($_POST['page']) : 1;
        $search = isset($_POST['search']) ? sanitize_text_field(wp_unslash($_POST['search'])) : '';
        $user_id = isset($_POST['user_id']) ? intval($_POST['user_id']) : 0;
        $type = isset($_POST['type']) ? sanitize_text_field(wp_unslash($_POST['type'])) : '';
        $severity = isset($_POST['severity']) ? sanitize_text_field(wp_unslash($_POST['severity'])) : '';

        $limit = 20;
        $offset = ($page - 1) * $limit;

        require_once plugin_dir_path(dirname(__FILE__)) . 'utils/class-database.php';

        $args = array(
            'limit' => $limit,
            'offset' => $offset,
            'search' => $search,
            'user_id' => $user_id,
            'object_type' => $type,
            'severity' => $severity
        );

        $logs = Activity_Monitor_Pro_Database::get_logs($args);
        $total_logs = Activity_Monitor_Pro_Database::get_total_logs_count($args);
        $total_pages = ceil($total_logs / $limit);

        if (empty($logs)) {
            wp_send_json_success(array(
                'html' => '<div style="padding: 40px; text-align: center; color: var(--amp-text-muted);">No activities found matching your criteria.</div>',
                'total_pages' => 0
            ));
        }

        ob_start();
        ?>
        <ul class="amp-activity-list">
            <?php foreach ($logs as $log): ?>
                <li class="amp-activity-item">
                    <div class="amp-avatar">
                        <?php echo esc_html(strtoupper(substr($log['user_name'], 0, 1))); ?>
                    </div>
                    <div class="amp-content">
                        <div class="amp-user-name">
                            <?php echo esc_html($log['user_name']); ?>
                            <span class="amp-severity-indicator severity-<?php echo esc_attr($log['severity']); ?>"
                                title="<?php echo esc_attr($log['severity']); ?>"></span>
                            <span style="font-weight:normal; color:var(--amp-text-muted); font-size:12px; margin-left:5px;">
                                <?php echo esc_html(ucfirst($log['object_type'])); ?>
                            </span>
                        </div>
                        <span class="amp-action-desc">
                            <?php echo esc_html($log['description']); ?>
                        </span>
                        <span class="amp-time">
                            <?php echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($log['created_at']))); ?>
                            (<?php echo esc_html(human_time_diff(strtotime($log['created_at']), current_time('timestamp')) . ' ago'); ?>)
                        </span>
                    </div>
                    <div class="amp-actions">
                        <?php if ($log['is_undoable']): ?>
                            <?php if ($log['undo_status'] === 'undone'): ?>
                                <button class="amp-btn amp-btn-sm amp-btn-outline amp-redo-btn"
                                    data-id="<?php echo esc_attr($log['id']); ?>">
                                    <span class="dashicons dashicons-redo"></span> Redo
                                </button>
                            <?php else: ?>
                                <button class="amp-btn amp-btn-sm amp-btn-outline amp-undo-btn"
                                    data-id="<?php echo esc_attr($log['id']); ?>">
                                    <span class="dashicons dashicons-undo"></span> Undo
                                </button>
                            <?php endif; ?>
                        <?php endif; ?>
                    </div>
                </li>
            <?php endforeach; ?>
        </ul>
        <?php
        $html = ob_get_clean();

        wp_send_json_success(array(
            'html' => $html,
            'total_pages' => $total_pages
        ));
    }

    /**
     * Handle CSV Export.
     */
    public function ajax_export_csv()
    {
        check_ajax_referer('activity_monitor_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_die('Permission denied');
        }

        require_once plugin_dir_path(dirname(__FILE__)) . 'utils/class-database.php';

        // Get all logs (limit to last 1000 for performance)
        $args = array('limit' => 1000);

        // Free version: CSV export is available
        $logs = Activity_Monitor_Pro_Database::get_logs($args);

        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="activity-log-' . gmdate('Y-m-d') . '.csv"');

        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fopen -- Using php://output stream for CSV export
        $output = fopen('php://output', 'w');

        // Header row
        fputcsv($output, array('ID', 'Date', 'User', 'Action', 'Object Type', 'Object Name', 'Description', 'IP Address', 'Severity'));

        foreach ($logs as $log) {
            fputcsv($output, array(
                $log['id'],
                $log['created_at'],
                $log['user_name'],
                $log['action_type'],
                $log['object_type'],
                $log['object_name'],
                $log['description'],
                $log['ip_address'],
                $log['severity']
            ));
        }

        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- Closing php://output stream
        fclose($output);
        exit;
    }

    /**
     * Handle AJAX request to clear logs.
     */
    public function ajax_clear_logs()
    {
        check_ajax_referer('activity_monitor_nonce', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Permission denied');
        }

        require_once plugin_dir_path(dirname(__FILE__)) . 'utils/class-database.php';
        global $wpdb;
        $table_logs = $wpdb->prefix . 'activitymon_logs';
        $table_undo = $wpdb->prefix . 'activitymon_undo_data';

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name safely derived from $wpdb->prefix, TRUNCATE cannot use prepare()
        $wpdb->query("TRUNCATE TABLE $table_logs");
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared, PluginCheck.Security.DirectDB.UnescapedDBParameter -- Table name safely derived from $wpdb->prefix, TRUNCATE cannot use prepare()
        $wpdb->query("TRUNCATE TABLE $table_undo");

        wp_send_json_success('Logs cleared successfully');
    }
}
