<?php
/*
Plugin Name: ACF Display
Plugin URI: https://www.darrenlambert.com/acf-display-wordpress-plugin/
Description: Automatically displays fields on posts, pages and custom post types created with ACF.
Version: 1.0
Author: Darren Lambert
Author URI: https://www.darrenlambert.com/
License:     GPLv2 or later
License URI: http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
Text Domain: acf-display
*/

/**
 * Load styles and scripts
 */
add_action('wp_enqueue_scripts', 'acfd_enqueue_scripts');
function acfd_enqueue_scripts()
{
	// Load out stylesheet
	wp_enqueue_style( 'acfd', plugins_url( '/css/acfd.css', __FILE__ ) );
}

/**
 * Display fields for the current post
 */
add_filter( 'the_content', 'acfd_render_fields', 11, 1);
function acfd_render_fields( $content ) {

	// Don't render in the dashboard
  if ( is_admin() ) return $content;

	// Get all of the fields
	$fields = get_field_objects();

	// Sort by menu_order, i.e. the admin UI order
	usort($fields, function ($item1, $item2) {
    return $item1['menu_order'] <=> $item2['menu_order'];
	});

	// Bail if no fields
	if (! $fields) return $content;

	// If we have fields add them to the end of the content

	// Buffer the output
	ob_start();

	$post_id = get_the_ID();

	// Render each field
  foreach( $fields as $field ) {
		acfd_render_field( $post_id, $field );
	}

	// Get the HTML
  $form = ob_get_contents();
  ob_end_clean();

	// Return the content and out fields
  return $content . $form;

}

/**
 * Render the given field
 * Adds a wrapper around the field value
 */
function acfd_render_field( $post_id, $field ) {

	// Don't render some field types
	if ( $field['type'] == 'password') {
		return;
	}

	?>
	<div class='acfd-field'>
		<label class='acfd-label'><?php echo esc_html($field['label']); ?></label>
		<?php acfd_render_field_value( $post_id, $field ); ?>
	</div>
	<?php

}

/**
 * Renders a field value
 */
function acfd_render_field_value( $post_id, $field ) {

	if ( $field['type'] == 'image') {

		$image = null;

		// There are multiple return formats
		$return_format = acfd_get_array_value('return_format', $field);

		// URL
		if ($return_format == 'url') {

			// Output via ACF function.  Doesn't add an wrapper
			$image = "<img src='". esc_url( get_field( $field['name'] ) ) ."' />";
		}

		// ID
		if ($return_format == 'id') {
			// Render the image
			$image = wp_get_attachment_image( acfd_get_array_value('value', $field), 'medium');
		}

		// Array
		if ($return_format == 'array') {

			// Get the value array
			$value = acfd_get_array_value('value', $field);

			// Render the image
			$image = wp_get_attachment_image( acfd_get_array_value('id', $value), 'medium');

		}

		// Bail if no image
		if (! $image) return;

		$allowed_html = ['img' => [
			'src'      => true,
			'srcset'   => true,
			'sizes'    => true,
			'class'    => true,
			'id'       => true,
			'width'    => true,
			'height'   => true,
			'alt'      => true,
			'align'    => true,
			'loading'    => true,
		]];

		?>
		<div class='acfd-field-value'>
			<?php echo wp_kses($image, $allowed_html); ?>
		</div>
		<?php

		return;
	}

	// File
	if ( $field['type'] == 'file') {

		$url = acfd_get_file_field_url_value($field);

		// Bail if no file URL
		if (! $url) return;

		?>
		<div class='acfd-field-value'>
			<a href='<?php echo esc_url($url) ?>' download>Download</a>
		</div>
		<?php

		return;
	}

	// All other fields
	?>
	<div class='acfd-field-value'>
		<?php the_field($field['name'], $post_id) ?>
	</div>
	<?php

}


/**
 * Returns the URL from a File field
 * Handles the different Return Formats
 */
function acfd_get_file_field_url_value($field) {

	// There are multiple return formats
	$return_format = acfd_get_array_value('return_format', $field);

	// URL
	if ($return_format == 'url') {
		return acfd_get_array_value('value', $field);
	}

	// ID
	if ($return_format == 'id') {
		return  wp_get_attachment_url(acfd_get_array_value('value', $field));
	}

	// Array
	if ($return_format == 'array') {

		// Get the value array
		$value = acfd_get_array_value('value', $field);

		// Return the URL of ID
		return  wp_get_attachment_url(acfd_get_array_value('id', $value));

	}

	// Default
	return null;

}

// Safely gets an array value
// Returns null if not present
function acfd_get_array_value($index, $array, $default = null) {

	if (isset($array[$index])) {
		return $array[$index];
	}

	return $default;
}
