<?php
/**
 * Frontend output and asset loading.
 *
 * @package AccessiBar
 */

defined( 'ABSPATH' ) || exit;

/**
 * Enqueues frontend assets and renders the accessibility button markup.
 */
class AccessiBar_Frontend {

	/**
	 * Registers frontend hooks.
	 */
	public function __construct() {
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_assets' ) );
		add_action( 'wp_footer', array( $this, 'render_button' ) );
	}

	/**
	 * Enqueues CSS, JS and passes config to the frontend script.
	 *
	 * @return void
	 */
	public function enqueue_assets() {
		$options = accessibar_get_options();

		wp_enqueue_style(
			'accessibar-front',
			ACCESSIBAR_PLUGIN_URL . 'assets/css/accessibar-front.css',
			array(),
			ACCESSIBAR_VERSION
		);

		wp_enqueue_script(
			'accessibar-front',
			ACCESSIBAR_PLUGIN_URL . 'assets/js/accessibar-front.js',
			array(),
			ACCESSIBAR_VERSION,
			false
		);

		$tool_labels = array(
			'font_size'       => __( 'Font Size', 'accessibar' ),
			'high_contrast'   => __( 'High Contrast', 'accessibar' ),
			'highlight_links' => __( 'Highlight Links', 'accessibar' ),
			'readable_font'   => __( 'Readable Font', 'accessibar' ),
			'stop_animations' => __( 'Stop Animations', 'accessibar' ),
			'big_cursor'      => __( 'Large Cursor', 'accessibar' ),
		);

		wp_localize_script(
			'accessibar-front',
			'accessibarConfig',
			array(
				'position'   => $options['position'],
				'color'      => $options['color'],
				'size'       => $options['size'],
				'buttonText' => $options['button_text'],
				'hideMobile' => $options['hide_mobile'],
				'tools'      => $options['tools'],
				'toolLabels' => $tool_labels,
				'fontUrl'    => ACCESSIBAR_PLUGIN_URL . 'assets/fonts/OpenDyslexic-Regular.woff2',
				'i18n'       => array(
					'open'      => __( 'Open accessibility toolbar', 'accessibar' ),
					'close'     => __( 'Close accessibility toolbar', 'accessibar' ),
					'reset'     => __( 'Reset All', 'accessibar' ),
					'increase'  => __( 'Increase font size', 'accessibar' ),
					'decrease'  => __( 'Decrease font size', 'accessibar' ),
					'resetFs'   => __( 'Reset font size', 'accessibar' ),
					'panelName' => __( 'Accessibility toolbar', 'accessibar' ),
				),
			)
		);
	}

	/**
	 * Outputs the accessibility button container in the footer.
	 *
	 * @return void
	 */
	public function render_button() {
		$options = accessibar_get_options();
		$sizes   = array(
			'small'  => '40',
			'medium' => '50',
			'large'  => '60',
		);

		$size_px    = isset( $sizes[ $options['size'] ] ) ? $sizes[ $options['size'] ] : '50';
		$position   = $options['position'];
		$color      = $options['color'];
		$hide_class = $options['hide_mobile'] ? ' accessibar-hide-mobile' : '';
		?>
		<div id="accessibar-root"
			class="accessibar-pos-<?php echo esc_attr( $position . $hide_class ); ?>"
			data-size="<?php echo esc_attr( $size_px ); ?>"
			style="--accessibar-color: <?php echo esc_attr( $color ); ?>; --accessibar-size: <?php echo esc_attr( $size_px ); ?>px;">
		</div>
		<?php
	}
}
