<?php
/**
 * Admin settings page.
 *
 * @package AccessiBar
 */

defined( 'ABSPATH' ) || exit;

/**
 * Handles the plugin settings page in wp-admin.
 */
class AccessiBar_Admin {

	/**
	 * Registers admin hooks.
	 */
	public function __construct() {
		add_action( 'admin_menu', array( $this, 'add_menu' ) );
		add_action( 'admin_init', array( $this, 'register_settings' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_assets' ) );
	}

	/**
	 * Adds the settings page under Settings menu.
	 *
	 * @return void
	 */
	public function add_menu() {
		add_options_page(
			__( 'AccessiBar Settings', 'accessibar' ),
			__( 'AccessiBar', 'accessibar' ),
			'manage_options',
			'accessibar',
			array( $this, 'render_page' )
		);
	}

	/**
	 * Registers the plugin setting with sanitization callback.
	 *
	 * @return void
	 */
	public function register_settings() {
		register_setting(
			'accessibar_group',
			ACCESSIBAR_OPTION_KEY,
			array(
				'type'              => 'array',
				'sanitize_callback' => array( $this, 'sanitize_options' ),
				'default'           => accessibar_get_defaults(),
			)
		);
	}

	/**
	 * Enqueues admin CSS and JS on the plugin settings page only.
	 *
	 * @param string $hook Current admin page hook suffix.
	 * @return void
	 */
	public function enqueue_assets( $hook ) {
		if ( 'settings_page_accessibar' !== $hook ) {
			return;
		}

		wp_enqueue_style( 'wp-color-picker' );
		wp_enqueue_style(
			'accessibar-admin',
			ACCESSIBAR_PLUGIN_URL . 'assets/css/accessibar-admin.css',
			array(),
			ACCESSIBAR_VERSION
		);
		wp_enqueue_script(
			'accessibar-admin',
			ACCESSIBAR_PLUGIN_URL . 'assets/js/accessibar-admin.js',
			array( 'wp-color-picker' ),
			ACCESSIBAR_VERSION,
			true
		);
	}

	/**
	 * Sanitizes all option values before saving.
	 *
	 * @param mixed $input Raw form input.
	 * @return array Sanitized options.
	 */
	public function sanitize_options( $input ) {
		if ( ! is_array( $input ) ) {
			return accessibar_get_defaults();
		}

		$defaults  = accessibar_get_defaults();
		$positions = array( 'bottom-right', 'bottom-left', 'top-right', 'top-left' );
		$sizes     = array( 'small', 'medium', 'large' );

		$clean = array();

		$pos               = isset( $input['position'] ) ? $input['position'] : '';
		$clean['position'] = in_array( $pos, $positions, true ) ? $pos : $defaults['position'];

		$color          = isset( $input['color'] ) ? $input['color'] : '';
		$clean['color'] = sanitize_hex_color( $color ) ? sanitize_hex_color( $color ) : $defaults['color'];

		$size          = isset( $input['size'] ) ? $input['size'] : '';
		$clean['size'] = in_array( $size, $sizes, true ) ? $size : $defaults['size'];

		$clean['button_text'] = sanitize_text_field( isset( $input['button_text'] ) ? $input['button_text'] : '' );
		$clean['hide_mobile'] = ! empty( $input['hide_mobile'] );

		$clean['tools'] = array();
		foreach ( array_keys( $defaults['tools'] ) as $tool ) {
			$clean['tools'][ $tool ] = ! empty( $input['tools'][ $tool ] );
		}

		return $clean;
	}

	/**
	 * Renders the settings page HTML.
	 *
	 * @return void
	 */
	public function render_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$options = accessibar_get_options();

		$tool_labels = array(
			'font_size'       => __( 'Font Size Controls', 'accessibar' ),
			'high_contrast'   => __( 'High Contrast Mode', 'accessibar' ),
			'highlight_links' => __( 'Highlight Links', 'accessibar' ),
			'readable_font'   => __( 'Readable Font (OpenDyslexic)', 'accessibar' ),
			'stop_animations' => __( 'Stop Animations', 'accessibar' ),
			'big_cursor'      => __( 'Large Cursor', 'accessibar' ),
		);
		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'AccessiBar Settings', 'accessibar' ); ?></h1>

			<form method="post" action="options.php">
				<?php settings_fields( 'accessibar_group' ); ?>

				<table class="form-table" role="presentation">

					<tr>
						<th scope="row">
							<label for="accessibar-position"><?php esc_html_e( 'Button Position', 'accessibar' ); ?></label>
						</th>
						<td>
							<select id="accessibar-position" name="<?php echo esc_attr( ACCESSIBAR_OPTION_KEY ); ?>[position]">
								<option value="bottom-right" <?php selected( $options['position'], 'bottom-right' ); ?>><?php esc_html_e( 'Bottom Right', 'accessibar' ); ?></option>
								<option value="bottom-left" <?php selected( $options['position'], 'bottom-left' ); ?>><?php esc_html_e( 'Bottom Left', 'accessibar' ); ?></option>
								<option value="top-right" <?php selected( $options['position'], 'top-right' ); ?>><?php esc_html_e( 'Top Right', 'accessibar' ); ?></option>
								<option value="top-left" <?php selected( $options['position'], 'top-left' ); ?>><?php esc_html_e( 'Top Left', 'accessibar' ); ?></option>
							</select>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label for="accessibar-color"><?php esc_html_e( 'Button Color', 'accessibar' ); ?></label>
						</th>
						<td>
							<input type="text" id="accessibar-color" name="<?php echo esc_attr( ACCESSIBAR_OPTION_KEY ); ?>[color]" value="<?php echo esc_attr( $options['color'] ); ?>" class="accessibar-color-field" />
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label for="accessibar-size"><?php esc_html_e( 'Button Size', 'accessibar' ); ?></label>
						</th>
						<td>
							<select id="accessibar-size" name="<?php echo esc_attr( ACCESSIBAR_OPTION_KEY ); ?>[size]">
								<option value="small" <?php selected( $options['size'], 'small' ); ?>><?php esc_html_e( 'Small (40px)', 'accessibar' ); ?></option>
								<option value="medium" <?php selected( $options['size'], 'medium' ); ?>><?php esc_html_e( 'Medium (50px)', 'accessibar' ); ?></option>
								<option value="large" <?php selected( $options['size'], 'large' ); ?>><?php esc_html_e( 'Large (60px)', 'accessibar' ); ?></option>
							</select>
						</td>
					</tr>

					<tr>
						<th scope="row">
							<label for="accessibar-button-text"><?php esc_html_e( 'Button Text', 'accessibar' ); ?></label>
						</th>
						<td>
							<input type="text" id="accessibar-button-text" name="<?php echo esc_attr( ACCESSIBAR_OPTION_KEY ); ?>[button_text]" value="<?php echo esc_attr( $options['button_text'] ); ?>" class="regular-text" />
							<p class="description"><?php esc_html_e( 'Optional text next to the icon. Leave empty to show icon only.', 'accessibar' ); ?></p>
						</td>
					</tr>

					<tr>
						<th scope="row"><?php esc_html_e( 'Toolbar Features', 'accessibar' ); ?></th>
						<td>
							<fieldset>
								<?php foreach ( $tool_labels as $key => $label ) : ?>
									<label>
										<input type="checkbox"
											name="<?php echo esc_attr( ACCESSIBAR_OPTION_KEY ); ?>[tools][<?php echo esc_attr( $key ); ?>]"
											value="1"
											<?php checked( ! empty( $options['tools'][ $key ] ) ); ?> />
										<?php echo esc_html( $label ); ?>
									</label><br>
								<?php endforeach; ?>
							</fieldset>
						</td>
					</tr>

					<tr>
						<th scope="row"><?php esc_html_e( 'Hide on Mobile', 'accessibar' ); ?></th>
						<td>
							<label>
								<input type="checkbox" name="<?php echo esc_attr( ACCESSIBAR_OPTION_KEY ); ?>[hide_mobile]" value="1" <?php checked( $options['hide_mobile'] ); ?> />
								<?php esc_html_e( 'Hide the accessibility button on mobile devices', 'accessibar' ); ?>
							</label>
						</td>
					</tr>

				</table>

				<?php submit_button(); ?>
			</form>
		</div>
		<?php
	}
}
