(function () {
	'use strict';

	var STORAGE_KEY = 'accessibar_settings';
	var config = window.accessibarConfig || {};
	var root = null;
	var panel = null;
	var toggleBtn = null;
	var isOpen = false;
	var fontLoaded = false;
	var focusableEls = [];
	var firstFocusable = null;
	var lastFocusable = null;
	var html = document.documentElement;

	var state = {
		fontSize: 0,
		highContrast: false,
		highlightLinks: false,
		readableFont: false,
		stopAnimations: false,
		bigCursor: false
	};

	function loadState() {
		try {
			var saved = localStorage.getItem(STORAGE_KEY);
			if (saved) {
				var parsed = JSON.parse(saved);
				for (var key in state) {
					if (parsed.hasOwnProperty(key)) {
						state[key] = parsed[key];
					}
				}
			}
		} catch (e) {}
	}

	function saveState() {
		try {
			localStorage.setItem(STORAGE_KEY, JSON.stringify(state));
		} catch (e) {}
	}

	function applyAll() {
		html.style.fontSize = state.fontSize !== 0
			? 'calc(1em + ' + state.fontSize + 'px)'
			: '';

		html.classList.toggle('accessibar-contrast', state.highContrast);
		html.classList.toggle('accessibar-links', state.highlightLinks);
		html.classList.toggle('accessibar-readable', state.readableFont);
		html.classList.toggle('accessibar-no-animations', state.stopAnimations);
		html.classList.toggle('accessibar-big-cursor', state.bigCursor);

		if (state.readableFont && !fontLoaded) {
			loadDyslexicFont();
		}

		syncToggles();
	}

	function loadDyslexicFont() {
		if (fontLoaded || !config.fontUrl) return;

		var style = document.createElement('style');
		style.textContent = '@font-face{font-family:"OpenDyslexic";src:url("' + config.fontUrl + '") format("woff2");font-display:swap;}';
		document.head.appendChild(style);
		fontLoaded = true;
	}

	function syncToggles() {
		if (!panel) return;

		var inputs = panel.querySelectorAll('[data-tool]');
		for (var i = 0; i < inputs.length; i++) {
			var input = inputs[i];
			var tool = input.getAttribute('data-tool');
			if (tool && state.hasOwnProperty(tool)) {
				input.checked = !!state[tool];
			}
		}
	}

	function createIcon() {
		var svg = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
		svg.setAttribute('viewBox', '0 0 24 24');
		svg.setAttribute('aria-hidden', 'true');
		svg.setAttribute('focusable', 'false');
		svg.innerHTML = '<circle cx="12" cy="4" r="2"/><path d="M5 8c0-.6.4-1 1-1h12c.6 0 1 .4 1 1s-.4 1-1 1H6c-.6 0-1-.4-1-1z"/><rect x="11" y="8" width="2" height="5.5"/><path d="M12 13.5l-3.5 8h2l1.5-4.5 1.5 4.5h2l-3.5-8z"/>';
		return svg;
	}

	function createToggle() {
		var btn = document.createElement('button');
		btn.className = 'accessibar-toggle';
		btn.setAttribute('aria-label', config.i18n.open);
		btn.setAttribute('aria-expanded', 'false');
		btn.setAttribute('aria-controls', 'accessibar-panel');
		btn.appendChild(createIcon());

		if (config.buttonText) {
			btn.classList.add('accessibar-toggle--with-text');
			var span = document.createElement('span');
			span.className = 'accessibar-toggle__text';
			span.textContent = config.buttonText;
			btn.appendChild(span);
		}

		btn.addEventListener('click', function () {
			if (isOpen) {
				closePanel();
			} else {
				openPanel();
			}
		});

		return btn;
	}

	function createSwitch(toolKey, label) {
		var item = document.createElement('div');
		item.className = 'accessibar-tool';

		var lbl = document.createElement('span');
		lbl.className = 'accessibar-tool__label';
		lbl.textContent = label;
		lbl.id = 'accessibar-label-' + toolKey;

		var sw = document.createElement('label');
		sw.className = 'accessibar-switch';

		var input = document.createElement('input');
		input.type = 'checkbox';
		input.setAttribute('data-tool', toolKey);
		input.setAttribute('aria-labelledby', lbl.id);
		input.checked = !!state[toolKey];

		input.addEventListener('change', function () {
			state[toolKey] = input.checked;
			saveState();
			applyAll();
		});

		var slider = document.createElement('span');
		slider.className = 'accessibar-switch__slider';

		sw.appendChild(input);
		sw.appendChild(slider);
		item.appendChild(lbl);
		item.appendChild(sw);
		return item;
	}

	function createFontSizeControl() {
		var item = document.createElement('div');
		item.className = 'accessibar-tool';

		var lbl = document.createElement('span');
		lbl.className = 'accessibar-tool__label';
		lbl.textContent = config.toolLabels.font_size;

		var controls = document.createElement('div');
		controls.className = 'accessibar-fontsize';
		controls.setAttribute('role', 'group');
		controls.setAttribute('aria-label', config.toolLabels.font_size);

		var minus = document.createElement('button');
		minus.className = 'accessibar-fontsize__btn';
		minus.textContent = 'A\u2212';
		minus.setAttribute('aria-label', config.i18n.decrease);
		minus.addEventListener('click', function () {
			if (state.fontSize > -4) {
				state.fontSize -= 2;
				saveState();
				applyAll();
			}
		});

		var reset = document.createElement('button');
		reset.className = 'accessibar-fontsize__btn';
		reset.textContent = 'A';
		reset.setAttribute('aria-label', config.i18n.resetFs);
		reset.addEventListener('click', function () {
			state.fontSize = 0;
			saveState();
			applyAll();
		});

		var plus = document.createElement('button');
		plus.className = 'accessibar-fontsize__btn';
		plus.textContent = 'A+';
		plus.setAttribute('aria-label', config.i18n.increase);
		plus.addEventListener('click', function () {
			if (state.fontSize < 10) {
				state.fontSize += 2;
				saveState();
				applyAll();
			}
		});

		controls.appendChild(minus);
		controls.appendChild(reset);
		controls.appendChild(plus);
		item.appendChild(lbl);
		item.appendChild(controls);
		return item;
	}

	function createPanel() {
		var el = document.createElement('div');
		el.className = 'accessibar-panel';
		el.id = 'accessibar-panel';
		el.setAttribute('role', 'dialog');
		el.setAttribute('aria-label', config.i18n.panelName);

		var header = document.createElement('div');
		header.className = 'accessibar-panel__header';

		var title = document.createElement('span');
		title.className = 'accessibar-panel__title';
		title.textContent = 'AccessiBar';

		var closeBtn = document.createElement('button');
		closeBtn.className = 'accessibar-panel__close';
		closeBtn.textContent = '\u00D7';
		closeBtn.setAttribute('aria-label', config.i18n.close);
		closeBtn.addEventListener('click', closePanel);

		header.appendChild(title);
		header.appendChild(closeBtn);
		el.appendChild(header);

		var tools = config.tools || {};
		var labels = config.toolLabels || {};

		if (tools.font_size) {
			el.appendChild(createFontSizeControl());
		}

		var toggleTools = [
			['highContrast', 'high_contrast'],
			['highlightLinks', 'highlight_links'],
			['readableFont', 'readable_font'],
			['stopAnimations', 'stop_animations'],
			['bigCursor', 'big_cursor']
		];

		for (var i = 0; i < toggleTools.length; i++) {
			var stateKey = toggleTools[i][0];
			var configKey = toggleTools[i][1];
			if (tools[configKey]) {
				el.appendChild(createSwitch(stateKey, labels[configKey] || configKey));
			}
		}

		var resetBtn = document.createElement('button');
		resetBtn.className = 'accessibar-reset';
		resetBtn.textContent = config.i18n.reset;
		resetBtn.addEventListener('click', function () {
			state.fontSize = 0;
			state.highContrast = false;
			state.highlightLinks = false;
			state.readableFont = false;
			state.stopAnimations = false;
			state.bigCursor = false;
			try { localStorage.removeItem(STORAGE_KEY); } catch (e) {}
			applyAll();
		});

		el.appendChild(resetBtn);
		return el;
	}

	function updateFocusableList() {
		if (!panel) return;
		focusableEls = panel.querySelectorAll('button, input, [tabindex]:not([tabindex="-1"])');
		firstFocusable = focusableEls[0] || null;
		lastFocusable = focusableEls[focusableEls.length - 1] || null;
	}

	function openPanel() {
		isOpen = true;
		panel.classList.add('accessibar-panel--open');
		toggleBtn.setAttribute('aria-expanded', 'true');
		toggleBtn.setAttribute('aria-label', config.i18n.close);
		updateFocusableList();
		if (firstFocusable) firstFocusable.focus();
	}

	function closePanel() {
		isOpen = false;
		panel.classList.remove('accessibar-panel--open');
		toggleBtn.setAttribute('aria-expanded', 'false');
		toggleBtn.setAttribute('aria-label', config.i18n.open);
		toggleBtn.focus();
	}

	function handleKeydown(e) {
		if (e.key === 'Escape' && isOpen) {
			e.preventDefault();
			closePanel();
			return;
		}

		if (e.key === 'Tab' && isOpen) {
			updateFocusableList();
			if (focusableEls.length === 0) return;

			if (e.shiftKey) {
				if (document.activeElement === firstFocusable) {
					e.preventDefault();
					lastFocusable.focus();
				}
			} else {
				if (document.activeElement === lastFocusable) {
					e.preventDefault();
					firstFocusable.focus();
				}
			}
		}
	}

	function handleClickOutside(e) {
		if (isOpen && root && !root.contains(e.target)) {
			closePanel();
		}
	}

	function init() {
		if (config.hideMobile && window.innerWidth <= 768) return;

		root = document.getElementById('accessibar-root');
		if (!root) return;

		toggleBtn = createToggle();
		panel = createPanel();

		root.appendChild(panel);
		root.appendChild(toggleBtn);

		syncToggles();

		document.addEventListener('keydown', handleKeydown);
		document.addEventListener('click', handleClickOutside);
	}

	loadState();
	applyAll();

	if (document.readyState === 'loading') {
		document.addEventListener('DOMContentLoaded', init);
	} else {
		init();
	}
})();
