<?php
/**
 * Admin Settings Class
 *
 * @package Change_Login_URL
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class ADALC_Admin
 */
class ADALC_Admin {

    /**
     * Option name for settings.
     *
     * @var string
     */
    private $option_name = 'ADALC_settings';

    /**
     * Constructor.
     */
    public function __construct() {
        if ( ! is_multisite() ) {
            add_action( 'admin_menu', array( $this, 'add_settings_page' ) );
        }
        add_action( 'admin_init', array( $this, 'register_settings' ) );
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );
        add_action( 'admin_notices', array( $this, 'display_admin_notices' ) );
        add_filter( 'plugin_action_links_' . plugin_basename( ADALC_PATH . 'accessdoor-smart-admin-login-url-control.php' ), array( $this, 'add_settings_link' ) );
        // Flush permalinks on admin_init if needed
        add_action( 'admin_init', function() {
            if ( get_transient( 'ADALC_permalinks_flush_needed' ) ) {
                flush_rewrite_rules();
                delete_transient( 'ADALC_permalinks_flush_needed' );
            }
        });

        // add action update_option_ADALC_settings to apply htaccess changes immediately when security settings are updated
        add_action( 'update_option_' . $this->option_name, array( $this, 'handle_settings_update' ), 10, 2 );
        // Apply some security services on init
        add_action( 'init', array( $this, 'apply_security_services' ), 20 );
        // Apply some security services early on plugins_loaded
        add_action( 'plugins_loaded', array( $this, 'apply_early_security_services' ), 20 );

        //rest api init hook for rest api disabling
        add_action( 'rest_api_init', array( $this, 'apply_rest_api_disabling' ), 0 );
    }

    //function to disable rest api for non-authenticated users
    public function apply_rest_api_disabling() {
        
            $settings = get_option( 'ADALC_settings', [] );
            if ( ! empty( $settings['disable_rest_api'] ) ) {
                add_filter( 'rest_authentication_errors', function( $result ) {
                    if ( ! empty( $result ) ) {
                        return $result;
                    }
                    if ( ! is_user_logged_in() ) {
                        return new WP_Error( 'rest_cannot_access', __( 'REST API restricted to authenticated users only.', 'accessdoor-smart-admin-login-url-control' ), array( 'status' => 401 ) );
                    }
                    return $result;
                });
            }
        
    }

    //function to handle settings update for xmlrpc services
    public function apply_early_security_services() {
        $settings = get_option( 'ADALC_settings', [] );

        // Disable File Editing in Dashboard
        if ( isset( $settings['disable_file_editing'] ) && $settings['disable_file_editing'] ) {
            if ( ! defined( 'DISALLOW_FILE_EDIT' ) ) {
                define( 'DISALLOW_FILE_EDIT', true );
            }
        }

        //disable xmlrpc completely
        if ( ! empty( $settings['disable_xmlrpc'] ) ) {
            add_filter( 'xmlrpc_enabled', '__return_false' );
             if ( defined( 'XMLRPC_REQUEST' ) && XMLRPC_REQUEST ) {
                status_header( 403 );
                header( 'Content-Type: text/plain; charset=UTF-8' );
                echo 'XML-RPC is disabled.';
                exit;
            }
        }
        // Disable XML-RPC Pingbacks
        if ( ! empty( $settings['disable_xmlrpc_pingback'] ) ) {

            add_filter( 'xmlrpc_methods', function( $methods ) {
                unset( $methods['pingback.ping'] );
                unset( $methods['pingback.extensions.getPingbacks'] );
                return $methods;
            });

            add_action( 'xmlrpc_call', function( $method ) {
                if ( strpos( $method, 'pingback' ) !== false ) {
                    status_header( 403 );
                    header( 'Content-Type: text/plain; charset=UTF-8' );
                    echo 'Pingbacks are disabled.';
                    exit;
                }
            });
        }

    }

    //function handle_settings_update to check if any security-related settings were changed and apply changes immediately
    public function handle_settings_update( $old_value, $new_value ) {
        // print_r($old_value);       

        // Stop if no security settings changed
        if ( $old_value === $new_value ) {
            return;
        }

        if ( ! function_exists( 'request_filesystem_credentials' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }

        global $wp_filesystem;

        if ( empty( $wp_filesystem ) ) {
            $creds = request_filesystem_credentials( '', '', false, false, null );
            if ( ! WP_Filesystem( $creds ) ) {
                return;
            }
        }

        /**
         * -----------------------------
         * BLOCK DIRECTORY BROWSING
         * -----------------------------
         */
        $htaccess_path = ABSPATH . '.htaccess';

        if ( $wp_filesystem->exists( $htaccess_path ) && $wp_filesystem->is_writable( $htaccess_path ) ) {
            $htaccess = $wp_filesystem->get_contents( $htaccess_path );

            $block = "\n# BEGIN Accessdoor Directory Protection\nOptions -Indexes\n# END Accessdoor Directory Protection\n";

            // Enable
            if ( ! empty( $new_value['block_directory_browsing'] ) ) {
                if ( strpos( $htaccess, 'BEGIN Accessdoor Directory Protection' ) === false ) {
                    $htaccess .= $block;
                }
            }
            // Disable
            else {
                $htaccess = preg_replace( '/# BEGIN Accessdoor Directory Protection.*?# END Accessdoor Directory Protection\n?/s', '', $htaccess );
            }

            $wp_filesystem->put_contents( $htaccess_path, $htaccess, FS_CHMOD_FILE );
        }

        /**
         * -----------------------------
         * BLOCK PHP IN UPLOADS
         * -----------------------------
         */
        $uploads_dir = wp_get_upload_dir();
        $uploads_path = trailingslashit( $uploads_dir['basedir'] );
        $uploads_htaccess = $uploads_path . '.htaccess';

        $php_block = "# BEGIN Accessdoor Upload Protection\n<FilesMatch \.php$>\nDeny from all\n</FilesMatch>\n# END Accessdoor Upload Protection\n";

        $existing = $wp_filesystem->exists( $uploads_htaccess )
            ? $wp_filesystem->get_contents( $uploads_htaccess )
            : '';

        // Enable
        if ( ! empty( $new_value['forbid_php_uploads'] ) ) {
            if ( strpos( $existing, '# BEGIN Accessdoor Upload Protection' ) === false ) {
                // Remove any old block first, then append the new one
                $cleaned = preg_replace( '/# BEGIN Accessdoor Upload Protection.*?# END Accessdoor Upload Protection\n?/s', '', $existing );
                $wp_filesystem->put_contents( $uploads_htaccess, $cleaned . $php_block, FS_CHMOD_FILE );
            }
        }
        // Disable
        else {
            $cleaned = preg_replace( '/# BEGIN Accessdoor Upload Protection.*?# END Accessdoor Upload Protection\n?/s', '', $existing );
            $wp_filesystem->put_contents( $uploads_htaccess, $cleaned, FS_CHMOD_FILE );
        }
        
    }


    /**
     * Apply selected security services based on plugin settings.
     *
     * @since 1.0.0
     */
    public function apply_security_services() {

        $settings = get_option( $this->option_name, array() );

        // Disable Comments
        if ( isset( $settings['disable_comments'] ) && $settings['disable_comments'] ) {
            add_filter( 'comments_open', '__return_false', 20, 2 );
            add_filter( 'pings_open', '__return_false', 20, 2 );
        }

        // Turn Off Pingbacks (XML-RPC + for all posts)
        if ( isset( $settings['turn_off_pingbacks'] ) && $settings['turn_off_pingbacks'] ) {

            // 1. Disable pingback & trackback XML-RPC methods
            add_filter( 'xmlrpc_methods', function( $methods ) {
                unset( $methods['pingback.ping'] );
                unset( $methods['pingback.extensions.getPingbacks'] );
                return $methods;
            });

            // 2. Remove X-Pingback HTTP header
            add_filter( 'wp_headers', function( $headers ) {
                unset( $headers['X-Pingback'] );
                return $headers;
            });

            // 3. Remove pingback discovery links from <head>
            remove_action( 'wp_head', 'rsd_link' );
            remove_action( 'wp_head', 'wlwmanifest_link' );

            // 4. Disable pingbacks & trackbacks in settings (default behavior)
            add_filter( 'pre_option_default_ping_status', '__return_zero' );
            add_filter( 'pre_option_default_comment_status', '__return_zero' );

            // 5. Disable trackback rewrite rules
            add_filter( 'rewrite_rules_array', function( $rules ) {
                foreach ( $rules as $rule => $rewrite ) {
                    if ( strpos( $rewrite, 'trackback' ) !== false ) {
                        unset( $rules[ $rule ] );
                    }
                }
                return $rules;
            });

            // 6. Force close pings on all existing posts
            // add_filter( 'comments_open', function( $open, $post_id ) {
            //     return false;
            // }, 10, 2 );

            add_filter( 'pings_open', function() {
                return false;
            });
        }

        
    }

    /**
     * Add settings page to WordPress admin menu.
     */
    public function add_settings_page() {

        // Do nothing if multisite, so no settings page is added
        if ( is_multisite() ) {
            return;
        }
        add_options_page(
            __( 'AccessDoor Smart Admin Login & Security', 'accessdoor-smart-admin-login-url-control' ),
            __( 'AccessDoor Smart Admin Login & Security', 'accessdoor-smart-admin-login-url-control' ),
            'manage_options',
            'accessdoor-smart-admin-login-url-control',
            array( $this, 'render_settings_page' )
        );
    }

    /**
     * Register plugin settings.
     */
    public function register_settings() {
        // Register the same option for both tabs with proper sanitization
        register_setting(
            'ADALC_general_group',
            $this->option_name,
            array( $this, 'sanitize_settings' )
        );
        
        register_setting(
            'ADALC_branding_group',
            $this->option_name,
            array( $this, 'sanitize_settings' )
        );

        register_setting(
            'ADALC_admin_group',
            $this->option_name,
            array( $this, 'sanitize_settings' )
        );

        // General Settings Section.
        add_settings_section(
            'ADALC_general_section',
            __( 'General Settings', 'accessdoor-smart-admin-login-url-control' ),
            array( $this, 'general_section_callback' ),
            'adalc-general-settings'
        );

        add_settings_field(
            'enable_custom_login',
             '<i class="fa-solid fa-door-open"></i>'. __( 'Enable Custom Login URL', 'accessdoor-smart-admin-login-url-control' ),
            array( $this, 'enable_custom_login_callback' ),
            'adalc-general-settings',
            'ADALC_general_section'
        );

        add_settings_field(
            'role_login_urls',
            '<span class="adalc-tooltip-heading"><i class="fa-solid fa-users-gear
        "></i>' . __( 'Role-Based Login URLs', 'accessdoor-smart-admin-login-url-control' ) .
                ' <span class="dashicons dashicons-info-outline" style="font-size:16px;vertical-align:middle;"></span>' .
                '<span class="adalc-tooltip-text">' . __( 'Set a custom login slug for each role. Leave empty to use the Administrator login URL.', 'accessdoor-smart-admin-login-url-control' ) . '</span>' .
            '</span>',
            array( $this, 'role_login_urls_callback' ),
            'adalc-general-settings',
            'ADALC_general_section'
        );

        // Redirect Settings Section.
        add_settings_section(
            'ADALC_redirect_section',
            __( 'Redirect wp-login.php', 'accessdoor-smart-admin-login-url-control' ),
            array( $this, 'redirect_section_callback' ),
            'adalc-general-settings'
        );

        add_settings_field(
            'redirect_wp_login',
            '<i class="fa-solid fa-right-to-bracket"></i>'. __( 'Redirect Default WordPress Login', 'accessdoor-smart-admin-login-url-control' ),
            array( $this, 'redirect_wp_login_callback' ),
            'adalc-general-settings',
            'ADALC_redirect_section'
        );

        add_settings_field(
            'redirect_url',
            '<span class="adalc-tooltip-heading">' . __( 'Custom Redirect URL', 'accessdoor-smart-admin-login-url-control' ) .
                ' <span class="dashicons dashicons-info-outline" style="font-size:16px;vertical-align:middle;"></span>' .
                '<span class="adalc-tooltip-text">' . __( 'Set the URL where users will be redirected if they try to access wp-login.php directly.', 'accessdoor-smart-admin-login-url-control' ) . '</span>' .
            '</span>',
            array( $this, 'redirect_url_callback' ),
            'adalc-general-settings',
            'ADALC_redirect_section'
        );

        // Custom Branding Section.
        add_settings_section(
            'ADALC_branding_section',
            __( 'Custom Branding per Role', 'accessdoor-smart-admin-login-url-control' ),
            array( $this, 'branding_section_callback' ),
            'adalc-branding-settings'
        );

        add_settings_field(
            'custom_branding',
            '',
            array( $this, 'role_branding_callback' ),
            'adalc-branding-settings',
            'ADALC_branding_section'
        );

        // Change Admin Info.
        add_settings_section(
            'ADALC_admin_info_section',
            __( 'Change Admin Info', 'accessdoor-smart-admin-login-url-control' ),
            array( $this, 'admin_info_section_callback' ),
            'adalc-admin-settings'
        );

        // Move admin email field to Admin Settings tab (last/third tab)
        add_settings_field(
            'change_admin_email',
            '<i class="fa-solid fa-envelope-circle-check"></i>'. __( 'Change Admin Email Address', 'accessdoor-smart-admin-login-url-control' ),
            array( $this, 'admin_email_field_callback' ),
            'adalc-admin-settings',
            'ADALC_admin_info_section'
        );

        // User email and username remain in Admin Settings
        add_settings_field(
            'change_user_email',
            '<i class="fa-solid fa-envelope-circle-check"></i>'. __( 'Change Admin User Email', 'accessdoor-smart-admin-login-url-control' ),
            array( $this, 'current_user_email_field_callback' ),
            'adalc-admin-settings',
            'ADALC_admin_info_section'
        );

        add_settings_field(
            'change_admin_username',
            '<i class="fa-solid fa-user-pen"></i>'. __( 'Change Admin Username', 'accessdoor-smart-admin-login-url-control' ),
            array( $this, 'admin_username_field_callback' ),
            'adalc-admin-settings',
            'ADALC_admin_info_section'
        );
            

            // Security Measures Section (new tab)
            register_setting(
                'ADALC_security_group',
                $this->option_name,
                array( $this, 'sanitize_settings' )
            );
            

            add_settings_section(
                'ADALC_security_section',
                __( 'Security Measures', 'accessdoor-smart-admin-login-url-control' ),
                array( $this, 'security_section_callback' ),
                'adalc-security-settings'
            );

            // Security options fields
            add_settings_field(
                'disable_comments',
                __( 'Disable Comments', 'accessdoor-smart-admin-login-url-control' ),
                array( $this, 'disable_comments_callback' ),
                'adalc-security-settings',
                'ADALC_security_section'
            );

            add_settings_field(
                'disable_xmlrpc_pingback',
                __( 'Disable XML-RPC Pingback', 'accessdoor-smart-admin-login-url-control' ),
                array( $this, 'disable_xmlrpc_pingback_callback' ),
                'adalc-security-settings',
                'ADALC_security_section'
            );

            add_settings_field(
                'disable_rest_api',
                __( 'Disable REST API', 'accessdoor-smart-admin-login-url-control' ),
                array( $this, 'disable_rest_api_callback' ),
                'adalc-security-settings',
                'ADALC_security_section'
            );

            add_settings_field(
                'disable_xmlrpc',
                __( 'Disable XML-RPC', 'accessdoor-smart-admin-login-url-control' ),
                array( $this, 'disable_xmlrpc_callback' ),
                'adalc-security-settings',
                'ADALC_security_section'
            );

            // Block Directory Browsing
            add_settings_field(
                'block_directory_browsing',
                __( 'Block Directory Browsing', 'accessdoor-smart-admin-login-url-control' ),
                array( $this, 'block_directory_browsing_callback' ),
                'adalc-security-settings',
                'ADALC_security_section'
            );

            // Forbid PHP in Uploads
            add_settings_field(
                'forbid_php_uploads',
                __( 'Forbid PHP in Uploads', 'accessdoor-smart-admin-login-url-control' ),
                array( $this, 'forbid_php_uploads_callback' ),
                'adalc-security-settings',
                'ADALC_security_section'
            );

            // Turn Off Pingbacks
            add_settings_field(
                'turn_off_pingbacks',
                __( 'Turn Off Pingbacks', 'accessdoor-smart-admin-login-url-control' ),
                array( $this, 'turn_off_pingbacks_callback' ),
                'adalc-security-settings',
                'ADALC_security_section'
            );

            // Disable File Editing
            add_settings_field(
                'disable_file_editing',
                __( 'Disable File Editing in Dashboard', 'accessdoor-smart-admin-login-url-control' ),
                array( $this, 'disable_file_editing_callback' ),
                'adalc-security-settings',
                'ADALC_security_section'
            );
      
        
    }

    /**
     * Get all roles that have at least one user.
     *
     * @return array Array of role slugs and names with user counts.
     */
    public function get_roles_with_users() {
        $roles_with_users = array();
        $user_counts      = count_users();
        $all_roles        = wp_roles()->get_names();

        foreach ( $user_counts['avail_roles'] as $role_slug => $count ) {
            if ( $count > 0 && isset( $all_roles[ $role_slug ] ) ) {
                $roles_with_users[ $role_slug ] = array(
                    'name'  => $all_roles[ $role_slug ],
                    'count' => $count,
                );
            }
        }

        return $roles_with_users;
    }

    /**
     * Role-based login URLs callback.
     */
    public function role_login_urls_callback() {
        $settings = get_option( $this->option_name, array() );
        $roles    = $this->get_roles_with_users();

        if ( empty( $roles ) ) {
            echo '<p>' . esc_html__( 'No roles with users found.', 'accessdoor-smart-admin-login-url-control' ) . '</p>';
            return;
        }
        echo '<div class="calu-table-wrapper">';
        echo '<table class="calu-custom-table">';
        echo '<thead>';
        echo '<tr>';
        echo '<th>' . esc_html__( 'Role', 'accessdoor-smart-admin-login-url-control' ) . '</th>';
        echo '<th>' . esc_html__( 'Custom Login Slug', 'accessdoor-smart-admin-login-url-control' ) . '</th>';
        echo '<th>' . esc_html__( 'Login URL', 'accessdoor-smart-admin-login-url-control' ) . '</th>';
        echo '<th>' . esc_html__( 'Users', 'accessdoor-smart-admin-login-url-control' ) . '</th>';

        echo '</tr>';
        echo '</thead>';
        echo '<tbody>';

        foreach ( $roles as $role_slug => $role_data ) {
            $slug_value = isset( $settings['role_slugs'][ $role_slug ] ) ? $settings['role_slugs'][ $role_slug ] : '';
            $login_url  = $slug_value ? home_url( $slug_value ) : __( 'Not set (Uses administrator login URL)', 'accessdoor-smart-admin-login-url-control' );
            if($role_data['name'] == 'Administrator'){
                $login_url  = $slug_value ? home_url( $slug_value ) : __( 'Not set', 'accessdoor-smart-admin-login-url-control' );
            }
            echo '<tr>';
            echo '<td><strong>' . esc_html( $role_data['name'] ) . '</strong></td>';
            echo '<td>';
            printf(
                '<input type="text" name="%s[role_slugs][%s]" value="%s" class="calu-slug-input" placeholder="%s" />',
                esc_attr( $this->option_name ),
                esc_attr( $role_slug ),
                esc_attr( $slug_value ),
                esc_attr( strtolower( str_replace( ' ', '-', $role_data['name'] ) ) . '-login' )
            );
            echo '</td>';
            $url_to_copy = is_string( $login_url ) ? $login_url : home_url( $slug_value );
            echo '<td>';
            echo '<code class="calu-url-display" id="calu-url-' . esc_attr( $role_slug ) . '">' . esc_html( $url_to_copy ) . '</code>';
            echo ' <button type="button" class="calu-copy-url" data-url="' . esc_attr( esc_url( $url_to_copy ) ) . '" data-target="calu-url-' . esc_attr( $role_slug ) . '" style="background:none;border:none;cursor:pointer;" title="Copy URL">';
            echo '<span class="dashicons dashicons-admin-page"></span>';
            echo '</button>';
            echo '</td>';
            echo '<td><span class="calu-user-count">' . esc_html( $role_data['count'] ) . '</span></td>';
            echo '</tr>';
        }

        echo '</tbody>';
        echo '</table>';
        echo '<p class="description">
            <img src="' . esc_url( plugin_dir_url( __DIR__ ) . 'assets/images/error-icon-25239.png' ) . '" alt="" />' . esc_html__( 'Do not use any of the following slugs as custom login URLs:', 'accessdoor-smart-admin-login-url-control' ) . '
wp-admin, wp-login, wp-login.php, login, admin, logout, lostpassword, resetpassword, register, wp-json, xmlrpc, xmlrpc.php, my-account, cart, checkout, shop, account, profile, signin, signup, etc.
        </p>';
        echo '</div>';
    }

    /**
     * Sanitize settings before saving.
     *
     * @param array $input Settings input.
     * @return array Sanitized settings.
     */
    public function sanitize_settings( $input ) {
   

        
        // Get existing settings to merge with new input
        $existing = get_option( $this->option_name, array() );
        $sanitized = $existing; // Start with existing data

        if ( isset( $input['enable_custom_login'] ) ) {
            $sanitized['enable_custom_login'] = 1;
        } elseif ( isset( $input['redirect_wp_login'] ) || isset( $input['role_slugs'] ) ) {
            // Only set to 0 if we're saving the general tab (has these fields)
            $sanitized['enable_custom_login'] = 0;
        }

        // Sanitize role-based slugs and block restricted/existing slugs.
        if ( isset( $input['role_slugs'] ) && is_array( $input['role_slugs'] ) ) {
            $sanitized['role_slugs'] = array();
            $restricted_slugs = array(
                'wp-admin', 'wp-login', 'wp-login.php', 'login', 'admin',
                'logout', 'lostpassword', 'resetpassword', 'register',
                'wp-json', 'xmlrpc', 'xmlrpc.php',
                'my-account', 'cart', 'checkout', 'shop',
                'account', 'profile', 'signin', 'signup',
            );
            foreach ( $input['role_slugs'] as $role => $slug ) {
                if ( empty( $slug ) ) {
                    continue;
                }
                $slug = sanitize_title( $slug );
                // Block restricted slugs
                if ( in_array( $slug, $restricted_slugs, true ) ) {
                    
                    add_settings_error(
                        $this->option_name,
                        'restricted_slug_' . $role,
                        /* translators: %s: The slug entered by the user */
                        sprintf( __( 'The slug "%s" is not allowed for security reasons.', 'accessdoor-smart-admin-login-url-control' ), esc_html( $slug ) ),
                        'error'
                    );
                    continue;
                }
                // Block if slug matches an existing page
                if ( get_page_by_path( $slug ) ) {
                        
                    add_settings_error(
                        $this->option_name,
                        'existing_page_slug_' . $role,
                        /* translators: %s: The slug entered by the user */
                        sprintf( __( 'The slug "%s" is already used by a page. Please choose another.', 'accessdoor-smart-admin-login-url-control' ), esc_html( $slug ) ),
                        'error'
                    );
                    continue;
                }
                $sanitized['role_slugs'][ $role ] = $slug;
            }
        }

        if ( isset( $input['redirect_wp_login'] ) ) {
            $sanitized['redirect_wp_login'] = 1;
        } elseif ( isset( $input['role_slugs'] ) ) {
            // Only set to 0 if we're saving the general tab
            $sanitized['redirect_wp_login'] = 0;
        }

        if ( isset( $input['redirect_url'] ) ) {
            $sanitized['redirect_url'] = esc_url_raw( $input['redirect_url'] );
        }

        // Sanitize role logo IDs.
        if ( isset( $input['role_logo'] ) && is_array( $input['role_logo'] ) ) {
            if ( ! isset( $sanitized['role_logo'] ) ) {
                $sanitized['role_logo'] = array();
            }
            foreach ( $input['role_logo'] as $role => $attachment_id ) {
                if ( ! empty( $attachment_id ) ) {
                    $sanitized['role_logo'][ $role ] = absint( $attachment_id );
                } else {
                    // Remove if empty
                    unset( $sanitized['role_logo'][ $role ] );
                }
            }
        }

        // Save Security tab options, but validate .htaccess changes for relevant features
        $security_fields = array(
            'disable_comments',
            'disable_xmlrpc_pingback',
            'disable_rest_api',
            'disable_xmlrpc',
            'block_directory_browsing',
            'forbid_php_uploads',
            'turn_off_pingbacks',
            'disable_file_editing',
        );
        foreach ( $security_fields as $field ) {
            // Default: set from input
            $sanitized[$field] = isset($input[$field]) ? 1 : 0;
        }

        // Sanitize role background image IDs.
        if ( isset( $input['role_bg_image'] ) && is_array( $input['role_bg_image'] ) ) {
            if ( ! isset( $sanitized['role_bg_image'] ) ) {
                $sanitized['role_bg_image'] = array();
            }
            foreach ( $input['role_bg_image'] as $role => $attachment_id ) {
                if ( ! empty( $attachment_id ) ) {
                    $sanitized['role_bg_image'][ $role ] = absint( $attachment_id );
                } else {
                    // Remove if empty
                    unset( $sanitized['role_bg_image'][ $role ] );
                }
            }
        }

        // Sanitize role background colors.
        if ( isset( $input['role_bg_color'] ) && is_array( $input['role_bg_color'] ) ) {
            if ( ! isset( $sanitized['role_bg_color'] ) ) {
                $sanitized['role_bg_color'] = array();
            }
            foreach ( $input['role_bg_color'] as $role => $color ) {
                if ( ! empty( $color ) ) {
                    $sanitized['role_bg_color'][ $role ] = sanitize_hex_color( $color );
                } else {
                    // Remove if empty
                    unset( $sanitized['role_bg_color'][ $role ] );
                }
            }
        }

    if (
        isset( $input['change_admin_email'] ) ||
        isset( $input['change_user_email'] ) ||
        isset( $input['change_admin_username'] )
    ) {
        $current_user = wp_get_current_user();
        $user_id      = $current_user->ID;
        $is_admin     = user_can( $current_user, 'manage_options' );
        if ( empty( $user_id ) || ! $is_admin ) {
            return $sanitized;
        }

        // Track which fields were actually changed
        $changed_fields = array();

        /**
         * ---------------------------------------
         * 1. Update ADMIN EMAIL (Settings → General)
         * No confirmation email
         * ---------------------------------------
         */
        if ( isset( $input['change_admin_email'] ) ) {
            $new_admin_email     = sanitize_email( $input['change_admin_email'] );
            $current_admin_email = get_option( 'admin_email' );
            if($new_admin_email != $current_admin_email){
                if ( ! is_email( $new_admin_email ) ) {
                add_settings_error(
                    $this->option_name,
                    'invalid_admin_email',
                    __( 'Please enter a valid admin email address.', 'accessdoor-smart-admin-login-url-control' ),
                    'error'
                );
            }else {
                // Force update without confirmation, allow any email (even if used by another user)
                update_option( 'admin_email', $new_admin_email );
                delete_option( 'new_admin_email' );
                remove_action( 'update_option_admin_email', 'wp_send_new_admin_email', 10 );
                add_settings_error(
                    $this->option_name,
                    'admin_email_updated',
                    __( 'Admin email updated successfully.', 'accessdoor-smart-admin-login-url-control' ),
                    'updated'
                );
            }
                $sanitized['change_admin_email'] = $new_admin_email;
            }

        }

        /**
         * ---------------------------------------
         * 2. Update ADMIN USER EMAIL
         * ---------------------------------------
         */
        if ( isset( $input['change_user_email'] ) ) {

            $new_user_email = sanitize_email( $input['change_user_email'] );
            // Only process if username is changed
            if ( $new_user_email !== $current_user->user_email ) {

            if ( ! is_email( $new_user_email ) ) {

                add_settings_error(
                    $this->option_name,
                    'invalid_user_email',
                    __( 'Please enter a valid user email address.', 'accessdoor-smart-admin-login-url-control' ),
                    'error'
                );

                $sanitized['change_user_email'] = $current_user->user_email;

            } elseif ( email_exists( $new_user_email ) && $new_user_email !== $current_user->user_email ) {

                add_settings_error(
                    $this->option_name,
                    'user_email_exists',
                    __( 'This email is already in use by another user.', 'accessdoor-smart-admin-login-url-control' ),
                    'error'
                );

                $sanitized['change_user_email'] = $current_user->user_email;

            }else {

                $result = wp_update_user( array(
                    'ID'         => $user_id,
                    'user_email' => $new_user_email,
                ) );

                if ( is_wp_error( $result ) ) {

                    add_settings_error(
                        $this->option_name,
                        'user_email_update_failed',
                        __( 'Failed to update user email: ', 'accessdoor-smart-admin-login-url-control' ) . $result->get_error_message(),
                        'error'
                    );

                    $sanitized['change_user_email'] = $current_user->user_email;

                } else {

                    add_settings_error(
                        $this->option_name,
                        'user_email_updated',
                        __( 'User email updated successfully.', 'accessdoor-smart-admin-login-url-control' ),
                        'updated'
                    );

                    $sanitized['change_user_email'] = $new_user_email;
                }
            }
            }

        }

        /**
         * ---------------------------------------
         * 3. Update ADMIN USERNAME (Advanced)
         * ---------------------------------------
         */
        if ( isset( $input['change_admin_username'] ) ) {
            $new_username = sanitize_user( $input['change_admin_username'], true );
            // Only process if username is changed
            if ( $new_username !== $current_user->user_login ) {
                // Block for super admins in multisite
                if ( is_multisite() && is_super_admin( $user_id ) ) {
                    add_settings_error(
                        $this->option_name,
                        'superadmin_username_change_blocked',
                        __( 'Changing the username for a network (super) admin is not allowed for security reasons.', 'accessdoor-smart-admin-login-url-control' ),
                        'error'
                    );
                    $sanitized['change_admin_username'] = $current_user->user_login;
                } elseif ( empty( $new_username ) ) {
                    add_settings_error(
                        $this->option_name,
                        'invalid_username',
                        __( 'Username cannot be empty.', 'accessdoor-smart-admin-login-url-control' ),
                        'error'
                    );
                    $sanitized['change_admin_username'] = $current_user->user_login;
                } elseif ( username_exists( $new_username ) ) {
                    add_settings_error(
                        $this->option_name,
                        'username_exists',
                        __( 'This username already exists.', 'accessdoor-smart-admin-login-url-control' ),
                        'error'
                    );
                    $sanitized['change_admin_username'] = $current_user->user_login;
                } else {
                    global $wpdb;
                    $updated = $wpdb->update(
                        $wpdb->users,
                        array( 'user_login' => $new_username ),
                        array( 'ID' => $user_id ),
                        array( '%s' ),
                        array( '%d' )
                    );
                    if ( false === $updated ) {
                        add_settings_error(
                            $this->option_name,
                            'username_update_failed',
                            __( 'Failed to update username.', 'accessdoor-smart-admin-login-url-control' ),
                            'error'
                        );
                        $sanitized['change_admin_username'] = $current_user->user_login;
                    } else {
                        // Refresh login session
                        wp_clear_auth_cookie();
                        wp_set_current_user( $user_id );
                        wp_set_auth_cookie( $user_id );
                        add_settings_error(
                            $this->option_name,
                            'username_updated',
                            __( 'Username updated successfully.', 'accessdoor-smart-admin-login-url-control' ),
                            'updated'
                        );
                        $sanitized['change_admin_username'] = $new_username;
                    }
                }
            }
        }
    }


        // Improved: Only set transient if values actually changed (robust for unset/empty cases)
        $settings_changed = false;
        // Compare role_slugs
        $old_slugs = isset( $existing['role_slugs'] ) ? $existing['role_slugs'] : array();
        $new_slugs = isset( $sanitized['role_slugs'] ) ? $sanitized['role_slugs'] : array();
        if ( $old_slugs !== $new_slugs ) {
            $settings_changed = true;
        }
        // Compare enable_custom_login
        $old_enable = isset( $existing['enable_custom_login'] ) ? (int)$existing['enable_custom_login'] : 0;
        $new_enable = isset( $sanitized['enable_custom_login'] ) ? (int)$sanitized['enable_custom_login'] : 0;
        if ( $old_enable !== $new_enable ) {
            $settings_changed = true;
        }
        if ( $settings_changed ) {
            set_transient( 'ADALC_permalinks_flush_needed', true, 60 );
            set_transient( 'ADALC_permalinks_flushed', true, 30 );
        }

        return $sanitized;
    }

    /**
     * Display admin notices.
     */
    public function display_admin_notices() {
        static $notice_shown = false;
        if ( $notice_shown ) {
            return;
        }
        // Check if we just flushed permalinks.
        if ( get_transient( 'ADALC_permalinks_flushed' ) ) {
            delete_transient( 'ADALC_permalinks_flushed' );
            $notice_shown = true;
            ?>
            <div class="notice notice-success is-dismissible">
                <p>
                    <strong><?php esc_html_e( 'Permalinks Flushed!', 'accessdoor-smart-admin-login-url-control' ); ?></strong>
                </p>
            </div>
            <?php
        }
    }

    /**
     * General section callback.
     */
    public function general_section_callback() {
    //  echo '<p style="margin-left:15px;">' . esc_html__( 'Configure your custom login URL settings below.', 'accessdoor-smart-admin-login-url-control' ) . '</p>';
    }
    /**
     * Redirect settings section callback.
     */
    public function redirect_section_callback() {
        // echo '<p style="margin-left:32px;">' . esc_html__( 'Configure what happens when someone tries to access the default wp-login.php directly.', 'accessdoor-smart-admin-login-url-control' ) . '</p>';
    }

    /**
     * Branding settings section callback.
     */
    public function branding_section_callback() {
        // echo '<p style="margin-left:32px;">' . esc_html__( 'Customize the login page design for each user role. Set a custom logo and background.', 'accessdoor-smart-admin-login-url-control' ) . '</p>';
    }

     /**
     * Admin Information settings section callback.
     */
    public function admin_info_section_callback() {
        // echo '<p style="margin-left:32px;">' . esc_html__( 'Change Admin Information below.', 'accessdoor-smart-admin-login-url-control' ) . '</p>';
    }
    /**
     * Admin email change callback.
     */
        public function admin_email_field_callback() {
        $current_admin_email = get_option('admin_email');
        ?>
        <label>
            <input type="email" name="<?php echo esc_attr( $this->option_name ); ?>[change_admin_email]" value="<?php echo esc_attr( $current_admin_email ); ?>" />
        </label>
        <p class="description"><?php esc_html_e( 'This is the site admin email (Settings > General).', 'accessdoor-smart-admin-login-url-control' ); ?></p>
        <?php
    }

    /**
     * Current user email change callback.
     */
    public function current_user_email_field_callback() {
        $current_user = wp_get_current_user();
        $user_email = $current_user ? $current_user->user_email : '';
        ?>
        <label>
            <input type="email" name="<?php echo esc_attr( $this->option_name ); ?>[change_user_email]" value="<?php echo esc_attr( $user_email ); ?>" />
        </label>
        <p class="description"><?php esc_html_e( 'This is your user email (current logged-in admin).', 'accessdoor-smart-admin-login-url-control' ); ?></p>
        <?php
        }

    /**
     * Admin username change callback.
     */
    public function admin_username_field_callback() {
        $current_user = wp_get_current_user();
        $value = $current_user ? $current_user->user_login : '';
        ?>
        <label>
            <input type="text" name="<?php echo esc_attr( $this->option_name ); ?>[change_admin_username]" value="<?php echo esc_attr( $value ); ?>" />
        </label>
        <p class="description"><?php esc_html_e( 'This is your username (current logged-in admin). Please note: after changing the username, you will be logged out immediately and redirected to the homepage.', 'accessdoor-smart-admin-login-url-control' ); ?></p>
        <?php
    }

    /**
     * Get current admin username.
     */
    private function get_admin_username() {
        // Deprecated: replaced by current logged-in user logic
        return '';
    }

    /**
     * Enable custom login checkbox callback.
     */
    public function enable_custom_login_callback() {
        $settings = get_option( $this->option_name, array() );
        $value    = isset( $settings['enable_custom_login'] ) ? $settings['enable_custom_login'] : 0;
        ?>
        <div class="calu-toggle-wrapper">
        <label class="calu-switch">
            <input type="checkbox" name="<?php echo esc_attr( $this->option_name ); ?>[enable_custom_login]" value="1" <?php checked( 1, $value ); ?> />
            <span class="calu-slider"></span>
        </label>
        </div>  
        <p><?php esc_html_e('Turn the custom login URL feature on or off. When enabled, the default WordPress login page will be replaced with your custom login URL.', 'accessdoor-smart-admin-login-url-control'); ?></p>
    <?php
    }

    /**
     * Custom login slug field callback.
     */
    public function custom_login_slug_callback() {
        $settings = get_option( $this->option_name, array() );
        $value    = isset( $settings['custom_login_slug'] ) ? $settings['custom_login_slug'] : 'custom-login';
        ?>
        <input type="text" name="<?php echo esc_attr( $this->option_name ); ?>[custom_login_slug]" value="<?php echo esc_attr( $value ); ?>" class="regular-text" />
        <p class="description">
            <?php
            printf(
                /* translators: %s: Custom login URL */
                esc_html__( 'Your custom login URL will be: %s', 'accessdoor-smart-admin-login-url-control' ),
                '<code>' . esc_html( home_url( $value ) ) . '</code>'
            );
            ?>
        </p>
        <?php
    }

    /**
     * Redirect wp-login.php checkbox callback.
     */
    public function redirect_wp_login_callback() {
        $settings = get_option( $this->option_name, array() );
        $value    = isset( $settings['redirect_wp_login'] ) ? $settings['redirect_wp_login'] : 0;
        ?>
        <div class="calu-toggle-wrapper">
        <label class="calu-switch">
            <input type="checkbox" name="<?php echo esc_attr( $this->option_name ); ?>[redirect_wp_login]" value="1" <?php checked( 1, $value ); ?> />
            <span class="calu-slider"></span>
        </label>
        </div>
           <p class="simple-description"><?php esc_html_e('By default, anyone attempting to access the standard WordPress login URL will be shown a 404 Not Found page.
When this option is enabled, requests to the default login URL will be automatically redirected to the custom login URL specified below.', 'accessdoor-smart-admin-login-url-control'); ?> </p>
        <?php
    }

    /**
     * Redirect URL field callback.
     */
    public function redirect_url_callback() {
        $settings = get_option( $this->option_name, array() );
        $value    = isset( $settings['redirect_url'] ) ? $settings['redirect_url'] : home_url();
        ?>
        <input type="url" name="<?php echo esc_attr( $this->option_name ); ?>[redirect_url]" value="<?php echo esc_attr( $value ); ?>" class="regular-text" />
        <p class="description">
            <img src="<?php echo esc_url( plugin_dir_url( __DIR__ ) . 'assets/images/error-icon-25239.png' ); ?>" alt=""/><?php esc_html_e( 'Configure what happens when someone tries to access the default wp-login.php directly.', 'accessdoor-smart-admin-login-url-control' ); ?>

        </p>
        <?php
    }

    /**
     * Custom branding per role
     */
    public function role_branding_callback() {
        $settings = get_option( $this->option_name, array() );
        $roles = $this->get_roles_with_users();
        
        if ( empty( $roles ) ) {
            echo '<p>' . esc_html__( 'No roles with users found.', 'accessdoor-smart-admin-login-url-control' ) . '</p>';
            return;
        }
        
        $first_role = true;
        ?>
        <div class="calu-accordion-wrapper">
        <?php 
            foreach ( $roles as $role_slug => $role_data ) :
                $is_administrator = false;
                $logo_id = isset( $settings['role_logo'][ $role_slug ] ) ? $settings['role_logo'][ $role_slug ] : '';
                $bg_image_id = isset( $settings['role_bg_image'][ $role_slug ] ) ? $settings['role_bg_image'][ $role_slug ] : '';
                $bg_color = isset( $settings['role_bg_color'][ $role_slug ] ) ? $settings['role_bg_color'][ $role_slug ] : '#f0f0f1';
                
                $logo_url = $logo_id ? wp_get_attachment_url( $logo_id ) : '';
                $bg_image_url = $bg_image_id ? wp_get_attachment_url( $bg_image_id ) : '';
                if($role_data['name'] == 'Administrator'){
                    $is_administrator = true;
                }
            ?>
            <div class="calu-accordion-item <?php echo $first_role ? 'active' : ''; ?>">
                <div class="calu-accordion-header" data-role="<?php echo esc_attr( $role_slug ); ?>">
                    <span class="calu-accordion-title"><?php echo esc_html( $role_data['name'] ); ?></span>
                    <span class="calu-accordion-icon">▼</span>
                </div>
                <div class="calu-accordion-content">
                    <div class="calu-role-branding-box">
                        <div class="calu-setting-wrap">
                           <?php if ( !$is_administrator ) : ?>
                                <button type="button" class="button calu-copy-admin-design" data-role="<?php echo esc_attr( $role_slug ); ?>"><?php esc_html_e( 'Use Admin Login Design', 'accessdoor-smart-admin-login-url-control' ); ?></button>
                            <?php endif; ?>
                            <div>
                                <label><?php esc_html_e( 'Logo', 'accessdoor-smart-admin-login-url-control' ); ?></label>
                                <input type="hidden" class="calu-logo-id" name="<?php echo esc_attr( $this->option_name ); ?>[role_logo][<?php echo esc_attr( $role_slug ); ?>]" value="<?php echo esc_attr( $logo_id ); ?>" data-role="<?php echo esc_attr( $role_slug ); ?>" />
                                <button type="button" class="button calu-upload-logo" data-role="<?php echo esc_attr( $role_slug ); ?>"><?php esc_html_e( 'Upload Logo', 'accessdoor-smart-admin-login-url-control' ); ?></button>
                                <button type="button" class="button calu-remove-logo" data-role="<?php echo esc_attr( $role_slug ); ?>" style="<?php echo $logo_id ? '' : 'display:none;'; ?>"><?php esc_html_e( 'Remove', 'accessdoor-smart-admin-login-url-control' ); ?></button>
                                <div class="calu-logo-preview" data-role="<?php echo esc_attr( $role_slug ); ?>" style="margin-top: 10px;">
                                    <?php if ( $logo_url ) : ?>
                                        <img src="<?php echo esc_url( $logo_url ); ?>" style="max-width: 200px; height: auto; display: block;" />
                                    <?php endif; ?>
                                </div>
                            </div>
                            <div>
                                <label><?php esc_html_e( 'Background Image:', 'accessdoor-smart-admin-login-url-control' ); ?></label><br>
                                <input type="hidden" class="calu-bg-image-id" name="<?php echo esc_attr( $this->option_name ); ?>[role_bg_image][<?php echo esc_attr( $role_slug ); ?>]" value="<?php echo esc_attr( $bg_image_id ); ?>" data-role="<?php echo esc_attr( $role_slug ); ?>" />
                                <button type="button" class="button calu-upload-bg-image" data-role="<?php echo esc_attr( $role_slug ); ?>"><?php esc_html_e( 'Upload Background Image', 'accessdoor-smart-admin-login-url-control' ); ?></button>
                                <button type="button" class="button calu-remove-bg-image" data-role="<?php echo esc_attr( $role_slug ); ?>" style="<?php echo $bg_image_id ? '' : 'display:none;'; ?>"><?php esc_html_e( 'Remove', 'accessdoor-smart-admin-login-url-control' ); ?></button>
                                <span class="description"><?php esc_html_e( 'Or use background color below', 'accessdoor-smart-admin-login-url-control' ); ?></span>
                                <div class="calu-bg-image-preview" data-role="<?php echo esc_attr( $role_slug ); ?>" style="margin-top: 10px;">
                                    <?php if ( $bg_image_url ) : ?>
                                        <img src="<?php echo esc_url( $bg_image_url ); ?>" style="max-width: 200px; height: auto; display: block;" />
                                    <?php endif; ?>
                                </div>
                            </div> 
                            <div>
                                <label><?php esc_html_e( 'Background Color:', 'accessdoor-smart-admin-login-url-control' ); ?></label><br>
                                <input type="text" class="calu-color-value" name="<?php echo esc_attr( $this->option_name ); ?>[role_bg_color][<?php echo esc_attr( $role_slug ); ?>]" value="<?php echo esc_attr( $bg_color ); ?>" data-role="<?php echo esc_attr( $role_slug ); ?>" />
                                <input type="color" class="calu-color-picker" value="<?php echo esc_attr( $bg_color ); ?>" data-role="<?php echo esc_attr( $role_slug ); ?>" />
                                <button type="button" class="button calu-clear-color" data-role="<?php echo esc_attr( $role_slug ); ?>"><?php esc_html_e( 'Clear', 'accessdoor-smart-admin-login-url-control' ); ?></button>
                                <span class="description"><?php esc_html_e( 'Used if no background image is set', 'accessdoor-smart-admin-login-url-control' ); ?></span>
                            </div>
                        </div>
                        <div class="accessdoor-login-preview" data-role="<?php echo esc_attr( $role_slug ); ?>" >
                            <div class="login-card" style="background-image: <?php echo $bg_image_url ? 'url(' . esc_url( $bg_image_url ) . ')' : 'none'; ?>; background-color: <?php echo esc_attr( $bg_color ); ?>;">
                                <?php if ( $logo_url ) : ?>
                                    <img class="login-logo" src="<?php echo esc_url( $logo_url ); ?>" />
                                <?php else : ?>
                                    <img class="login-logo" src="<?php echo esc_url( plugin_dir_url( __DIR__ ) . 'assets/images/wordpress-logo.svg' ); ?>" />
                                <?php endif; ?>
                                <div class="login-form-preview">
                                    <label>Username or Email Address </label>
                                    <input type="text" disabled>
                                    <label>Password </label>
                                    <input type="password"  disabled>
                                    <button disabled>Log In</button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <?php 
                $first_role = false;
            endforeach; 
            ?>
        </div>
        <?php
    }

    /**
     * Render settings page.
     */
    public function render_settings_page() {
        $current_user = wp_get_current_user();
        // if ( is_multisite() && is_network_admin() ) {
        //     if ( ! current_user_can( 'manage_network_options' ) ) {
        //         wp_die( esc_html__( 'You do not have permission to access this page.', 'accessdoor-smart-admin-login-url-control' ) );
        //     }
        // } else {
        //     if ( ! current_user_can( 'manage_options' ) || ! in_array( 'administrator', (array) $current_user->roles, true ) ) {
        //         wp_die( esc_html__( 'You do not have permission to access this page.', 'accessdoor-smart-admin-login-url-control' ) );
        //     }
        // }
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        $active_tab = isset( $_GET['tab'] ) ? sanitize_key( wp_unslash( $_GET['tab'] ) ) : 'general';

        $current_user = wp_get_current_user();
        $username  = $current_user ? $current_user->user_login : '';
        $user_tab_title = 'Account Settings';
        ?>
        <div class="wrap calu-settings-page">
            <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>

            <!-- TABS -->
            <h2 class="nav-tab-wrapper">
                <a href="?page=accessdoor-smart-admin-login-url-control&tab=general"
                class="nav-tab <?php echo $active_tab === 'general' ? 'nav-tab-active' : ''; ?>">
                    General Settings
                </a>

                <a href="?page=accessdoor-smart-admin-login-url-control&tab=advanced"
                class="nav-tab <?php echo $active_tab === 'advanced' ? 'nav-tab-active' : ''; ?>">
                    Redesign Login Page
                </a>

                <a href="?page=accessdoor-smart-admin-login-url-control&tab=admin_settings"
                class="nav-tab <?php echo $active_tab === 'admin_settings' ? 'nav-tab-active' : ''; ?>">
                    <?php echo esc_html( $user_tab_title ); ?>
                </a>

                    <a href="?page=accessdoor-smart-admin-login-url-control&tab=security"
                    class="nav-tab <?php echo $active_tab === 'security' ? 'nav-tab-active' : ''; ?>">
                        <?php esc_html_e( 'Security Measures', 'accessdoor-smart-admin-login-url-control' ); ?>
                    </a>
            </h2>
            
            <form action="options.php" method="post" class="calu-settings-form">
                <?php
                if ( $active_tab === 'general' ) {
                    settings_fields( 'ADALC_general_group' );
                    do_settings_sections( 'adalc-general-settings' );
                }

                if ( $active_tab === 'advanced' ) {
                    settings_fields( 'ADALC_branding_group' );
                    do_settings_sections( 'adalc-branding-settings' );
                }

                if ( $active_tab === 'admin_settings' ) {
                    settings_fields( 'ADALC_admin_group' );
                    do_settings_sections( 'adalc-admin-settings' );
                }

                    if ( $active_tab === 'security' ) {
                        echo '<div class="calu-security-wrap">';
                        settings_fields( 'ADALC_security_group' );
                        do_settings_sections( 'adalc-security-settings' );
                        echo '</div>';
                    }
                submit_button( __( 'Save Settings', 'accessdoor-smart-admin-login-url-control' ) );
                ?>
            </form>
        </div>
        <?php
    }
    /**
     * Security section callback.
     */
    public function security_section_callback() {
        echo '<h2 style="margin-top:0;">' . esc_html__( 'Security Settings', 'accessdoor-smart-admin-login-url-control' ) . '</h2>';
        /**
         * Disable Comments field callback.
         */
    }
        public function disable_comments_callback() {
            $settings = get_option( $this->option_name, array() );
            $status = '';
            $value = !empty($settings) && isset($settings['disable_comments']) ? (int)$settings['disable_comments'] : 0;
            if ( has_filter( 'comments_open', '__return_false' ) && ! $value ) {
                $status = 'disabled';
            }
           
            ?>
            <div style="margin-bottom:10px;">
                
                <div style="margin-bottom:5px;color:#666;">
                    <?php esc_html_e( 'Disable all comments site-wide for better security and spam prevention.', 'accessdoor-smart-admin-login-url-control' ); ?>
                    <?php if($status) { ?><p class="notice"><?php esc_html_e( 'Some other plugin or code has already disabled comments site-wide.', 'accessdoor-smart-admin-login-url-control' ); ?></p><?php } ?>
                </div>
                <div class="calu-toggle-wrapper">
                    <label class="calu-switch">
                        <input type="checkbox" name="<?php echo esc_attr( $this->option_name ); ?>[disable_comments]" value="1" <?php checked( 1, $value ); ?> <?php echo $status ?>/>
                        <span class="calu-slider"></span>
                    </label>
                </div>
            </div>
            <?php
        }

        /**
         * Disable XML-RPC Pingback field callback.
         */
        public function disable_xmlrpc_pingback_callback() {
            $settings = get_option( $this->option_name, array() );
            $status = '';
            $value = !empty($settings) && isset($settings['disable_xmlrpc_pingback']) ? (int)$settings['disable_xmlrpc_pingback'] : 0;
            if ( has_filter( 'xmlrpc_methods', '__return_false' ) && ! $value ) {
                $status = 'disabled';
            }
           ?>
            <div style="margin-bottom:10px;">
                
                <div style="margin-bottom:5px;color:#666;">
                    <?php esc_html_e( 'Block XML-RPC pingback requests to prevent DDoS and spam attacks.', 'accessdoor-smart-admin-login-url-control' ); ?>
                    <?php if($status) { ?><p class="notice"><?php esc_html_e( 'Some other plugin or code has already blocked XML-RPC pingback.', 'accessdoor-smart-admin-login-url-control' ); ?></p><?php } ?>
                </div>
                <div class="calu-toggle-wrapper">
                    <label class="calu-switch">
                        <input type="checkbox" name="<?php echo esc_attr( $this->option_name ); ?>[disable_xmlrpc_pingback]" value="1" <?php checked( 1, $value ); ?> <?php echo $status ?>/>
                        <span class="calu-slider"></span>
                    </label>
                </div>
            </div>
            <?php
        }

        /**
         * Disable REST API field callback.
         */
        public function disable_rest_api_callback() {
            $settings = get_option( $this->option_name, array() );
            $status = '';
            $value = !empty($settings) && isset($settings['disable_rest_api']) ? (int)$settings['disable_rest_api'] : 0;
            if ( has_filter( 'rest_authentication_errors', '__return_false' ) && ! $value ) {
                $status = 'disabled';
            }
            ?>
            <div style="margin-bottom:10px;">
               
                <div style="margin-bottom:5px;color:#666;">
                    <?php esc_html_e( 'Disable the WordPress REST API for non-authenticated users to reduce attack surface.', 'accessdoor-smart-admin-login-url-control' ); ?>
                    <?php if($status) { ?><p class="notice"><?php esc_html_e( 'Some other plugin or code has already blocked REST API.', 'accessdoor-smart-admin-login-url-control' ); ?></p><?php } ?>
                </div>
                <div class="calu-toggle-wrapper">
                    <label class="calu-switch">
                        <input type="checkbox" name="<?php echo esc_attr( $this->option_name ); ?>[disable_rest_api]" value="1" <?php checked( 1, $value ); ?> <?php echo $status ?>/>
                        <span class="calu-slider"></span>
                    </label>
                </div>
            </div>
            <?php
        }

        /**
         * Disable XML-RPC field callback.
         */
        public function disable_xmlrpc_callback() {
            $settings = get_option( $this->option_name, array() );
            $status = '';
            $value = !empty($settings) && isset($settings['disable_xmlrpc']) ? (int)$settings['disable_xmlrpc'] : 0;
            if ( has_filter( 'xmlrpc_enabled', '__return_false' ) && ! $value ) {
                $status = 'disabled';
            }
            ?>
            <div style="margin-bottom:10px;">
                
                <div style="margin-bottom:5px;color:#666;">
                    <?php esc_html_e( 'Completely disable XML-RPC functionality to block remote access attempts.', 'accessdoor-smart-admin-login-url-control' ); ?>
                    <?php if($status) { ?><p class="notice"><?php esc_html_e( 'Some other plugin or code has already disabled XML-RPC.', 'accessdoor-smart-admin-login-url-control' ); ?></p><?php } ?>
                </div>
                <div class="calu-toggle-wrapper">
                    <label class="calu-switch">
                        <input type="checkbox" name="<?php echo esc_attr( $this->option_name ); ?>[disable_xmlrpc]" value="1" <?php checked( 1, $value ); ?> <?php echo $status ?>/>
                        <span class="calu-slider"></span>
                    </label>
                </div>
            </div>
            <?php
        }


      /**
     * Block Directory Browsing field callback.
     */
    public function block_directory_browsing_callback() {
        $settings = get_option( $this->option_name, array() );
        $value = !empty($settings) && isset($settings['block_directory_browsing']) ? (int)$settings['block_directory_browsing'] : 0;
        $htaccess_path = ABSPATH . '.htaccess';
        $htaccess_error = false;

        if ( ! function_exists( 'request_filesystem_credentials' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        global $wp_filesystem;
        if ( empty( $wp_filesystem ) ) {
            $creds = request_filesystem_credentials( '', '', false, false, null );
            if ( ! WP_Filesystem( $creds ) ) {
                $wp_filesystem = null;
            }
        }
        if ( ! $wp_filesystem || ! $wp_filesystem->exists( $htaccess_path ) || ! $wp_filesystem->is_writable( $htaccess_path ) ) {
            $htaccess_error = true;
            $value = 0; // Prevent enabling the feature
        }
        ?>
        <div class="calu-toggle-wrapper">
            <label class="calu-switch">
                <input type="checkbox" name="<?php echo esc_attr( $this->option_name ); ?>[block_directory_browsing]" value="1" <?php checked( 1, $value ); ?> <?php if($htaccess_error) echo 'disabled'; ?> />
                <span class="calu-slider"></span>
            </label>
        </div>
        <p><?php esc_html_e('Prevents directory listing via .htaccess (Apache only).', 'accessdoor-smart-admin-login-url-control'); ?></p>
        <?php if ( $htaccess_error ) : ?>
            <div class="-notice -notice-error" style="margin:10px 0 0 0;padding:8px 12px;background:#fff0f0;border-left:4px solid #d63638;">
                <strong><?php esc_html_e('Error:', 'accessdoor-smart-admin-login-url-control'); ?></strong>
                <?php esc_html_e('.htaccess file is not writable or accessible. This feature cannot be enabled until permissions are fixed.', 'accessdoor-smart-admin-login-url-control'); ?>
            </div>
        <?php endif;
    }

    /**
     * Forbid PHP in Uploads field callback.
     */
    public function forbid_php_uploads_callback() {
        $settings = get_option( $this->option_name, array() );
        $value = !empty($settings) && isset($settings['forbid_php_uploads']) ? (int)$settings['forbid_php_uploads'] : 0;
        $uploads_dir = wp_get_upload_dir();
        $uploads_path = trailingslashit( $uploads_dir['basedir'] );
        $uploads_htaccess = $uploads_path . '.htaccess';
        $uploads_error = false;

        if ( ! function_exists( 'request_filesystem_credentials' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        global $wp_filesystem;
        if ( empty( $wp_filesystem ) ) {
            $creds = request_filesystem_credentials( '', '', false, false, null );
            if ( ! WP_Filesystem( $creds ) ) {
                $wp_filesystem = null;
            }
        }
        if ( ! $wp_filesystem || ! $wp_filesystem->exists( $uploads_htaccess ) || ! $wp_filesystem->is_writable( $uploads_htaccess ) ) {
            $uploads_error = true;
            $value = 0; // Prevent enabling the feature
        }
        ?>
        <div class="calu-toggle-wrapper">
            <label class="calu-switch">
                <input type="checkbox" name="<?php echo esc_attr( $this->option_name ); ?>[forbid_php_uploads]" value="1" <?php checked( 1, $value ); ?> <?php if($uploads_error) echo 'disabled'; ?> />
                <span class="calu-slider"></span>
            </label>
        </div>
        <p><?php esc_html_e('Prevents execution of PHP files in wp-content/uploads via .htaccess (Apache only).', 'accessdoor-smart-admin-login-url-control'); ?></p>
        <?php if ( $uploads_error ) : ?>
            <div class="-notice -notice-error" style="margin:10px 0 0 0;padding:8px 12px;background:#fff0f0;border-left:4px solid #d63638;">
                <strong><?php esc_html_e('Error:', 'accessdoor-smart-admin-login-url-control'); ?></strong>
                <?php esc_html_e('Uploads .htaccess file is not writable or accessible. This feature cannot be enabled until permissions are fixed.', 'accessdoor-smart-admin-login-url-control'); ?>
            </div>
        <?php endif;
}

    /**
     * Turn Off Pingbacks field callback.
     */
    public function turn_off_pingbacks_callback() {
        $settings = get_option( $this->option_name, array() );
        $status = '';
        $value = !empty($settings) && isset($settings['turn_off_pingbacks']) ? (int)$settings['turn_off_pingbacks'] : 0;

        // Check if pingbacks are already disabled by another plugin or code
        $pingback_disabled = has_filter( 'xmlrpc_methods', '__return_false' );
        if ( $pingback_disabled && ! $value ) {
            $status = 'disabled';
        }
        ?>
        <div class="calu-toggle-wrapper">
            <label class="calu-switch">
                <input type="checkbox" name="<?php echo esc_attr( $this->option_name ); ?>[turn_off_pingbacks]" value="1" <?php checked( 1, $value ); ?> <?php echo $status; ?> />
                <span class="calu-slider"></span>
            </label>
        </div>
        <p class=""><?php esc_html_e('Disables XML-RPC pingbacks and closes pingbacks on all posts.', 'accessdoor-smart-admin-login-url-control'); ?></p>
        <?php if($status) { ?>
            <p class="notice"><?php esc_html_e( 'Some other plugin or code has already disabled pingbacks site-wide.', 'accessdoor-smart-admin-login-url-control' ); ?></p>
        <?php }
    }

    /**
     * Disable File Editing in Dashboard field callback.
     */
    public function disable_file_editing_callback() {
        $settings = get_option( $this->option_name, array() );
        $status = '';
        $value = !empty($settings) && isset($settings['disable_file_editing']) ? (int)$settings['disable_file_editing'] : 0;

        // Check if DISALLOW_FILE_EDIT is already defined and true
        if ( defined( 'DISALLOW_FILE_EDIT' ) && DISALLOW_FILE_EDIT && ! $value ) {
            $status = 'disabled';
        }
        ?>
        <div class="calu-toggle-wrapper">
            <label class="calu-switch">
                <input type="checkbox" name="<?php echo esc_attr( $this->option_name ); ?>[disable_file_editing]" value="1" <?php checked( 1, $value ); ?> <?php echo $status; ?> />
                <span class="calu-slider"></span>
            </label>
        </div>
        <p class=""><?php esc_html_e('Prevents editing of plugin and theme files from the WordPress dashboard.', 'accessdoor-smart-admin-login-url-control'); ?></p>
        <?php if($status) { ?>
            <p class="notice"><?php esc_html_e( 'File editing is already disabled by another plugin or your wp-config.php.', 'accessdoor-smart-admin-login-url-control' ); ?></p>
        <?php }
    }

    /**
     * Enqueue admin scripts and styles.
     *
     * @param string $hook Current admin page hook.
     */
    public function enqueue_admin_scripts( $hook ) {
        if ( 'settings_page_accessdoor-smart-admin-login-url-control' !== $hook ) {
            return;
        }

        wp_enqueue_media();
        wp_enqueue_style( 'calu-admin-style', ADALC_URL . 'assets/css/admin.css?time=' . time(), array(), ADALC_VERSION );
        wp_enqueue_script( 'calu-admin-script', ADALC_URL . 'assets/js/admin.js?v=1', array( 'jquery' ), ADALC_VERSION, true );
        wp_enqueue_style( 'dashicons' );
        wp_enqueue_style( 'adalc-fontawesome', 'https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.2/css/all.min.css', array(), '6.4.2' );



    }

    /**
     * Add settings link to plugin actions.
     *
     * @param array $links Plugin action links.
     * @return array Modified plugin action links.
     */
    public function add_settings_link( $links ) {
        $settings_link = '<a href="' . admin_url( 'options-general.php?page=accessdoor-smart-admin-login-url-control' ) . '">' . __( 'Settings', 'accessdoor-smart-admin-login-url-control' ) . '</a>';
        array_unshift( $links, $settings_link );
        return $links;
    }

    public function change_admin_email( $new_email ) {

        // 1. Validate email
        if ( ! is_email( $new_email ) ) {
            return new WP_Error( 'invalid_email', 'Invalid email address.' );
        }

       update_option('admin_email', sanitize_email( $new_email ));
       update_option('new_admin_email', sanitize_email($new_email));

       // Remove WordPress pending confirmation
        delete_option( 'new_admin_email' );
        delete_option( 'adminhash' );

        return true;
    }

}

