<?php 
/*
Plugin Name: AA Live Your Site
Plugin URI: https://wordpress.org/plugins/aalivesite/
Description: AA Health Calculator is a fast, easy-to-use plugin that helps you track and improve your health with accurate insights.
Version: 1.0.1
Author: Syed Ashik Mahmud
Author URI: https://aaextensions.com
License: GPL2
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Text Domain: aa-live-site
Domain Path: /languages
*/

if (!defined('ABSPATH')) exit;

function aa_ls_add_admin_menu() {
    add_menu_page(
        __('AA Live Site', 'aa-live-site'),
        __('Live Site', 'aa-live-site'),
        'manage_options',
        'aa-live-site',
        'aa_ls_settings_page',
        'dashicons-update',
        80
    );
}
add_action('admin_menu', 'aa_ls_add_admin_menu');

function aa_ls_settings_page() {
    if (!current_user_can('manage_options')) {
        wp_die(__('You do not have sufficient permissions to access this page.', 'aa-live-site'));
    }

    global $wpdb;
    $message = '';

    $screen = get_current_screen();

    $screen->add_help_tab([
        'id'      => 'aa_ls_help_usage',
        'title'   => __('Usage Guide', 'aa-live-site'),
        'content' =>
            '<h3>' . __('How to Use', 'aa-live-site') . '</h3>' .
            '<ul>
                <li><strong>Old URL:</strong> e.g., https://staging.example.com</li>
                <li><strong>New URL:</strong> e.g., https://www.example.com</li>
                <li><strong>Safe Mode:</strong> Previews the number of changes.</li>
                <li><strong>Backup:</strong> Creates a full database .sql file.</li>
                <li><strong>Include Custom Tables:</strong> Looks inside additional plugin tables.</li>
            </ul>'
    ]);

    $screen->add_help_tab([
        'id'      => 'aa_ls_help_troubleshooting',
        'title'   => __('Troubleshooting', 'aa-live-site'),
        'content' =>
            '<h3>' . __('Troubleshooting', 'aa-live-site') . '</h3>' .
            '<ul>
                <li><strong>URLs not changing?</strong> Make sure the "Old URL" matches exactly.</li>
                <li><strong>Backup not created?</strong> Your server may block `exec()` or `mysqldump`.</li>
                <li><strong>Plugin data missing?</strong> Try enabling "Include Custom Tables".</li>
                <li><strong>Serialized data?</strong> This plugin doesn’t yet support deep serialization.</li>
            </ul>' .
            '<p>' . __('Still stuck? Email us at support@aaplugins.com', 'aa-live-site') . '</p>'
    ]);

    $screen->set_help_sidebar(
        '<p><strong>' . __('Need Support?', 'aa-live-site') . '</strong></p>' .
        '<p><a href="mailto:support@aaplugins.com">support@aaplugins.com</a></p>' .
        '<p><a href="https://wordpress.org/plugins/aa-live-site/" target="_blank">' . __('Plugin Docs', 'aa-live-site') . '</a></p>'
    );

    if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['aa_ls_submit']) && check_admin_referer('aa_ls_change_urls', 'aa_ls_nonce')) {
        $fromurl = esc_url_raw(trim($_POST['fromurl']));
        $tourl = esc_url_raw(trim($_POST['tourl']));
        $do_backup = !empty($_POST['backup']);
        $safe_mode = !empty($_POST['safe_mode']);
        $include_custom = !empty($_POST['include_custom']);

        $prefix = $wpdb->prefix;

        if (empty($fromurl) || empty($tourl)) {
            $message .= "<div class='notice notice-error'><p>" . esc_html__('Both URLs are required.', 'aa-live-site') . "</p></div>";
        } else {
            if ($do_backup) {
                $backup_file = plugin_dir_path(__FILE__) . 'backup-' . date("Ymd-His") . '.sql';
                $cmd = "mysqldump -h " . escapeshellarg(DB_HOST) . " -u " . escapeshellarg(DB_USER) . " -p" . escapeshellarg(DB_PASSWORD) . " " . escapeshellarg(DB_NAME) . " > " . escapeshellarg($backup_file);
                exec($cmd, $output, $result_code);
                $message .= $result_code === 0
                    ? "<div class='notice notice-success'><p>" . esc_html__('Database backup created successfully.', 'aa-live-site') . "</p></div>"
                    : "<div class='notice notice-error'><p>" . esc_html__('Database backup failed. Please check server configuration.', 'aa-live-site') . "</p></div>";
            }

            $tables = [
                "{$prefix}options" => ["option_value"],
                "{$prefix}posts" => ["guid", "post_content"],
                "{$prefix}postmeta" => ["meta_value"]
            ];

            if ($include_custom) {
                $extra_tables = $wpdb->get_results("SHOW TABLES LIKE '%meta%'");
                foreach ($extra_tables as $table_obj) {
                    $table_name = reset($table_obj);
                    if (!isset($tables[$table_name])) {
                        $columns = $wpdb->get_col("DESC `$table_name`", 0);
                        foreach ($columns as $col) {
                            if (in_array($col, ['meta_value', 'value', 'settings'])) {
                                $tables[$table_name][] = $col;
                            }
                        }
                    }
                }
            }

            $total_updated = 0;
            $simulated = [];

            foreach ($tables as $table => $columns) {
                foreach ($columns as $column) {
                    $sql = $wpdb->prepare("SELECT COUNT(*) FROM `$table` WHERE `$column` LIKE %s", '%' . $wpdb->esc_like($fromurl) . '%');
                    $count = $wpdb->get_var($sql);

                    if ($safe_mode) {
                        if ($count > 0) {
                            $simulated[] = "<li><code>$table.$column</code>: $count matches</li>";
                        }
                    } else {
                        $query = $wpdb->prepare("UPDATE `$table` SET `$column` = REPLACE(`$column`, %s, %s)", $fromurl, $tourl);
                        $rows = $wpdb->query($query);
                        if (!is_wp_error($rows)) {
                            $total_updated += intval($rows);
                        }
                    }
                }
            }

            if ($safe_mode) {
                $message .= "<div class='notice notice-info'><p><strong>Safe Mode:</strong> No changes made.<br>Would update the following:</p><ul>" . implode('', $simulated) . "</ul></div>";
            } else {
                $message .= "<div class='notice notice-success is-dismissible'><p>" .
                    esc_html__('URL replacement completed.', 'aa-live-site') . "<br>" .
                    esc_html__('From:', 'aa-live-site') . " <code>" . esc_html($fromurl) . "</code><br>" .
                    esc_html__('To:', 'aa-live-site') . " <code>" . esc_html($tourl) . "</code><br>" .
                    sprintf(esc_html__('%d rows updated.', 'aa-live-site'), $total_updated) .
                    "</p></div>";
            }
        }
    }

    echo "<div class='wrap'>
        <h1>" . esc_html__('AA Live Site URL Updater', 'aa-live-site') . "</h1>
        $message
        <form method='post' style='background:#fff; padding:20px; max-width:700px; border-radius:8px; box-shadow:0 0 10px rgba(0,0,0,0.05);'>";
            wp_nonce_field('aa_ls_change_urls', 'aa_ls_nonce');
            echo "<p>
                <label><strong>" . __('Old URL:', 'aa-live-site') . "</strong></label><br>
                <input name='fromurl' type='url' required class='regular-text' placeholder='https://oldsite.com' />
            </p>
            <p>
                <label><strong>" . __('New URL:', 'aa-live-site') . "</strong></label><br>
                <input name='tourl' type='url' required class='regular-text' placeholder='https://newsite.com' />
            </p>
            <p><label><input type='checkbox' name='backup' /> " . __('Create database backup before updating', 'aa-live-site') . "</label></p>
            <p><label><input type='checkbox' name='safe_mode' /> " . __('Enable Safe Mode (preview only, no changes)', 'aa-live-site') . "</label></p>
            <p><label><input type='checkbox' name='include_custom' /> " . __('Include custom tables (Premium Feature)', 'aa-live-site') . "</label></p>
            <p><input type='submit' name='aa_ls_submit' class='button button-primary button-large' value='" . esc_attr__('Update URLs', 'aa-live-site') . "' /></p>
            <hr>
            <details style='margin-top:1em;'>
                <summary style='cursor:pointer; font-weight:bold;'>" . __('FAQ – Click to Expand', 'aa-live-site') . "</summary>
                <div style='margin-top:10px;'>
                    <p><strong>" . __('Q: Will this change Elementor URLs?', 'aa-live-site') . "</strong><br>" . __('Yes, it replaces URLs in post_content and meta_value, covering Elementor data.', 'aa-live-site') . "</p>
                    <p><strong>" . __('Q: What if I make a mistake?', 'aa-live-site') . "</strong><br>" . __('Use the backup option to restore your database if needed.', 'aa-live-site') . "</p>
                    <p><strong>" . __('Q: Does this work with multilingual plugins?', 'aa-live-site') . "</strong><br>" . __('Yes, it updates URLs across all post types and meta tables.', 'aa-live-site') . "</p>
                </div>
            </details>
        </form>
    </div>";
}
