<?php
/*
Plugin Name: AA Condition
Plugin URI: http://wordpress.org/plugins/aa-condition/
Description: Custom condition shortcode with advanced options and shortcode generator.
Version: 1.0.1
Author: Syed Ashik Mahmud
Author URI: https://aaextensions.com
License: GPL2
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Text Domain: aa-condition
Domain Path: /languages
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

// Define if premium features are enabled — toggle to true when premium is activated.
if ( ! defined( 'AA_CONDITION_PREMIUM' ) ) {
    define( 'AA_CONDITION_PREMIUM', false );
}

add_shortcode( 'aa_condition_button', 'aa_condition_button_handler' );

function aa_condition_button_handler( $atts ) {
    $atts = shortcode_atts(
        array(
            'show_get_help'        => 'yes',
            'get_help_url'         => '#',
            'get_help_label'       => __( 'GET HELP', 'aa-condition' ),
            'show_give_back'       => 'yes',
            'give_back_url'        => '#',
            'give_back_label'      => __( 'GIVE BACK', 'aa-condition' ),
            'button_target_blank'  => 'no',
            'button_class'         => 'button-medium',
            'show_login_msg'       => 'yes',
            'login_url'            => wp_login_url(),
            'register_url'         => wp_registration_url(),
        ),
        $atts,
        'aa_condition_button'
    );

    $target = ( strtolower( $atts['button_target_blank'] ) === 'yes' ) ? ' target="_blank" rel="noopener noreferrer"' : '';

    ob_start();

    if ( is_user_logged_in() && strtolower( $atts['show_get_help'] ) === 'yes' ) {
        ?>
        <p style="text-align:center;">
            <a href="<?php echo esc_url( $atts['get_help_url'] ); ?>"<?php echo $target; ?>>
                <input 
                    type="button" 
                    value="<?php echo esc_attr( $atts['get_help_label'] ); ?>" 
                    class="<?php echo esc_attr( $atts['button_class'] ); ?>" 
                    style="font-size:14px;"
                    readonly
                />
            </a>
        </p>
        <?php
    }

    if ( strtolower( $atts['show_give_back'] ) === 'yes' ) {
        ?>
        <p style="text-align:center;">
            <a href="<?php echo esc_url( $atts['give_back_url'] ); ?>"<?php echo $target; ?>>
                <input 
                    type="button" 
                    value="<?php echo esc_attr( $atts['give_back_label'] ); ?>" 
                    class="<?php echo esc_attr( $atts['button_class'] ); ?>" 
                    style="font-size:14px;"
                    readonly
                />
            </a>
        </p>
        <?php
    }

    if ( ! is_user_logged_in() && strtolower( $atts['show_login_msg'] ) === 'yes' ) {
        ?>
        <p style="text-align:center; font-weight:bold;">
            <?php
            echo wp_kses_post(
                sprintf(
                    /* translators: %1$s=Login link, %2$s=Register link */
                    __( 'If you would like to create a campaign please <a href="%1$s" style="color:red;">Log in</a> or <a href="%2$s" style="color:red;">Register</a> below.', 'aa-condition' ),
                    esc_url( $atts['login_url'] ),
                    esc_url( $atts['register_url'] )
                )
            );
            ?>
        </p>

        <style type="text/css">
            .aa-login-block p {
                display: inline-block; 
                padding-right: 15px; 
                font-weight: bold;
            }
            .aa-login-block {
                text-align: center;
            }
        </style>

        <div class="aa-login-block">
            <p><a href="<?php echo esc_url( $atts['login_url'] ); ?>"><?php esc_html_e( 'Login', 'aa-condition' ); ?></a></p>
            <p><a href="<?php echo esc_url( $atts['register_url'] ); ?>"><?php esc_html_e( 'Register', 'aa-condition' ); ?></a></p>
        </div>
        <?php
    }

    return ob_get_clean();
}

add_action( 'admin_menu', 'aa_condition_admin_menu' );
function aa_condition_admin_menu() {
    add_menu_page(
        __( 'AA Condition', 'aa-condition' ),
        __( 'AA Condition', 'aa-condition' ),
        'manage_options',
        'aa-condition',
        'aa_condition_admin_page_callback',
        'dashicons-editor-code',
        100
    );
}

function aa_condition_admin_page_callback() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'aa-condition' ) );
    }

    $defaults = array(
        'show_get_help'        => 'yes',
        'get_help_url'         => '#',
        'get_help_label'       => __( 'GET HELP', 'aa-condition' ),
        'show_give_back'       => 'yes',
        'give_back_url'        => '#',
        'give_back_label'      => __( 'GIVE BACK', 'aa-condition' ),
        'button_target_blank'  => 'no',
        'button_class'         => 'button-medium',
        'show_login_msg'       => 'yes',
        'login_url'            => wp_login_url(),
        'register_url'         => wp_registration_url(),
    );

    $input = $defaults;

    $message = '';
    // Check if form submitted
    if ( isset( $_POST['aa_condition_nonce'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['aa_condition_nonce'] ) ), 'aa_condition_save' ) ) {
        foreach ( $defaults as $key => $default ) {
            if ( isset( $_POST[ $key ] ) ) {
                $input[ $key ] = sanitize_text_field( wp_unslash( $_POST[ $key ] ) );
            }
        }
        $message = __( 'Shortcode generated successfully.', 'aa-condition' );
    } elseif ( isset( $_POST['aa_condition_nonce'] ) ) {
        $message = __( 'Security check failed. Please try again.', 'aa-condition' );
    }

    // Premium check
    $is_premium = AA_CONDITION_PREMIUM;

    // Disable premium fields if not premium
    $disabled_attr = $is_premium ? '' : 'disabled';

    // Build shortcode string
    $shortcode = '[aa_condition_button';
    foreach ( $input as $key => $value ) {
        // Always include URLs and labels, others only if changed from default
        if (
            in_array( $key, array( 'get_help_url', 'get_help_label', 'give_back_url', 'give_back_label', 'login_url', 'register_url' ), true )
            || strtolower( $value ) !== strtolower( $defaults[ $key ] )
        ) {
            $shortcode .= sprintf( ' %1$s="%2$s"', esc_attr( $key ), esc_attr( $value ) );
        }
    }
    $shortcode .= ']';

    ?>

    <div class="wrap">
        <h1><?php esc_html_e( 'AA Condition Shortcode Generator', 'aa-condition' ); ?></h1>

        <?php if ( ! $is_premium ) : ?>
            <div style="background:#ffefc6; border-left:4px solid #ffb900; padding:10px; margin-bottom:20px;">
                <strong><?php esc_html_e( 'Premium Features:', 'aa-condition' ); ?></strong> <?php esc_html_e( 'Some options are disabled. Upgrade to unlock all features.', 'aa-condition' ); ?>
            </div>
        <?php endif; ?>

        <?php if ( $message ) : ?>
            <div id="message" class="updated notice is-dismissible"><p><?php echo esc_html( $message ); ?></p></div>
        <?php endif; ?>

        <form method="post" action="">
            <?php wp_nonce_field( 'aa_condition_save', 'aa_condition_nonce' ); ?>

            <table class="form-table" role="presentation">
                <tbody>
                <tr>
                    <th scope="row"><?php esc_html_e( 'Show GET HELP button', 'aa-condition' ); ?></th>
                    <td>
                        <select name="show_get_help" <?php echo $disabled_attr; ?>>
                            <option value="yes" <?php selected( $input['show_get_help'], 'yes' ); ?>><?php esc_html_e( 'Yes', 'aa-condition' ); ?></option>
                            <option value="no" <?php selected( $input['show_get_help'], 'no' ); ?>><?php esc_html_e( 'No', 'aa-condition' ); ?></option>
                        </select>
                        <?php if ( ! $is_premium ) echo '<p class="description" style="color:#999;">' . esc_html__( 'Premium only', 'aa-condition' ) . '</p>'; ?>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e( 'GET HELP Button URL', 'aa-condition' ); ?></th>
                    <td>
                        <input type="url" name="get_help_url" value="<?php echo esc_attr( $input['get_help_url'] ); ?>" class="regular-text" <?php echo $disabled_attr; ?> />
                        <?php if ( ! $is_premium ) echo '<p class="description" style="color:#999;">' . esc_html__( 'Premium only', 'aa-condition' ) . '</p>'; ?>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e( 'GET HELP Button Label', 'aa-condition' ); ?></th>
                    <td>
                        <input type="text" name="get_help_label" value="<?php echo esc_attr( $input['get_help_label'] ); ?>" class="regular-text" <?php echo $disabled_attr; ?> />
                        <?php if ( ! $is_premium ) echo '<p class="description" style="color:#999;">' . esc_html__( 'Premium only', 'aa-condition' ) . '</p>'; ?>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><?php esc_html_e( 'Show GIVE BACK button', 'aa-condition' ); ?></th>
                    <td>
                        <select name="show_give_back">
                            <option value="yes" <?php selected( $input['show_give_back'], 'yes' ); ?>><?php esc_html_e( 'Yes', 'aa-condition' ); ?></option>
                            <option value="no" <?php selected( $input['show_give_back'], 'no' ); ?>><?php esc_html_e( 'No', 'aa-condition' ); ?></option>
                        </select>
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e( 'GIVE BACK Button URL', 'aa-condition' ); ?></th>
                    <td>
                        <input type="url" name="give_back_url" value="<?php echo esc_attr( $input['give_back_url'] ); ?>" class="regular-text" />
                    </td>
                </tr>
                <tr>
                    <th scope="row"><?php esc_html_e( 'GIVE BACK Button Label', 'aa-condition' ); ?></th>
                    <td>
                        <input type="text" name="give_back_label" value="<?php echo esc_attr( $input['give_back_label'] ); ?>" class="regular-text" />
                    </td>
                </tr>

                <tr>
                    <th scope="row"><?php esc_html_e( 'Open buttons in new tab', 'aa-condition' ); ?></th>
                    <td>
                        <select name="button_target_blank">
                            <option value="no" <?php selected( $input['button_target_blank'], 'no' ); ?>><?php esc_html_e( 'No', 'aa-condition' ); ?></option>
                            <option value="yes" <?php selected( $input['button_target_blank'], 'yes' ); ?>><?php esc_html_e( 'Yes', 'aa-condition' ); ?></option>
                        </select>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><?php esc_html_e( 'Button CSS class', 'aa-condition' ); ?></th>
                    <td>
                        <input type="text" name="button_class" value="<?php echo esc_attr( $input['button_class'] ); ?>" class="regular-text" />
                        <p class="description"><?php esc_html_e( 'Additional CSS classes for buttons, space separated.', 'aa-condition' ); ?></p>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><?php esc_html_e( 'Show Login/Register message', 'aa-condition' ); ?></th>
                    <td>
                        <select name="show_login_msg" <?php echo $disabled_attr; ?>>
                            <option value="yes" <?php selected( $input['show_login_msg'], 'yes' ); ?>><?php esc_html_e( 'Yes', 'aa-condition' ); ?></option>
                            <option value="no" <?php selected( $input['show_login_msg'], 'no' ); ?>><?php esc_html_e( 'No', 'aa-condition' ); ?></option>
                        </select>
                        <?php if ( ! $is_premium ) echo '<p class="description" style="color:#999;">' . esc_html__( 'Premium only', 'aa-condition' ) . '</p>'; ?>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><?php esc_html_e( 'Login URL', 'aa-condition' ); ?></th>
                    <td>
                        <input type="url" name="login_url" value="<?php echo esc_attr( $input['login_url'] ); ?>" class="regular-text" <?php echo $disabled_attr; ?> />
                        <?php if ( ! $is_premium ) echo '<p class="description" style="color:#999;">' . esc_html__( 'Premium only', 'aa-condition' ) . '</p>'; ?>
                    </td>
                </tr>

                <tr>
                    <th scope="row"><?php esc_html_e( 'Register URL', 'aa-condition' ); ?></th>
                    <td>
                        <input type="url" name="register_url" value="<?php echo esc_attr( $input['register_url'] ); ?>" class="regular-text" <?php echo $disabled_attr; ?> />
                        <?php if ( ! $is_premium ) echo '<p class="description" style="color:#999;">' . esc_html__( 'Premium only', 'aa-condition' ) . '</p>'; ?>
                    </td>
                </tr>

                </tbody>
            </table>

            <p><input type="submit" class="button button-primary" value="<?php esc_attr_e( 'Generate Shortcode', 'aa-condition' ); ?>"></p>
        </form>

        <h2><?php esc_html_e( 'Generated Shortcode', 'aa-condition' ); ?></h2>
        <textarea rows="3" style="width: 100%; font-family: monospace;" readonly><?php echo esc_textarea( $shortcode ); ?></textarea>

        <h2><?php esc_html_e( 'Shortcode Usage Example', 'aa-condition' ); ?></h2>
        <pre style="background:#f9f9f9; padding:10px; border:1px solid #ddd;">
[aa_condition_button show_get_help="yes" get_help_url="https://yourhelpurl.com" get_help_label="Help Now" show_give_back="no"]
        </pre>

    </div>

    <?php
}
