<?php
/**
 * Fired when the plugin is uninstalled.
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    A1AI
 */

// If uninstall not called from WordPress, then exit.
if (!defined('WP_UNINSTALL_PLUGIN')) {
    exit;
}

// If the user doesn't have sufficient permissions, exit.
if (!current_user_can('activate_plugins')) {
    exit;
}

// Check if the user wants to keep data
// This would typically be controlled via an admin setting
$keep_data = get_option('a1ai_keep_data_on_uninstall', false);

if (!$keep_data) {
    // Remove all options
    $options = array(
        'a1ai_version',
        'a1ai_openai_api_key',
        'a1ai_global_chatbot_enabled',
        'a1ai_chatbot_position',
        'a1ai_default_model',
        'a1ai_default_temperature',
        'a1ai_default_max_tokens',
        'a1ai_keep_conversation_history',
        'a1ai_conversation_retention_days',
        'a1ai_keep_data_on_uninstall',
    );

    foreach ($options as $option) {
        delete_option($option);
    }
    
    // Remove database tables
    global $wpdb;
    
    $tables = array(
        $wpdb->prefix . 'a1ai_chatbots',
        $wpdb->prefix . 'a1ai_conversations',
        $wpdb->prefix . 'a1ai_messages',
        $wpdb->prefix . 'a1ai_adjustments',
    );
    
    // Cache any relevant data that might be needed later
    $tables_exist_cache = wp_cache_get('a1ai_tables_exist', 'a1ai');
    if (false !== $tables_exist_cache) {
        wp_cache_delete('a1ai_tables_exist', 'a1ai');
    }
    
    // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery
    // Direct database queries are necessary for uninstall operations
    // phpcs:disable WordPress.DB.DirectDatabaseQuery.NoCaching
    // No caching needed for uninstall operations
    // phpcs:disable WordPress.DB.DirectDatabaseQuery.SchemaChange
    // Schema changes are necessary for uninstall operations
    
    foreach ($tables as $table) {
        // Use backtick escaping for table names (safe as derived from trusted prefix + hardcoded strings)
        $safe_table = '`' . str_replace('`', '``', $table) . '`';
        // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared -- Table name is safely escaped with backticks
        $wpdb->query("DROP TABLE IF EXISTS $safe_table");
    }
    
    // phpcs:enable WordPress.DB.DirectDatabaseQuery.SchemaChange
    // phpcs:enable WordPress.DB.DirectDatabaseQuery.NoCaching
    // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery
    
    // Remove any transients
    delete_transient('a1ai_activation_redirect');
    delete_transient('a1ai_usage_stats_cache');
    
    // Clear scheduled hooks
    wp_clear_scheduled_hook('a1ai_daily_maintenance');
}

// If we decide to add any user capabilities, we would remove them here 