<?php
/**
 * Partial template for rendering the shortcode chatbot
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    A1AI
 * @subpackage A1AI/public/partials
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

// Extract variables passed to template
$chatbot_id = isset($chatbot_id) ? intval($chatbot_id) : 0;
$chatbot = isset($chatbot) ? $chatbot : null;
$session_id = isset($session_id) ? sanitize_text_field($session_id) : '';

// If no chatbot found, show error message
if (empty($chatbot)) {
    return '<div class="a1ai-error">' . esc_html__('Chatbot not found.', 'a1ai-chatbot') . '</div>';
}

// Get settings
$settings = array();
if (is_string($chatbot['settings'])) {
    // Try to decode as JSON first
    $decoded = json_decode($chatbot['settings'], true);
    if (json_last_error() === JSON_ERROR_NONE) {
        $settings = $decoded;
    } else {
        // Fall back to unserialize if JSON fails
        $settings = maybe_unserialize($chatbot['settings']);
    }
} elseif (is_array($chatbot['settings'])) {
    // Settings is already an array
    $settings = $chatbot['settings'];
}
$chatbot_name = isset($settings['name']) ? sanitize_text_field($settings['name']) : __('AI Assistant', 'a1ai-chatbot');
$chatbot_title = isset($settings['chatbot_title']) ? sanitize_text_field($settings['chatbot_title']) : __('Chat Support', 'a1ai-chatbot');
$welcome_message = isset($settings['welcome_message']) ? wp_kses_post($settings['welcome_message']) : __('Hello! How can I help you today?', 'a1ai-chatbot');
$placeholder_text = isset($settings['placeholder']) ? sanitize_text_field($settings['placeholder']) : __('Type your message here...', 'a1ai-chatbot');
$primary_color = isset($settings['primary_color']) ? sanitize_hex_color($settings['primary_color']) : '#0073aa';
$chatbot_width = isset($settings['chat_width']) ? sanitize_text_field($settings['chat_width']) : '350px';
$chatbot_height = isset($settings['chat_height']) ? sanitize_text_field($settings['chat_height']) : '500px';
$font_size = isset($settings['font_size']) ? sanitize_text_field($settings['font_size']) : 'medium';

// GDPR settings
$gdpr_enabled = isset($settings['gdpr_enabled']) && $settings['gdpr_enabled'] ? true : false;
$gdpr_consent_text = isset($settings['gdpr_consent_text']) ? wp_kses_post($settings['gdpr_consent_text']) : __('By clicking "I Agree", you consent to the use of AI-powered chat services and acknowledge that your conversations may be processed to provide you with assistance. You can withdraw your consent at any time.', 'a1ai-chatbot');

// Debug output for GDPR settings
if (defined('WP_DEBUG') && WP_DEBUG) {
    echo '<!-- GDPR Debug: enabled=' . ($gdpr_enabled ? 'true' : 'false') . ' -->';
}

// Generate unique container ID
$container_id = 'a1ai-chatbot-' . $chatbot_id . '-' . wp_rand();
?>

<div id="<?php echo esc_attr($container_id); ?>" class="a1ai-chatbot-container a1ai-font-size-<?php echo esc_attr($font_size); ?>" 
     data-chatbot-id="<?php echo esc_attr($chatbot_id); ?>"
     data-session-id="<?php echo esc_attr($session_id); ?>"
     data-gdpr-enabled="<?php echo esc_attr($gdpr_enabled ? 'true' : 'false'); ?>"
     style="width: <?php echo esc_attr($chatbot_width); ?>; height: <?php echo esc_attr($chatbot_height); ?>;">
    
    <?php if ($gdpr_enabled) : ?>
    <!-- GDPR Consent Overlay -->
    <div class="a1ai-gdpr-overlay" id="gdpr-overlay-<?php echo esc_attr($container_id); ?>">
        <div class="a1ai-gdpr-content">
            <h4><?php echo esc_html__('Privacy Consent', 'a1ai-chatbot'); ?></h4>
            <div class="a1ai-gdpr-text">
                <?php echo wp_kses_post($gdpr_consent_text); ?>
            </div>
            <div class="a1ai-gdpr-buttons">
                <button type="button" class="a1ai-gdpr-agree" style="background-color: <?php echo esc_attr($primary_color); ?>">
                    <?php echo esc_html__('I Agree', 'a1ai-chatbot'); ?>
                </button>
                <button type="button" class="a1ai-gdpr-decline">
                    <?php echo esc_html__('Decline', 'a1ai-chatbot'); ?>
                </button>
            </div>
        </div>
    </div>
    <?php endif; ?>
    
    <div class="a1ai-chatbot-header" style="background-color: <?php echo esc_attr($primary_color); ?>">
        <h4 class="a1ai-chatbot-title"><?php echo esc_html($chatbot_title); ?></h4>
    </div>
    
    <div class="a1ai-chatbot-messages">
        <?php
        // Check if upgrade plugin is active (reviews are a premium feature)
        $upgrade_active = class_exists('A1AI_Upgrade_Helper') && A1AI_Upgrade_Helper::is_upgrade_active();

        // Check if this is the designated review chatbot - show review form immediately
        $review_chatbot_id = get_option('a1ai_review_chatbot_id', 0);
        $is_review_chatbot = ($chatbot_id == $review_chatbot_id && $review_chatbot_id > 0);

        // Only show welcome message if NOT a review chatbot
        if (!$is_review_chatbot || !$upgrade_active) {
            ?>
            <!-- AI welcome message -->
            <div class="a1ai-message a1ai-message-ai">
                <div class="a1ai-message-content"><?php echo wp_kses_post($welcome_message); ?></div>
                <div class="a1ai-message-time"><?php echo esc_html(date_i18n(get_option('time_format'))); ?></div>
            </div>
            <?php
        }

        if ($upgrade_active && $is_review_chatbot) {
            // Use the proper generate_review_form() method for consistent design
            // Wrap in standalone container for proper display
            echo '<div class="a1ai-standalone-review-container">';
            
            // Generate the review form using the proper method
            if (isset($public_instance) && method_exists($public_instance, 'generate_review_form')) {
                // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- generate_review_form returns trusted HTML
                echo $public_instance->generate_review_form($chatbot_id);
            } else {
                // Fallback if instance not available (shouldn't happen)
                echo '<div class="a1ai-review-form">';
                echo '<p>' . esc_html__('Review form error. Please refresh the page.', 'a1ai-chatbot') . '</p>';
                echo '</div>';
            }
            
            echo '</div>'; // Close a1ai-standalone-review-container
            
            // Add review form submission handling (still needed for AJAX)
            echo '<script>';
            echo 'function handleReviewSubmissionShortcode() {';
            // Scope to this specific chatbot container
            echo 'var chatbotContainer = document.querySelector(".a1ai-chatbot-container[data-chatbot-id=\'' . esc_js($chatbot_id) . '\']");';
            echo 'if (!chatbotContainer) return;';
            echo 'var reviewForm = chatbotContainer.querySelector(".a1ai-review-submission");';
            echo 'if (reviewForm) {';
            echo 'reviewForm.addEventListener("submit", function(e) {';
            echo 'e.preventDefault();';
            echo 'e.stopPropagation();';
            echo 'var formData = new FormData(this);';
            echo 'formData.append("action", "a1ai_submit_review");';
            echo 'formData.append("nonce", "' . esc_js(wp_create_nonce('a1ai_submit_review')) . '");';
            echo 'formData.append("chatbot_id", "' . esc_js($chatbot_id) . '");';
            echo 'var chatbotContainer = document.querySelector(".a1ai-chatbot-container");';
            echo 'if (chatbotContainer) formData.append("session_id", chatbotContainer.getAttribute("data-session-id"));';

            echo 'fetch("' . esc_url(admin_url('admin-ajax.php')) . '", {';
            echo 'method: "POST", body: formData';
            echo '}).then(response => response.json())';
            echo '.then(data => {';
            echo 'if (data.success) {';
            echo 'var messageDiv = document.createElement("div");';
            echo 'messageDiv.className = "a1ai-message a1ai-message-ai";';
            echo '// Process message to make links clickable';
            echo 'var processedMessage = data.data.message.replace(/(https?:\\/\\/[^\\s<>"{}|\\\\^`[\\]]+)/gi, function(url) {';
            echo '    var currentDomain = window.location.hostname;';
            echo '    try {';
            echo '        var urlObj = new URL(url);';
            echo '        var isInternal = urlObj.hostname === currentDomain;';
            echo '        var linkClass = isInternal ? "a1ai-link-internal" : "a1ai-link-external";';
            echo '        var target = isInternal ? "_self" : "_blank";';
            echo '        var rel = isInternal ? "" : "noopener noreferrer";';
            echo '        return "<a href=\\"" + url + "\\" class=\\"" + linkClass + "\\" target=\\"" + target + "\\" rel=\\"" + rel + "\\">" + url + "</a>";';
            echo '    } catch (e) {';
            echo '        return url;';
            echo '    }';
            echo '});';
            echo 'messageDiv.innerHTML = "<div class=\\"a1ai-message-content\\">" + processedMessage + "</div>";';

            echo 'if (data.data.show_copy_button && data.data.review_summary) {';
            echo 'var summaryDiv = document.createElement("div");';
            echo 'summaryDiv.className = "a1ai-review-summary";';
            echo 'summaryDiv.style = "margin-top: 10px; padding: 10px; background: #f0f8ff; border-radius: 6px; border-left: 4px solid #0073aa;";';
            echo 'var copyButton = document.createElement("button");';
            echo 'copyButton.textContent = "Click to Copy";';
            echo 'copyButton.style.cssText = "background: #0073aa; color: white; padding: 8px 15px; border: none; border-radius: 4px; cursor: pointer; margin-top: 10px;";';
            echo 'copyButton.onclick = function() { navigator.clipboard.writeText(data.data.review_summary); this.textContent = "Copied!"; setTimeout(() => this.textContent = "Click to Copy", 2000); };';
            echo 'summaryDiv.innerHTML = "<strong>Your Review Summary:</strong><br>" + data.data.review_summary + "<br><br>";';
            echo 'summaryDiv.appendChild(copyButton);';
            echo 'messageDiv.appendChild(summaryDiv);';
            echo '}';

            echo 'var messagesContainer = document.querySelector(".a1ai-chatbot-messages");';
            echo 'if (messagesContainer) {';
            echo 'messagesContainer.appendChild(messageDiv);';
            echo 'setTimeout(function() {';
            echo 'messagesContainer.scrollTo({ top: messagesContainer.scrollHeight, behavior: "smooth" });';
            echo '}, 100);';
            echo '}';

            echo 'var reviewForm = document.querySelector(".a1ai-review-form");';
            echo 'if (reviewForm) {';
            echo 'reviewForm.style.display = "none";';
            echo 'console.log("Review form hidden after submission");';
            echo '}';

            echo 'if (data.data.external_links && data.data.external_links.length > 0) {';
            echo 'setTimeout(function() {';
            echo 'data.data.external_links.forEach(function(link) {';
            echo 'window.open(link.url, "_blank", "width=800,height=600,scrollbars=yes,resizable=yes,toolbar=yes,location=yes");';
            echo '});';
            echo '}, 2000);'; // 2 second delay before opening windows
            echo '}';
            echo '} else {';
            echo 'alert("Error: " + (data.data.error || "Unknown error"));';
            echo '}';
            echo '}).catch(error => {';
            echo 'console.error("Review submission error:", error);';
            echo 'alert("An error occurred while submitting your review.");';
            echo '});';
            echo '}); }';
            echo '}';

            echo 'if (document.readyState === "loading") {';
            echo 'document.addEventListener("DOMContentLoaded", handleReviewSubmissionShortcode);';
            echo '} else { handleReviewSubmissionShortcode(); }';

            // Add review mode activation - CSS will handle hiding input and footer
            echo 'function activateReviewModeShortcode() {';
            echo 'var container = document.querySelector(".a1ai-chatbot-container[data-chatbot-id=\'' . esc_js($chatbot_id) . '\']");';
            echo 'if (container) {';
            echo 'container.classList.add("a1ai-review-mode");';
            // CSS handles hiding input and footer via .a1ai-review-mode class
            echo '}';
            echo '}';
            echo 'if (document.readyState === "loading") {';
            echo 'document.addEventListener("DOMContentLoaded", activateReviewModeShortcode);';
            echo '} else { activateReviewModeShortcode(); }';

            echo '</script>';
        }
        ?>

        <!-- Messages will be added here dynamically -->
    </div>
    
    <div class="a1ai-chatbot-input">
        <form class="a1ai-chatbot-form">
            <?php wp_nonce_field('a1ai_submit_message', 'a1ai_message_nonce'); ?>
            <input type="text" class="a1ai-chatbot-input-field" placeholder="<?php echo esc_attr($placeholder_text); ?>" aria-label="<?php esc_attr_e('Message', 'a1ai-chatbot'); ?>">
            <button type="submit" class="a1ai-chatbot-submit" style="background-color: <?php echo esc_attr($primary_color); ?>">
                <span class="screen-reader-text"><?php esc_html_e('Send', 'a1ai-chatbot'); ?></span>
                <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="24" height="24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <line x1="22" y1="2" x2="11" y2="13"></line>
                    <polygon points="22 2 15 22 11 13 2 9 22 2"></polygon>
                </svg>
            </button>
        </form>
    </div>
    
    <?php if (get_option('a1ai_show_powered_by', false)) : ?>
    <div class="a1ai-chatbot-powered-by">
        <?php 
        /* translators: %s: Plugin name with link */
        echo sprintf(esc_html__('Powered by %s', 'a1ai-chatbot'), '<a href="https://a1aidigital.com/ai-implementation/" target="_blank" rel="noopener noreferrer">A1AI</a>'); 
        ?>
    </div>
    <?php endif; ?>
</div> 