/**
 * Public-facing JavaScript for the A1AI Chatbot plugin
 *
 * @package    A1AI
 * @subpackage A1AI/public/js
 */

(function($) {
    'use strict';

    /**
     * A1AI Chatbot handler
     */
    const A1AIChatbot = {
        
        /**
         * Initialize the chatbot functionality
         * @param {string} selector - The selector for chatbot containers
         */
        init: function(selector) {
            const chatbots = document.querySelectorAll(selector);
            
            if (!chatbots.length) {
                return;
            }
            
            chatbots.forEach(chatbot => {
                this.setupChatbot(chatbot);
            });
        },
        
        /**
         * Set up a specific chatbot instance
         * @param {HTMLElement} container - The chatbot container
         */
        setupChatbot: function(container) {
            const chatbotId = container.dataset.chatbotId;
            const sessionId = container.dataset.sessionId;
            const gdprEnabled = container.dataset.gdprEnabled === 'true';
            const form = container.querySelector('.a1ai-chatbot-form');
            const messagesContainer = container.querySelector('.a1ai-chatbot-messages');
            
            
            console.log('Setting up chatbot:', chatbotId, 'GDPR enabled:', gdprEnabled);
            
            if (!form || !messagesContainer) {
                console.error('Chatbot form or messages container not found');
                return;
            }
            
            const welcomeMessages = messagesContainer.querySelectorAll('.a1ai-message-ai');
            
            // Set up GDPR consent if enabled
            if (gdprEnabled) {
                console.log('Setting up GDPR consent');
                this.setupGDPRConsent(container);
                
                // Add click event to show GDPR overlay when user tries to interact with the chatbot
                container.addEventListener('click', (e) => {
                    // If the click is not on the GDPR overlay or its children
                    const overlay = container.querySelector('.a1ai-gdpr-overlay');
                    if (!this.hasGDPRConsent(container) && 
                        overlay && 
                        overlay.style.display !== 'flex' && 
                        !e.target.closest('.a1ai-gdpr-overlay')) {
                        
                        console.log('User clicked chatbot without consent, showing GDPR overlay');
                        overlay.style.display = 'flex';
                        container.classList.add('a1ai-gdpr-active');
                        e.stopPropagation();
                    }
                });
            } else {
                console.log('GDPR consent not enabled');
            }
            
            // Scroll to bottom of messages
            this.scrollToBottom(messagesContainer);
            
            // Add event listener for form submission
            form.addEventListener('submit', event => {
                event.preventDefault();
                
                // Check GDPR consent if enabled
                if (gdprEnabled && !this.hasGDPRConsent(container)) {
                    console.log('GDPR consent required but not given');
                    // Show GDPR overlay if not already visible
                    const overlay = container.querySelector('.a1ai-gdpr-overlay');
                    if (overlay && overlay.style.display !== 'flex') {
                        overlay.style.display = 'flex';
                        container.classList.add('a1ai-gdpr-active');
                    }
                    return;
                }
                
                const input = form.querySelector('.a1ai-chatbot-input-field');
                const message = input.value.trim();
                
                if (message === '') {
                    return;
                }
                
                // Add user message to chat
                this.addUserMessage(messagesContainer, message);
                
                // Clear input
                input.value = '';
                
                // Show loading indicator
                this.showLoadingIndicator(messagesContainer);
                
                // Send message to server
                this.sendMessage(chatbotId, sessionId, message, messagesContainer);
            });

            // Add delegated event listener for review form submission
            container.addEventListener('submit', event => {
                if (event.target.classList.contains('a1ai-review-submission')) {
                    event.preventDefault();
                    console.log('Review form submitted');
                    this.handleReviewSubmission(event.target, chatbotId, messagesContainer);
                }
            });
            
            // Also add click handler directly to submit button as backup
            container.addEventListener('click', event => {
                if (event.target.classList.contains('a1ai-submit-btn') || event.target.closest('.a1ai-submit-btn')) {
                    const submitButton = event.target.closest('.a1ai-submit-btn') || event.target;
                    const reviewForm = submitButton.closest('.a1ai-review-submission') || submitButton.closest('form');
                    if (reviewForm && reviewForm.classList.contains('a1ai-review-submission')) {
                        event.preventDefault();
                        event.stopPropagation();
                        console.log('Submit button clicked directly, triggering form submission');
                        this.handleReviewSubmission(reviewForm, chatbotId, messagesContainer);
                    }
                }
            });
            
            // Initialize star rating for any review forms already present on page load
            if (container.querySelector('.a1ai-review-form')) {
                this.initStarRating(container);
            }
        },
        
        /**
         * Set up GDPR consent functionality
         * @param {HTMLElement} container - The chatbot container
         */
        setupGDPRConsent: function(container) {
            const overlay = container.querySelector('.a1ai-gdpr-overlay');
            const agreeButton = overlay?.querySelector('.a1ai-gdpr-agree');
            const declineButton = overlay?.querySelector('.a1ai-gdpr-decline');
            
            if (!overlay || !agreeButton || !declineButton) {
                console.error('GDPR consent elements not found');
                return;
            }
            
            // Log GDPR status for debugging
            console.log('GDPR enabled:', container.dataset.gdprEnabled);
            console.log('GDPR consent status:', this.hasGDPRConsent(container));
            
            // Check if consent was already given
            if (this.hasGDPRConsent(container)) {
                overlay.style.display = 'none';
                container.classList.remove('a1ai-gdpr-active');
                this.enableChatInput(container);
                return;
            }
            
            // For shortcode chatbots, show GDPR overlay immediately
            if (!container.classList.contains('a1ai-global-chatbot-container')) {
                console.log('Shortcode chatbot: Showing GDPR overlay immediately');
                overlay.style.display = 'flex';
                container.classList.add('a1ai-gdpr-active');
            } else {
                // For global chatbots, keep the existing behavior (hidden by default)
                overlay.style.display = 'none';
                container.classList.remove('a1ai-gdpr-active');
            }
            
            // Disable chat input and messages until consent is given
            this.disableChatInput(container);
            
            // Handle agree button
            agreeButton.addEventListener('click', () => {
                this.setGDPRConsent(container, true);
                overlay.style.display = 'none';
                container.classList.remove('a1ai-gdpr-active');
                
                // Re-enable chat input and messages
                this.enableChatInput(container);
            });
            
            // Handle decline button
            declineButton.addEventListener('click', () => {
                this.setGDPRConsent(container, false);
                overlay.style.display = 'none';
                container.classList.remove('a1ai-gdpr-active');
                // Hide the entire chatbot
                container.style.display = 'none';
            });
        },
        
        /**
         * Disable chat input and messages for GDPR overlay
         * @param {HTMLElement} container - The chatbot container
         */
        disableChatInput: function(container) {
            console.log('Disabling chat input');
            
            const chatInput = container.querySelector('.a1ai-chatbot-input');
            const messagesContainer = container.querySelector('.a1ai-chatbot-messages');
            const inputField = container.querySelector('.a1ai-chatbot-input-field');
            const submitButton = container.querySelector('.a1ai-chatbot-submit');
            
            if (chatInput) {
                chatInput.style.pointerEvents = 'none';
                chatInput.style.opacity = '0.3';
            }
            
            if (messagesContainer) {
                messagesContainer.style.pointerEvents = 'none';
                messagesContainer.style.opacity = '0.3';
            }
            
            if (inputField) {
                inputField.disabled = true;
                inputField.style.pointerEvents = 'none';
                inputField.style.opacity = '0.3';
                inputField.style.cursor = 'not-allowed';
            }
            
            if (submitButton) {
                submitButton.disabled = true;
                submitButton.style.pointerEvents = 'none';
                submitButton.style.opacity = '0.3';
                submitButton.style.cursor = 'not-allowed';
            }
            
            // Add GDPR active class to container
            container.classList.add('a1ai-gdpr-active');
        },
        
        /**
         * Enable chat input and messages after GDPR consent
         * @param {HTMLElement} container - The chatbot container
         */
        enableChatInput: function(container) {
            console.log('Enabling chat input');
            
            const chatInput = container.querySelector('.a1ai-chatbot-input');
            const messagesContainer = container.querySelector('.a1ai-chatbot-messages');
            const inputField = container.querySelector('.a1ai-chatbot-input-field');
            const submitButton = container.querySelector('.a1ai-chatbot-submit');
            
            if (chatInput) {
                chatInput.style.pointerEvents = 'auto';
                chatInput.style.opacity = '1';
            }
            
            if (messagesContainer) {
                messagesContainer.style.pointerEvents = 'auto';
                messagesContainer.style.opacity = '1';
            }
            
            if (inputField) {
                inputField.disabled = false;
                inputField.style.pointerEvents = 'auto';
                inputField.style.opacity = '1';
                inputField.style.cursor = 'text';
                inputField.style.backgroundColor = 'white';
                inputField.focus(); // Focus the input field for better UX
            }
            
            if (submitButton) {
                submitButton.disabled = false;
                submitButton.style.pointerEvents = 'auto';
                submitButton.style.opacity = '1';
                submitButton.style.cursor = 'pointer';
            }
            
            // Remove GDPR active class from container
            container.classList.remove('a1ai-gdpr-active');
        },
        
        /**
         * Check if GDPR consent has been given
         * @param {HTMLElement} container - The chatbot container
         * @returns {boolean} - True if consent given
         */
        hasGDPRConsent: function(container) {
            const chatbotId = container.dataset.chatbotId;
            const sessionId = container.dataset.sessionId;
            const consentKey = `a1ai_gdpr_consent_${chatbotId}_${sessionId}`;
            const consentValue = localStorage.getItem(consentKey);
            
            console.log('Checking GDPR consent:', consentKey, 'Value:', consentValue);
            
            return consentValue === 'true';
        },
        
        /**
         * Set GDPR consent
         * @param {HTMLElement} container - The chatbot container
         * @param {boolean} consented - Whether user consented
         */
        setGDPRConsent: function(container, consented) {
            const chatbotId = container.dataset.chatbotId;
            const sessionId = container.dataset.sessionId;
            const consentKey = `a1ai_gdpr_consent_${chatbotId}_${sessionId}`;
            localStorage.setItem(consentKey, consented.toString());
            
            // Clean up old consent data (keep only last 10 sessions per chatbot)
            this.cleanupOldConsentData(chatbotId);
        },
        
        /**
         * Clean up old GDPR consent data to prevent localStorage overflow
         * @param {number} chatbotId - The chatbot ID
         */
        cleanupOldConsentData: function(chatbotId) {
            const maxSessions = 10;
            const consentKeys = [];
            
            // Find all consent keys for this chatbot
            for (let i = 0; i < localStorage.length; i++) {
                const key = localStorage.key(i);
                if (key && key.startsWith(`a1ai_gdpr_consent_${chatbotId}_`)) {
                    consentKeys.push(key);
                }
            }
            
            // If we have more than maxSessions, remove the oldest ones
            if (consentKeys.length > maxSessions) {
                consentKeys.sort(); // Sort by key (which includes timestamp)
                const keysToRemove = consentKeys.slice(0, consentKeys.length - maxSessions);
                keysToRemove.forEach(key => localStorage.removeItem(key));
            }
        },
        
        /**
         * Add a user message to the chat
         * @param {HTMLElement} container - The messages container
         * @param {string} message - The message text
         */
        addUserMessage: function(container, message) {
            const time = new Date().toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'});
            const messageHtml = `
                <div class="a1ai-message a1ai-message-user">
                    <div class="a1ai-message-content">${message}</div>
                    <div class="a1ai-message-time">${time}</div>
                </div>
            `;
            
            container.insertAdjacentHTML('beforeend', messageHtml);
            this.scrollToBottom(container);
        },
        
        /**
         * Convert URLs in text to clickable links
         * @param {string} text - The text to process
         * @returns {string} - Text with URLs converted to clickable links
         */
        makeLinksClickable: function(text) {
            // Regular expression to match URLs
            const urlRegex = /(https?:\/\/[^\s<>"{}|\\^`[\]]+)/gi;
            
            return text.replace(urlRegex, function(url) {
                // Check if it's an internal link (same domain)
                const currentDomain = window.location.hostname;
                const urlObj = new URL(url);
                const isInternal = urlObj.hostname === currentDomain;
                
                // Create clickable link
                const linkClass = isInternal ? 'a1ai-link-internal' : 'a1ai-link-external';
                const target = isInternal ? '_self' : '_blank';
                const rel = isInternal ? '' : 'noopener noreferrer';
                
                return `<a href="${url}" class="${linkClass}" target="${target}" rel="${rel}">${url}</a>`;
            });
        },

        /**
         * Add an AI message to the chat
         * @param {HTMLElement} container - The messages container
         * @param {string} message - The message text
         */
        addAIMessage: function(container, message) {
            const time = new Date().toLocaleTimeString([], {hour: '2-digit', minute:'2-digit'});
            
            // Process message to make links clickable
            const processedMessage = this.makeLinksClickable(message);
            
            const messageHtml = `
                <div class="a1ai-message a1ai-message-ai">
                    <div class="a1ai-message-content">${processedMessage}</div>
                    <div class="a1ai-message-time">${time}</div>
                </div>
            `;
            
            // Remove loading indicator if present
            const loadingIndicator = container.querySelector('.a1ai-message-loading');
            if (loadingIndicator) {
                loadingIndicator.remove();
            }
            
            container.insertAdjacentHTML('beforeend', messageHtml);
            this.scrollToBottom(container);
        },
        
        /**
         * Show loading indicator
         * @param {HTMLElement} container - The messages container
         */
        showLoadingIndicator: function(container) {
            const loadingHtml = `
                <div class="a1ai-message a1ai-message-ai a1ai-message-loading">
                    <div class="a1ai-loading-dot"></div>
                    <div class="a1ai-loading-dot"></div>
                    <div class="a1ai-loading-dot"></div>
                </div>
            `;
            
            container.insertAdjacentHTML('beforeend', loadingHtml);
            this.scrollToBottom(container);
        },
        
        /**
         * Send a message to the server
         * @param {number} chatbotId - The chatbot ID
         * @param {string} sessionId - The session ID
         * @param {string} message - The message text
         * @param {HTMLElement} container - The messages container
         */
        sendMessage: function(chatbotId, sessionId, message, container) {
            $.ajax({
                url: a1ai_public_params.ajax_url,
                type: 'POST',
                data: {
                    action: 'a1ai_send_message',
                    chatbot_id: chatbotId,
                    session_id: sessionId,
                    message: message,
                    nonce: a1ai_public_params.nonce
                },
                success: response => {
                    console.log('A1AI Review Debug: Server response:', response);

                    if (response.success) {
                        this.addAIMessage(container, response.data.message);

                        // Check if we should show review form
                        if (response.data.show_review_form && response.data.review_form_html) {
                            console.log('A1AI Review Debug: Showing review form');
                            this.addReviewForm(container, response.data.review_form_html);
                        } else {
                            console.log('A1AI Review Debug: No review form. show_review_form:', response.data.show_review_form, 'review_form_html:', !!response.data.review_form_html);
                        }
                    } else {
                        this.addAIMessage(container, a1ai_public_params.error_message);
                        console.error('Error sending message:', response.data.error);
                    }
                },
                error: (xhr, status, error) => {
                    this.addAIMessage(container, a1ai_public_params.error_message);
                    console.error('AJAX error:', error);
                }
            });
        },
        
        /**
         * Add a review form to the chat
         * @param {HTMLElement} container - The messages container
         * @param {string} formHTML - The review form HTML
         */
        addReviewForm: function(container, formHTML) {
            // Wrap in standalone container without chat message styling
            const reviewFormHtml = `
                <div class="a1ai-standalone-review-container">
                    ${formHTML}
                </div>
            `;

            container.insertAdjacentHTML('beforeend', reviewFormHtml);

            // Hide chat interface elements when review form is shown
            // Try both container class names (shortcode uses .a1ai-chatbot-container, global uses .a1ai-global-chatbot-container)
            const chatbotContainer = container.closest('.a1ai-chatbot-container, .a1ai-global-chatbot-container');
            if (chatbotContainer) {
                // Add review mode class to container - CSS will handle hiding input and footer
                chatbotContainer.classList.add('a1ai-review-mode');

                // Update header to show review mode
                const header = chatbotContainer.querySelector('.a1ai-chatbot-header');
                const title = chatbotContainer.querySelector('.a1ai-chatbot-title');
                if (header && title) {
                    // Save original title for restoration later
                    if (!chatbotContainer.dataset.originalTitle) {
                        chatbotContainer.dataset.originalTitle = title.textContent;
                    }
                    title.textContent = 'Share Your Experience';
                }
            }

            // Initialize star rating for dynamically added form
            this.initStarRating(container);

            this.scrollToBottom(container);
        },

        /**
         * Initialize star rating functionality for review forms
         * @param {HTMLElement} container - The container to search for review forms
         */
        initStarRating: function(container) {
            console.log('Initializing star rating');
            
            // Find the review form within the container
            const reviewForm = container.querySelector('.a1ai-review-form');
            if (!reviewForm) {
                console.log('No review form found in container');
                return;
            }
            
            // Find all star rating containers in this review form
            const stars = reviewForm.querySelectorAll('.a1ai-star');
            const ratingInput = reviewForm.querySelector('#a1ai-rating-value');
            
            console.log('Found stars:', stars.length, 'Rating input:', !!ratingInput);
            
            if (stars.length > 0 && ratingInput) {
                // Store reference to input for closure
                const inputField = ratingInput;
                
                stars.forEach((star, index) => {
                    // Remove any existing click handlers to avoid duplicates
                    const newStar = star.cloneNode(true);
                    star.parentNode.replaceChild(newStar, star);
                    
                    newStar.addEventListener('click', (e) => {
                        e.preventDefault();
                        e.stopPropagation();
                        
                        const rating = parseInt(newStar.getAttribute('data-rating'));
                        inputField.value = rating;
                        
                        console.log('Star clicked, rating set to:', rating, 'Input value is now:', inputField.value);
                        
                        // Update visual state of all stars (get fresh reference after cloning)
                        const allStars = reviewForm.querySelectorAll('.a1ai-star');
                        allStars.forEach((s, i) => {
                            s.classList.remove('active');
                            if (i < rating) {
                                s.classList.add('active');
                            }
                        });
                    });
                });
                
                console.log('Star rating initialized successfully for', stars.length, 'stars');
            } else {
                console.log('Star rating not initialized - missing stars or rating input');
            }
            
            // Handle 10-scale rating if present
            const scaleInput = reviewForm.querySelector('#a1ai-rating-scale');
            const scaleValue = reviewForm.querySelector('#a1ai-scale-value');
            
            if (scaleInput && scaleValue) {
                scaleInput.addEventListener('input', function() {
                    scaleValue.textContent = this.value;
                });
                console.log('10-scale rating initialized');
            }
        },

        /**
         * Handle review form submission
         * @param {HTMLFormElement} form - The review form
         * @param {number} chatbotId - The chatbot ID
         * @param {HTMLElement} container - The messages container
         */
        handleReviewSubmission: function(form, chatbotId, container) {
            console.log('handleReviewSubmission called', { chatbotId, form });

            // Get form data
            const formData = new FormData(form);
            const rating = formData.get('rating');
            const comment = formData.get('comment');
            const reviewerName = formData.get('reviewer_name');
            const reviewerEmail = formData.get('reviewer_email');

            console.log('Review form data:', { rating, comment, reviewerName, reviewerEmail });
            console.log('Rating value type:', typeof rating, 'Rating is empty?', !rating);

            // Validate required fields with specific messages
            if (!rating || rating === '' || rating === '0') {
                alert('Please select a rating by clicking on the stars.');
                console.log('Validation failed: No rating selected');
                return;
            }
            
            if (!reviewerName || reviewerName.trim() === '') {
                alert('Please enter your name.');
                console.log('Validation failed: No name provided');
                return;
            }
            
            if (!reviewerEmail || reviewerEmail.trim() === '') {
                alert('Please enter your email address.');
                console.log('Validation failed: No email provided');
                return;
            }

            // Disable form during submission
            const submitButton = form.querySelector('button[type="submit"]');
            if (submitButton) {
                submitButton.disabled = true;
                submitButton.textContent = 'Submitting...';
            }
            
            // Get session ID from container
            const chatbotContainer = container.closest('.a1ai-chatbot-container, .a1ai-global-chatbot-container');
            const sessionId = chatbotContainer ? chatbotContainer.dataset.sessionId : '';
            
            console.log('Submitting review with session ID:', sessionId);

            // Send review to server
            $.ajax({
                url: a1ai_public_params.ajax_url,
                type: 'POST',
                data: {
                    action: 'a1ai_submit_review',
                    nonce: a1ai_public_params.review_nonce || a1ai_public_params.nonce,
                    chatbot_id: chatbotId,
                    rating: rating,
                    comment: comment,
                    reviewer_name: reviewerName,
                    reviewer_email: reviewerEmail,
                    session_id: sessionId
                },
                success: response => {
                    console.log('Review submission response:', response);

                    if (response.success) {
                        // Remove the review form
                        const reviewContainer = form.closest('.a1ai-standalone-review-container');
                        if (reviewContainer) {
                            reviewContainer.remove();
                        }

                        // Add thank you message
                        this.addAIMessage(container, response.data.message || 'Thank you for your review!');

                        // Restore chat interface elements
                        const chatbotContainer = container.closest('.a1ai-chatbot-container, .a1ai-global-chatbot-container');
                        if (chatbotContainer) {
                            // Remove review mode class
                            chatbotContainer.classList.remove('a1ai-review-mode');

                            // Show the chat input area again (try both possible class names)
                            const inputWrapper = chatbotContainer.querySelector('.a1ai-chatbot-input, .a1ai-input-wrapper');
                            if (inputWrapper) {
                                inputWrapper.style.display = '';
                            }

                            // Restore original header title
                            const title = chatbotContainer.querySelector('.a1ai-chatbot-title');
                            if (title) {
                                const originalTitle = chatbotContainer.dataset.originalTitle || 'Chat Support';
                                title.textContent = originalTitle;
                            }
                        }

                        // Fire conversion tracking
                        if (typeof A1AIConversionTracking !== 'undefined') {
                            A1AIConversionTracking.trackEvent('review_submission', {
                                chatbotId: chatbotId,
                                rating: rating
                            });
                        }

                        // Open external review links if provided (with 2 second delay)
                        if (response.data.external_links && response.data.external_links.length > 0) {
                            setTimeout(function() {
                                response.data.external_links.forEach(link => {
                                    window.open(link.url, '_blank');
                                });
                            }, 2000); // 2 second delay before opening windows
                        }
                    } else {
                        alert('Error submitting review: ' + (response.data.error || 'Unknown error'));
                        if (submitButton) {
                            submitButton.disabled = false;
                            submitButton.textContent = 'Submit Review';
                        }
                    }
                },
                error: (xhr, status, error) => {
                    console.error('Review submission error:', error);
                    alert('Error submitting review. Please try again.');
                    if (submitButton) {
                        submitButton.disabled = false;
                        submitButton.textContent = 'Submit Review';
                    }
                }
            });
        },

        /**
         * Scroll to the bottom of a container
         * @param {HTMLElement} container - The container to scroll
         */
        scrollToBottom: function(container) {
            container.scrollTop = container.scrollHeight;
        }
    };

    /**
     * Global Chatbot Handler
     */
    const A1AIGlobalChatbot = {
        
        /**
         * Initialize the global chatbot
         */
        init: function() {
            const globalChatbot = document.querySelector('.a1ai-global-chatbot-container');
            
            if (!globalChatbot) {
                return;
            }
            
            console.log('Initializing global chatbot');
            
            // Set up toggle functionality
            const toggleButton = globalChatbot.querySelector('.a1ai-global-chatbot-toggle');
            const chatbotDialog = globalChatbot.querySelector('.a1ai-global-chatbot-dialog');
            const closeButton = globalChatbot.querySelector('.a1ai-chatbot-close');
            const gdprEnabled = globalChatbot.dataset.gdprEnabled === 'true';
            
            console.log('Global chatbot GDPR enabled:', gdprEnabled);
            
            if (toggleButton && chatbotDialog) {
                toggleButton.addEventListener('click', () => {
                    // Check GDPR consent if enabled
                    if (gdprEnabled && !A1AIChatbot.hasGDPRConsent(globalChatbot)) {
                        console.log('Global chatbot: GDPR consent required');
                        // Show GDPR overlay instead of chatbot
                        const overlay = globalChatbot.querySelector('.a1ai-gdpr-overlay');
                        if (overlay) {
                            console.log('Showing GDPR overlay');
                            overlay.style.display = 'flex';
                            globalChatbot.classList.add('a1ai-gdpr-active');
                        }
                        return;
                    }
                    
                    if (chatbotDialog.style.display === 'none' || !chatbotDialog.style.display) {
                        chatbotDialog.style.display = 'flex';
                        globalChatbot.classList.add('a1ai-active');
                    } else {
                        chatbotDialog.style.display = 'none';
                        globalChatbot.classList.remove('a1ai-active');
                    }
                });
                
                if (closeButton) {
                    closeButton.addEventListener('click', () => {
                        chatbotDialog.style.display = 'none';
                        globalChatbot.classList.remove('a1ai-active');
                    });
                }
            }
            
            // Set up chatbot functionality
            A1AIChatbot.setupChatbot(globalChatbot);
            
            // Set up GDPR consent for global chatbot specifically
            if (gdprEnabled) {
                const overlay = globalChatbot.querySelector('.a1ai-gdpr-overlay');
                const agreeButton = overlay?.querySelector('.a1ai-gdpr-agree');
                const declineButton = overlay?.querySelector('.a1ai-gdpr-decline');
                const chatbotDialog = globalChatbot.querySelector('.a1ai-global-chatbot-dialog');
                
                if (overlay && agreeButton && declineButton) {
                    // Handle agree button for global chatbot
                    agreeButton.addEventListener('click', () => {
                        A1AIChatbot.setGDPRConsent(globalChatbot, true);
                        overlay.style.display = 'none';
                        globalChatbot.classList.remove('a1ai-gdpr-active');
                        
                        // Re-enable chat input and messages
                        A1AIChatbot.enableChatInput(globalChatbot);
                        
                        // Show the chatbot dialog
                        if (chatbotDialog) {
                            chatbotDialog.style.display = 'flex';
                            globalChatbot.classList.add('a1ai-active');
                        }
                    });
                    
                    // Handle decline button for global chatbot
                    declineButton.addEventListener('click', () => {
                        A1AIChatbot.setGDPRConsent(globalChatbot, false);
                        overlay.style.display = 'none';
                        globalChatbot.classList.remove('a1ai-gdpr-active');
                        // Hide the entire global chatbot
                        globalChatbot.style.display = 'none';
                    });
                }
            }
        }
    };

    /**
     * Initialize everything when DOM is ready
     */
    $(document).ready(function() {
        console.log('A1AI: Initializing chatbots');
        
        // Initialize shortcode chatbots
        const shortcodeChatbots = document.querySelectorAll('.a1ai-chatbot-container:not(.a1ai-global-chatbot-container .a1ai-chatbot-container)');
        console.log('A1AI: Found shortcode chatbots:', shortcodeChatbots.length);
        A1AIChatbot.init('.a1ai-chatbot-container:not(.a1ai-global-chatbot-container .a1ai-chatbot-container)');
        
        // Initialize global chatbot
        const globalChatbot = document.querySelector('.a1ai-global-chatbot-container');
        console.log('A1AI: Global chatbot found:', !!globalChatbot);
        if (globalChatbot) {
            console.log('A1AI: Global chatbot ID:', globalChatbot.dataset.chatbotId);
            console.log('A1AI: Global chatbot GDPR enabled:', globalChatbot.dataset.gdprEnabled);
        }
        A1AIGlobalChatbot.init();
    });

})(jQuery); 