<?php
/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    A1AI
 * @subpackage A1AI/public
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and hooks for the public-facing side of the site.
 *
 * @package    A1AI
 * @subpackage A1AI/public
 * @author     A1AI <info@example.com>
 */
class A1AI_Public {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * The data manager instance.
     *
     * @since    1.0.0
     * @access   private
     * @var      A1AI_Data_Manager    $data_manager    Handles database operations.
     */
    private $data_manager;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param    string                $plugin_name       The name of this plugin.
     * @param    string                $version           The version of this plugin.
     * @param    A1AI_Data_Manager     $data_manager      The data manager instance.
     */
    public function __construct($plugin_name = 'a1ai', $version = '1.3.0', $data_manager = null) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
        $this->data_manager = $data_manager;
        
        // If data manager wasn't passed, create one
        if (null === $this->data_manager) {
            require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-a1ai-data-manager.php';
            $this->data_manager = new A1AI_Data_Manager();
        }
    }

    /**
     * Register the stylesheets for the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function enqueue_styles() {
        wp_enqueue_style(
            $this->plugin_name,
            plugin_dir_url(__FILE__) . 'css/a1ai-public.css',
            array(),
            $this->version,
            'all'
        );
    }

    /**
     * Register the JavaScript for the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function enqueue_scripts() {
        wp_enqueue_script(
            $this->plugin_name,
            plugin_dir_url(__FILE__) . 'js/a1ai-public.js',
            array('jquery'),
            $this->version,
            true
        );
        
        // Localize script with necessary data
        wp_localize_script(
            $this->plugin_name,
            'a1ai_public_params',
            array(
                'ajax_url' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('a1ai_public_nonce'),
                'review_nonce' => wp_create_nonce('a1ai_submit_review'),
                'error_message' => esc_html__('Sorry, I encountered an error processing your request. Have the admin review the dashboard and please try again later.', 'a1ai-chatbot'),
            )
        );
    }

    /**
     * Register the shortcodes.
     *
     * @since    1.0.0
     */
    public function register_shortcodes() {
        add_shortcode('a1ai_chatbot', array($this, 'render_chatbot_shortcode'));
    }

    /**
     * Render the chatbot shortcode.
     *
     * @since    1.0.0
     * @param    array    $atts    Shortcode attributes.
     * @return   string            Shortcode output.
     */
    public function render_chatbot_shortcode($atts) {
        $atts = shortcode_atts(
            array(
                'id' => 0,
            ),
            $atts,
            'a1ai_chatbot'
        );
        
        // Sanitize attributes
        $chatbot_id = intval($atts['id']);
        
        if ($chatbot_id <= 0) {
            return '<div class="a1ai-error">' . esc_html__('Invalid chatbot ID.', 'a1ai-chatbot') . '</div>';
        }
        
        // Get chatbot data
        $chatbot = $this->data_manager->get_chatbot($chatbot_id);
        
        if (empty($chatbot)) {
            return '<div class="a1ai-error">' . esc_html__('Chatbot not found.', 'a1ai-chatbot') . '</div>';
        }
        
        // Get session ID for user
        $session_id = $this->get_session_id();
        
        // Pass Public instance to template
        $public_instance = $this;
        
        // Start output buffering
        ob_start();
        
        // Include template
        $template_path = plugin_dir_path(__FILE__) . 'partials/a1ai-shortcode-chatbot.php';
        
        if (file_exists($template_path)) {
            include $template_path;
        } else {
            return '<div class="a1ai-error">' . esc_html__('Chatbot template not found.', 'a1ai-chatbot') . '</div>';
        }
        
        // Get the buffered content
        $output = ob_get_clean();

        // Apply filter to allow other plugins to modify the chatbot output
        $output = apply_filters('a1ai_chatbot_output', $output, $chatbot_id);

        return $output;
    }

    /**
     * Render the global chatbot.
     *
     * @since    1.0.0
     */
    public function render_global_chatbot() {
        // Simple test output
        echo '<!-- A1AI Global Chatbot: Method called -->';
        
        // Check if global chatbot is enabled
        $enabled = get_option('a1ai_global_chatbot_enabled', false);
        
        // Debug: Global chatbot enabled status
        if (defined('WP_DEBUG') && WP_DEBUG) {
            echo '<!-- Global Chatbot Debug: enabled=' . ($enabled ? 'true' : 'false') . ' -->';
        }
        
        if (!$enabled) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                echo '<!-- Global Chatbot: Not enabled -->';
            }
            return;
        }
        
        // Get global chatbot ID
        $chatbot_id = intval(get_option('a1ai_global_chatbot_id', 0));
        
        if (defined('WP_DEBUG') && WP_DEBUG) {
            echo '<!-- Global Chatbot Debug: chatbot_id=' . esc_html($chatbot_id) . ' -->';
        }
        
        if ($chatbot_id <= 0) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                echo '<!-- Global Chatbot: No chatbot ID set -->';
            }
            return;
        }
        
        // Get chatbot data
        $chatbot = $this->data_manager->get_chatbot($chatbot_id);
        
        if (empty($chatbot)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                echo '<!-- Global Chatbot: Chatbot not found for ID ' . esc_html($chatbot_id) . ' -->';
            }
            return;
        }
        
        if (defined('WP_DEBUG') && WP_DEBUG) {
            echo '<!-- Global Chatbot: Rendering chatbot ' . esc_html($chatbot_id) . ' -->';
        }
        
        // Get global position setting
        $global_position = get_option('a1ai_global_chatbot_position', 'bottom-right');
        
        // Get session ID for user
        $session_id = $this->get_session_id();
        
        // Pass Public instance to template
        $public_instance = $this;
        
        // Include template
        $template_path = plugin_dir_path(__FILE__) . 'partials/a1ai-global-chatbot.php';
        
        if (file_exists($template_path)) {
            include $template_path;
        } else {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                echo '<!-- Global Chatbot: Template file not found -->';
            }
        }
    }
    
    /**
     * Handle AJAX request to send a message.
     *
     * @since    1.0.0
     */
    public function handle_send_message() {
        // Basic debug logging
        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging wrapped in WP_DEBUG check
            error_log("A1AI: handle_send_message called at " . gmdate('Y-m-d H:i:s'));
        }

        // Check nonce for security
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_key(wp_unslash($_POST['nonce'])), 'a1ai_public_nonce')) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging wrapped in WP_DEBUG check
                error_log("A1AI: Nonce check failed");
            }
            wp_send_json_error(array('error' => 'Security check failed.'));
        }

        // Rate limiting: Prevent API abuse
        $user_ip = isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : '';
        $rate_limit_key = 'a1ai_rate_limit_' . md5($user_ip);
        $request_count = get_transient($rate_limit_key);
        $rate_limit = apply_filters('a1ai_rate_limit_messages_per_minute', 60); // Allow filtering

        if ($request_count && $request_count >= $rate_limit) {
            wp_send_json_error(array('error' => 'Rate limit exceeded. Please wait before sending more messages.'));
        }

        // Increment request count
        set_transient($rate_limit_key, ($request_count ? $request_count + 1 : 1), MINUTE_IN_SECONDS);

        // Check required parameters
        if (!isset($_POST['chatbot_id']) || !isset($_POST['session_id']) || !isset($_POST['message'])) {
            wp_send_json_error(array('error' => 'Missing required parameters.'));
        }
        
        // Sanitize inputs
        $chatbot_id = intval($_POST['chatbot_id']);
        $session_id = sanitize_text_field(wp_unslash($_POST['session_id']));
        $message = sanitize_textarea_field(wp_unslash($_POST['message']));
        
        if (empty($message)) {
            wp_send_json_error(array('error' => 'Message cannot be empty.'));
        }
        
        // Get chatbot
        $chatbot = $this->data_manager->get_chatbot($chatbot_id);
        
        if (empty($chatbot)) {
            wp_send_json_error(array('error' => 'Chatbot not found.'));
        }
        
        // Get or create conversation
        $conversation = $this->data_manager->get_conversation_by_session($session_id, $chatbot_id);
        
        if (empty($conversation)) {
            $user_id = is_user_logged_in() ? get_current_user_id() : 0;
            // Fix: correct parameter order (chatbot_id, session_id, user_id)
            $conversation_id = $this->data_manager->create_conversation($chatbot_id, $session_id, $user_id);
            $conversation = $this->data_manager->get_conversation($conversation_id);
            
            // Send email notification for new chat
            if (!empty($conversation)) {
                $this->data_manager->send_new_chat_notification($chatbot_id, $session_id, $message);
            }
        }
        
        if (empty($conversation)) {
            wp_send_json_error(array('error' => 'Failed to create conversation.'));
        }
        
        // Store user message
        $this->data_manager->add_message($conversation['id'], 'user', $message);
        
        // Check if this is the review chatbot
        $review_chatbot_id = get_option('a1ai_review_chatbot_id', 0);
        $is_review_chatbot = ($chatbot_id == $review_chatbot_id && $review_chatbot_id > 0);

        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging wrapped in WP_DEBUG check
            error_log("A1AI Review: chatbot_id=$chatbot_id, review_chatbot_id=$review_chatbot_id, is_review_chatbot=" . ($is_review_chatbot ? 'true' : 'false'));
        }

        // Process with AI and get response
        try {
            $settings = maybe_unserialize($chatbot['settings']);
            $ai_response_data = $this->process_ai_message($message, $conversation['id'], $settings, $chatbot_id);
        } catch (Exception $e) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging wrapped in WP_DEBUG check
                error_log("A1AI Error in AI processing: " . $e->getMessage());
            }
            wp_send_json_error(array('error' => 'AI processing failed.'));
            return;
        }

        // Store AI response with token usage
        $tokens_used = isset($ai_response_data['tokens_used']) ? intval($ai_response_data['tokens_used']) : 0;
        $this->data_manager->add_message($conversation['id'], 'assistant', $ai_response_data['message'], $tokens_used);

        // Check for conversation inactivity and send full conversation email if needed (premium feature)
        if (class_exists('A1AI_Upgrade_Helper') && A1AI_Upgrade_Helper::is_upgrade_active()) {
            // Load email manager if not already loaded
            $email_manager_path = WP_PLUGIN_DIR . '/a1ai-chatbot-upgrade/includes/class-a1ai-email-manager.php';
            if (file_exists($email_manager_path) && !class_exists('A1AI_Email_Manager')) {
                require_once $email_manager_path;
            }
            
            if (class_exists('A1AI_Email_Manager')) {
                $email_manager = new A1AI_Email_Manager();
                $email_manager->check_and_send_inactivity_email($conversation['id'], $chatbot_id);
            }
        }

        // Check if we should show review form after this conversation
        $response_data = array('message' => $ai_response_data['message']);

        // Check if upgrade plugin is active (reviews are a premium feature)
        $upgrade_active = class_exists('A1AI_Upgrade_Helper') && A1AI_Upgrade_Helper::is_upgrade_active();

        // Only show review forms if upgrade plugin is active
        if ($upgrade_active) {
            // Check if user is asking for a review (works on any chatbot)
            $is_review_request = $this->is_review_request($message, $session_id);

            // Show review form if:
            // 1. This is the designated review chatbot and conditions are met, OR
            // 2. User specifically asks for a review on any chatbot
            if (($is_review_chatbot && $this->should_show_review_form($conversation['id'], $message, $session_id)) || $is_review_request) {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging wrapped in WP_DEBUG check
                    error_log("A1AI Review: Showing review form. is_review_chatbot=$is_review_chatbot, is_review_request=$is_review_request");
                }
                $response_data['show_review_form'] = true;
                $response_data['review_form_html'] = $this->generate_review_form($chatbot_id);
            } else {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging wrapped in WP_DEBUG check
                    error_log("A1AI Review: NOT showing review form. is_review_chatbot=$is_review_chatbot, is_review_request=$is_review_request");
                }
            }
        }

        // Send response
        wp_send_json_success($response_data);
    }
    
    /**
     * Process message with AI.
     *
     * @since    1.0.0
     * @param    string    $message          The user message.
     * @param    int       $conversation_id  The conversation ID.
     * @param    array     $settings         The chatbot settings.
     * @param    int       $chatbot_id       The chatbot ID.
     * @return   string                      The AI response.
     */
    private function process_ai_message($message, $conversation_id, $settings, $chatbot_id = 0) {
        // Load OpenAI integration class
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-a1ai-openai.php';
        $openai = new A1AI_OpenAI($this->data_manager);
        
        // Add chatbot ID to settings
        $settings['chatbot_id'] = $chatbot_id;
        
        // Process with OpenAI
        $response = $openai->process_conversation($conversation_id, $message, $settings);
        
        // Check if successful
        if ($response['success']) {
            return array(
                'message' => $response['message'],
                'tokens_used' => isset($response['tokens_used']) ? intval($response['tokens_used']) : 0,
            );
        } else {
            // Log error for admin using WordPress debug logging
            if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                // Only log when debugging is enabled
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('A1AI OpenAI Error: ' . $response['error']);
            }

            // Return friendly error message to user
            return array(
                'message' => esc_html__('Sorry, I encountered an error processing your request. Have the admin review the dashboard and please try again later.', 'a1ai-chatbot'),
                'tokens_used' => 0,
            );
        }
    }
    
    /**
     * Generate a unique session ID for the user.
     *
     * @since    1.0.0
     * @return   string    Unique session ID.
     */
    private function get_session_id() {
        // Check if user is logged in
        if (is_user_logged_in()) {
            $user_id = get_current_user_id();
            $session_id = 'user_' . $user_id . '_' . wp_generate_uuid4();
        } else {
            // For guests, use cookies to track sessions
            if (!isset($_COOKIE['a1ai_session_id'])) {
                $session_id = 'guest_' . wp_generate_uuid4();
                setcookie('a1ai_session_id', $session_id, time() + (86400 * 30), COOKIEPATH, COOKIE_DOMAIN, is_ssl());
            } else {
                $session_id = sanitize_text_field(wp_unslash($_COOKIE['a1ai_session_id']));
            }
        }
        
        return $session_id;
    }

    /**
     * Determine if review form should be shown
     *
     * @since    1.0.0
     * @param    int       $conversation_id  The conversation ID
     * @param    string    $message          The user message
     * @return   bool                        Whether to show review form
     */
    private function should_show_review_form($conversation_id, $message, $session_id = '') {
        // Check if a review has already been submitted in this session
        if (!empty($session_id) && get_transient('a1ai_review_submitted_' . $session_id)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging wrapped in WP_DEBUG check
                error_log("A1AI Review: Review already submitted for session $session_id, not showing form");
            }
            return false;
        }

        // Get message count in this conversation
        $messages = $this->data_manager->get_messages($conversation_id);
        $message_count = count($messages);

        if (defined('WP_DEBUG') && WP_DEBUG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging wrapped in WP_DEBUG check
            error_log("A1AI Review: Conversation $conversation_id has $message_count messages");
        }

        // For the designated review chatbot, show form immediately on first interaction
        // For other chatbots, show after meaningful conversation
        $min_messages = 2; // Show immediately after first exchange (1 user + 1 AI)

        if ($message_count >= $min_messages) {
            $session_key = 'a1ai_review_shown_' . $conversation_id;
            $already_shown = get_transient($session_key);

            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging wrapped in WP_DEBUG check
                error_log("A1AI Review: Session key: $session_key, Already shown: " . ($already_shown ? 'true' : 'false'));
            }

            if (!$already_shown) {
                // Mark as shown for 24 hours to avoid repeated requests
                set_transient($session_key, true, 24 * HOUR_IN_SECONDS);
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging wrapped in WP_DEBUG check
                    error_log("A1AI Review: Setting transient and returning true");
                }
                return true;
            } else {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging wrapped in WP_DEBUG check
                    error_log("A1AI Review: Already shown, returning false");
                }
            }
        } else {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging wrapped in WP_DEBUG check
                error_log("A1AI Review: Not enough messages ($message_count < $min_messages), returning false");
            }
        }

        return false;
    }

    /**
     * Check if user message is requesting a review
     *
     * @since    1.0.0
     * @param    string    $message  The user message
     * @return   bool                Whether message is requesting a review
     */
    private function is_review_request($message, $session_id = '') {
        // Check if a review has already been submitted in this session
        if (!empty($session_id) && get_transient('a1ai_review_submitted_' . $session_id)) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging only active when WP_DEBUG is enabled
                error_log("A1AI Review: Review already submitted for session $session_id, skipping");
            }
            return false;
        }

        $message_lower = strtolower($message);

        // Keywords that indicate user wants to leave a review
        $review_keywords = array(
            'review',
            'rating',
            'feedback',
            'rate',
            'leave a review',
            'write a review',
            'give a review',
            'leave feedback',
            'rate you',
            'rate this',
            'rate your service',
            'how was',
            'experience'
        );

        foreach ($review_keywords as $keyword) {
            if (strpos($message_lower, $keyword) !== false) {
                if (defined('WP_DEBUG') && WP_DEBUG) {
                    // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging wrapped in WP_DEBUG check
                    error_log("A1AI Review: Detected review request keyword: $keyword in message: $message");
                }
                return true;
            }
        }

        return false;
    }

    /**
     * Generate HTML for review form
     *
     * @since    1.0.0
     * @param    int       $chatbot_id  The chatbot ID
     * @return   string                 Review form HTML
     */
    public function generate_review_form($chatbot_id) {
        $rating_system = get_option('a1ai_rating_system', '5-star');
        $intro_text = get_option('a1ai_review_intro_text', 'Thank you for taking the time to review us, please let us know how your experience has been!');

        // Modern card-based form generation
        $form_html = '<div class="a1ai-review-form">';
        $form_html .= '<div class="a1ai-review-header">';
        $form_html .= '<div class="a1ai-review-icon">&#9733;</div>';
        $form_html .= '<h3 class="a1ai-review-title">' . esc_html__('Share Your Experience', 'a1ai-chatbot') . '</h3>';
        $form_html .= '</div>';

        $form_html .= '<div class="a1ai-intro-message">';
        $form_html .= '<p>' . esc_html($intro_text) . '</p>';
        $form_html .= '</div>';

        $form_html .= '<form class="a1ai-review-submission">';

        // Add rating input based on system
        if ($rating_system === '10-scale') {
            $form_html .= '<div class="a1ai-form-group a1ai-rating-input">';
            $form_html .= '<label class="a1ai-form-label">' . esc_html__('How would you rate us?', 'a1ai-chatbot') . '</label>';
            $form_html .= '<div class="a1ai-scale-container">';
            $form_html .= '<input type="range" id="a1ai-rating-scale" name="rating" min="1" max="10" value="5" class="a1ai-range-slider">';
            $form_html .= '<div class="a1ai-scale-display"><span id="a1ai-scale-value">5</span><span class="a1ai-scale-max"> / 10</span></div>';
            $form_html .= '</div>';
            $form_html .= '</div>';
        } else {
            $form_html .= '<div class="a1ai-form-group a1ai-rating-input">';
            $form_html .= '<label class="a1ai-form-label">' . esc_html__('How would you rate your experience?', 'a1ai-chatbot') . '</label>';
            $form_html .= '<div class="a1ai-rating-box">';
            $form_html .= '<div class="a1ai-stars">';
            for ($i = 1; $i <= 5; $i++) {
                $form_html .= '<span class="a1ai-star" data-rating="' . $i . '">&#9733;</span>';
            }
            $form_html .= '</div>';
            $form_html .= '<div class="a1ai-click-to-rate">' . esc_html__('Click to rate', 'a1ai-chatbot') . '</div>';
            $form_html .= '</div>';
            $form_html .= '<input type="hidden" name="rating" id="a1ai-rating-value">';
            $form_html .= '</div>';
        }

        // Initially hide form fields - they'll appear after star rating is selected
        $form_html .= '<div class="a1ai-review-form-fields" style="display: none;">';
        
        $form_html .= '<div class="a1ai-form-group">';
        $form_html .= '<label class="a1ai-form-label">' . esc_html__('Your Feedback', 'a1ai-chatbot') . '</label>';
        $form_html .= '<textarea name="comment" rows="4" class="a1ai-form-input a1ai-form-textarea" placeholder="' . esc_attr__('Tell us about your experience...', 'a1ai-chatbot') . '"></textarea>';
        $form_html .= '</div>';

        $form_html .= '<div class="a1ai-form-row">';
        $form_html .= '<div class="a1ai-form-group a1ai-form-half">';
        $form_html .= '<label class="a1ai-form-label">' . esc_html__('Your Name', 'a1ai-chatbot') . ' <span class="a1ai-required">*</span></label>';
        $form_html .= '<input type="text" name="reviewer_name" class="a1ai-form-input" placeholder="' . esc_attr__('John Doe', 'a1ai-chatbot') . '" required>';
        $form_html .= '</div>';
        $form_html .= '<div class="a1ai-form-group a1ai-form-half">';
        $form_html .= '<label class="a1ai-form-label">' . esc_html__('Your Email', 'a1ai-chatbot') . ' <span class="a1ai-required">*</span></label>';
        $form_html .= '<input type="email" name="reviewer_email" class="a1ai-form-input" placeholder="' . esc_attr__('john@example.com', 'a1ai-chatbot') . '" required>';
        $form_html .= '</div>';
        $form_html .= '</div>';

        $form_html .= '<button type="submit" class="a1ai-submit-btn">';
        $form_html .= '<span class="a1ai-btn-icon">&#10003;</span> ';
        $form_html .= esc_html__('Submit Review', 'a1ai-chatbot');
        $form_html .= '</button>';
        $form_html .= '</div>'; // Close a1ai-review-form-fields
        $form_html .= '</form>';
        $form_html .= '</div>';

        // Add modern CSS styling
        $form_html .= '<style>';
        // Main container
        $form_html .= '.a1ai-review-form { ';
        $form_html .= 'background: linear-gradient(135deg, #ffffff 0%, #f8fafc 100%); ';
        $form_html .= 'border-radius: 16px; ';
        $form_html .= 'box-shadow: 0 4px 20px rgba(0,0,0,0.08); ';
        $form_html .= 'padding: 24px; ';
        $form_html .= 'margin: 12px 0; ';
        $form_html .= 'border: 1px solid #e2e8f0; ';
        $form_html .= 'font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, "Helvetica Neue", Arial, sans-serif; ';
        $form_html .= '}';

        // Header section
        $form_html .= '.a1ai-review-header { ';
        $form_html .= 'display: flex; ';
        $form_html .= 'align-items: center; ';
        $form_html .= 'gap: 12px; ';
        $form_html .= 'margin-bottom: 16px; ';
        $form_html .= 'padding-bottom: 16px; ';
        $form_html .= 'border-bottom: 2px solid #f1f5f9; ';
        $form_html .= '}';

        $form_html .= '.a1ai-review-icon { ';
        $form_html .= 'width: 44px; ';
        $form_html .= 'height: 44px; ';
        $form_html .= 'background: linear-gradient(135deg, #fbbf24 0%, #f59e0b 100%); ';
        $form_html .= 'border-radius: 12px; ';
        $form_html .= 'display: flex; ';
        $form_html .= 'align-items: center; ';
        $form_html .= 'justify-content: center; ';
        $form_html .= 'font-size: 22px; ';
        $form_html .= 'color: white; ';
        $form_html .= 'box-shadow: 0 4px 12px rgba(251, 191, 36, 0.3); ';
        $form_html .= '}';

        $form_html .= '.a1ai-review-title { ';
        $form_html .= 'margin: 0; ';
        $form_html .= 'font-size: 18px; ';
        $form_html .= 'font-weight: 600; ';
        $form_html .= 'color: #1e293b; ';
        $form_html .= '}';

        // Intro message
        $form_html .= '.a1ai-intro-message { ';
        $form_html .= 'background: linear-gradient(135deg, #eff6ff 0%, #dbeafe 100%); ';
        $form_html .= 'border-left: 4px solid #3b82f6; ';
        $form_html .= 'border-radius: 8px; ';
        $form_html .= 'padding: 14px 16px; ';
        $form_html .= 'margin-bottom: 20px; ';
        $form_html .= '}';

        $form_html .= '.a1ai-intro-message p { ';
        $form_html .= 'margin: 0; ';
        $form_html .= 'color: #1e40af; ';
        $form_html .= 'font-size: 14px; ';
        $form_html .= 'line-height: 1.6; ';
        $form_html .= '}';

        // Form groups
        $form_html .= '.a1ai-form-group { ';
        $form_html .= 'margin-bottom: 18px; ';
        $form_html .= '}';

        $form_html .= '.a1ai-form-row { ';
        $form_html .= 'display: flex; ';
        $form_html .= 'gap: 16px; ';
        $form_html .= 'flex-wrap: wrap; ';
        $form_html .= '}';

        $form_html .= '.a1ai-form-half { ';
        $form_html .= 'flex: 1; ';
        $form_html .= 'min-width: 140px; ';
        $form_html .= '}';

        $form_html .= '.a1ai-form-label { ';
        $form_html .= 'display: block; ';
        $form_html .= 'margin-bottom: 8px; ';
        $form_html .= 'font-weight: 600; ';
        $form_html .= 'font-size: 14px; ';
        $form_html .= 'color: #374151; ';
        $form_html .= '}';

        $form_html .= '.a1ai-required { ';
        $form_html .= 'color: #ef4444; ';
        $form_html .= '}';

        // Form inputs
        $form_html .= '.a1ai-form-input { ';
        $form_html .= 'width: 100%; ';
        $form_html .= 'padding: 12px 14px; ';
        $form_html .= 'border: 2px solid #e2e8f0; ';
        $form_html .= 'border-radius: 10px; ';
        $form_html .= 'font-size: 14px; ';
        $form_html .= 'color: #1e293b; ';
        $form_html .= 'background: #ffffff; ';
        $form_html .= 'transition: all 0.2s ease; ';
        $form_html .= 'box-sizing: border-box; ';
        $form_html .= '}';

        $form_html .= '.a1ai-form-input:focus { ';
        $form_html .= 'outline: none; ';
        $form_html .= 'border-color: #3b82f6; ';
        $form_html .= 'box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.15); ';
        $form_html .= '}';

        $form_html .= '.a1ai-form-input::placeholder { ';
        $form_html .= 'color: #9ca3af; ';
        $form_html .= '}';

        $form_html .= '.a1ai-form-textarea { ';
        $form_html .= 'resize: vertical; ';
        $form_html .= 'min-height: 90px; ';
        $form_html .= '}';

        // Rating box (yellow container)
        $form_html .= '.a1ai-rating-box { ';
        $form_html .= 'background: #fffacd; ';
        $form_html .= 'border-radius: 12px; ';
        $form_html .= 'padding: 16px; ';
        $form_html .= 'margin: 12px 0; ';
        $form_html .= 'border: 1px solid #fef3c7; ';
        $form_html .= '}';

        // Star rating
        $form_html .= '.a1ai-stars { ';
        $form_html .= 'display: flex; ';
        $form_html .= 'gap: 8px; ';
        $form_html .= 'user-select: none; ';
        $form_html .= 'padding: 8px 0; ';
        $form_html .= 'justify-content: center; ';
        $form_html .= '}';

        // Click to rate text
        $form_html .= '.a1ai-click-to-rate { ';
        $form_html .= 'text-align: center; ';
        $form_html .= 'color: #d97706; ';
        $form_html .= 'font-size: 13px; ';
        $form_html .= 'margin-top: 4px; ';
        $form_html .= 'font-weight: 500; ';
        $form_html .= '}';

        $form_html .= '.a1ai-star { ';
        $form_html .= 'font-size: 32px; ';
        $form_html .= 'color: #d1d5db; ';
        $form_html .= 'cursor: pointer; ';
        $form_html .= 'transition: all 0.2s ease; ';
        $form_html .= 'text-shadow: 0 2px 4px rgba(0,0,0,0.1); ';
        $form_html .= '}';

        $form_html .= '.a1ai-star:hover { ';
        $form_html .= 'color: #fbbf24; ';
        $form_html .= 'transform: scale(1.15); ';
        $form_html .= '}';

        $form_html .= '.a1ai-star.active { ';
        $form_html .= 'color: #fbbf24; ';
        $form_html .= 'text-shadow: 0 2px 8px rgba(251, 191, 36, 0.4); ';
        $form_html .= '}';

        // Scale slider
        $form_html .= '.a1ai-scale-container { ';
        $form_html .= 'display: flex; ';
        $form_html .= 'align-items: center; ';
        $form_html .= 'gap: 16px; ';
        $form_html .= 'padding: 8px 0; ';
        $form_html .= '}';

        $form_html .= '.a1ai-range-slider { ';
        $form_html .= 'flex: 1; ';
        $form_html .= 'height: 8px; ';
        $form_html .= 'border-radius: 4px; ';
        $form_html .= 'background: #e2e8f0; ';
        $form_html .= 'cursor: pointer; ';
        $form_html .= '-webkit-appearance: none; ';
        $form_html .= '}';

        $form_html .= '.a1ai-range-slider::-webkit-slider-thumb { ';
        $form_html .= '-webkit-appearance: none; ';
        $form_html .= 'width: 24px; ';
        $form_html .= 'height: 24px; ';
        $form_html .= 'border-radius: 50%; ';
        $form_html .= 'background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%); ';
        $form_html .= 'box-shadow: 0 2px 8px rgba(59, 130, 246, 0.4); ';
        $form_html .= 'cursor: pointer; ';
        $form_html .= '}';

        $form_html .= '.a1ai-scale-display { ';
        $form_html .= 'background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%); ';
        $form_html .= 'color: white; ';
        $form_html .= 'padding: 8px 16px; ';
        $form_html .= 'border-radius: 8px; ';
        $form_html .= 'font-weight: 600; ';
        $form_html .= 'min-width: 70px; ';
        $form_html .= 'text-align: center; ';
        $form_html .= 'box-shadow: 0 2px 8px rgba(59, 130, 246, 0.3); ';
        $form_html .= '}';

        $form_html .= '.a1ai-scale-max { ';
        $form_html .= 'opacity: 0.8; ';
        $form_html .= 'font-size: 12px; ';
        $form_html .= '}';

        // Submit button
        $form_html .= '.a1ai-submit-btn { ';
        $form_html .= 'width: 100%; ';
        $form_html .= 'padding: 14px 24px; ';
        $form_html .= 'background: linear-gradient(135deg, #10b981 0%, #059669 100%); ';
        $form_html .= 'color: white; ';
        $form_html .= 'border: none; ';
        $form_html .= 'border-radius: 10px; ';
        $form_html .= 'font-size: 16px; ';
        $form_html .= 'font-weight: 600; ';
        $form_html .= 'cursor: pointer; ';
        $form_html .= 'transition: all 0.2s ease; ';
        $form_html .= 'display: flex; ';
        $form_html .= 'align-items: center; ';
        $form_html .= 'justify-content: center; ';
        $form_html .= 'gap: 8px; ';
        $form_html .= 'box-shadow: 0 4px 14px rgba(16, 185, 129, 0.3); ';
        $form_html .= 'margin-top: 8px; ';
        $form_html .= '}';

        $form_html .= '.a1ai-submit-btn:hover { ';
        $form_html .= 'background: linear-gradient(135deg, #059669 0%, #047857 100%); ';
        $form_html .= 'transform: translateY(-2px); ';
        $form_html .= 'box-shadow: 0 6px 20px rgba(16, 185, 129, 0.4); ';
        $form_html .= '}';

        $form_html .= '.a1ai-submit-btn:active { ';
        $form_html .= 'transform: translateY(0); ';
        $form_html .= '}';

        $form_html .= '.a1ai-btn-icon { ';
        $form_html .= 'font-size: 18px; ';
        $form_html .= '}';

        // Responsive adjustments
        $form_html .= '@media (max-width: 480px) { ';
        $form_html .= '.a1ai-review-form { padding: 18px; } ';
        $form_html .= '.a1ai-form-row { flex-direction: column; gap: 0; } ';
        $form_html .= '.a1ai-star { font-size: 28px; } ';
        $form_html .= '.a1ai-review-header { flex-direction: column; text-align: center; } ';
        $form_html .= '}';

        $form_html .= '</style>';

        // Add working JavaScript
        $form_html .= '<script>';
        $form_html .= 'setTimeout(function() {'; // Wait for DOM to be ready

        if ($rating_system === '10-scale') {
            $form_html .= 'var scaleInput = document.getElementById("a1ai-rating-scale");';
            $form_html .= 'var scaleValue = document.getElementById("a1ai-scale-value");';
            $form_html .= 'var formFields = document.querySelector(".a1ai-review-form-fields");';
            $form_html .= 'if (scaleInput && scaleValue) {';
            $form_html .= 'scaleInput.addEventListener("input", function() {';
            $form_html .= 'scaleValue.textContent = this.value; }); ';
            // Show form fields when scale value changes
            $form_html .= 'scaleInput.addEventListener("change", function() {';
            $form_html .= 'if (this.value && formFields) { formFields.style.display = "block"; formFields.scrollIntoView({ behavior: "smooth", block: "nearest" }); }';
            $form_html .= '}); }';
        } else {
            $form_html .= 'var reviewForm = document.querySelector(".a1ai-review-form");';
            $form_html .= 'if (!reviewForm) { reviewForm = document.querySelector(".a1ai-review-submission").closest(".a1ai-review-form") || document.querySelector(".a1ai-standalone-review-container"); }';
            $form_html .= 'if (reviewForm) {';
            $form_html .= 'var stars = reviewForm.querySelectorAll(".a1ai-star");';
            $form_html .= 'var ratingInput = reviewForm.querySelector("#a1ai-rating-value");';
            $form_html .= 'var formFields = reviewForm.querySelector(".a1ai-review-form-fields");';
            $form_html .= 'if (stars.length > 0 && ratingInput) {';
            $form_html .= 'stars.forEach(function(star, index) {';
            $form_html .= 'star.addEventListener("click", function(e) {';
            $form_html .= 'e.preventDefault();';
            $form_html .= 'e.stopPropagation();';
            $form_html .= 'var rating = parseInt(this.getAttribute("data-rating"));';
            $form_html .= 'ratingInput.value = rating;';
            $form_html .= 'var allStars = reviewForm.querySelectorAll(".a1ai-star");';
            $form_html .= 'allStars.forEach(function(s, i) {';
            $form_html .= 'if (i < rating) { s.classList.add("active"); } else { s.classList.remove("active"); }';
            $form_html .= '}); ';
            // Show form fields after rating is selected - get fresh reference to ensure we have the right one
            $form_html .= 'var fieldsToShow = reviewForm.querySelector(".a1ai-review-form-fields");';
            $form_html .= 'if (fieldsToShow) {';
            $form_html .= 'fieldsToShow.style.display = "block";';
            $form_html .= 'fieldsToShow.style.setProperty("display", "block", "important");';
            $form_html .= 'setTimeout(function() { fieldsToShow.scrollIntoView({ behavior: "smooth", block: "nearest" }); }, 100);';
            $form_html .= '}';
            $form_html .= '}); }); }';
            $form_html .= '}';
        }

        $form_html .= '}, 100);'; // Give DOM time to load
        // Also run on DOMContentLoaded if not already loaded
        $form_html .= 'if (document.readyState === "loading") {';
        $form_html .= 'document.addEventListener("DOMContentLoaded", function() {';
        $form_html .= 'setTimeout(function() {';
        // Re-initialize star rating handlers in case DOM wasn't ready before
        if ($rating_system === '10-scale') {
            $form_html .= 'var scaleInput = document.getElementById("a1ai-rating-scale");';
            $form_html .= 'var scaleValue = document.getElementById("a1ai-scale-value");';
            $form_html .= 'var reviewForm = document.querySelector(".a1ai-review-form");';
            $form_html .= 'if (!reviewForm) { reviewForm = document.querySelector(".a1ai-review-submission").closest(".a1ai-review-form") || document.querySelector(".a1ai-standalone-review-container"); }';
            $form_html .= 'var formFields = reviewForm ? reviewForm.querySelector(".a1ai-review-form-fields") : null;';
            $form_html .= 'if (scaleInput && scaleValue && reviewForm) {';
            $form_html .= 'scaleInput.addEventListener("input", function() {';
            $form_html .= 'scaleValue.textContent = this.value; }); ';
            $form_html .= 'scaleInput.addEventListener("change", function() {';
            $form_html .= 'if (this.value && formFields) { formFields.style.display = "block"; formFields.style.setProperty("display", "block", "important"); formFields.scrollIntoView({ behavior: "smooth", block: "nearest" }); }';
            $form_html .= '}); }';
        } else {
            $form_html .= 'var reviewForm = document.querySelector(".a1ai-review-form");';
            $form_html .= 'if (!reviewForm) { reviewForm = document.querySelector(".a1ai-review-submission").closest(".a1ai-review-form") || document.querySelector(".a1ai-standalone-review-container"); }';
            $form_html .= 'if (reviewForm) {';
            $form_html .= 'var stars = reviewForm.querySelectorAll(".a1ai-star");';
            $form_html .= 'var ratingInput = reviewForm.querySelector("#a1ai-rating-value");';
            $form_html .= 'var formFields = reviewForm.querySelector(".a1ai-review-form-fields");';
            $form_html .= 'if (stars.length > 0 && ratingInput) {';
            $form_html .= 'stars.forEach(function(star, index) {';
            $form_html .= 'star.addEventListener("click", function(e) {';
            $form_html .= 'e.preventDefault();';
            $form_html .= 'e.stopPropagation();';
            $form_html .= 'var rating = parseInt(this.getAttribute("data-rating"));';
            $form_html .= 'ratingInput.value = rating;';
            $form_html .= 'var allStars = reviewForm.querySelectorAll(".a1ai-star");';
            $form_html .= 'allStars.forEach(function(s, i) {';
            $form_html .= 'if (i < rating) { s.classList.add("active"); } else { s.classList.remove("active"); }';
            $form_html .= '}); ';
            $form_html .= 'var fieldsToShow = reviewForm.querySelector(".a1ai-review-form-fields");';
            $form_html .= 'if (fieldsToShow) {';
            $form_html .= 'fieldsToShow.style.display = "block";';
            $form_html .= 'fieldsToShow.style.setProperty("display", "block", "important");';
            $form_html .= 'setTimeout(function() { fieldsToShow.scrollIntoView({ behavior: "smooth", block: "nearest" }); }, 100);';
            $form_html .= '}';
            $form_html .= '}); }); }';
            $form_html .= '}';
        }
        $form_html .= '}, 100);';
        $form_html .= '}); }';
        $form_html .= '</script>';

        return $form_html;
    }

    /**
     * Handle review submission via AJAX
     *
     * @since    1.0.0
     */
    public function handle_submit_review() {
        // Check if upgrade plugin is active (reviews are a premium feature)
        if (!class_exists('A1AI_Upgrade_Helper') || !A1AI_Upgrade_Helper::is_upgrade_active()) {
            wp_send_json_error(array('error' => 'Review submission requires the premium upgrade.'));
            return;
        }

        // Check nonce for security
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_key(wp_unslash($_POST['nonce'])), 'a1ai_submit_review')) {
            wp_send_json_error(array('error' => 'Security check failed.'));
            return;
        }

        // Check required parameters
        if (!isset($_POST['chatbot_id']) || !isset($_POST['rating']) || !isset($_POST['reviewer_name']) || !isset($_POST['reviewer_email'])) {
            wp_send_json_error(array('error' => 'Missing required parameters.'));
            return;
        }

        // Sanitize inputs
        $chatbot_id = intval($_POST['chatbot_id']);
        $rating = intval($_POST['rating']);
        $comment = isset($_POST['comment']) ? sanitize_textarea_field(wp_unslash($_POST['comment'])) : '';
        $reviewer_name = sanitize_text_field(wp_unslash($_POST['reviewer_name']));
        $reviewer_email = sanitize_email(wp_unslash($_POST['reviewer_email']));

        // Validate data
        if ($chatbot_id <= 0) {
            wp_send_json_error(array('error' => 'Invalid chatbot ID.'));
            return;
        }

        if ($rating < 1 || $rating > 10) {
            wp_send_json_error(array('error' => 'Invalid rating.'));
            return;
        }

        if (empty($reviewer_name) || empty($reviewer_email)) {
            wp_send_json_error(array('error' => 'Name and email are required.'));
            return;
        }

        if (!is_email($reviewer_email)) {
            wp_send_json_error(array('error' => 'Invalid email address.'));
            return;
        }

        // Get chatbot to verify it exists
        $chatbot = $this->data_manager->get_chatbot($chatbot_id);
        if (empty($chatbot)) {
            wp_send_json_error(array('error' => 'Chatbot not found.'));
            return;
        }

        // Store the review (you'll need to create this table or extend existing tables)
        $review_data = array(
            'chatbot_id' => $chatbot_id,
            'rating' => $rating,
            'comment' => $comment,
            'reviewer_name' => $reviewer_name,
            'reviewer_email' => $reviewer_email,
            'date' => current_time('mysql'),
            'status' => 'pending' // Reviews start as pending for moderation
        );

        // For now, store as option until we have a proper reviews table
        $existing_reviews = get_option('a1ai_reviews', array());
        $review_data['id'] = time() . '_' . $chatbot_id; // Simple ID for now
        $existing_reviews[] = $review_data;
        update_option('a1ai_reviews', $existing_reviews);

        // Fire conversion tracking event (if upgrade plugin active)
        if (class_exists('A1AI_Upgrade_Helper') && A1AI_Upgrade_Helper::is_upgrade_active()) {
            do_action('a1ai_conversion_event', 'review_submission', array(
                'chatbot_id' => $chatbot_id,
                'rating' => $rating,
                'review_text' => $comment
            ));
        }

        // Mark this session as having submitted a review to prevent duplicate forms
        $session_id = isset($_POST['session_id']) ? sanitize_text_field(wp_unslash($_POST['session_id'])) : '';
        if (!empty($session_id)) {
            set_transient('a1ai_review_submitted_' . $session_id, true, 3600); // 1 hour expiry
        }

        // Get threshold settings
        $rating_system = get_option('a1ai_rating_system', '5-star');
        $min_star_threshold = get_option('a1ai_min_star_threshold', 3);
        $min_scale_threshold = get_option('a1ai_min_scale_threshold', 6);

        // Determine if rating meets threshold
        $meets_threshold = false;
        if ($rating_system === '10-scale') {
            $meets_threshold = ($rating >= $min_scale_threshold);
        } else {
            $meets_threshold = ($rating >= $min_star_threshold);
        }

        // Prepare response based on threshold
        if ($meets_threshold) {
            // High rating - positive response and external links
            $ai_message = __('Thank you so much for the positive feedback! We\'re thrilled to hear about your great experience. Was there anything else you particularly enjoyed? We\'d love for you to share your experience on our review platforms - the windows should be opening shortly!', 'a1ai-chatbot');

            // Get external review links
            $external_links = array();
            $google_link = get_option('a1ai_review_google_link', '');
            $yelp_link = get_option('a1ai_review_yelp_link', '');
            $facebook_link = get_option('a1ai_review_facebook_link', '');
            $other_link = get_option('a1ai_review_other_link', '');
            $other_name = get_option('a1ai_review_other_name', 'Other Platform');

            if (!empty($google_link)) {
                $external_links[] = array('name' => 'Google', 'url' => $google_link);
            }
            if (!empty($yelp_link)) {
                $external_links[] = array('name' => 'Yelp', 'url' => $yelp_link);
            }
            if (!empty($facebook_link)) {
                $external_links[] = array('name' => 'Facebook', 'url' => $facebook_link);
            }
            if (!empty($other_link)) {
                $external_links[] = array('name' => $other_name, 'url' => $other_link);
            }

            // Send success response with external links and copy functionality
            wp_send_json_success(array(
                'message' => $ai_message,
                'external_links' => $external_links,
                'show_copy_button' => true,
                'review_summary' => $comment,
                'rating_display' => ($rating_system === '10-scale') ? "$rating/10" : str_repeat('⭐', $rating)
            ));
        } else {
            // Low rating - supportive response, no external links
            $ai_message = __('Thank you for your honest feedback. We truly appreciate you taking the time to share your experience with us. We will look into the issues you\'ve mentioned and will get back to you if needed. Is there anything else you would like to discuss or any way we can better assist you?', 'a1ai-chatbot');

            // Send response without external links
            wp_send_json_success(array(
                'message' => $ai_message,
                'external_links' => array(), // No external links for low ratings
                'show_copy_button' => false,
                'continue_conversation' => true // Flag to indicate conversation should continue
            ));
        }
    }
} 