<?php
/**
 * The core plugin class.
 *
 * @since      1.0.0
 * @package    A1AI
 * @subpackage A1AI/includes
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    A1AI
 * @subpackage A1AI/includes
 * @author     A1AI <info@example.com>
 */
class A1AI {

    /**
     * The loader that's responsible for maintaining and registering all hooks that power
     * the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      A1AI_Loader    $loader    Maintains and registers all hooks for the plugin.
     */
    protected $loader;

    /**
     * The unique identifier of this plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $plugin_name    The string used to uniquely identify this plugin.
     */
    protected $plugin_name;

    /**
     * The current version of the plugin.
     *
     * @since    1.0.0
     * @access   protected
     * @var      string    $version    The current version of the plugin.
     */
    protected $version;

    /**
     * The data manager instance.
     *
     * @since    1.0.0
     * @access   protected
     * @var      A1AI_Data_Manager    $data_manager    Handles database operations.
     */
    protected $data_manager;

    /**
     * Define the core functionality of the plugin.
     *
     * Set the plugin name and the plugin version that can be used throughout the plugin.
     * Load the dependencies, define the locale, and set the hooks for the admin area and
     * the public-facing side of the site.
     *
     * @since    1.0.0
     */
    public function __construct() {
        if (defined('A1AI_VERSION')) {
            $this->version = A1AI_VERSION;
        } else {
            $this->version = '1.3.0';
        }
        
        $this->plugin_name = 'a1ai';
        
        $this->load_dependencies();
        $this->set_locale();
        $this->define_admin_hooks();
        $this->define_public_hooks();
        $this->define_api_hooks();
        
        // Check for database upgrades
        $this->check_database_upgrade();
    }

    /**
     * Load the required dependencies for this plugin.
     *
     * Include the following files that make up the plugin:
     *
     * - A1AI_Loader. Orchestrates the hooks of the plugin.
     * - A1AI_i18n. Defines internationalization functionality.
     * - A1AI_Admin. Defines all hooks for the admin area.
     * - A1AI_Public. Defines all hooks for the public side of the site.
     *
     * Create an instance of the loader which will be used to register the hooks
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function load_dependencies() {
        /**
         * The class responsible for orchestrating the actions and filters of the
         * core plugin.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-a1ai-loader.php';

        /**
         * The class responsible for defining internationalization functionality
         * of the plugin.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-a1ai-i18n.php';

        /**
         * The class responsible for managing data operations.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-a1ai-data-manager.php';

        /**
         * The class responsible for handling database migrations.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-a1ai-migration.php';

        /**
         * Helper class for upgrade plugin detection.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-a1ai-upgrade-helper.php';

        /**
         * The class responsible for defining all actions that occur in the admin area.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'admin/class-a1ai-admin.php';

        /**
         * The class responsible for defining all actions that occur in the public-facing
         * side of the site.
         */
        require_once plugin_dir_path(dirname(__FILE__)) . 'public/class-a1ai-public.php';

        $this->loader = new A1AI_Loader();
        $this->data_manager = new A1AI_Data_Manager();
    }

    /**
     * Define the locale for this plugin for internationalization.
     *
     * Uses the A1AI_i18n class in order to set the domain and to register the hook
     * with WordPress.
     *
     * @since    1.0.0
     * @access   private
     */
    private function set_locale() {
        $plugin_i18n = new A1AI_i18n();
        $this->loader->add_action('plugins_loaded', $plugin_i18n, 'load_plugin_textdomain');
    }

    /**
     * Register all of the hooks related to the admin area functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_admin_hooks() {
        $plugin_admin = new A1AI_Admin($this->get_plugin_name(), $this->get_version(), $this->data_manager);
        
        // Register admin menu and settings
        $this->loader->add_action('admin_menu', $plugin_admin, 'register_admin_menu');
        
        // Add settings link on plugin page
        $plugin_basename = plugin_basename(plugin_dir_path(dirname(__FILE__)) . 'a1ai.php');
        $this->loader->add_filter('plugin_action_links_' . $plugin_basename, $plugin_admin, 'add_settings_link');
        
        // Admin scripts and styles
        $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_styles');
        $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts');
        
        // Admin AJAX handler
        $this->loader->add_action('wp_ajax_a1ai_admin_action', $plugin_admin, 'handle_admin_ajax');
    }

    /**
     * Register all of the hooks related to the public-facing functionality
     * of the plugin.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_public_hooks() {
        $plugin_public = new A1AI_Public($this->get_plugin_name(), $this->get_version(), $this->data_manager);
        
        // Enqueue scripts and styles
        $this->loader->add_action('wp_enqueue_scripts', $plugin_public, 'enqueue_styles');
        $this->loader->add_action('wp_enqueue_scripts', $plugin_public, 'enqueue_scripts');
        
        // Register shortcode
        $this->loader->add_action('init', $plugin_public, 'register_shortcodes');
        
        // Add global chatbot to footer
        $this->loader->add_action('wp_footer', $plugin_public, 'render_global_chatbot');
        
        // AJAX handlers for the chatbot
        $this->loader->add_action('wp_ajax_a1ai_send_message', $plugin_public, 'handle_send_message');
        $this->loader->add_action('wp_ajax_nopriv_a1ai_send_message', $plugin_public, 'handle_send_message');

        // AJAX handlers for reviews
        $this->loader->add_action('wp_ajax_a1ai_submit_review', $plugin_public, 'handle_submit_review');
        $this->loader->add_action('wp_ajax_nopriv_a1ai_submit_review', $plugin_public, 'handle_submit_review');
    }

    /**
     * Register API-related hooks.
     *
     * @since    1.0.0
     * @access   private
     */
    private function define_api_hooks() {
        // Register AJAX actions here
        $this->loader->add_action('wp_ajax_a1ai_public_action', $this, 'handle_public_ajax');
        $this->loader->add_action('wp_ajax_nopriv_a1ai_public_action', $this, 'handle_public_ajax');
        
        // Schedule maintenance tasks
        $this->loader->add_action('a1ai_daily_maintenance', $this, 'run_daily_maintenance');
        
        if (!wp_next_scheduled('a1ai_daily_maintenance')) {
            wp_schedule_event(time(), 'daily', 'a1ai_daily_maintenance');
        }
    }



    /**
     * Handle public AJAX requests.
     *
     * This is a placeholder that will be implemented in Phase 4.
     *
     * @since    1.0.0
     */
    public function handle_public_ajax() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_key($_POST['nonce']), 'a1ai_public_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'a1ai-chatbot')));
        }
        
        // Handle different actions
        $action = isset($_POST['action_type']) ? sanitize_text_field(wp_unslash($_POST['action_type'])) : '';
        
        switch ($action) {
            // Add cases here as needed
            
            default:
                wp_send_json_error(array('message' => __('Invalid action.', 'a1ai-chatbot')));
                break;
        }
    }

    /**
     * Run daily maintenance tasks.
     *
     * @since    1.0.0
     */
    public function run_daily_maintenance() {
        // This function will clear old conversations based on retention settings
        $retention_days = get_option('a1ai_conversation_retention_days', 30);
        $keep_history = get_option('a1ai_keep_conversation_history', true);
        
        if (!$keep_history || $retention_days <= 0) {
            return;
        }
        
        // Check cache first
        $cache_key = 'a1ai_maintenance_' . gmdate('Ymd');
        $maintenance_ran = wp_cache_get($cache_key, 'a1ai');
        
        if (false !== $maintenance_ran) {
            return; // Maintenance already ran today
        }
        
        global $wpdb;
        
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery
        // Direct database queries are necessary for maintenance operations
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.NoCaching
        // Caching is implemented with wp_cache_get and wp_cache_set
        
        // Delete old conversations and their messages
        $date = gmdate('Y-m-d H:i:s', strtotime("-{$retention_days} days"));
        
        // Get IDs of old conversations
        $old_conversations = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT id FROM {$wpdb->prefix}a1ai_conversations WHERE created_at < %s",
                $date
            )
        );
        
        if (!empty($old_conversations)) {
            // Sanitize conversation IDs
            $old_conversations = array_map('intval', $old_conversations);

            // Build placeholders for prepared statement
            $placeholders = implode(',', array_fill(0, count($old_conversations), '%d'));

            // Delete associated messages first
            // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            // phpcs:disable WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare
            $wpdb->query($wpdb->prepare(
                "DELETE FROM {$wpdb->prefix}a1ai_messages WHERE conversation_id IN ($placeholders)",
                $old_conversations
            ));
            // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            // phpcs:enable WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare

            // Delete the conversations
            // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            // phpcs:disable WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare
            $wpdb->query($wpdb->prepare(
                "DELETE FROM {$wpdb->prefix}a1ai_conversations WHERE id IN ($placeholders)",
                $old_conversations
            ));
            // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            // phpcs:enable WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare

            // Clear related caches
            wp_cache_delete('a1ai_usage_stats_month', 'a1ai');
            wp_cache_delete('a1ai_usage_stats_week', 'a1ai');
            wp_cache_delete('a1ai_usage_stats_day', 'a1ai');
            wp_cache_delete('a1ai_usage_stats_year', 'a1ai');
        }
        
        // Set cache to avoid running again today
        wp_cache_set($cache_key, true, 'a1ai', DAY_IN_SECONDS);
        
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.NoCaching
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery
    }

    /**
     * Check for database upgrades.
     *
     * This method will check if the database schema version matches the plugin version.
     * If not, it will trigger the upgrade process.
     *
     * @since    1.0.0
     */
    private function check_database_upgrade() {
        $current_db_version = get_option('a1ai_db_version', '1.0.0');
        
        if (version_compare($current_db_version, $this->version, '<')) {
            // Trigger upgrade process
            $this->run_database_upgrade();
        }
    }

    /**
     * Run the database upgrade process.
     *
     * This method will handle the actual database schema changes.
     *
     * @since    1.0.0
     */
    private function run_database_upgrade() {
        global $wpdb;
        
        // Fix AUTO_INCREMENT issues for all tables
        $this->fix_auto_increment_issues();
        
        // Add missing indexes
        $this->add_missing_indexes();
        
        // Run migrations
        if (class_exists('A1AI_Migration')) {
            $migration = new A1AI_Migration($this->data_manager);
            $migration_results = $migration->run_migrations();
            
            // Log migration results for debugging
            if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                // phpcs:disable WordPress.PHP.DevelopmentFunctions.error_log_error_log
                // These logs are intentionally only shown during debugging
                foreach ($migration_results as $version => $result) {
                    if (isset($result['messages'])) {
                        foreach ($result['messages'] as $message) {
                            error_log("A1AI Migration {$version}: {$message}");
                        }
                    }
                    if (isset($result['errors'])) {
                        foreach ($result['errors'] as $error) {
                            error_log("A1AI Migration {$version} Error: {$error}");
                        }
                    }
                }
                // phpcs:enable WordPress.PHP.DevelopmentFunctions.error_log_error_log
            }
        }
        
        // Update the option to the new version
        update_option('a1ai_db_version', $this->version);
    }
    
    /**
     * Fix AUTO_INCREMENT issues in database tables.
     *
     * @since    1.0.0
     */
    private function fix_auto_increment_issues() {
        global $wpdb;
        
        $tables = array(
            'a1ai_chatbots',
            'a1ai_conversations', 
            'a1ai_messages',
            'a1ai_adjustments'
        );
        
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery
        // Direct database queries are necessary for schema changes
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.NoCaching
        // Schema changes don't require caching
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.SchemaChange
        // Schema changes are necessary for database upgrades
        
        foreach ($tables as $table) {
            $table_name = $wpdb->prefix . $table;
            
            // Check if table exists and fix AUTO_INCREMENT
            $result = $wpdb->get_var($wpdb->prepare(
                "SHOW TABLES LIKE %s",
                $table_name
            ));
            if ($result === $table_name) {
                // Use backtick escaping for table name (safe as it's derived from trusted prefix + hardcoded string)
                $safe_table_name = '`' . str_replace('`', '``', $table_name) . '`';
                // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                $wpdb->query("ALTER TABLE $safe_table_name MODIFY `id` mediumint(9) NOT NULL AUTO_INCREMENT");
                // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            }
        }
        
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.SchemaChange
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.NoCaching
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery
    }
    
    /**
     * Add missing indexes to database tables.
     *
     * @since    1.0.0
     */
    private function add_missing_indexes() {
        global $wpdb;
        
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery
        // Direct database queries are necessary for schema changes
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.NoCaching
        // Schema changes don't require caching
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.SchemaChange
        // Schema changes are necessary for database upgrades
        
        // Add indexes to conversations table
        $conversations_table = $wpdb->prefix . 'a1ai_conversations';
        $safe_conversations_table = '`' . str_replace('`', '``', $conversations_table) . '`';
        // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $wpdb->query("ALTER TABLE $safe_conversations_table ADD KEY `chatbot_id` (`chatbot_id`)");
        $wpdb->query("ALTER TABLE $safe_conversations_table ADD KEY `user_id` (`user_id`)");
        $wpdb->query("ALTER TABLE $safe_conversations_table ADD KEY `session_id` (`session_id`)");
        // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared

        // Add index to messages table
        $messages_table = $wpdb->prefix . 'a1ai_messages';
        $safe_messages_table = '`' . str_replace('`', '``', $messages_table) . '`';
        // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $wpdb->query("ALTER TABLE $safe_messages_table ADD KEY `conversation_id` (`conversation_id`)");
        // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared

        // Add index to adjustments table
        $adjustments_table = $wpdb->prefix . 'a1ai_adjustments';
        $safe_adjustments_table = '`' . str_replace('`', '``', $adjustments_table) . '`';
        // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $wpdb->query("ALTER TABLE $safe_adjustments_table ADD KEY `chatbot_id` (`chatbot_id`)");
        // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.SchemaChange
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.NoCaching
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery
    }

    /**
     * Run the loader to execute all of the hooks with WordPress.
     *
     * @since    1.0.0
     */
    public function run() {
        $this->loader->run();
    }

    /**
     * The name of the plugin used to uniquely identify it within the context of
     * WordPress and to define internationalization functionality.
     *
     * @since     1.0.0
     * @return    string    The name of the plugin.
     */
    public function get_plugin_name() {
        return $this->plugin_name;
    }

    /**
     * The reference to the class that orchestrates the hooks with the plugin.
     *
     * @since     1.0.0
     * @return    A1AI_Loader    Orchestrates the hooks of the plugin.
     */
    public function get_loader() {
        return $this->loader;
    }

    /**
     * Retrieve the version number of the plugin.
     *
     * @since     1.0.0
     * @return    string    The version number of the plugin.
     */
    public function get_version() {
        return $this->version;
    }
    
    /**
     * Get the data manager instance.
     *
     * @since     1.0.0
     * @return    A1AI_Data_Manager    The data manager instance.
     */
    public function get_data_manager() {
        return $this->data_manager;
    }
} 