<?php
/**
 * Helper class for upgrade plugin detection
 *
 * @link       https://a1websitepro.com
 * @since      1.0.0
 *
 * @package    A1ai_Chatbot
 * @subpackage A1ai_Chatbot/includes
 */

/**
 * Helper class for upgrade plugin detection.
 *
 * This class provides a centralized method to check if the upgrade plugin is active,
 * handling multiple possible installation paths.
 *
 * @since      1.0.0
 * @package    A1ai_Chatbot
 * @subpackage A1ai_Chatbot/includes
 * @author     A1 Website Pro <support@a1websitepro.com>
 */
class A1AI_Upgrade_Helper {

    /**
     * Check if the upgrade plugin is active AND has a valid license
     *
     * Checks multiple possible installation paths for the upgrade plugin
     * and validates that a valid license is active
     *
     * @since    1.0.0
     * @return   bool    True if upgrade plugin is active with valid license, false otherwise
     */
    public static function is_upgrade_active() {
        // Possible plugin paths
        $upgrade_paths = array(
            'a1ai-chatbot-upgrade/a1ai-chatbot-upgrade.php',
            'a1ai-chatbot-upgrade-new/a1ai-chatbot-upgrade.php',
            'a1ai-chatbot-upgrade-new-x/a1ai-chatbot-upgrade.php'
        );

        // Check if is_plugin_active function exists
        if (!function_exists('is_plugin_active')) {
            include_once(ABSPATH . 'wp-admin/includes/plugin.php');
        }

        // Check each possible path
        $plugin_active = false;
        foreach ($upgrade_paths as $path) {
            if (function_exists('is_plugin_active') && is_plugin_active($path)) {
                $plugin_active = true;
                break;
            }
        }

        // If plugin not active, return false
        if (!$plugin_active) {
            return false;
        }

        // Plugin is active, now check if license is valid
        return self::has_valid_license();
    }

    /**
     * Check if the upgrade plugin is installed (files exist)
     *
     * @since    1.0.0
     * @return   bool    True if upgrade plugin files exist, false otherwise
     */
    public static function is_plugin_installed() {
        // Possible plugin paths
        $upgrade_paths = array(
            'a1ai-chatbot-upgrade/a1ai-chatbot-upgrade.php',
            'a1ai-chatbot-upgrade-new/a1ai-chatbot-upgrade.php',
            'a1ai-chatbot-upgrade-new-x/a1ai-chatbot-upgrade.php'
        );

        // Check if any of the plugin files exist
        foreach ($upgrade_paths as $path) {
            $full_path = WP_PLUGIN_DIR . '/' . $path;
            if (file_exists($full_path)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Check if the upgrade plugin has a valid license
     *
     * @since    1.0.0
     * @return   bool    True if valid license exists, false otherwise
     */
    private static function has_valid_license() {
        // Check EDD license status
        $license_status = get_option('a1ai_upgrade_license_status', '');
        return $license_status === 'valid';
    }
}

