<?php
/**
 * The OpenAI integration functionality of the plugin.
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    A1AI
 * @subpackage A1AI/includes
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * The OpenAI integration functionality of the plugin.
 *
 * Handles all interactions with the OpenAI API including message formatting,
 * API requests, error handling, and response processing.
 *
 * @package    A1AI
 * @subpackage A1AI/includes
 * @author     A1AI <info@example.com>
 */
class A1AI_OpenAI {

    /**
     * The data manager instance.
     *
     * @since    1.0.0
     * @access   private
     * @var      A1AI_Data_Manager    $data_manager    Handles database operations.
     */
    private $data_manager;

    /**
     * The conversation manager instance.
     *
     * @since    1.4.0
     * @access   private
     * @var      A1AI_Conversation_Manager    $conversation_manager    Handles conversation state and guidance.
     */
    private $conversation_manager;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param    A1AI_Data_Manager     $data_manager    The data manager instance.
     */
    public function __construct($data_manager = null) {
        // If data manager wasn't passed, create one
        if (null === $data_manager) {
            require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-a1ai-data-manager.php';
            $this->data_manager = new A1AI_Data_Manager();
        } else {
            $this->data_manager = $data_manager;
        }

        // Initialize conversation manager
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-a1ai-conversation-manager.php';
        $this->conversation_manager = new A1AI_Conversation_Manager($this->data_manager);
    }

    /**
     * Process a conversation with OpenAI.
     *
     * @since    1.0.0
     * @param    int       $conversation_id    The conversation ID.
     * @param    string    $message            The user message.
     * @param    array     $settings           The chatbot settings.
     * @return   array                         The response with success status and message or error.
     */
    public function process_conversation($conversation_id, $message, $settings) {
        // Get chatbot ID from settings
        $chatbot_id = isset($settings['chatbot_id']) ? intval($settings['chatbot_id']) : 0;
        
        // Add conversation ID to settings for conversation manager
        $settings['conversation_id'] = $conversation_id;
        
        // Check for matching adjustments first
        if ($chatbot_id > 0) {
            $adjustments = $this->data_manager->get_adjustments($chatbot_id);
            
            if (!empty($adjustments)) {
                foreach ($adjustments as $adjustment) {
                    // Simple string comparison for now (can be enhanced later)
                    $user_input = trim(strtolower($adjustment['user_input']));
                    $current_message = trim(strtolower($message));
                    
                    // Check for exact match or contains match based on match type
                    $is_match = false;
                    
                    switch ($adjustment['ui_match_type']) {
                        case 'Exact':
                            $is_match = ($user_input === $current_message);
                            break;
                        case 'Phrase':
                            $is_match = (strpos($current_message, $user_input) !== false);
                            break;
                        case 'Flexible':
                        case 'Broad':
                        default:
                            // Simple contains check for broad matching
                            $is_match = (
                                (strpos($current_message, $user_input) !== false) || 
                                (strpos($user_input, $current_message) !== false)
                            );
                            
                            // Only use levenshtein for shorter strings to avoid performance issues
                            if (!$is_match && strlen($current_message) <= 255 && strlen($user_input) <= 255) {
                                $max_length = max(strlen($current_message), strlen($user_input));
                                if ($max_length > 0) {
                                    $similarity = levenshtein($current_message, $user_input) / $max_length;
                                    $is_match = $similarity < 0.3; // Less than 30% difference
                                }
                            }
                            break;
                    }
                    
                    // If we have a match, return the adjustment output
                    if ($is_match) {
                        // Update conversation state even for adjustments
                        $this->conversation_manager->update_conversation_state($conversation_id, array(
                            'last_response_type' => 'adjustment',
                            'last_updated' => current_time('mysql')
                        ));
                        
                        return array(
                            'success' => true,
                            'message' => $adjustment['ai_output'],
                            'tokens_used' => 0, // No tokens used for adjustments
                            'from_adjustment' => true,
                        );
                    }
                }
            }
        }
        
        // No adjustment matched, proceed with normal OpenAI request
        // Get message history for this conversation
        $history = $this->data_manager->get_conversation_messages($conversation_id);
        
        // Prepare messages for OpenAI
        $messages = $this->prepare_messages($history, $message, $settings);
        
        // Send request to OpenAI
        $response = $this->send_request($messages, $settings);
        
        // Process response
        if ($response['success']) {
            $ai_message = $this->extract_message($response['data']);
            $token_usage = $this->extract_token_usage($response['data']);
            
            // Validate and enhance response if needed
            $enhanced_response = $this->validate_and_enhance_response($ai_message, $conversation_id, $message, $settings);
            
            // Update conversation state
            $this->conversation_manager->update_conversation_state($conversation_id, array(
                'last_response_type' => 'ai',
                'last_user_message' => $message,
                'last_updated' => current_time('mysql')
            ));
            
            return array(
                'success' => true,
                'message' => $enhanced_response,
                'tokens_used' => $token_usage,
            );
        } else {
            return array(
                'success' => false,
                'error' => $response['message'],
            );
        }
    }

    /**
     * Prepare messages for OpenAI API request.
     *
     * @since    1.0.0
     * @param    array     $history     Previous messages in the conversation.
     * @param    string    $message     The new user message.
     * @param    array     $settings    The chatbot settings.
     * @return   array                  Formatted messages for OpenAI API.
     */
    private function prepare_messages($history, $message, $settings) {
        $messages = array();
        
        // Get chatbot ID from settings or conversation
        $chatbot_id = isset($settings['chatbot_id']) ? intval($settings['chatbot_id']) : 0;
        $conversation_id = isset($settings['conversation_id']) ? intval($settings['conversation_id']) : 0;
        
        // Get system prompt from custom instructions table
        $system_prompt = '';
        if ($chatbot_id > 0) {
            $system_instructions = $this->data_manager->get_custom_instructions($chatbot_id, 'system');
            if (!empty($system_instructions) && is_array($system_instructions)) {
                foreach ($system_instructions as $instruction) {
                    if ($instruction['instruction_name'] === 'System Prompt' && !empty($instruction['instruction_content'])) {
                        $system_prompt = $instruction['instruction_content'];
                        break;
                    }
                }
            }
        }
        
        // Get general custom instructions from dedicated table
        $custom_instructions = '';
        if ($chatbot_id > 0) {
            $instructions_array = $this->data_manager->get_custom_instructions($chatbot_id, 'general');
            if (!empty($instructions_array) && is_array($instructions_array)) {
                $instructions_text = '';
                foreach ($instructions_array as $instruction) {
                    if (!empty($instruction['instruction_content'])) {
                        $instructions_text .= $instruction['instruction_content'] . "\n\n";
                    }
                }
                $custom_instructions = trim($instructions_text);
            }
        }
        
        // Combine system prompt and custom instructions
        $system_content = '';
        if (!empty($system_prompt)) {
            $system_content .= $system_prompt . "\n\n";
        }
        if (!empty($custom_instructions)) {
            $system_content .= $custom_instructions . "\n\n";
        }

        // Inject business information context
        if ($chatbot_id > 0) {
            $business_info = $this->data_manager->get_business_info($chatbot_id);
            if (!empty($business_info)) {
                $system_content .= "BUSINESS INFORMATION:\n";
                $system_content .= "You are an AI assistant for " . (!empty($business_info['business_name']) ? $business_info['business_name'] : 'this business') . ".\n\n";

                if (!empty($business_info['tagline'])) {
                    $system_content .= "About: " . $business_info['tagline'] . "\n\n";
                }

                if (!empty($business_info['business_type'])) {
                    $system_content .= "Business Type: " . ucfirst($business_info['business_type']) . "\n\n";
                }

                // Services
                if (!empty($business_info['services'])) {
                    $services = json_decode($business_info['services'], true);
                    if (!empty($services) && is_array($services)) {
                        $system_content .= "Services Offered:\n";
                        foreach ($services as $service) {
                            $system_content .= "- " . $service . "\n";
                        }
                        $system_content .= "\n";
                    }
                }

                // Products
                if (!empty($business_info['products'])) {
                    $products = json_decode($business_info['products'], true);
                    if (!empty($products) && is_array($products)) {
                        $system_content .= "Products Available:\n";
                        foreach ($products as $product) {
                            $system_content .= "- " . $product . "\n";
                        }
                        $system_content .= "\n";
                    }
                }

                // Business Hours
                if (!empty($business_info['hours'])) {
                    $hours = json_decode($business_info['hours'], true);
                    if (!empty($hours) && is_array($hours)) {
                        $system_content .= "Business Hours:\n";
                        foreach ($hours as $day => $time) {
                            if (!empty($time)) {
                                $system_content .= ucfirst($day) . ": " . $time . "\n";
                            }
                        }
                        $system_content .= "\n";
                    }
                }

                // Contact Information
                $has_contact = false;
                if (!empty($business_info['address']) || !empty($business_info['phone']) || !empty($business_info['email'])) {
                    $system_content .= "Contact Information:\n";
                    if (!empty($business_info['address'])) {
                        $system_content .= "Address: " . str_replace("\n", ", ", $business_info['address']) . "\n";
                        $has_contact = true;
                    }
                    if (!empty($business_info['phone'])) {
                        $system_content .= "Phone: " . $business_info['phone'] . "\n";
                        $has_contact = true;
                    }
                    if (!empty($business_info['email'])) {
                        $system_content .= "Email: " . $business_info['email'] . "\n";
                        $has_contact = true;
                    }
                    if (!empty($business_info['website'])) {
                        $system_content .= "Website: " . $business_info['website'] . "\n";
                        $has_contact = true;
                    }
                    if ($has_contact) {
                        $system_content .= "\n";
                    }
                }

                // Policies
                if (!empty($business_info['policies'])) {
                    $system_content .= "Important Policies & Information:\n";
                    $system_content .= $business_info['policies'] . "\n\n";
                }

                $system_content .= "IMPORTANT: Only provide information based on the business information above. If asked about something not listed here, politely say you don't have that specific information and suggest contacting the business directly.\n\n";
            }
        }

        // Add minimal conversation guidance only if there's significant history
        if ($conversation_id > 0 && !empty($history) && count($history) > 2) {
            $system_content .= "IMPORTANT: This is an ongoing conversation. Maintain context and continuity with previous messages. Do not restart the conversation or ask generic greetings.\n\n";
        }

        // Add core conversation principles
        $system_content .= "Respond naturally and directly to the user's questions. Stay on topic and maintain conversation flow.\n\n";

        // Add accuracy and honesty principles for business use
        $system_content .= "ACCURACY PRINCIPLES:\n";
        $system_content .= "- Only provide information you are confident is accurate\n";
        $system_content .= "- If you don't have specific information, be honest and say so\n";
        $system_content .= "- Never make up facts, prices, availability, or business policies\n";
        $system_content .= "- When uncertain, suggest contacting the business directly for accurate details\n";
        $system_content .= "- Be helpful, professional, and transparent about your limitations\n\n";

        // Add system message
        if (!empty($system_content)) {
            $messages[] = array(
                'role' => 'system',
                'content' => sanitize_textarea_field($system_content),
            );
        } else {
            // Default system message (business-focused with accuracy emphasis)
            $default_message = "You are a professional business assistant. Your role is to help customers by:\n\n";
            $default_message .= "1. ACCURACY FIRST\n";
            $default_message .= "   - Only state facts you know to be true\n";
            $default_message .= "   - If unsure, say 'I don't have that specific information. Please contact us directly.'\n";
            $default_message .= "   - Never guess prices, availability, hours, or specific business details\n\n";
            $default_message .= "2. HELPFUL GUIDANCE\n";
            $default_message .= "   - Answer general questions professionally\n";
            $default_message .= "   - Direct customers to contact methods for detailed inquiries\n";
            $default_message .= "   - Be friendly, concise, and clear\n\n";
            $default_message .= "3. LEAD GENERATION\n";
            $default_message .= "   - When customers show interest, offer to help them connect\n";
            $default_message .= "   - Suggest contacting the business for quotes or appointments\n";
            $default_message .= "   - Be proactive but respectful\n\n";
            $default_message .= "Remember: It's better to say 'I don't know' than provide incorrect information.";

            $messages[] = array(
                'role' => 'system',
                'content' => $default_message,
            );
        }
        
        // Add initial messages if provided
        if (!empty($settings['initial_messages']) && is_array($settings['initial_messages'])) {
            foreach ($settings['initial_messages'] as $initial_message) {
                if (!empty($initial_message['role']) && !empty($initial_message['content'])) {
                    $messages[] = array(
                        'role' => sanitize_text_field($initial_message['role']),
                        'content' => sanitize_textarea_field($initial_message['content']),
                    );
                }
            }
        }
        
        // Add conversation history
        if (!empty($history) && is_array($history)) {
            foreach ($history as $msg) {
                $messages[] = array(
                    'role' => sanitize_text_field($msg['role']),
                    'content' => $msg['content'],
                );
            }
        }
        
        // Add current user message
        $messages[] = array(
            'role' => 'user',
            'content' => $message,
        );
        
        return $messages;
    }

    /**
     * Send request to OpenAI API.
     *
     * @since    1.0.0
     * @param    array     $messages    The formatted messages.
     * @param    array     $settings    The API settings.
     * @return   array                  The response with success status and data or error message.
     */
    private function send_request($messages, $settings) {
        // Get API key from settings
        $api_key = get_option('a1ai_openai_api_key', '');
        
        if (empty($api_key)) {
            return array(
                'success' => false,
                'message' => esc_html__('OpenAI API key is not configured.', 'a1ai-chatbot'),
            );
        }
        
        // Default settings
        $defaults = array(
            'model' => 'gpt-4o',
            'temperature' => 0.7,
            'max_tokens' => 1000,
        );
        
        // Merge with provided settings
        $api_settings = wp_parse_args(
            array(
                'model' => isset($settings['model']) ? sanitize_text_field($settings['model']) : $defaults['model'],
                'temperature' => isset($settings['temperature']) ? (float) $settings['temperature'] : $defaults['temperature'],
                'max_tokens' => isset($settings['max_tokens']) ? (int) $settings['max_tokens'] : $defaults['max_tokens'],
            ),
            $defaults
        );
        
        // Prepare request data
        $request_data = array(
            'model' => $api_settings['model'],
            'messages' => $messages,
            'temperature' => $api_settings['temperature'],
            'max_tokens' => $api_settings['max_tokens'],
        );
        
        // Prepare HTTP request
        $args = array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type' => 'application/json',
            ),
            'timeout' => 60,
            'body' => wp_json_encode($request_data),
        );
        
        // Send request to OpenAI
        $response = wp_remote_post('https://api.openai.com/v1/chat/completions', $args);
        
        // Check for errors
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'message' => $response->get_error_message(),
            );
        }
        
        // Get response code and body
        $response_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        // Check if response is successful
        if (200 !== $response_code) {
            $error_message = isset($data['error']['message']) 
                ? sanitize_text_field($data['error']['message']) 
                : esc_html__('Unknown error occurred.', 'a1ai-chatbot');
                
            return array(
                'success' => false,
                'message' => $error_message,
                'code' => $response_code,
            );
        }
        
        // Return success with data
        return array(
            'success' => true,
            'data' => $data,
        );
    }

    /**
     * Extract message from OpenAI response.
     *
     * @since    1.0.0
     * @param    array     $data    The response data.
     * @return   string             The extracted message content.
     */
    private function extract_message($data) {
        if (isset($data['choices'][0]['message']['content'])) {
            return $data['choices'][0]['message']['content'];
        }
        
        return esc_html__('No response generated.', 'a1ai-chatbot');
    }

    /**
     * Extract token usage from OpenAI response.
     *
     * @since    1.0.0
     * @param    array     $data    The response data.
     * @return   int                The total token usage or 0 if not available.
     */
    private function extract_token_usage($data) {
        if (isset($data['usage']['total_tokens'])) {
            return (int) $data['usage']['total_tokens'];
        }
        
        return 0;
    }

    /**
     * Validate and enhance AI response based on conversation state.
     *
     * @since    1.4.0
     * @param    string   $ai_message        The AI response message.
     * @param    int      $conversation_id   The conversation ID.
     * @param    string   $user_message      The user message.
     * @param    array    $settings          The chatbot settings.
     * @return   string                      The enhanced response.
     */
    private function validate_and_enhance_response($ai_message, $conversation_id, $user_message, $settings) {
        $enhanced_message = $ai_message;
        
        // Check for repetitive content
        if ($this->conversation_manager->is_repetitive_content($conversation_id, $ai_message)) {
            // Get suggested actions to provide alternative content
            $suggested_actions = $this->conversation_manager->get_suggested_actions($conversation_id, $settings['chatbot_id']);
            
            if (!empty($suggested_actions)) {
                $enhanced_message .= "\n\nTo move our conversation forward, I'd be happy to help you with:";
                foreach (array_slice($suggested_actions, 0, 3) as $action) {
                    $enhanced_message .= "\n• " . $action;
                }
            }
        }
        
        // Check if response includes action guidance
        $state = $this->conversation_manager->get_conversation_state($conversation_id);
        if ($state['conversation_stage'] === 'action' && !$this->contains_action_guidance($ai_message)) {
            $enhanced_message = $this->add_action_guidance($enhanced_message, $settings);
        }
        
        return $enhanced_message;
    }

    /**
     * Check if response contains action guidance.
     *
     * @since    1.4.0
     * @param    string   $message    The message to check.
     * @return   bool                 True if contains action guidance.
     */
    private function contains_action_guidance($message) {
        $action_keywords = array(
            'contact', 'call', 'email', 'reach out', 'get in touch',
            'purchase', 'buy', 'order', 'sign up', 'subscribe',
            'schedule', 'book', 'appointment', 'consultation',
            'learn more', 'find out', 'discover', 'explore'
        );
        
        $message_lower = strtolower($message);
        foreach ($action_keywords as $keyword) {
            if (strpos($message_lower, $keyword) !== false) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Add action guidance to response.
     *
     * @since    1.4.0
     * @param    string   $message    The original message.
     * @param    array    $settings   The chatbot settings.
     * @return   string               The message with action guidance.
     */
    private function add_action_guidance($message, $settings) {
        $chatbot_id = isset($settings['chatbot_id']) ? intval($settings['chatbot_id']) : 0;
        
        // Get custom instructions for action guidance
        $custom_instructions = $this->data_manager->get_custom_instructions($chatbot_id, 'general');
        $action_guidance = '';
        
        foreach ($custom_instructions as $instruction) {
            if (strpos(strtolower($instruction['instruction_content']), 'call to action') !== false ||
                strpos(strtolower($instruction['instruction_content']), 'next step') !== false) {
                $action_guidance = $instruction['instruction_content'];
                break;
            }
        }
        
        if (!empty($action_guidance)) {
            $message .= "\n\n" . $action_guidance;
        } else {
            // Default action guidance
            $message .= "\n\nHow can I help you take the next step? Would you like to learn more about our services or get in touch?";
        }
        
        return $message;
    }
} 