<?php
/**
 * Conversation Manager for A1AI Chatbot
 *
 * Handles conversation state management, action guidance, and repetition prevention.
 *
 * @package    A1AI
 * @subpackage A1AI/includes
 * @since      1.4.0
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Conversation Manager Class
 *
 * Manages conversation flow, tracks user intent, and ensures responses guide toward
 * website owner's specified actions while avoiding repetitive information.
 *
 * @since      1.4.0
 * @package    A1AI
 * @subpackage A1AI/includes
 */
class A1AI_Conversation_Manager {

    /**
     * The data manager instance.
     *
     * @since    1.4.0
     * @access   private
     * @var      A1AI_Data_Manager    $data_manager    Handles database operations.
     */
    private $data_manager;

    /**
     * Conversation state cache.
     *
     * @since    1.4.0
     * @access   private
     * @var      array    $conversation_states    Cached conversation states.
     */
    private $conversation_states = array();

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.4.0
     * @param    A1AI_Data_Manager     $data_manager    The data manager instance.
     */
    public function __construct($data_manager = null) {
        if (null === $data_manager) {
            require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-a1ai-data-manager.php';
            $this->data_manager = new A1AI_Data_Manager();
        } else {
            $this->data_manager = $data_manager;
        }
    }

    /**
     * Get conversation state for a specific conversation.
     *
     * @since    1.4.0
     * @param    int    $conversation_id    The conversation ID.
     * @return   array                      The conversation state.
     */
    public function get_conversation_state($conversation_id) {
        // Check cache first
        if (isset($this->conversation_states[$conversation_id])) {
            return $this->conversation_states[$conversation_id];
        }

        // Get conversation details
        $conversation = $this->data_manager->get_conversation($conversation_id);
        if (empty($conversation)) {
            return $this->get_default_state();
        }

        // Get messages to analyze conversation
        $messages = $this->data_manager->get_messages($conversation_id);
        
        // Analyze conversation state
        $state = $this->analyze_conversation_state($messages, $conversation);
        
        // Cache the state
        $this->conversation_states[$conversation_id] = $state;
        
        return $state;
    }

    /**
     * Update conversation state.
     *
     * @since    1.4.0
     * @param    int    $conversation_id    The conversation ID.
     * @param    array  $updates            State updates to apply.
     * @return   bool                      True on success, false on failure.
     */
    public function update_conversation_state($conversation_id, $updates) {
        $current_state = $this->get_conversation_state($conversation_id);
        $new_state = array_merge($current_state, $updates);
        
        // Update cache
        $this->conversation_states[$conversation_id] = $new_state;
        
        // Store in database for persistence
        return $this->store_conversation_state($conversation_id, $new_state);
    }

    /**
     * Analyze conversation to determine current state.
     *
     * @since    1.4.0
     * @param    array  $messages       The conversation messages.
     * @param    array  $conversation   The conversation details.
     * @return   array                  The analyzed state.
     */
    private function analyze_conversation_state($messages, $conversation) {
        $state = $this->get_default_state();
        
        if (empty($messages)) {
            return $state;
        }

        // Analyze user intent and conversation flow
        $user_messages = array_filter($messages, function($msg) {
            return $msg['role'] === 'user';
        });

        $assistant_messages = array_filter($messages, function($msg) {
            return $msg['role'] === 'assistant';
        });

        // Track mentioned topics and actions
        $mentioned_topics = array();
        $mentioned_actions = array();
        $user_intent = 'general';
        $conversation_stage = 'greeting';

        foreach ($user_messages as $msg) {
            $content = strtolower($msg['content']);
            
            // Detect user intent
            if (strpos($content, 'help') !== false || strpos($content, 'support') !== false) {
                $user_intent = 'help';
            } elseif (strpos($content, 'buy') !== false || strpos($content, 'purchase') !== false || strpos($content, 'order') !== false) {
                $user_intent = 'purchase';
            } elseif (strpos($content, 'contact') !== false || strpos($content, 'email') !== false || strpos($content, 'phone') !== false) {
                $user_intent = 'contact';
            } elseif (strpos($content, 'about') !== false || strpos($content, 'information') !== false) {
                $user_intent = 'information';
            }

            // Extract mentioned topics
            $topics = $this->extract_topics($content);
            $mentioned_topics = array_merge($mentioned_topics, $topics);
        }

        // Determine conversation stage - more lenient thresholds
        $message_count = count($user_messages);
        if ($message_count <= 1) {
            $conversation_stage = 'greeting';
        } elseif ($message_count <= 5) {
            $conversation_stage = 'exploration';
        } elseif ($message_count <= 10) {
            $conversation_stage = 'engagement';
        } else {
            $conversation_stage = 'action';
        }

        // Track repeated information
        $repeated_info = $this->detect_repeated_information($assistant_messages);

        $state['user_intent'] = $user_intent;
        $state['conversation_stage'] = $conversation_stage;
        $state['mentioned_topics'] = array_unique($mentioned_topics);
        $state['mentioned_actions'] = array_unique($mentioned_actions);
        $state['repeated_information'] = $repeated_info;
        $state['message_count'] = $message_count;
        $state['last_updated'] = current_time('mysql');

        return $state;
    }

    /**
     * Extract topics from user message.
     *
     * @since    1.4.0
     * @param    string $content    The message content.
     * @return   array              Array of extracted topics.
     */
    private function extract_topics($content) {
        $topics = array();
        
        // Common topic keywords
        $topic_keywords = array(
            'pricing' => array('price', 'cost', 'fee', 'payment', 'subscription'),
            'features' => array('feature', 'function', 'capability', 'benefit'),
            'support' => array('help', 'support', 'assist', 'issue', 'problem'),
            'contact' => array('contact', 'email', 'phone', 'reach', 'speak'),
            'about' => array('about', 'company', 'team', 'story', 'mission'),
            'products' => array('product', 'service', 'solution', 'offer'),
        );

        foreach ($topic_keywords as $topic => $keywords) {
            foreach ($keywords as $keyword) {
                if (strpos($content, $keyword) !== false) {
                    $topics[] = $topic;
                    break;
                }
            }
        }

        return $topics;
    }

    /**
     * Detect repeated information in assistant messages.
     *
     * @since    1.4.0
     * @param    array $messages    The assistant messages.
     * @return   array              Array of repeated information.
     */
    private function detect_repeated_information($messages) {
        $repeated_info = array();
        $message_contents = array();

        foreach ($messages as $msg) {
            $content = strtolower($msg['content']);
            $message_contents[] = $content;
        }

        // Check for similar content patterns
        for ($i = 0; $i < count($message_contents) - 1; $i++) {
            for ($j = $i + 1; $j < count($message_contents); $j++) {
                $similarity = $this->calculate_similarity($message_contents[$i], $message_contents[$j]);
                if ($similarity > 0.85) { // 85% similarity threshold (less sensitive)
                    $repeated_info[] = array(
                        'message_index' => $i,
                        'similarity' => $similarity,
                        'content_sample' => substr($message_contents[$i], 0, 100)
                    );
                }
            }
        }

        return $repeated_info;
    }

    /**
     * Calculate similarity between two strings.
     *
     * @since    1.4.0
     * @param    string $str1    First string.
     * @param    string $str2    Second string.
     * @return   float           Similarity score (0-1).
     */
    private function calculate_similarity($str1, $str2) {
        $words1 = explode(' ', $str1);
        $words2 = explode(' ', $str2);
        
        $common_words = array_intersect($words1, $words2);
        $total_words = array_unique(array_merge($words1, $words2));
        
        if (empty($total_words)) {
            return 0;
        }
        
        return count($common_words) / count($total_words);
    }

    /**
     * Get default conversation state.
     *
     * @since    1.4.0
     * @return   array    The default state.
     */
    private function get_default_state() {
        return array(
            'user_intent' => 'general',
            'conversation_stage' => 'greeting',
            'mentioned_topics' => array(),
            'mentioned_actions' => array(),
            'repeated_information' => array(),
            'message_count' => 0,
            'last_updated' => current_time('mysql'),
            'action_guidance_provided' => false,
            'next_suggested_action' => '',
        );
    }

    /**
     * Store conversation state in database.
     *
     * @since    1.4.0
     * @param    int    $conversation_id    The conversation ID.
     * @param    array  $state             The state to store.
     * @return   bool                      True on success, false on failure.
     */
    private function store_conversation_state($conversation_id, $state) {
        // Store in conversation metadata or create a separate table
        // For now, we'll use WordPress transients for caching
        $cache_key = 'a1ai_conversation_state_' . $conversation_id;
        return set_transient($cache_key, $state, HOUR_IN_SECONDS);
    }

    /**
     * Generate action guidance based on conversation state and system prompt.
     *
     * @since    1.4.0
     * @param    int    $conversation_id    The conversation ID.
     * @param    int    $chatbot_id         The chatbot ID.
     * @param    string $user_message       The current user message.
     * @return   string                     Action guidance to include in system prompt.
     */
    public function generate_action_guidance($conversation_id, $chatbot_id, $user_message) {
        $state = $this->get_conversation_state($conversation_id);
        $custom_instructions = $this->data_manager->get_custom_instructions($chatbot_id, 'general');
        
        $guidance = '';
        
        // Get action-oriented instructions
        $action_instructions = array();
        foreach ($custom_instructions as $instruction) {
            if (strpos(strtolower($instruction['instruction_content']), 'action') !== false ||
                strpos(strtolower($instruction['instruction_content']), 'guide') !== false ||
                strpos(strtolower($instruction['instruction_content']), 'direct') !== false) {
                $action_instructions[] = $instruction['instruction_content'];
            }
        }

        // Prevent repetition only if severe
        if (!empty($state['repeated_information']) && count($state['repeated_information']) > 3) {
            $guidance .= "Avoid repeating previous information. Provide new value or perspective. ";
        }

        // Minimal stage-specific guidance - focus on maintaining context
        if ($state['message_count'] > 0) {
            $guidance .= "Continue the ongoing conversation naturally. Remember the context of previous messages. ";
        }

        return $guidance;
    }

    /**
     * Check if information has been repeated recently.
     *
     * @since    1.4.0
     * @param    int    $conversation_id    The conversation ID.
     * @param    string $new_content        The new content to check.
     * @return   bool                      True if content is repetitive.
     */
    public function is_repetitive_content($conversation_id, $new_content) {
        $state = $this->get_conversation_state($conversation_id);
        
        if (empty($state['repeated_information'])) {
            return false;
        }

        // Check against recent repeated content
        foreach ($state['repeated_information'] as $repeated) {
            $similarity = $this->calculate_similarity(
                strtolower($new_content),
                $repeated['content_sample']
            );
            
            if ($similarity > 0.8) { // 80% similarity threshold (less sensitive)
                return true;
            }
        }

        return false;
    }

    /**
     * Get suggested next actions based on conversation state.
     *
     * @since    1.4.0
     * @param    int    $conversation_id    The conversation ID.
     * @param    int    $chatbot_id         The chatbot ID.
     * @return   array                      Array of suggested actions.
     */
    public function get_suggested_actions($conversation_id, $chatbot_id) {
        $state = $this->get_conversation_state($conversation_id);
        $custom_instructions = $this->data_manager->get_custom_instructions($chatbot_id, 'general');
        
        $suggested_actions = array();
        
        // Extract action suggestions from custom instructions
        foreach ($custom_instructions as $instruction) {
            if (strpos(strtolower($instruction['instruction_content']), 'suggest') !== false ||
                strpos(strtolower($instruction['instruction_content']), 'recommend') !== false) {
                $suggested_actions[] = $instruction['instruction_content'];
            }
        }

        // Add stage-specific suggestions
        switch ($state['conversation_stage']) {
            case 'greeting':
                $suggested_actions[] = 'Ask about their specific needs or goals';
                break;
            case 'exploration':
                $suggested_actions[] = 'Provide relevant information about your services';
                break;
            case 'engagement':
                $suggested_actions[] = 'Share success stories or case studies';
                break;
            case 'action':
                $suggested_actions[] = 'Provide clear call-to-action (contact, purchase, etc.)';
                break;
        }

        return $suggested_actions;
    }
}
