<?php
/**
 * Fired during plugin activation.
 *
 * @link       https://example.com
 * @since      1.0.0
 *
 * @package    A1AI
 * @subpackage A1AI/includes
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Fired during plugin activation.
 *
 * This class defines all code necessary to run during the plugin's activation.
 *
 * @since      1.0.0
 * @package    A1AI
 * @subpackage A1AI/includes
 * @author     A1AI <info@example.com>
 */
class A1AI_Activator {

    /**
     * Run plugin activation tasks.
     *
     * Create necessary database tables and set default options.
     *
     * @since    1.0.0
     */
    public static function activate() {
        // Set default options on activation
        self::set_default_options();
        
        // Create database tables
        self::create_tables();
        
        // Set database version
        update_option('a1ai_db_version', A1AI_VERSION);
        
        // Set a transient to redirect to welcome page
        set_transient('a1ai_activation_redirect', true, 30);
    }
    
    /**
     * Set default plugin options.
     *
     * @since    1.0.0
     */
    private static function set_default_options() {
        $default_options = array(
            'version' => A1AI_VERSION,
            'api_key' => '',
            'global_chatbot_enabled' => false,
            'chatbot_position' => 'bottom-right',
            'default_model' => 'gpt-4o',
            'default_temperature' => 0.7,
            'default_max_tokens' => 1000,
            'keep_conversation_history' => true,
            'conversation_retention_days' => 30,
        );

        // Only add options if they don't already exist
        foreach ($default_options as $option_name => $default_value) {
            if (false === get_option('a1ai_' . $option_name)) {
                add_option('a1ai_' . $option_name, $default_value);
            }
        }
    }
    
    /**
     * Create database tables.
     *
     * @since    1.0.0
     */
    private static function create_tables() {
        // Load the data manager class
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-a1ai-data-manager.php';
        
        // Initialize the data manager
        $data_manager = new A1AI_Data_Manager();
        
        // Create tables
        $data_manager->create_tables();
        
        // Create a default chatbot if none exists
        self::create_default_chatbot($data_manager);
    }
    
    /**
     * Create a default chatbot if none exists.
     *
     * @since    1.0.0
     * @param    A1AI_Data_Manager    $data_manager    The data manager instance.
     */
    private static function create_default_chatbot($data_manager) {
        // Check if tables exist and if there are any chatbots
        if ($data_manager->tables_exist()) {
            $chatbots = $data_manager->get_chatbots();
            
            // If no chatbots exist, create a default one
            if (empty($chatbots)) {
                $default_settings = array(
                    'model' => 'gpt-4o',
                    'temperature' => 0.7,
                    'max_tokens' => 1000,
                    'system_message' => 'You are a helpful assistant.',
                    'initial_messages' => array(
                        array(
                            'role' => 'assistant',
                            'content' => 'Hello! How can I assist you today?',
                        ),
                    ),
                    'welcome_message' => 'Welcome to our chatbot! How can I help you?',
                    'placeholder_text' => 'Type your message here...',
                    'button_text' => 'Send',
                    'theme' => 'light',
                    'position' => 'bottom-right',
                    'width' => '350px',
                    'height' => '500px',
                    'is_global' => false,
                    'pages' => array(),
                    'exclude_pages' => array(),
                );
                
                $data_manager->save_chatbot(0, 'Default Chatbot', $default_settings);
            }
        }
    }
} 