<?php
/**
 * Admin settings template
 *
 * @link       https://a1aidigital.com
 * @since      1.0.0
 *
 * @package    A1AI
 * @subpackage A1AI/admin/partials
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

// Get current settings
$api_key = get_option('a1ai_openai_api_key', '');
$default_model = get_option('a1ai_default_model', 'gpt-4o');
$default_temperature = get_option('a1ai_default_temperature', 0.7);
$default_max_tokens = get_option('a1ai_default_max_tokens', 1000);
$keep_history = get_option('a1ai_keep_conversation_history', true);
$retention_days = get_option('a1ai_conversation_retention_days', 30);
$global_enabled = get_option('a1ai_global_chatbot_enabled', false);
$global_chatbot_id = get_option('a1ai_global_chatbot_id', 0);
$global_position = get_option('a1ai_global_chatbot_position', 'bottom-right');
$show_powered_by = get_option('a1ai_show_powered_by', false);

// Get all chatbots for the dropdown
$chatbots = $data_manager->get_chatbots();

// Get business information for the first/default chatbot (for global settings)
$default_chatbot_id = !empty($chatbots) ? $chatbots[0]['id'] : 0;
$business_info = $default_chatbot_id > 0 ? $data_manager->get_business_info($default_chatbot_id) : array();

// Parse business info with defaults
$business_name = isset($business_info['business_name']) ? $business_info['business_name'] : get_bloginfo('name');
$tagline = isset($business_info['tagline']) ? $business_info['tagline'] : get_bloginfo('description');
$business_type = isset($business_info['business_type']) ? $business_info['business_type'] : '';
$services = isset($business_info['services']) ? json_decode($business_info['services'], true) : array();
$products = isset($business_info['products']) ? json_decode($business_info['products'], true) : array();
$hours = isset($business_info['hours']) ? json_decode($business_info['hours'], true) : array();
$address = isset($business_info['address']) ? $business_info['address'] : '';
$phone = isset($business_info['phone']) ? $business_info['phone'] : '';
$email = isset($business_info['email']) ? $business_info['email'] : get_option('admin_email');
$website = isset($business_info['website']) ? $business_info['website'] : get_site_url();
$policies = isset($business_info['policies']) ? $business_info['policies'] : '';
?>

<div class="a1ai-wrap">
    <div class="a1ai-header">
        <h1><?php echo esc_html__('A1AI Chatbot Settings', 'a1ai-chatbot'); ?></h1>
    </div>

    <div class="a1ai-notices"></div>

    <div class="a1ai-container">
        <form id="a1ai-settings-form" class="a1ai-settings-form">
            <input type="hidden" name="nonce" value="<?php echo esc_attr(wp_create_nonce('a1ai_admin_nonce')); ?>" />
            <div class="a1ai-tabs-nav">
                <a href="#tab-general" class="active"><?php echo esc_html__('General', 'a1ai-chatbot'); ?></a>
                <a href="#tab-ai"><?php echo esc_html__('AI Settings', 'a1ai-chatbot'); ?></a>
                <a href="#tab-business"><?php echo esc_html__('Business Information', 'a1ai-chatbot'); ?></a>
                <a href="#tab-global"><?php echo esc_html__('Global Chatbot', 'a1ai-chatbot'); ?></a>
                <a href="#tab-advanced"><?php echo esc_html__('Advanced', 'a1ai-chatbot'); ?></a>
                <a href="#tab-upgrades" class="a1ai-upgrades-tab"><?php echo esc_html__('Upgrades, Add-Ons & Services', 'a1ai-chatbot'); ?></a>
            </div>
            
            <div class="a1ai-tab-content" id="tab-general">
                <h2><?php echo esc_html__('General Settings', 'a1ai-chatbot'); ?></h2>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="a1ai_openai_api_key"><?php echo esc_html__('OpenAI API Key', 'a1ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <input type="password"
                                id="a1ai_openai_api_key"
                                name="a1ai_openai_api_key"
                                value="<?php echo esc_attr($api_key); ?>"
                                class="a1ai-api-key-field"
                                autocomplete="new-password"
                                data-lpignore="true"
                                data-1p-ignore="true"
                            />
                            <p class="description">
                                <?php echo esc_html__('Enter your OpenAI API key from', 'a1ai-chatbot'); ?>
                                <a href="https://platform.openai.com/api-keys" target="_blank"><?php echo esc_html__('OpenAI Dashboard', 'a1ai-chatbot'); ?></a>
                            </p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="a1ai_edd_api_key"><?php echo esc_html__('Chatbot Upgrade License Key', 'a1ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <input type="password"
                                id="a1ai_edd_api_key"
                                name="a1ai_edd_api_key"
                                value="<?php echo esc_attr(get_option('a1ai_edd_api_key', '')); ?>"
                                class="a1ai-api-key-field"
                                autocomplete="new-password"
                                data-lpignore="true"
                                data-1p-ignore="true"
                                placeholder="Enter your license key to unlock premium features"
                            />
                            <p class="description">
                                <?php echo esc_html__('Enter your A1AI Chatbot Upgrade license key to unlock premium features (Contact Forms, Reviews, Content Awareness, etc.). You received this key via email after purchase.', 'a1ai-chatbot'); ?>
                            </p>
                            <?php
                            // Show license status if upgrade plugin is active
                            if (class_exists('A1AI_Upgrade_Helper') && A1AI_Upgrade_Helper::is_upgrade_active()) {
                                echo '<p style="color: #46b450; font-weight: 500;">✓ ' . esc_html__('License Active - Premium features unlocked!', 'a1ai-chatbot') . '</p>';
                            } elseif (class_exists('A1AI_Upgrade_Helper') && A1AI_Upgrade_Helper::is_plugin_installed()) {
                                $license_status = get_option('a1ai_upgrade_license_status', '');
                                if (!empty($license_status) && $license_status !== 'valid') {
                                    echo '<p style="color: #dc3232; font-weight: 500;">✗ ' . esc_html__('License not active. Please check your license key.', 'a1ai-chatbot') . '</p>';
                                }
                            }
                            ?>
                        </td>
                    </tr>
                </table>

                <div class="a1ai-setup-guide" style="background: #f0f8ff; border: 1px solid #0073aa; border-radius: 6px; padding: 20px; margin: 20px 0;">
                    <h3 style="margin: 0 0 15px 0; color: #0073aa;"><?php echo esc_html__('Setup Instructions', 'a1ai-chatbot'); ?></h3>
                    <p style="margin: 0 0 10px 0;"><strong><?php echo esc_html__('Follow these steps to get your OpenAI API key:', 'a1ai-chatbot'); ?></strong></p>
                    <ol style="margin: 0 0 15px 20px; line-height: 1.8;">
                        <li><?php echo esc_html__('Visit the OpenAI Dashboard at', 'a1ai-chatbot'); ?> <a href="https://platform.openai.com/api-keys" target="_blank">platform.openai.com/api-keys</a></li>
                        <li><?php echo esc_html__('Sign up or log in to your OpenAI account', 'a1ai-chatbot'); ?></li>
                        <li><?php echo esc_html__('Click "Create new secret key" to generate your API key', 'a1ai-chatbot'); ?></li>
                        <li><?php echo esc_html__('Copy the key and paste it in the field above', 'a1ai-chatbot'); ?></li>
                        <li><?php echo esc_html__('Save your settings and create your first chatbot', 'a1ai-chatbot'); ?></li>
                    </ol>
                    <p style="margin: 0;">
                        <a href="https://a1aidigital.com/ai-wordpress-plugin/setup/" target="_blank" style="color: #0073aa; text-decoration: none; font-weight: 500;">
                            <?php echo esc_html__('For more detailed instructions CLICK HERE', 'a1ai-chatbot'); ?> →
                        </a>
                    </p>
                </div>

                <div class="a1ai-management-option" style="background: #fff8e5; border: 1px solid #f0b429; border-radius: 6px; padding: 20px; margin: 20px 0;">
                    <h3 style="margin: 0 0 15px 0; color: #b87900;"><?php echo esc_html__('AI Management & Credit Options', 'a1ai-chatbot'); ?></h3>
                    <p style="margin: 0 0 10px 0; line-height: 1.6;">
                        <?php echo esc_html__('Managing your own AI chatbot is like hiring on a team that you manage yourself - you have full control and flexibility. However, if you prefer a hands-off approach, having us manage it for you is like hiring on a team AND a manager to oversee that team - we handle everything for you.', 'a1ai-chatbot'); ?>
                    </p>
                    <p style="margin: 0 0 15px 0; line-height: 1.6;">
                        <?php echo esc_html__('We also offer options to simply pay us for your OpenAI credits, making billing easier and more predictable.', 'a1ai-chatbot'); ?>
                    </p>
                    <p style="margin: 0;">
                        <a href="https://a1aidigital.com/ai-wordpress-plugin/ai-management/" target="_blank" style="color: #b87900; text-decoration: none; font-weight: 500;">
                            <?php echo esc_html__('Want to have us manage it or just pay us for the credits? CLICK HERE', 'a1ai-chatbot'); ?> →
                        </a>
                    </p>
                </div>

                <p class="submit">
                    <button type="submit" class="button button-primary">
                        <?php echo esc_html__('Save Settings', 'a1ai-chatbot'); ?>
                    </button>
                    <span class="spinner" style="float: none; margin: 0 0 0 10px;"></span>
                </p>
            </div>

            <div class="a1ai-tab-content" id="tab-ai" style="display: none;">
                <h2><?php echo esc_html__('AI Settings', 'a1ai-chatbot'); ?></h2>
                
                <p><?php echo esc_html__('These settings will be used as defaults for new chatbots. You can override them in individual chatbot settings.', 'a1ai-chatbot'); ?></p>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="a1ai_default_model"><?php echo esc_html__('Default Model', 'a1ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <select id="a1ai_default_model" name="a1ai_default_model">
                                <option value="gpt-4o" <?php selected($default_model, 'gpt-4o'); ?>>GPT-4o</option>
                                <option value="gpt-4-turbo" <?php selected($default_model, 'gpt-4-turbo'); ?>>GPT-4 Turbo</option>
                                <option value="gpt-3.5-turbo" <?php selected($default_model, 'gpt-3.5-turbo'); ?>>GPT-3.5 Turbo</option>
                            </select>
                            <p class="description">
                                <?php echo esc_html__('Select the default AI model to use for chatbots.', 'a1ai-chatbot'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label><?php echo esc_html__('AI Response Style', 'a1ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <fieldset id="a1ai_temperature_presets">
                                <label style="display: block; margin-bottom: 10px;">
                                    <input type="radio" name="a1ai_temperature_preset" value="0.3" <?php checked($default_temperature >= 0 && $default_temperature <= 0.4); ?> />
                                    <strong><?php echo esc_html__('Factual & Precise (0.3)', 'a1ai-chatbot'); ?></strong>
                                    <span style="color: #46b450; font-weight: bold;"><?php echo esc_html__('[RECOMMENDED FOR BUSINESS]', 'a1ai-chatbot'); ?></span>
                                    <br />
                                    <span class="description" style="margin-left: 24px;">
                                        <?php echo esc_html__('Best for: Customer service, FAQs, accurate business information', 'a1ai-chatbot'); ?>
                                    </span>
                                </label>

                                <label style="display: block; margin-bottom: 10px;">
                                    <input type="radio" name="a1ai_temperature_preset" value="0.7" <?php checked($default_temperature > 0.4 && $default_temperature <= 0.9); ?> />
                                    <strong><?php echo esc_html__('Balanced (0.7)', 'a1ai-chatbot'); ?></strong>
                                    <span style="color: #666;"><?php echo esc_html__('[DEFAULT]', 'a1ai-chatbot'); ?></span>
                                    <br />
                                    <span class="description" style="margin-left: 24px;">
                                        <?php echo esc_html__('Best for: General conversation, friendly chat, versatile responses', 'a1ai-chatbot'); ?>
                                    </span>
                                </label>

                                <label style="display: block; margin-bottom: 10px;">
                                    <input type="radio" name="a1ai_temperature_preset" value="1.2" <?php checked($default_temperature > 0.9 && $default_temperature <= 1.4); ?> />
                                    <strong><?php echo esc_html__('Creative & Conversational (1.2)', 'a1ai-chatbot'); ?></strong>
                                    <br />
                                    <span class="description" style="margin-left: 24px;">
                                        <?php echo esc_html__('Best for: Storytelling, marketing content, brainstorming', 'a1ai-chatbot'); ?>
                                    </span>
                                </label>

                                <label style="display: block; margin-bottom: 10px;">
                                    <input type="radio" name="a1ai_temperature_preset" value="custom" <?php checked($default_temperature < 0 || ($default_temperature > 1.4 && $default_temperature <= 2)); ?> />
                                    <strong><?php echo esc_html__('⚙️ Advanced: Custom Temperature', 'a1ai-chatbot'); ?></strong>
                                </label>

                                <div id="a1ai_custom_temperature_wrapper" style="margin-left: 24px; margin-top: 10px; display: none;">
                                    <input type="number"
                                        id="a1ai_custom_temperature_input"
                                        value="<?php echo esc_attr($default_temperature); ?>"
                                        min="0"
                                        max="2"
                                        step="0.1"
                                        style="width: 80px;"
                                    />
                                    <span class="description">
                                        <?php echo esc_html__('Enter a value between 0 (deterministic) and 2 (very creative)', 'a1ai-chatbot'); ?>
                                    </span>
                                </div>

                                <!-- Hidden field to store the actual temperature value (this is what gets submitted) -->
                                <input type="hidden" id="a1ai_temperature_value" name="a1ai_default_temperature" value="<?php echo esc_attr($default_temperature); ?>" />
                            </fieldset>
                            <p class="description" style="margin-top: 10px;">
                                <?php echo esc_html__('Temperature controls how creative vs. factual the AI responses are. For business use, we recommend "Factual & Precise" to prevent the AI from making things up.', 'a1ai-chatbot'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="a1ai_default_max_tokens"><?php echo esc_html__('Default Max Tokens', 'a1ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <input type="number"
                                id="a1ai_default_max_tokens"
                                name="a1ai_default_max_tokens"
                                value="<?php echo esc_attr($default_max_tokens); ?>"
                                min="100"
                                max="4000"
                                step="100"
                            />
                            <p class="description">
                                <?php echo esc_html__('Maximum length of the response. Higher values may result in more comprehensive answers but can increase costs.', 'a1ai-chatbot'); ?>
                            </p>
                        </td>
                    </tr>
                </table>

                <?php if (class_exists('A1AI_Upgrade_Helper') && A1AI_Upgrade_Helper::is_upgrade_active()): ?>
                <?php
                // Load saved content awareness settings
                $content_settings = get_option('a1ai_content_awareness_settings', array(
                    'auto_analyze' => 0,
                    'generate_sitemap' => 0,
                    'analysis_frequency' => 'weekly',
                    'min_content_length' => 100,
                    'tier2_enabled' => 0,
                    'analysis_strategy' => 'tier1_fallback'
                ));
                ?>
                <h3><?php echo esc_html__('Content Awareness', 'a1ai-chatbot'); ?></h3>
                <p><?php echo esc_html__('Analyze your website content to help AI provide more relevant responses based on your posts, pages, and other content.', 'a1ai-chatbot'); ?></p>

                <div class="a1ai-content-awareness-section">
                    <table class="form-table">
                        <tr>
                            <th scope="row">
                                <label for="a1ai_auto_analyze"><?php echo esc_html__('Auto Analysis', 'a1ai-chatbot'); ?></label>
                            </th>
                            <td>
                                <input type="checkbox" id="a1ai_auto_analyze" name="a1ai_auto_analyze" value="1" <?php checked($content_settings['auto_analyze'], 1); ?>>
                                <label for="a1ai_auto_analyze"><?php echo esc_html__('Automatically analyze new content', 'a1ai-chatbot'); ?></label>
                                <br><br>
                                <input type="checkbox" id="a1ai_generate_sitemap" name="a1ai_generate_sitemap" value="1" <?php checked($content_settings['generate_sitemap'], 1); ?>>
                                <label for="a1ai_generate_sitemap"><?php echo esc_html__('Generate sitemaps for crawling and AI assistance', 'a1ai-chatbot'); ?></label>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label for="a1ai_analysis_frequency"><?php echo esc_html__('Analysis Frequency', 'a1ai-chatbot'); ?></label>
                            </th>
                            <td>
                                <select id="a1ai_analysis_frequency" name="a1ai_analysis_frequency">
                                    <option value="daily" <?php selected($content_settings['analysis_frequency'], 'daily'); ?>><?php echo esc_html__('Daily', 'a1ai-chatbot'); ?></option>
                                    <option value="weekly" <?php selected($content_settings['analysis_frequency'], 'weekly'); ?>><?php echo esc_html__('Weekly', 'a1ai-chatbot'); ?></option>
                                    <option value="monthly" <?php selected($content_settings['analysis_frequency'], 'monthly'); ?>><?php echo esc_html__('Monthly', 'a1ai-chatbot'); ?></option>
                                </select>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">
                                <label for="a1ai_min_content_length"><?php echo esc_html__('Minimum Content Length', 'a1ai-chatbot'); ?></label>
                            </th>
                            <td>
                                <input type="number" id="a1ai_min_content_length" name="a1ai_min_content_length" value="<?php echo esc_attr($content_settings['min_content_length']); ?>" min="0">
                                <p class="description"><?php echo esc_html__('Minimum content length to analyze (in characters).', 'a1ai-chatbot'); ?></p>
                            </td>
                        </tr>
                    </table>

                    <!-- Tier 1 Content Analysis -->
                    <div class="a1ai-tier1-section" style="margin-top: 30px; padding: 20px; background: #fff; border: 1px solid #ddd; border-radius: 5px;">
                        <h3 style="margin: 0 0 10px 0; color: #0073aa;"><?php echo esc_html__('Tier 1 Content Analysis', 'a1ai-chatbot'); ?></h3>
                        <p style="margin: 0 0 15px 0; color: #666; font-size: 13px;"><?php echo esc_html__('Fast, lightweight analysis for quick content understanding and user guidance.', 'a1ai-chatbot'); ?></p>

                        <h4><?php echo esc_html__('Content Analysis Tools', 'a1ai-chatbot'); ?></h4>
                        <div class="a1ai-analysis-controls" style="margin-bottom: 20px;">
                            <button type="button" id="a1ai-analyze-all-content-tier1" class="button button-primary" data-tier="1">
                                <?php echo esc_html__('Analyze All Content', 'a1ai-chatbot'); ?>
                            </button>
                            <button type="button" id="a1ai-analyze-recent-content-tier1" class="button" data-tier="1">
                                <?php echo esc_html__('Analyze Recent Content', 'a1ai-chatbot'); ?>
                            </button>
                            <button type="button" id="a1ai-clear-analysis-data-tier1" class="button" data-tier="1">
                                <?php echo esc_html__('Clear Analysis Data', 'a1ai-chatbot'); ?>
                            </button>
                            <button type="button" id="a1ai-view-analysis-results-tier1" class="button" data-tier="1">
                                <?php echo esc_html__('View Analysis Results', 'a1ai-chatbot'); ?>
                            </button>
                        </div>

                        <div id="a1ai-analysis-progress-tier1" style="display: none; margin-bottom: 20px;">
                            <div class="a1ai-progress-bar" style="width: 100%; height: 20px; background: #f1f1f1; border-radius: 10px; overflow: hidden;">
                                <div class="a1ai-progress-fill" style="height: 100%; background: #0073aa; width: 0%; transition: width 0.3s ease;"></div>
                            </div>
                            <p class="a1ai-progress-text" style="margin-top: 10px;"><?php echo esc_html__('Analyzing content...', 'a1ai-chatbot'); ?></p>
                        </div>

                        <h4><?php echo esc_html__('Tier 1 Statistics', 'a1ai-chatbot'); ?></h4>
                        <div class="a1ai-stats-grid" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 15px; margin-bottom: 15px;">
                            <div class="stat-item" style="background: #f9f9f9; padding: 15px; border-radius: 5px; text-align: center;">
                                <span class="stat-number" id="a1ai-total-analyzed-tier1" style="font-size: 24px; font-weight: bold; color: #0073aa;">-</span>
                                <span class="stat-label" style="display: block; font-size: 12px; color: #666;"><?php echo esc_html__('Total Analyzed', 'a1ai-chatbot'); ?></span>
                            </div>
                            <div class="stat-item" style="background: #f9f9f9; padding: 15px; border-radius: 5px; text-align: center;">
                                <span class="stat-number" id="a1ai-pending-analysis-tier1" style="font-size: 24px; font-weight: bold; color: #d63638;">-</span>
                                <span class="stat-label" style="display: block; font-size: 12px; color: #666;"><?php echo esc_html__('Pending Analysis', 'a1ai-chatbot'); ?></span>
                            </div>
                            <div class="stat-item" style="background: #f9f9f9; padding: 15px; border-radius: 5px; text-align: center;">
                                <span class="stat-number" id="a1ai-coverage-tier1" style="font-size: 24px; font-weight: bold; color: #00a32a;">-</span>
                                <span class="stat-label" style="display: block; font-size: 12px; color: #666;"><?php echo esc_html__('Coverage', 'a1ai-chatbot'); ?></span>
                            </div>
                            <div class="stat-item" style="background: #f9f9f9; padding: 15px; border-radius: 5px; text-align: center;">
                                <span class="stat-number" id="a1ai-last-analysis-tier1" style="font-size: 14px; font-weight: bold; color: #666;">-</span>
                                <span class="stat-label" style="display: block; font-size: 12px; color: #666;"><?php echo esc_html__('Last Analysis', 'a1ai-chatbot'); ?></span>
                            </div>
                        </div>

                        <div class="a1ai-tier1-features" style="margin-top: 15px; padding: 10px; background: #f0f8ff; border-radius: 4px;">
                            <strong><?php echo esc_html__('Tier 1 Features:', 'a1ai-chatbot'); ?></strong>
                            <span style="font-size: 13px; color: #666;"><?php echo esc_html__('Basic keyword extraction, topic analysis, sentiment analysis, content summaries, basic search functionality', 'a1ai-chatbot'); ?></span>
                        </div>
                    </div>

                    <!-- Tier 2 Content Analysis -->
                    <div class="a1ai-tier2-section" style="margin-top: 20px; padding: 20px; background: #fff; border: 1px solid #ddd; border-radius: 5px;">
                        <h3 style="margin: 0 0 10px 0; color: #0073aa;"><?php echo esc_html__('Tier 2 Content Analysis', 'a1ai-chatbot'); ?></h3>
                        <p style="margin: 0 0 15px 0; color: #666; font-size: 13px;"><?php echo esc_html__('In-depth analysis for complex websites with comprehensive content mapping and semantic understanding.', 'a1ai-chatbot'); ?></p>

                        <div class="a1ai-tier2-enable-section" style="background: #f0f0f1; padding: 15px; border-radius: 4px; margin-bottom: 20px;">
                            <label style="display: flex; align-items: center; gap: 10px; margin-bottom: 10px;">
                                <input type="checkbox" id="a1ai-tier2-enabled" value="1" <?php checked($content_settings['tier2_enabled'], 1); ?> />
                                <strong><?php echo esc_html__('Enable Tier 2 Analysis', 'a1ai-chatbot'); ?></strong>
                            </label>
                            <div style="margin-left: 30px; display: flex; align-items: center; gap: 10px;">
                                <label for="a1ai-analysis-strategy" style="font-weight: 500;"><?php echo esc_html__('Strategy:', 'a1ai-chatbot'); ?></label>
                                <select id="a1ai-analysis-strategy" <?php disabled($content_settings['tier2_enabled'], 0); ?> style="padding: 4px 8px;">
                                    <option value="tier1_fallback" <?php selected($content_settings['analysis_strategy'], 'tier1_fallback'); ?>><?php echo esc_html__('Smart: Start with Tier 1, use Tier 2 if needed', 'a1ai-chatbot'); ?></option>
                                    <option value="tier2_only" <?php selected($content_settings['analysis_strategy'], 'tier2_only'); ?>><?php echo esc_html__('Comprehensive: Always use Tier 2', 'a1ai-chatbot'); ?></option>
                                </select>
                            </div>
                        </div>

                        <div class="a1ai-tier2-controls tier2-disabled">
                        <h4><?php echo esc_html__('Content Analysis Tools', 'a1ai-chatbot'); ?></h4>
                        <div class="a1ai-analysis-controls" style="margin-bottom: 20px;">
                            <button type="button" id="a1ai-analyze-all-content-tier2" class="button button-primary" data-tier="2" disabled>
                                <?php echo esc_html__('Analyze All Content', 'a1ai-chatbot'); ?>
                            </button>
                            <button type="button" id="a1ai-analyze-recent-content-tier2" class="button" data-tier="2" disabled>
                                <?php echo esc_html__('Analyze Recent Content', 'a1ai-chatbot'); ?>
                            </button>
                            <button type="button" id="a1ai-clear-analysis-data-tier2" class="button" data-tier="2" disabled>
                                <?php echo esc_html__('Clear Analysis Data', 'a1ai-chatbot'); ?>
                            </button>
                            <button type="button" id="a1ai-view-analysis-results-tier2" class="button" data-tier="2" disabled>
                                <?php echo esc_html__('View Analysis Results', 'a1ai-chatbot'); ?>
                            </button>
                        </div>

                        <div id="a1ai-analysis-progress-tier2" style="display: none; margin-bottom: 20px;">
                            <div class="a1ai-progress-bar" style="width: 100%; height: 20px; background: #f1f1f1; border-radius: 10px; overflow: hidden;">
                                <div class="a1ai-progress-fill" style="height: 100%; background: #0073aa; width: 0%; transition: width 0.3s ease;"></div>
                            </div>
                            <p class="a1ai-progress-text" style="margin-top: 10px;"><?php echo esc_html__('Analyzing content...', 'a1ai-chatbot'); ?></p>
                        </div>

                        <h4><?php echo esc_html__('Tier 2 Statistics', 'a1ai-chatbot'); ?></h4>
                        <div class="a1ai-stats-grid" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 15px; margin-bottom: 15px;">
                            <div class="stat-item" style="background: #f9f9f9; padding: 15px; border-radius: 5px; text-align: center;">
                                <span class="stat-number" id="a1ai-total-analyzed-tier2" style="font-size: 24px; font-weight: bold; color: #0073aa;">-</span>
                                <span class="stat-label" style="display: block; font-size: 12px; color: #666;"><?php echo esc_html__('Total Analyzed', 'a1ai-chatbot'); ?></span>
                            </div>
                            <div class="stat-item" style="background: #f9f9f9; padding: 15px; border-radius: 5px; text-align: center;">
                                <span class="stat-number" id="a1ai-pending-analysis-tier2" style="font-size: 24px; font-weight: bold; color: #d63638;">-</span>
                                <span class="stat-label" style="display: block; font-size: 12px; color: #666;"><?php echo esc_html__('Pending Analysis', 'a1ai-chatbot'); ?></span>
                            </div>
                            <div class="stat-item" style="background: #f9f9f9; padding: 15px; border-radius: 5px; text-align: center;">
                                <span class="stat-number" id="a1ai-coverage-tier2" style="font-size: 24px; font-weight: bold; color: #00a32a;">-</span>
                                <span class="stat-label" style="display: block; font-size: 12px; color: #666;"><?php echo esc_html__('Coverage', 'a1ai-chatbot'); ?></span>
                            </div>
                            <div class="stat-item" style="background: #f9f9f9; padding: 15px; border-radius: 5px; text-align: center;">
                                <span class="stat-number" id="a1ai-last-analysis-tier2" style="font-size: 14px; font-weight: bold; color: #666;">-</span>
                                <span class="stat-label" style="display: block; font-size: 12px; color: #666;"><?php echo esc_html__('Last Analysis', 'a1ai-chatbot'); ?></span>
                            </div>
                        </div>

                        <div class="a1ai-tier2-features" style="margin-top: 15px; padding: 10px; background: #f0f8ff; border-radius: 4px;">
                            <strong><?php echo esc_html__('Tier 2 Features:', 'a1ai-chatbot'); ?></strong>
                            <span style="font-size: 13px; color: #666;"><?php echo esc_html__('Advanced semantic analysis, named entity recognition, content relationships mapping, SEO metadata analysis, readability scoring, advanced search with context, performance metrics tracking', 'a1ai-chatbot'); ?></span>
                        </div>
                        </div>
                    </div>

                    <!-- Analysis Results Modal -->
                    <div id="a1ai-analysis-modal" style="display: none; position: fixed; z-index: 999999; left: 0; top: 0; width: 100%; height: 100%; background-color: rgba(0,0,0,0.5);">
                        <div style="position: relative; background-color: #fefefe; margin: 5% auto; padding: 20px; border: 1px solid #888; width: 90%; max-width: 800px; border-radius: 5px; max-height: 80%; overflow-y: auto;">
                            <span style="color: #aaa; float: right; font-size: 28px; font-weight: bold; cursor: pointer;" onclick="document.getElementById('a1ai-analysis-modal').style.display='none'">&times;</span>
                            <h3><?php echo esc_html__('Content Analysis Results', 'a1ai-chatbot'); ?></h3>
                            <div id="a1ai-analysis-results-content">
                                <p><?php echo esc_html__('Loading analysis results...', 'a1ai-chatbot'); ?></p>
                            </div>
                        </div>
                    </div>
                </div>

                <style>
                    .tier2-disabled {
                        opacity: 0.6;
                        position: relative;
                    }
                    .tier2-disabled::after {
                        content: '';
                        position: absolute;
                        top: 0;
                        left: 0;
                        right: 0;
                        bottom: 0;
                        background: rgba(255, 255, 255, 0.7);
                        z-index: 1;
                        pointer-events: none;
                        border-radius: 5px;
                    }
                    .tier2-disabled > * {
                        position: relative;
                    }
                    .tier2-disabled button:disabled,
                    .tier2-disabled select:disabled {
                        cursor: not-allowed;
                        opacity: 0.5;
                    }
                </style>
                <?php else: ?>
                <div class="a1ai-upgrade-notice" style="background: #f0f8ff; border: 1px solid #0073aa; border-radius: 6px; padding: 20px; margin: 20px 0;">
                    <h3 style="margin: 0 0 15px 0; color: #0073aa;">🚀 <?php echo esc_html__('Content Awareness - Premium Feature', 'a1ai-chatbot'); ?></h3>
                    <p style="margin: 0 0 15px 0; color: #333;">
                        <?php echo esc_html__('Unlock powerful content awareness features that analyze your website content to provide more relevant AI responses.', 'a1ai-chatbot'); ?>
                    </p>
                    <ul style="margin: 0 0 15px 20px; color: #666;">
                        <li><?php echo esc_html__('Automatic content analysis', 'a1ai-chatbot'); ?></li>
                        <li><?php echo esc_html__('Smart content recommendations', 'a1ai-chatbot'); ?></li>
                        <li><?php echo esc_html__('Advanced tier 2 analysis with entity extraction', 'a1ai-chatbot'); ?></li>
                        <li><?php echo esc_html__('SEO metadata integration', 'a1ai-chatbot'); ?></li>
                    </ul>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=a1ai-chatbots#tab-upgrade')); ?>" class="button button-primary" style="background: #0073aa;">
                        <?php echo esc_html__('Upgrade to Access Content Awareness', 'a1ai-chatbot'); ?>
                    </a>
                </div>
                <?php endif; ?>

                <!-- Save Settings Button for AI Settings Tab (Free Plugin) -->
                <p class="submit" style="margin-top: 20px;">
                    <button type="submit" class="button button-primary">
                        <?php echo esc_html__('Save Settings', 'a1ai-chatbot'); ?>
                    </button>
                </p>
            </div>

            <div class="a1ai-tab-content" id="tab-business" style="display: none;">
                <h2><?php echo esc_html__('Business Information', 'a1ai-chatbot'); ?></h2>
                <p class="description" style="margin-bottom: 20px;">
                    <?php echo esc_html__('Provide accurate information about your business. The AI will use this data to answer customer questions, preventing it from making things up. This dramatically improves accuracy and builds trust.', 'a1ai-chatbot'); ?>
                </p>

                <?php if (class_exists('A1AI_Upgrade_Helper') && A1AI_Upgrade_Helper::is_upgrade_active()): ?>
                <div class="a1ai-premium-feature" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: #fff; padding: 20px; border-radius: 8px; margin-bottom: 20px;">
                    <div style="display: flex; align-items: center; justify-content: space-between;">
                        <div>
                            <h3 style="margin: 0 0 10px 0; color: #fff;">
                                <span style="font-size: 24px;">✨</span> <?php echo esc_html__('Premium: Auto-Detect Business Information', 'a1ai-chatbot'); ?>
                            </h3>
                            <p style="margin: 0; opacity: 0.95;">
                                <?php echo esc_html__('Automatically scan your website to fill in business details. Saves time by detecting your business name, services, contact info, and more from your existing website content.', 'a1ai-chatbot'); ?>
                            </p>
                        </div>
                        <button type="button" id="a1ai-auto-detect-business" class="button button-primary button-hero" style="background: #fff; color: #667eea; border: none; font-weight: 600; white-space: nowrap; margin-left: 20px;">
                            <span class="dashicons dashicons-search" style="margin-top: 3px;"></span>
                            <?php echo esc_html__('Scan Website', 'a1ai-chatbot'); ?>
                        </button>
                    </div>
                    <div id="a1ai-auto-detect-status" style="margin-top: 15px; padding: 10px; background: rgba(255,255,255,0.1); border-radius: 4px; display: none;">
                        <span class="spinner" style="float: none; margin: 0 10px 0 0;"></span>
                        <span class="status-text"><?php echo esc_html__('Scanning your website for business information...', 'a1ai-chatbot'); ?></span>
                    </div>
                </div>
                <?php endif; ?>

                <input type="hidden" name="business_info_chatbot_id" value="<?php echo esc_attr($default_chatbot_id); ?>" />

                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="business_name"><?php echo esc_html__('Business/Company Name', 'a1ai-chatbot'); ?> <span style="color: #dc3232;">*</span></label>
                        </th>
                        <td>
                            <input type="text"
                                id="business_name"
                                name="business_info[business_name]"
                                value="<?php echo esc_attr($business_name); ?>"
                                class="regular-text"
                                placeholder="<?php echo esc_attr__('e.g., ABC Web Design Co.', 'a1ai-chatbot'); ?>"
                            />
                            <p class="description">
                                <?php echo esc_html__('Your official business name as customers know it', 'a1ai-chatbot'); ?>
                            </p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="tagline"><?php echo esc_html__('Tagline/Description', 'a1ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <textarea
                                id="tagline"
                                name="business_info[tagline]"
                                rows="3"
                                class="large-text"
                                placeholder="<?php echo esc_attr__('e.g., Professional web design and digital marketing services since 2010', 'a1ai-chatbot'); ?>"
                            ><?php echo esc_textarea($tagline); ?></textarea>
                            <p class="description">
                                <?php echo esc_html__('A brief description of what your business does', 'a1ai-chatbot'); ?>
                            </p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="business_type"><?php echo esc_html__('Business Type', 'a1ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <select id="business_type" name="business_info[business_type]" class="regular-text">
                                <option value=""><?php echo esc_html__('Select type...', 'a1ai-chatbot'); ?></option>
                                <option value="service" <?php selected($business_type, 'service'); ?>><?php echo esc_html__('Service Business', 'a1ai-chatbot'); ?></option>
                                <option value="retail" <?php selected($business_type, 'retail'); ?>><?php echo esc_html__('Retail/E-commerce', 'a1ai-chatbot'); ?></option>
                                <option value="restaurant" <?php selected($business_type, 'restaurant'); ?>><?php echo esc_html__('Restaurant/Food Service', 'a1ai-chatbot'); ?></option>
                                <option value="professional" <?php selected($business_type, 'professional'); ?>><?php echo esc_html__('Professional Services', 'a1ai-chatbot'); ?></option>
                                <option value="healthcare" <?php selected($business_type, 'healthcare'); ?>><?php echo esc_html__('Healthcare', 'a1ai-chatbot'); ?></option>
                                <option value="education" <?php selected($business_type, 'education'); ?>><?php echo esc_html__('Education/Training', 'a1ai-chatbot'); ?></option>
                                <option value="other" <?php selected($business_type, 'other'); ?>><?php echo esc_html__('Other', 'a1ai-chatbot'); ?></option>
                            </select>
                            <p class="description">
                                <?php echo esc_html__('Helps the AI understand your business context', 'a1ai-chatbot'); ?>
                            </p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="services"><?php echo esc_html__('Services Offered', 'a1ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <textarea
                                id="services"
                                name="business_info[services]"
                                rows="5"
                                class="large-text"
                                placeholder="<?php echo esc_attr__('Enter one service per line, e.g.:\nWebsite Design\nSEO Optimization\nSocial Media Marketing\nContent Writing', 'a1ai-chatbot'); ?>"
                            ><?php echo esc_textarea(!empty($services) ? implode("\n", $services) : ''); ?></textarea>
                            <p class="description">
                                <?php echo esc_html__('List each service on a new line. The AI will only mention services you list here.', 'a1ai-chatbot'); ?>
                            </p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="products"><?php echo esc_html__('Products Offered', 'a1ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <textarea
                                id="products"
                                name="business_info[products]"
                                rows="5"
                                class="large-text"
                                placeholder="<?php echo esc_attr__('Enter one product per line (if applicable)', 'a1ai-chatbot'); ?>"
                            ><?php echo esc_textarea(!empty($products) ? implode("\n", $products) : ''); ?></textarea>
                            <p class="description">
                                <?php echo esc_html__('If you sell products, list them here (one per line)', 'a1ai-chatbot'); ?>
                            </p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label><?php echo esc_html__('Business Hours', 'a1ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <div class="business-hours-wrapper">
                                <?php
                                $days = array(
                                    'monday' => 'Monday',
                                    'tuesday' => 'Tuesday',
                                    'wednesday' => 'Wednesday',
                                    'thursday' => 'Thursday',
                                    'friday' => 'Friday',
                                    'saturday' => 'Saturday',
                                    'sunday' => 'Sunday'
                                );

                                foreach ($days as $day_key => $day_label) {
                                    $day_hours = isset($hours[$day_key]) ? $hours[$day_key] : '';
                                    // Translate day label individually
                                    $translated_label = '';
                                    switch ($day_key) {
                                        case 'monday':
                                            $translated_label = __('Monday', 'a1ai-chatbot');
                                            break;
                                        case 'tuesday':
                                            $translated_label = __('Tuesday', 'a1ai-chatbot');
                                            break;
                                        case 'wednesday':
                                            $translated_label = __('Wednesday', 'a1ai-chatbot');
                                            break;
                                        case 'thursday':
                                            $translated_label = __('Thursday', 'a1ai-chatbot');
                                            break;
                                        case 'friday':
                                            $translated_label = __('Friday', 'a1ai-chatbot');
                                            break;
                                        case 'saturday':
                                            $translated_label = __('Saturday', 'a1ai-chatbot');
                                            break;
                                        case 'sunday':
                                            $translated_label = __('Sunday', 'a1ai-chatbot');
                                            break;
                                        default:
                                            $translated_label = esc_html($day_label);
                                    }
                                    ?>
                                    <div style="margin-bottom: 8px;">
                                        <label style="display: inline-block; width: 100px; font-weight: 500;">
                                            <?php echo esc_html($translated_label); ?>:
                                        </label>
                                        <input type="text"
                                            name="business_info[hours][<?php echo esc_attr($day_key); ?>]"
                                            value="<?php echo esc_attr($day_hours); ?>"
                                            placeholder="<?php echo esc_attr__('e.g., 9:00 AM - 5:00 PM or Closed', 'a1ai-chatbot'); ?>"
                                            style="width: 300px;"
                                        />
                                    </div>
                                    <?php
                                }
                                ?>
                            </div>
                            <p class="description">
                                <?php echo esc_html__('Enter your operating hours for each day. Type "Closed" for days you\'re not open.', 'a1ai-chatbot'); ?>
                            </p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="address"><?php echo esc_html__('Physical Address', 'a1ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <textarea
                                id="address"
                                name="business_info[address]"
                                rows="3"
                                class="large-text"
                                placeholder="<?php echo esc_attr__('e.g., 123 Main Street\nSuite 100\nCity, State 12345', 'a1ai-chatbot'); ?>"
                            ><?php echo esc_textarea($address); ?></textarea>
                            <p class="description">
                                <?php echo esc_html__('Your business location (leave blank if online-only)', 'a1ai-chatbot'); ?>
                            </p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="phone"><?php echo esc_html__('Phone Number', 'a1ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <input type="tel"
                                id="phone"
                                name="business_info[phone]"
                                value="<?php echo esc_attr($phone); ?>"
                                class="regular-text"
                                placeholder="<?php echo esc_attr__('e.g., (555) 123-4567', 'a1ai-chatbot'); ?>"
                            />
                            <p class="description">
                                <?php echo esc_html__('Main contact phone number', 'a1ai-chatbot'); ?>
                            </p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="business_email"><?php echo esc_html__('Email Address', 'a1ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <input type="email"
                                id="business_email"
                                name="business_info[email]"
                                value="<?php echo esc_attr($email); ?>"
                                class="regular-text"
                                placeholder="<?php echo esc_attr__('e.g., info@yourbusiness.com', 'a1ai-chatbot'); ?>"
                            />
                            <p class="description">
                                <?php echo esc_html__('Main contact email address', 'a1ai-chatbot'); ?>
                            </p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="business_website"><?php echo esc_html__('Website URL', 'a1ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <input type="url"
                                id="business_website"
                                name="business_info[website]"
                                value="<?php echo esc_attr($website); ?>"
                                class="regular-text"
                                readonly
                            />
                            <p class="description">
                                <?php echo esc_html__('Your website URL (auto-filled from WordPress)', 'a1ai-chatbot'); ?>
                            </p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row">
                            <label for="policies"><?php echo esc_html__('Key Policies & FAQs', 'a1ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <textarea
                                id="policies"
                                name="business_info[policies]"
                                rows="8"
                                class="large-text"
                                placeholder="<?php echo esc_attr__('Enter important policies, one per line, e.g.:\nFree shipping on orders over $50\n30-day money-back guarantee\nWe accept Visa, Mastercard, PayPal\nAppointments required for consultations', 'a1ai-chatbot'); ?>"
                            ><?php echo esc_textarea($policies); ?></textarea>
                            <p class="description">
                                <?php echo esc_html__('Important policies, return info, payment methods, etc. The AI will reference this when answering questions.', 'a1ai-chatbot'); ?>
                            </p>
                        </td>
                    </tr>
                </table>

                <div class="a1ai-info-box" style="background: #e7f3ff; border-left: 4px solid #2271b1; padding: 15px; margin: 20px 0;">
                    <h3 style="margin-top: 0;"><?php echo esc_html__('💡 How This Improves Accuracy', 'a1ai-chatbot'); ?></h3>
                    <ul style="margin-bottom: 0;">
                        <li><?php echo esc_html__('The AI automatically receives this information in every conversation', 'a1ai-chatbot'); ?></li>
                        <li><?php echo esc_html__('It will only answer based on facts you provide here', 'a1ai-chatbot'); ?></li>
                        <li><?php echo esc_html__('When asked about something not listed, it will honestly say "I don\'t have that information"', 'a1ai-chatbot'); ?></li>
                        <li><?php echo esc_html__('This prevents the AI from making up services, prices, or hours', 'a1ai-chatbot'); ?></li>
                    </ul>
                </div>

                <p class="submit">
                    <button type="submit" class="button button-primary">
                        <?php echo esc_html__('Save Business Information', 'a1ai-chatbot'); ?>
                    </button>
                </p>
            </div>

            <div class="a1ai-tab-content" id="tab-global" style="display: none;">
                <h2><?php echo esc_html__('Global Chatbot Settings', 'a1ai-chatbot'); ?></h2>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="a1ai_global_chatbot_enabled"><?php echo esc_html__('Enable Global Chatbot', 'a1ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <input type="checkbox"
                                id="a1ai_global_chatbot_enabled"
                                name="a1ai_global_chatbot_enabled"
                                value="1"
                                <?php checked($global_enabled); ?>
                            />
                            <p class="description">
                                <?php echo esc_html__('Show a floating chatbot on all pages of your site.', 'a1ai-chatbot'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="a1ai_global_chatbot_id"><?php echo esc_html__('Select Chatbot', 'a1ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <select id="a1ai_global_chatbot_id" name="a1ai_global_chatbot_id">
                                <option value="0"><?php echo esc_html__('— Select a chatbot —', 'a1ai-chatbot'); ?></option>
                                <?php foreach ($chatbots as $chatbot) : ?>
                                    <option value="<?php echo esc_attr($chatbot['id']); ?>" <?php selected($global_chatbot_id, $chatbot['id']); ?>>
                                        <?php echo esc_html($chatbot['name']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <p class="description">
                                <?php echo esc_html__('Choose which chatbot to display globally.', 'a1ai-chatbot'); ?>
                                <?php if (empty($chatbots)) : ?>
                                    <?php echo esc_html__('No chatbots available. Please create one first.', 'a1ai-chatbot'); ?>
                                    <a href="<?php echo esc_url(admin_url('admin.php?page=a1ai-chatbots&action=add')); ?>"><?php echo esc_html__('Create Chatbot', 'a1ai-chatbot'); ?></a>
                                <?php endif; ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="a1ai_global_chatbot_position"><?php echo esc_html__('Position', 'a1ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <select id="a1ai_global_chatbot_position" name="a1ai_global_chatbot_position">
                                <option value="bottom-right" <?php selected($global_position, 'bottom-right'); ?>><?php echo esc_html__('Bottom Right', 'a1ai-chatbot'); ?></option>
                                <option value="bottom-left" <?php selected($global_position, 'bottom-left'); ?>><?php echo esc_html__('Bottom Left', 'a1ai-chatbot'); ?></option>
                            </select>
                            <p class="description">
                                <?php echo esc_html__('Choose where the chatbot appears on the screen.', 'a1ai-chatbot'); ?>
                            </p>
                        </td>
                    </tr>
                </table>

                <p class="submit">
                    <button type="submit" class="button button-primary">
                        <?php echo esc_html__('Save Settings', 'a1ai-chatbot'); ?>
                    </button>
                    <span class="spinner" style="float: none; margin: 0 0 0 10px;"></span>
                </p>
            </div>

            <div class="a1ai-tab-content" id="tab-advanced" style="display: none;">
                <h2><?php echo esc_html__('Advanced Settings', 'a1ai-chatbot'); ?></h2>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="a1ai_keep_conversation_history"><?php echo esc_html__('Keep Conversation History', 'a1ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <input type="checkbox"
                                id="a1ai_keep_conversation_history"
                                name="a1ai_keep_conversation_history"
                                value="1"
                                <?php checked($keep_history); ?>
                            />
                            <p class="description">
                                <?php echo esc_html__('Store conversation history in the database. Disabling will reduce database size but analytics will be limited.', 'a1ai-chatbot'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="a1ai_conversation_retention_days"><?php echo esc_html__('Data Retention Period', 'a1ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <input type="number"
                                id="a1ai_conversation_retention_days"
                                name="a1ai_conversation_retention_days"
                                value="<?php echo esc_attr($retention_days); ?>"
                                min="1"
                                max="365"
                            />
                            <span><?php echo esc_html__('days', 'a1ai-chatbot'); ?></span>
                            <p class="description">
                                <?php echo esc_html__('Number of days to keep conversation data. Older data will be automatically deleted.', 'a1ai-chatbot'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="a1ai_keep_data_on_uninstall"><?php echo esc_html__('Keep Data on Uninstall', 'a1ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <input type="checkbox"
                                id="a1ai_keep_data_on_uninstall"
                                name="a1ai_keep_data_on_uninstall"
                                value="1"
                                <?php checked(get_option('a1ai_keep_data_on_uninstall', false)); ?>
                            />
                            <p class="description">
                                <?php echo esc_html__('If checked, plugin data will be preserved when uninstalling. Uncheck to remove all data on uninstall.', 'a1ai-chatbot'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="a1ai_show_powered_by"><?php echo esc_html__('Show "Powered By" Attribution', 'a1ai-chatbot'); ?></label>
                        </th>
                        <td>
                            <input type="checkbox"
                                id="a1ai_show_powered_by"
                                name="a1ai_show_powered_by"
                                value="1"
                                <?php checked($show_powered_by); ?>
                            />
                            <p class="description">
                                <?php echo esc_html__('If checked, a "Powered by A1AI" link will be displayed in the chatbot interface. This attribution helps support the plugin development.', 'a1ai-chatbot'); ?>
                            </p>
                        </td>
                    </tr>
                </table>

                <p class="submit">
                    <button type="submit" class="button button-primary">
                        <?php echo esc_html__('Save Settings', 'a1ai-chatbot'); ?>
                    </button>
                    <span class="spinner" style="float: none; margin: 0 0 0 10px;"></span>
                </p>
            </div>

            <div class="a1ai-tab-content" id="tab-upgrades" style="display: none;">
                <div class="a1ai-upgrades-bg"></div>
                <h2 class="a1ai-upgrades-title"><?php echo esc_html__('Upgrades, Add-Ons & Services', 'a1ai-chatbot'); ?></h2>
                <div class="a1ai-upgrades-section">
                    <h3><span class="a1ai-icon a1ai-icon-rocket"></span><?php echo esc_html__('Upgrade and Services', 'a1ai-chatbot'); ?></h3>
                    <div class="a1ai-upgrades-columns">
                        <div class="a1ai-upgrade-card a1ai-upgrade-standard">
                            <span class="a1ai-badge a1ai-badge-popular"><?php echo esc_html__('Most Popular', 'a1ai-chatbot'); ?></span>
                            <h4><?php echo esc_html__('AI Chat Paid Upgrade - Premium', 'a1ai-chatbot'); ?></h4>
                            <ul>
                                <li><?php echo esc_html__('All core features', 'a1ai-chatbot'); ?></li>
                                <li><?php echo esc_html__('Better response control', 'a1ai-chatbot'); ?></li>
                                <li><?php echo esc_html__('Upgraded support', 'a1ai-chatbot'); ?></li>
                                <li><?php echo esc_html__('Content aware responses', 'a1ai-chatbot'); ?></li>
                                <li><?php echo esc_html__('Conversational contact forms', 'a1ai-chatbot'); ?></li>
                                <li><?php echo esc_html__('Prompting', 'a1ai-chatbot'); ?></li>
                                <li><?php echo esc_html__('Voice to text', 'a1ai-chatbot'); ?></li>
                                <li><?php echo esc_html__('And more', 'a1ai-chatbot'); ?></li>
                            </ul>
                            <a href="https://a1aidigital.com/ai-wordpress-plugin/upgrades-products/" class="button button-primary a1ai-upgrade-link" target="_blank"><span class="a1ai-icon a1ai-icon-arrow"></span><?php echo esc_html__('Upgrade Ai Plugin', 'a1ai-chatbot'); ?></a>
                        </div>
                        <div class="a1ai-upgrade-card a1ai-upgrade-enterprise a1ai-glow">
                            <span class="a1ai-badge a1ai-badge-value"><?php echo esc_html__('Best Value', 'a1ai-chatbot'); ?></span>
                            <h4><?php echo esc_html__('Managed AI Chat and Reporting', 'a1ai-chatbot'); ?></h4>
                            
                                <li><?php echo esc_html__('All Upgraded features', 'a1ai-chatbot'); ?></li>
                                <li><?php echo esc_html__('Setup and Management', 'a1ai-chatbot'); ?></li>
                                <li><?php echo esc_html__('Ongoing implementation and testing', 'a1ai-chatbot'); ?></li>                                
                                <li><?php echo esc_html__('Advanced analytics & reporting', 'a1ai-chatbot'); ?></li>
                                <li><?php echo esc_html__('Priority support', 'a1ai-chatbot'); ?></li>
                                <li><?php echo esc_html__('Custom integrations', 'a1ai-chatbot'); ?></li>
                                <li><?php echo esc_html__('Ai reviews and questionnaires', 'a1ai-chatbot'); ?></li>
                                <li><?php echo esc_html__('And More', 'a1ai-chatbot'); ?></li>
                            
                            <a href="https://a1aidigital.com/ai-wordpress-plugin/services/" class="button button-secondary a1ai-upgrade-link" target="_blank"><span class="a1ai-icon a1ai-icon-arrow"></span><?php echo esc_html__('Get Profesionally Managed Ai Services', 'a1ai-chatbot'); ?></a>
                        </div>
                    </div>
                </div>

                <hr class="a1ai-section-divider" />
                <div class="a1ai-services-section">
                    <h3><span class="a1ai-icon a1ai-icon-cog"></span><?php echo esc_html__('Services', 'a1ai-chatbot'); ?></h3>
                    <p><?php echo esc_html__('We offer personalized services that use a blend of Ai and "traditional", human, techniques. Our services include:', 'a1ai-chatbot'); ?></p>
                            <ul>
                                <li><?php echo esc_html__('Ai Chatbot Management', 'a1ai-chatbot'); ?></li>
                                <li><?php echo esc_html__('Google Ads Management', 'a1ai-chatbot'); ?></li>
                                <li><?php echo esc_html__('SEO Management', 'a1ai-chatbot'); ?></li>                                
                                <li><?php echo esc_html__('Social Media Management', 'a1ai-chatbot'); ?></li>
                                <li><?php echo esc_html__('Website Design and Management', 'a1ai-chatbot'); ?></li>
                                <li><?php echo esc_html__('Business Consulting', 'a1ai-chatbot'); ?></li>
                                <li><?php echo esc_html__('And More', 'a1ai-chatbot'); ?></li>
                            </ul>
                    
                    <a href="https://a1aidigital.com/all-services/" class="button a1ai-services-link a1ai-btn-flash" target="_blank"><span class="a1ai-icon a1ai-icon-arrow"></span><?php echo esc_html__('View All Services', 'a1ai-chatbot'); ?></a>
                </div>

                <hr class="a1ai-section-divider" />
                <div class="a1ai-addons-section">
                    <h3><span class="a1ai-icon a1ai-icon-puzzle"></span><?php echo esc_html__('Add-Ons', 'a1ai-chatbot'); ?></h3>
                    <p><?php echo esc_html__('Coming soon: AI plugins for customer tracking and analysis, Google Ads Management, integrating on-site tracking with Google Ads, Google Analytics, ai crm, ai SEO , and more. Planned add-ons include AI CRM connections, customer outreach, SEO, and more. Real value add-ons that will be like hiring a team of people', 'a1ai-chatbot'); ?></p>
                   
                                <li><?php echo esc_html__('Customer tracking and analysis', 'a1ai-chatbot'); ?></li>
                                <li><?php echo esc_html__('Integrating on-site tracking with PPC and SEO', 'a1ai-chatbot'); ?></li>
                                <li><?php echo esc_html__('Upgraded customer support', 'a1ai-chatbot'); ?></li>
                                <li><?php echo esc_html__('Google Analytics', 'a1ai-chatbot'); ?></li>
                                <li><?php echo esc_html__('Ai crm', 'a1ai-chatbot'); ?></li>
                                <li><?php echo esc_html__('Ai SEO', 'a1ai-chatbot'); ?></li>
                                <li><?php echo esc_html__('Ai Newsletters and Outreach', 'a1ai-chatbot'); ?></li>
                                <li><?php echo esc_html__('And more', 'a1ai-chatbot'); ?></li>
                                        
                    
                    <a href="https://a1aidigital.com/ai-ecosystem/" class="button a1ai-addons-link a1ai-btn-flash" target="_blank"><span class="a1ai-icon a1ai-icon-arrow"></span><?php echo esc_html__('Learn More About Add-Ons', 'a1ai-chatbot'); ?></a>
                </div>
                
            </div>
            
            
        </form>
    </div>
</div> 
