<?php
/**
 * Admin reviews template
 *
 * @link       https://a1aidigital.com
 * @since      1.0.0
 *
 * @package    A1AI
 * @subpackage A1AI/admin/partials
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

// Get all chatbots for the dropdown
$chatbots = $data_manager->get_chatbots();
$current_review_chatbot = get_option('a1ai_review_chatbot_id', 0);
$current_rating_system = get_option('a1ai_rating_system', '5-star');
$current_intro_text = get_option('a1ai_review_intro_text', 'Thank you for taking the time to review us, please let us know how your experience has been!');

// Handle review chatbot selection
if (isset($_POST['action']) && $_POST['action'] === 'select_review_chatbot') {
    if (isset($_POST['a1ai_review_nonce']) && wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['a1ai_review_nonce'])), 'a1ai_select_review_chatbot')) {
        $review_chatbot_id = isset($_POST['review_chatbot_id']) ? intval($_POST['review_chatbot_id']) : 0;
        $rating_system = isset($_POST['rating_system']) ? sanitize_text_field(wp_unslash($_POST['rating_system'])) : '5-star';
        $intro_text = isset($_POST['intro_text']) ? sanitize_textarea_field(wp_unslash($_POST['intro_text'])) : '';

        // Handle external review links
        $google_link = isset($_POST['google_review_link']) ? esc_url_raw(wp_unslash($_POST['google_review_link'])) : '';
        $yelp_link = isset($_POST['yelp_review_link']) ? esc_url_raw(wp_unslash($_POST['yelp_review_link'])) : '';
        $facebook_link = isset($_POST['facebook_review_link']) ? esc_url_raw(wp_unslash($_POST['facebook_review_link'])) : '';
        $other_link = isset($_POST['other_review_link']) ? esc_url_raw(wp_unslash($_POST['other_review_link'])) : '';
        $other_name = isset($_POST['other_review_name']) ? sanitize_text_field(wp_unslash($_POST['other_review_name'])) : 'Other Platform';

        // Handle minimum thresholds
        $min_star_threshold = isset($_POST['min_star_threshold']) ? intval($_POST['min_star_threshold']) : 3;
        $min_scale_threshold = isset($_POST['min_scale_threshold']) ? intval($_POST['min_scale_threshold']) : 6;

        update_option('a1ai_review_chatbot_id', $review_chatbot_id);
        update_option('a1ai_rating_system', $rating_system);
        update_option('a1ai_review_intro_text', $intro_text);
        update_option('a1ai_review_google_link', $google_link);
        update_option('a1ai_review_yelp_link', $yelp_link);
        update_option('a1ai_review_facebook_link', $facebook_link);
        update_option('a1ai_review_other_link', $other_link);
        update_option('a1ai_review_other_name', $other_name);
        update_option('a1ai_min_star_threshold', $min_star_threshold);
        update_option('a1ai_min_scale_threshold', $min_scale_threshold);
        $current_review_chatbot = $review_chatbot_id; // Update the current variable
        $current_intro_text = $intro_text; // Update the current variable

        // Show success message
        echo '<div class="notice notice-success is-dismissible"><p>';
        if ($review_chatbot_id > 0) {
            $selected_chatbot = array_filter($chatbots, function($c) use ($review_chatbot_id) {
                return $c['id'] == $review_chatbot_id;
            });
            $chatbot_name = !empty($selected_chatbot) ? reset($selected_chatbot)['name'] : 'Unknown';
            $rating_display = ($rating_system === '10-scale') ? '1-10 Scale' : '5-Star';
            // translators: %1$s is the chatbot name, %2$s is the rating system type
            echo sprintf(esc_html__('Successfully set "%1$s" as the review chatbot with %2$s rating system!', 'a1ai-chatbot'), esc_html($chatbot_name), esc_html($rating_display));
        } else {
            echo esc_html__('Review chatbot selection cleared.', 'a1ai-chatbot');
        }
        echo '</p></div>';
    }
}

// Get actual submitted reviews from WordPress options
$all_reviews = get_option('a1ai_reviews', array());
$sample_reviews = array();

// Filter reviews for the selected review chatbot if one is selected
if ($current_review_chatbot > 0) {
    $sample_reviews = array_filter($all_reviews, function($review) use ($current_review_chatbot) {
        return isset($review['chatbot_id']) && $review['chatbot_id'] == $current_review_chatbot;
    });
} else {
    // If no specific chatbot selected, show all reviews
    $sample_reviews = $all_reviews;
}

// Sort reviews by date (newest first)
usort($sample_reviews, function($a, $b) {
    $date_a = isset($a['date']) ? strtotime($a['date']) : 0;
    $date_b = isset($b['date']) ? strtotime($b['date']) : 0;
    return $date_b - $date_a;
});

?>

<h2><?php echo esc_html__('Reviews Management', 'a1ai-chatbot'); ?></h2>

<div class="a1ai-reviews-section">
    <div class="a1ai-reviews-controls">
        <style>
        .a1ai-reviews-selector {
            display: block !important;
            float: none !important;
        }
        .a1ai-reviews-selector div {
            display: block !important;
            float: none !important;
            clear: both !important;
            width: 100% !important;
        }
        .a1ai-reviews-selector label {
            display: block !important;
            float: none !important;
            width: 100% !important;
        }
        .a1ai-reviews-selector input,
        .a1ai-reviews-selector select,
        .a1ai-reviews-selector textarea {
            display: block !important;
            float: none !important;
            width: 100% !important;
            box-sizing: border-box !important;
        }
        </style>
        <form method="post" class="a1ai-reviews-selector" style="max-width: 600px; display: block; clear: both;">
            <?php wp_nonce_field('a1ai_select_review_chatbot', 'a1ai_review_nonce'); ?>
            <input type="hidden" name="action" value="select_review_chatbot" />

            <div style="margin-bottom: 25px;">
                <label for="review_chatbot_id" style="display: block; font-weight: 600; margin-bottom: 8px;"><?php echo esc_html__('Select Review Chatbot:', 'a1ai-chatbot'); ?></label>
                <select id="review_chatbot_id" name="review_chatbot_id" style="width: 100%;">
                    <option value="0"><?php echo esc_html__('-- Select a Chatbot for Reviews --', 'a1ai-chatbot'); ?></option>
                    <?php foreach ($chatbots as $chatbot) : ?>
                        <option value="<?php echo esc_attr($chatbot['id']); ?>" <?php selected($current_review_chatbot, $chatbot['id']); ?>>
                            <?php echo esc_html($chatbot['name']); ?>
                            <?php if ($current_review_chatbot == $chatbot['id']) : ?>
                                <?php echo esc_html__('(Current Review Chatbot)', 'a1ai-chatbot'); ?>
                            <?php endif; ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>

            <div style="margin-bottom: 25px;">
                <label for="rating_system" style="display: block; font-weight: 600; margin-bottom: 8px;"><?php echo esc_html__('Rating System:', 'a1ai-chatbot'); ?></label>
                <select id="rating_system" name="rating_system" style="width: 100%;">
                    <option value="5-star" <?php selected($current_rating_system, '5-star'); ?>>
                        <?php echo esc_html__('⭐ 5-Star Rating', 'a1ai-chatbot'); ?>
                    </option>
                    <option value="10-scale" <?php selected($current_rating_system, '10-scale'); ?>>
                        <?php echo esc_html__('📊 1-10 Scale', 'a1ai-chatbot'); ?>
                    </option>
                </select>
            </div>

            <div style="margin-bottom: 25px;">
                <label for="intro_text" style="display: block; font-weight: 600; margin-bottom: 8px;"><?php echo esc_html__('Introduction Message:', 'a1ai-chatbot'); ?></label>
                <textarea id="intro_text" name="intro_text" rows="3" style="width: 100%; resize: vertical;" placeholder="<?php echo esc_attr__('Enter the introduction message for your review form...', 'a1ai-chatbot'); ?>"><?php echo esc_textarea($current_intro_text); ?></textarea>
            </div>

            <div style="margin-bottom: 25px;">
                <h3 style="margin-bottom: 5px;"><?php echo esc_html__('External Review Links', 'a1ai-chatbot'); ?></h3>
                <p style="margin-bottom: 20px; color: #666; font-style: italic;"><?php echo esc_html__('Add links to external review platforms. After submitting the internal review, these links will open in new windows.', 'a1ai-chatbot'); ?></p>

                <?php
                $current_google_link = get_option('a1ai_review_google_link', '');
                $current_yelp_link = get_option('a1ai_review_yelp_link', '');
                $current_facebook_link = get_option('a1ai_review_facebook_link', '');
                $current_other_link = get_option('a1ai_review_other_link', '');
                $current_other_name = get_option('a1ai_review_other_name', 'Other Platform');
                ?>

                <div style="margin-bottom: 15px;">
                    <label for="google_review_link" style="display: block; font-weight: 600; margin-bottom: 5px;">🌟 Google Review Link:</label>
                    <input type="url" id="google_review_link" name="google_review_link" value="<?php echo esc_attr($current_google_link); ?>" placeholder="https://g.page/your-business/review" style="width: 100%;" />
                </div>

                <div style="margin-bottom: 15px;">
                    <label for="yelp_review_link" style="display: block; font-weight: 600; margin-bottom: 5px;">⭐ Yelp Review Link:</label>
                    <input type="url" id="yelp_review_link" name="yelp_review_link" value="<?php echo esc_attr($current_yelp_link); ?>" placeholder="https://www.yelp.com/writeareview/biz/your-business" style="width: 100%;" />
                </div>

                <div style="margin-bottom: 15px;">
                    <label for="facebook_review_link" style="display: block; font-weight: 600; margin-bottom: 5px;">📘 Facebook Review Link:</label>
                    <input type="url" id="facebook_review_link" name="facebook_review_link" value="<?php echo esc_attr($current_facebook_link); ?>" placeholder="https://www.facebook.com/your-business/reviews" style="width: 100%;" />
                </div>

                <div style="margin-bottom: 15px;">
                    <label for="other_review_name" style="display: block; font-weight: 600; margin-bottom: 5px;">🔗 Other Platform Name:</label>
                    <input type="text" id="other_review_name" name="other_review_name" value="<?php echo esc_attr($current_other_name); ?>" placeholder="Platform Name" style="width: 100%;" />
                </div>

                <div style="margin-bottom: 15px;">
                    <label for="other_review_link" style="display: block; font-weight: 600; margin-bottom: 5px;">🔗 Other Platform URL:</label>
                    <input type="url" id="other_review_link" name="other_review_link" value="<?php echo esc_attr($current_other_link); ?>" placeholder="https://your-custom-review-platform.com" style="width: 100%;" />
                </div>
            </div>

            <div style="margin-bottom: 25px;">
                <h3 style="margin-bottom: 5px;"><?php echo esc_html__('Minimum Review Threshold', 'a1ai-chatbot'); ?></h3>
                <p style="margin-bottom: 20px; color: #666; font-style: italic;"><?php echo esc_html__('Set minimum ratings required to open external review links. Reviews below these thresholds will receive internal feedback only.', 'a1ai-chatbot'); ?></p>

                <?php
                $min_star_threshold = get_option('a1ai_min_star_threshold', 3);
                $min_scale_threshold = get_option('a1ai_min_scale_threshold', 6);
                ?>

                <div style="margin-bottom: 15px;">
                    <label for="min_star_threshold" style="display: block; font-weight: 600; margin-bottom: 5px;">⭐ Minimum Stars (1-5):</label>
                    <select id="min_star_threshold" name="min_star_threshold" style="width: 100%;">
                        <?php for ($i = 1; $i <= 5; $i++) : ?>
                            <option value="<?php echo esc_attr($i); ?>" <?php selected($min_star_threshold, $i); ?>>
                                <?php echo esc_html($i); ?> Star<?php echo $i > 1 ? 's' : ''; ?>
                            </option>
                        <?php endfor; ?>
                    </select>
                </div>

                <div style="margin-bottom: 15px;">
                    <label for="min_scale_threshold" style="display: block; font-weight: 600; margin-bottom: 5px;">📊 Minimum Scale (1-10):</label>
                    <select id="min_scale_threshold" name="min_scale_threshold" style="width: 100%;">
                        <?php for ($i = 1; $i <= 10; $i++) : ?>
                            <option value="<?php echo esc_attr($i); ?>" <?php selected($min_scale_threshold, $i); ?>>
                                <?php echo esc_html($i); ?> out of 10
                            </option>
                        <?php endfor; ?>
                    </select>
                </div>
            </div>

            <button type="submit" class="button button-primary"><?php echo esc_html__('Save Review Settings', 'a1ai-chatbot'); ?></button>
        </form>

    </div>

    <?php if (empty($sample_reviews)) : ?>
        <div class="a1ai-empty-state">
            <div class="a1ai-empty-reviews">
                <h3><?php echo esc_html__('Review System Ready to Configure', 'a1ai-chatbot'); ?></h3>
                <p><?php echo esc_html__('Select a chatbot above to enable review collection, then generate a review form to integrate with your website.', 'a1ai-chatbot'); ?></p>

                <div class="a1ai-upgrade-notice">
                    <h4><?php echo esc_html__('🚀 Want Professional Review Management?', 'a1ai-chatbot'); ?></h4>
                    <p><?php echo esc_html__('Get advanced review collection, automated follow-ups, and detailed analytics with our premium services.', 'a1ai-chatbot'); ?></p>
                    <div class="a1ai-contact-info">
                        <p><strong><?php echo esc_html__('Contact A1AI for Premium Review Features:', 'a1ai-chatbot'); ?></strong></p>
                        <p><strong><?php echo esc_html__('Email:', 'a1ai-chatbot'); ?></strong> <a href="mailto:contact@a1aidigital.com">contact@a1aidigital.com</a></p>
                        <p><strong><?php echo esc_html__('Phone:', 'a1ai-chatbot'); ?></strong> <a href="tel:801-893-1686">801-893-1686</a></p>
                    </div>
                </div>
            </div>
        </div>
    <?php else : ?>
        <div class="a1ai-reviews-stats">
            <div class="a1ai-stat-item">
                <span class="a1ai-stat-number"><?php echo esc_html(count($sample_reviews)); ?></span>
                <span class="a1ai-stat-label"><?php echo esc_html__('Total Reviews', 'a1ai-chatbot'); ?></span>
            </div>
            <div class="a1ai-stat-item">
                <?php
                $avg_rating = array_sum(array_column($sample_reviews, 'rating')) / count($sample_reviews);
                ?>
                <span class="a1ai-stat-number"><?php echo esc_html(number_format($avg_rating, 1)); ?></span>
                <span class="a1ai-stat-label"><?php echo esc_html__('Average Rating', 'a1ai-chatbot'); ?></span>
            </div>
            <div class="a1ai-stat-item">
                <?php
                $pending_count = count(array_filter($sample_reviews, function($r) { return $r['status'] === 'pending'; }));
                ?>
                <span class="a1ai-stat-number"><?php echo esc_html($pending_count); ?></span>
                <span class="a1ai-stat-label"><?php echo esc_html__('Pending Review', 'a1ai-chatbot'); ?></span>
            </div>
        </div>

        <div class="a1ai-reviews-list">
            <?php foreach ($sample_reviews as $review) : ?>
                <div class="a1ai-review-item" data-review-id="<?php echo esc_attr($review['id']); ?>">
                    <div class="a1ai-review-header">
                        <div class="a1ai-review-rating">
                            <?php for ($i = 1; $i <= 5; $i++) : ?>
                                <span class="a1ai-star <?php echo $i <= $review['rating'] ? 'filled' : ''; ?>">★</span>
                            <?php endfor; ?>
                            <span class="a1ai-rating-number">(<?php echo esc_html($review['rating']); ?>/5)</span>
                        </div>
                        <div class="a1ai-review-status">
                            <span class="a1ai-status-badge a1ai-status-<?php echo esc_attr($review['status']); ?>">
                                <?php echo esc_html(ucfirst($review['status'])); ?>
                            </span>
                        </div>
                        <div class="a1ai-review-date">
                            <?php echo esc_html(date_i18n(get_option('date_format'), strtotime($review['date']))); ?>
                        </div>
                    </div>

                    <div class="a1ai-review-content">
                        <div class="a1ai-review-comment">
                            <p><?php echo esc_html($review['comment']); ?></p>
                        </div>
                        <div class="a1ai-review-meta">
                            <strong><?php echo esc_html($review['reviewer_name']); ?></strong>
                            <span class="a1ai-review-email"><?php echo esc_html($review['reviewer_email']); ?></span>
                            <?php
                            $chatbot = array_filter($chatbots, function($c) use ($review) {
                                return $c['id'] == $review['chatbot_id'];
                            });
                            $chatbot_name = !empty($chatbot) ? reset($chatbot)['name'] : 'Unknown Chatbot';
                            ?>
                            <span class="a1ai-review-chatbot"><?php echo esc_html__('via', 'a1ai-chatbot'); ?> <?php echo esc_html($chatbot_name); ?></span>
                        </div>
                    </div>

                    <div class="a1ai-review-actions">
                        <?php if ($review['status'] === 'pending') : ?>
                            <button type="button" class="button button-primary a1ai-approve-review" data-review-id="<?php echo esc_attr($review['id']); ?>">
                                <?php echo esc_html__('Approve', 'a1ai-chatbot'); ?>
                            </button>
                        <?php endif; ?>
                        <button type="button" class="button a1ai-edit-review" data-review-id="<?php echo esc_attr($review['id']); ?>">
                            <?php echo esc_html__('Edit', 'a1ai-chatbot'); ?>
                        </button>
                        <button type="button" class="button button-link-delete a1ai-delete-review" data-review-id="<?php echo esc_attr($review['id']); ?>">
                            <?php echo esc_html__('Delete', 'a1ai-chatbot'); ?>
                        </button>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>

        <div class="a1ai-upgrade-notice" style="margin-top: 30px;">
            <h4><?php echo esc_html__('🚀 Upgrade to Premium Review Features', 'a1ai-chatbot'); ?></h4>
            <p><?php echo esc_html__('Get advanced review collection, automated follow-ups, sentiment analysis, and detailed analytics.', 'a1ai-chatbot'); ?></p>
            <ul>
                <li><?php echo esc_html__('Automated review requests after conversations', 'a1ai-chatbot'); ?></li>
                <li><?php echo esc_html__('Customizable review forms and questions', 'a1ai-chatbot'); ?></li>
                <li><?php echo esc_html__('Sentiment analysis and trending topics', 'a1ai-chatbot'); ?></li>
                <li><?php echo esc_html__('Integration with Google Reviews and other platforms', 'a1ai-chatbot'); ?></li>
                <li><?php echo esc_html__('Detailed analytics and reporting', 'a1ai-chatbot'); ?></li>
            </ul>
            <div class="a1ai-contact-info">
                <p><strong><?php echo esc_html__('Contact A1AI for Premium Review Features:', 'a1ai-chatbot'); ?></strong></p>
                <p><strong><?php echo esc_html__('Email:', 'a1ai-chatbot'); ?></strong> <a href="mailto:contact@a1aidigital.com">contact@a1aidigital.com</a></p>
                <p><strong><?php echo esc_html__('Phone:', 'a1ai-chatbot'); ?></strong> <a href="tel:801-893-1686">801-893-1686</a></p>
            </div>
        </div>
    <?php endif; ?>
</div>


<style>
.a1ai-reviews-section {
    max-width: 100%;
}

.a1ai-reviews-controls {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
    padding: 15px;
    background: #f9f9f9;
    border-radius: 5px;
}

.a1ai-reviews-selector {
    display: flex;
    align-items: center;
    gap: 10px;
    flex: 1;
}

.a1ai-reviews-selector select {
    min-width: 200px;
}

.a1ai-reviews-selector textarea {
    min-width: 300px;
    resize: vertical;
    font-family: inherit;
}

.a1ai-reviews-selector label {
    font-weight: 600;
    color: #333;
}

.a1ai-reviews-stats {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
    gap: 15px;
    margin-bottom: 25px;
}

.a1ai-stat-item {
    background: white;
    padding: 20px;
    border-radius: 8px;
    text-align: center;
    border: 1px solid #ddd;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.a1ai-stat-number {
    display: block;
    font-size: 28px;
    font-weight: bold;
    color: #0073aa;
    margin-bottom: 5px;
}

.a1ai-stat-label {
    font-size: 12px;
    color: #666;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.a1ai-reviews-list {
    display: flex;
    flex-direction: column;
    gap: 15px;
}

.a1ai-review-item {
    background: white;
    border: 1px solid #ddd;
    border-radius: 8px;
    padding: 20px;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
    transition: box-shadow 0.2s ease;
}

.a1ai-review-item:hover {
    box-shadow: 0 4px 8px rgba(0,0,0,0.15);
}

.a1ai-review-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 15px;
    border-bottom: 1px solid #eee;
    padding-bottom: 10px;
}

.a1ai-review-rating {
    display: flex;
    align-items: center;
    gap: 5px;
}

.a1ai-star {
    color: #ddd;
    font-size: 18px;
}

.a1ai-star.filled {
    color: #ffa500;
}

.a1ai-rating-number {
    font-size: 14px;
    color: #666;
    margin-left: 5px;
}

.a1ai-status-badge {
    padding: 4px 12px;
    border-radius: 12px;
    font-size: 12px;
    font-weight: 500;
    text-transform: uppercase;
}

.a1ai-status-approved {
    background: #d4edda;
    color: #155724;
}

.a1ai-status-pending {
    background: #fff3cd;
    color: #856404;
}

.a1ai-review-date {
    font-size: 12px;
    color: #666;
}

.a1ai-review-content {
    margin-bottom: 15px;
}

.a1ai-review-comment p {
    margin: 0 0 10px 0;
    line-height: 1.6;
    color: #333;
}

.a1ai-review-meta {
    display: flex;
    align-items: center;
    gap: 15px;
    font-size: 13px;
    color: #666;
}

.a1ai-review-email {
    color: #0073aa;
}

.a1ai-review-chatbot {
    font-style: italic;
}

.a1ai-review-actions {
    display: flex;
    gap: 10px;
    justify-content: flex-end;
}

.a1ai-empty-reviews {
    text-align: center;
    padding: 60px 20px;
}

.a1ai-empty-reviews h3 {
    margin-bottom: 10px;
    color: #666;
}

.a1ai-upgrade-notice {
    background: #f0f8ff;
    border: 2px solid #0073aa;
    border-radius: 8px;
    padding: 20px;
    margin-top: 20px;
}

.a1ai-upgrade-notice h4 {
    margin: 0 0 15px 0;
    color: #0073aa;
}

.a1ai-upgrade-notice ul {
    margin: 15px 0;
    padding-left: 20px;
}

.a1ai-upgrade-notice li {
    margin-bottom: 5px;
}

.a1ai-contact-info {
    background: rgba(0,115,170,0.1);
    padding: 15px;
    border-radius: 5px;
    margin-top: 15px;
}

.a1ai-contact-info p {
    margin: 5px 0;
}

.a1ai-contact-info a {
    color: #0073aa;
    text-decoration: none;
}

.a1ai-contact-info a:hover {
    text-decoration: underline;
}


@media (max-width: 768px) {
    .a1ai-reviews-controls {
        flex-direction: column;
        gap: 15px;
        align-items: stretch;
    }

    .a1ai-review-header {
        flex-direction: column;
        align-items: flex-start;
        gap: 10px;
    }

    .a1ai-review-meta {
        flex-direction: column;
        align-items: flex-start;
        gap: 5px;
    }

    .a1ai-review-actions {
        justify-content: flex-start;
        flex-wrap: wrap;
    }
}
</style>

<script>
jQuery(document).ready(function($) {
    // Review action buttons
    $('.a1ai-approve-review').on('click', function() {
        alert('<?php echo esc_js(__('Review approval functionality requires premium features.', 'a1ai-chatbot')); ?>');
    });

    $('.a1ai-edit-review').on('click', function() {
        alert('<?php echo esc_js(__('Review editing functionality requires premium features.', 'a1ai-chatbot')); ?>');
    });

    $('.a1ai-delete-review').on('click', function() {
        if (confirm('<?php echo esc_js(__('Are you sure you want to delete this review?', 'a1ai-chatbot')); ?>')) {
            alert('<?php echo esc_js(__('Review deletion functionality requires premium features.', 'a1ai-chatbot')); ?>');
        }
    });
});
</script>
