<?php
/**
 * Admin contact forms management template
 *
 * @link       https://a1aidigital.com
 * @since      1.0.0
 *
 * @package    A1AI
 * @subpackage A1AI/admin/partials
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

// Get all chatbots for selection
$chatbots = $data_manager->get_chatbots();

// Get contact forms for the selected chatbot
$selected_chatbot_id = 0;
if ( isset($_GET['chatbot_id'], $_GET['a1ai_select_nonce']) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['a1ai_select_nonce'] ) ), 'a1ai_select_chatbot' ) ) {
    $selected_chatbot_id = intval($_GET['chatbot_id']);
}

$contact_forms = array();
if ($selected_chatbot_id > 0) {
    // Get contact form configuration for this chatbot
    $chatbot = $data_manager->get_chatbot($selected_chatbot_id);
    if ($chatbot && isset($chatbot['settings'])) {
        // Check if settings is already an array or needs to be unserialized
        if (is_array($chatbot['settings'])) {
            $settings = $chatbot['settings'];
        } else {
            $settings = maybe_unserialize($chatbot['settings']);
        }
        $contact_forms = isset($settings['contact_forms']) ? $settings['contact_forms'] : array();
    }
}

// Show success messages from redirects
if (isset($_GET['form_saved']) && $_GET['form_saved'] === '1') {
    echo '<div class="notice notice-success"><p>' . esc_html__('Contact form saved successfully!', 'a1ai-chatbot') . '</p></div>';
    echo '<script>
        jQuery(document).ready(function($) {
            // Ensure contact forms tab is activated after form save
            setTimeout(function() {
                if (window.location.hash === "#tab-contact-forms") {
                    console.log("A1AI: Forcing contact forms tab activation after save");
                    $(".a1ai-tabs-nav a[href=\'#tab-contact-forms\']").trigger("click");
                }
            }, 100);
        });
    </script>';
}
if (isset($_GET['form_deleted']) && $_GET['form_deleted'] === '1') {
    echo '<div class="notice notice-success"><p>' . esc_html__('Contact form deleted successfully!', 'a1ai-chatbot') . '</p></div>';
}

// Handle form submissions
if ( isset($_SERVER['REQUEST_METHOD']) && 'POST' === $_SERVER['REQUEST_METHOD'] ) {
    if ( isset($_POST['a1ai_action']) ) {
        $a1ai_action = sanitize_key( wp_unslash( $_POST['a1ai_action'] ) );
        switch ( $a1ai_action ) {
            case 'save_contact_form':
                if ( ! current_user_can('manage_options') ) {
                    echo '<div class="notice notice-error"><p>' . esc_html__('You do not have permission to perform this action.', 'a1ai-chatbot') . '</p></div>';
                    break;
                }
                if ( ! isset($_POST['a1ai_contact_form_nonce']) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['a1ai_contact_form_nonce'] ) ), 'a1ai_save_contact_form' ) ) {
                    echo '<div class="notice notice-error"><p>' . esc_html__('Security check failed. Please try again.', 'a1ai-chatbot') . '</p></div>';
                    break;
                }

                if ( isset($_POST['chatbot_id'], $_POST['form_name'], $_POST['form_type']) ) {
                    $chatbot_id = intval( $_POST['chatbot_id'] );
                    $form_name = sanitize_text_field( wp_unslash( $_POST['form_name'] ) );
                    $form_type = sanitize_text_field( wp_unslash( $_POST['form_type'] ) );
                    $save_submissions = isset($_POST['save_submissions']) ? 1 : 0;
                    $email_notifications = isset($_POST['email_notifications']) ? 1 : 0;
                    $notification_email = sanitize_email( wp_unslash( $_POST['notification_email'] ?? '' ) );

                    // Get form fields
                    $form_fields = array();
                    if (isset($_POST['field_name']) && is_array($_POST['field_name'])) {
                        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- We sanitize after unslashing
                        $field_names = wp_unslash($_POST['field_name']);
                        foreach ($field_names as $index => $field_name) {
                            if (!empty($field_name)) {
                                $form_fields[] = array(
                                    'name' => sanitize_text_field($field_name),
                                    'label' => sanitize_text_field(wp_unslash($_POST['field_label'][$index] ?? $field_name)),
                                    'type' => sanitize_text_field(wp_unslash($_POST['field_type'][$index] ?? 'text')),
                                    'required' => isset($_POST['field_required'][$index]) ? 1 : 0,
                                    'placeholder' => sanitize_text_field(wp_unslash($_POST['field_placeholder'][$index] ?? ''))
                                );
                            }
                        }
                    }

                    // Get current chatbot settings
                    $chatbot = $data_manager->get_chatbot($chatbot_id);
                    if ($chatbot) {
                        // Debug logging
                        if (defined('WP_DEBUG') && WP_DEBUG) {
                            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging wrapped in WP_DEBUG check
                            error_log('A1AI Contact Forms Debug - Form Name: ' . $form_name);
                            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging wrapped in WP_DEBUG check
                            error_log('A1AI Contact Forms Debug - Chatbot ID: ' . $chatbot_id);
                            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging wrapped in WP_DEBUG check
                            error_log('A1AI Contact Forms Debug - Form Fields Count: ' . count($form_fields));
                        }

                        // Check if settings is already an array or needs to be unserialized
                        if (is_array($chatbot['settings'])) {
                            $settings = $chatbot['settings'];
                        } else {
                            $settings = maybe_unserialize($chatbot['settings']);
                        }
                        if (!isset($settings['contact_forms'])) {
                            $settings['contact_forms'] = array();
                        }

                        // Add/update contact form
                        $form_id = isset($_POST['form_id']) ? intval($_POST['form_id']) : time();
                        $settings['contact_forms'][$form_id] = array(
                            'id' => $form_id,
                            'name' => $form_name,
                            'type' => $form_type,
                            'fields' => $form_fields,
                            'save_submissions' => $save_submissions,
                            'email_notifications' => $email_notifications,
                            'notification_email' => $notification_email,
                            'created_at' => isset($settings['contact_forms'][$form_id]['created_at']) ? $settings['contact_forms'][$form_id]['created_at'] : current_time('mysql'),
                            'updated_at' => current_time('mysql')
                        );

                        // Debug logging before save
                        if (defined('WP_DEBUG') && WP_DEBUG) {
                            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging wrapped in WP_DEBUG check
                            error_log('A1AI Contact Forms Debug - Form ID: ' . $form_id);
                            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log, WordPress.PHP.DevelopmentFunctions.error_log_print_r -- Debug logging wrapped in WP_DEBUG check
                            error_log('A1AI Contact Forms Debug - Settings before save: ' . print_r($settings['contact_forms'], true));
                        }

                        // Update chatbot settings
                        $result = $data_manager->save_chatbot($chatbot_id, $chatbot['name'], $settings);
                        if (defined('WP_DEBUG') && WP_DEBUG) {
                            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log -- Debug logging wrapped in WP_DEBUG check
                            error_log('A1AI Contact Forms Debug - Save result: ' . ($result ? 'SUCCESS' : 'FAILED'));
                        }
                        if ($result) {
                            // Force clear cache to ensure form appears immediately
                            wp_cache_delete('a1ai_chatbot_' . $chatbot_id, 'a1ai');
                            wp_cache_delete('a1ai_all_chatbots', 'a1ai');

                            // Redirect to chatbot edit page with contact forms tab
                            $redirect_url = admin_url('admin.php');
                            $params = array(
                                'page' => 'a1ai-chatbots',
                                'action' => 'edit',
                                'id' => $chatbot_id,
                                '_wpnonce' => wp_create_nonce('a1ai_chatbot_action'),
                                'form_saved' => '1'
                            );
                            $redirect_url = add_query_arg($params, $redirect_url) . '#tab-contact-forms';
                            wp_redirect($redirect_url);
                            exit;
                        } else {
                            echo '<div class="notice notice-error"><p>' . esc_html__('Failed to save contact form.', 'a1ai-chatbot') . '</p></div>';
                        }
                    }
                }
                break;

            case 'delete_contact_form':
                if ( ! current_user_can('manage_options') ) {
                    echo '<div class="notice notice-error"><p>' . esc_html__('You do not have permission to perform this action.', 'a1ai-chatbot') . '</p></div>';
                    break;
                }
                if ( ! isset($_POST['_wpnonce']) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'a1ai_delete_contact_form' ) ) {
                    echo '<div class="notice notice-error"><p>' . esc_html__('Security check failed. Please try again.', 'a1ai-chatbot') . '</p></div>';
                    break;
                }

                if ( isset($_POST['form_id'], $_POST['chatbot_id']) ) {
                    $form_id = intval( $_POST['form_id'] );
                    $chatbot_id = intval( $_POST['chatbot_id'] );

                    $chatbot = $data_manager->get_chatbot($chatbot_id);
                    if ($chatbot) {
                        // Check if settings is already an array or needs to be unserialized
                        if (is_array($chatbot['settings'])) {
                            $settings = $chatbot['settings'];
                        } else {
                            $settings = maybe_unserialize($chatbot['settings']);
                        }
                        if (isset($settings['contact_forms'][$form_id])) {
                            unset($settings['contact_forms'][$form_id]);
                            $result = $data_manager->save_chatbot($chatbot_id, $chatbot['name'], $settings);
                            if ($result) {
                                // Force clear cache
                                wp_cache_delete('a1ai_chatbot_' . $chatbot_id, 'a1ai');
                                wp_cache_delete('a1ai_all_chatbots', 'a1ai');

                                // Redirect to chatbot edit page with contact forms tab
                                $redirect_url = admin_url('admin.php');
                                $params = array(
                                    'page' => 'a1ai-chatbots',
                                    'action' => 'edit',
                                    'id' => $chatbot_id,
                                    '_wpnonce' => wp_create_nonce('a1ai_chatbot_action'),
                                    'form_deleted' => '1'
                                );
                                $redirect_url = add_query_arg($params, $redirect_url) . '#tab-contact-forms';
                                wp_redirect($redirect_url);
                                exit;
                            }
                        }
                    }
                }
                break;
        }
    }
}
?>

<div class="a1ai-contact-forms-container">
    <h2><?php echo esc_html__('Manage Contact Forms', 'a1ai-chatbot'); ?></h2>

    <div class="a1ai-contact-forms-description">
        <p><?php echo esc_html__('Create and manage contact forms for your chatbots. Forms can function as traditional contact forms or hybrid chat+form interfaces that dynamically ask questions before submission.', 'a1ai-chatbot'); ?></p>
    </div>

    <!-- Chatbot Selection -->
    <div class="a1ai-chatbot-selector">
        <form method="get" action="">
            <input type="hidden" name="page" value="a1ai-chatbots">
            <!-- Preserve the contact forms tab when form submits -->
            <input type="hidden" name="active_tab" value="contact-forms">
            <?php wp_nonce_field( 'a1ai_select_chatbot', 'a1ai_select_nonce' ); ?>
            <label for="chatbot_id"><?php echo esc_html__('Select Chatbot:', 'a1ai-chatbot'); ?></label>
            <select name="chatbot_id" id="contact_forms_chatbot_select">
                <option value=""><?php echo esc_html__('-- Select a Chatbot --', 'a1ai-chatbot'); ?></option>
                <?php foreach ($chatbots as $chatbot) : ?>
                    <option value="<?php echo esc_attr($chatbot['id']); ?>" <?php selected($selected_chatbot_id, $chatbot['id']); ?>>
                        <?php echo esc_html($chatbot['name']); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </form>
    </div>

    <?php if ($selected_chatbot_id > 0) : ?>
        <!-- Add New Contact Form -->
        <div class="a1ai-add-contact-form">
            <h3><?php echo esc_html__('Add New Contact Form', 'a1ai-chatbot'); ?></h3>
            <form method="post" action="" id="contact-form-editor">
                <?php wp_nonce_field('a1ai_save_contact_form', 'a1ai_contact_form_nonce'); ?>
                <input type="hidden" name="a1ai_action" value="save_contact_form">
                <input type="hidden" name="chatbot_id" value="<?php echo esc_attr($selected_chatbot_id); ?>">
                <input type="hidden" name="form_id" id="edit_form_id" value="">

                <div class="a1ai-form-row">
                    <div class="a1ai-form-field" style="width: 48%; display: inline-block; vertical-align: top;">
                        <label for="form_name"><?php echo esc_html__('Form Name:', 'a1ai-chatbot'); ?></label>
                        <input type="text" name="form_name" id="form_name" placeholder="<?php echo esc_attr__('e.g., Contact Us, Support Request', 'a1ai-chatbot'); ?>" required>
                    </div>
                    <div class="a1ai-form-field" style="width: 48%; display: inline-block; vertical-align: top; margin-left: 4%;">
                        <label for="form_type"><?php echo esc_html__('Form Type:', 'a1ai-chatbot'); ?></label>
                        <select name="form_type" id="form_type">
                            <option value="traditional"><?php echo esc_html__('Traditional Contact Form', 'a1ai-chatbot'); ?></option>
                            <option value="hybrid"><?php echo esc_html__('Hybrid Chat + Form', 'a1ai-chatbot'); ?></option>
                            <option value="organic_chat"><?php echo esc_html__('Organic Chat Contact Form', 'a1ai-chatbot'); ?></option>
                        </select>
                    </div>
                </div>

                <div class="a1ai-form-row">
                    <div class="a1ai-form-field" style="width: 48%; display: inline-block; vertical-align: top;">
                        <label for="notification_email"><?php echo esc_html__('Notification Email:', 'a1ai-chatbot'); ?></label>
                        <input type="email" name="notification_email" id="notification_email" placeholder="<?php echo esc_attr__('admin@yoursite.com', 'a1ai-chatbot'); ?>">
                    </div>
                    <div class="a1ai-form-field" style="width: 48%; display: inline-block; vertical-align: top; margin-left: 4%;">
                        <label><?php echo esc_html__('Options:', 'a1ai-chatbot'); ?></label>
                        <div style="margin-top: 5px;">
                            <label style="display: block; margin-bottom: 5px;">
                                <input type="checkbox" name="save_submissions" id="save_submissions" value="1" checked>
                                <?php echo esc_html__('Save submissions to database', 'a1ai-chatbot'); ?>
                            </label>
                            <label style="display: block;">
                                <input type="checkbox" name="email_notifications" id="email_notifications" value="1" checked>
                                <?php echo esc_html__('Send email notifications', 'a1ai-chatbot'); ?>
                            </label>
                        </div>
                    </div>
                </div>

                <!-- Form Fields Section -->
                <div class="a1ai-form-fields-section">
                    <h4><?php echo esc_html__('Form Fields', 'a1ai-chatbot'); ?></h4>
                    <p class="description"><?php echo esc_html__('Add the fields you want to include in your contact form. You can reorder fields by dragging and dropping.', 'a1ai-chatbot'); ?></p>

                    <div id="form-fields-container">
                        <!-- Default fields will be added by JavaScript -->
                    </div>

                    <button type="button" id="add-field-btn" class="button">
                        <?php echo esc_html__('+ Add Field', 'a1ai-chatbot'); ?>
                    </button>
                </div>

                <div class="a1ai-form-row" style="margin-top: 20px;">
                    <button type="submit" class="button button-primary" id="save-form-btn">
                        <?php echo esc_html__('Save Contact Form', 'a1ai-chatbot'); ?>
                    </button>
                    <button type="button" class="button" id="cancel-edit-btn" style="display: none; margin-left: 10px;">
                        <?php echo esc_html__('Cancel', 'a1ai-chatbot'); ?>
                    </button>
                </div>
            </form>
        </div>

        <!-- Existing Contact Forms -->
        <div class="a1ai-existing-contact-forms">
            <h3><?php echo esc_html__('Existing Contact Forms', 'a1ai-chatbot'); ?></h3>

            <?php if (empty($contact_forms)) : ?>
                <div class="a1ai-empty-state">
                    <p><?php echo esc_html__('No contact forms found for this chatbot. Create your first contact form above.', 'a1ai-chatbot'); ?></p>
                </div>
            <?php else : ?>
                <div class="a1ai-contact-forms-list">
                    <?php foreach ($contact_forms as $form) : ?>
                        <div class="a1ai-contact-form-item">
                            <div class="a1ai-contact-form-content">
                                <div class="a1ai-contact-form-header">
                                    <h4><?php echo esc_html($form['name']); ?></h4>
                                    <span class="a1ai-form-type-badge a1ai-form-type-<?php echo esc_attr($form['type']); ?>">
                                        <?php echo esc_html(ucfirst($form['type'])); ?>
                                    </span>
                                </div>

                                <div class="a1ai-contact-form-details">
                                    <div class="a1ai-form-fields-summary">
                                        <strong><?php echo esc_html__('Fields:', 'a1ai-chatbot'); ?></strong>
                                        <?php
                                        $field_names = array_map(function($field) { return $field['label']; }, $form['fields']);
                                        echo esc_html(implode(', ', $field_names));
                                        ?>
                                    </div>

                                    <div class="a1ai-form-meta">
                                        <span class="a1ai-form-setting">
                                            <strong><?php echo esc_html__('Save to DB:', 'a1ai-chatbot'); ?></strong>
                                            <?php echo $form['save_submissions'] ? esc_html__('Yes', 'a1ai-chatbot') : esc_html__('No', 'a1ai-chatbot'); ?>
                                        </span>
                                        <span class="a1ai-form-setting">
                                            <strong><?php echo esc_html__('Email Notifications:', 'a1ai-chatbot'); ?></strong>
                                            <?php echo $form['email_notifications'] ? esc_html__('Yes', 'a1ai-chatbot') : esc_html__('No', 'a1ai-chatbot'); ?>
                                        </span>
                                        <?php if (!empty($form['notification_email'])) : ?>
                                            <span class="a1ai-form-setting">
                                                <strong><?php echo esc_html__('Email:', 'a1ai-chatbot'); ?></strong>
                                                <?php echo esc_html($form['notification_email']); ?>
                                            </span>
                                        <?php endif; ?>
                                        <span class="a1ai-form-date">
                                            <?php echo esc_html__('Created:', 'a1ai-chatbot'); ?>
                                            <?php echo esc_html(date_i18n(get_option('date_format'), strtotime($form['created_at']))); ?>
                                        </span>
                                    </div>
                                </div>
                            </div>

                            <div class="a1ai-contact-form-actions">
                                <button type="button" class="button button-small a1ai-edit-contact-form"
                                        data-form='<?php echo esc_attr(json_encode($form)); ?>'>
                                    <?php echo esc_html__('Edit', 'a1ai-chatbot'); ?>
                                </button>
                                <form method="post" action="" style="display: inline-block;">
                                    <?php wp_nonce_field('a1ai_delete_contact_form'); ?>
                                    <input type="hidden" name="a1ai_action" value="delete_contact_form">
                                    <input type="hidden" name="form_id" value="<?php echo esc_attr($form['id']); ?>">
                                    <input type="hidden" name="chatbot_id" value="<?php echo esc_attr($selected_chatbot_id); ?>">
                                    <button type="submit" class="button button-small a1ai-delete-contact-form"
                                            onclick="return confirm('<?php echo esc_js(__('Are you sure you want to delete this contact form?', 'a1ai-chatbot')); ?>')">
                                        <?php echo esc_html__('Delete', 'a1ai-chatbot'); ?>
                                    </button>
                                </form>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
    <?php else : ?>
        <div class="a1ai-notice a1ai-notice-info">
            <p><?php echo esc_html__('Please select a chatbot to manage its contact forms.', 'a1ai-chatbot'); ?></p>
        </div>
    <?php endif; ?>
</div>

<style>
.a1ai-contact-forms-container {
    max-width: 1200px;
}

.a1ai-contact-form-item {
    border: 1px solid #ddd;
    border-radius: 4px;
    padding: 20px;
    margin-bottom: 15px;
    background: #fff;
    box-shadow: 0 1px 3px rgba(0,0,0,0.1);
}

.a1ai-contact-form-header {
    display: flex;
    align-items: center;
    margin-bottom: 15px;
}

.a1ai-contact-form-header h4 {
    margin: 0;
    margin-right: 10px;
    font-size: 16px;
}

.a1ai-form-type-badge {
    background: #f0f0f0;
    padding: 4px 8px;
    border-radius: 3px;
    font-size: 12px;
    font-weight: bold;
}

.a1ai-form-type-traditional {
    background-color: #e7f3ff;
    color: #0073aa;
}

.a1ai-form-type-hybrid {
    background-color: #fff2e7;
    color: #d54e21;
}

.a1ai-form-fields-summary {
    margin-bottom: 10px;
    color: #666;
}

.a1ai-form-meta {
    display: flex;
    flex-wrap: wrap;
    gap: 15px;
    font-size: 13px;
    color: #666;
}

.a1ai-form-setting {
    display: inline-block;
}

.a1ai-contact-form-actions {
    margin-top: 15px;
}

.a1ai-form-fields-section {
    margin: 20px 0;
    padding: 20px;
    border: 1px solid #ddd;
    border-radius: 4px;
    background: #f9f9f9;
}

.form-field-row {
    display: flex;
    align-items: center;
    margin-bottom: 10px;
    padding: 10px;
    background: #fff;
    border: 1px solid #ddd;
    border-radius: 3px;
}

.form-field-row .field-input {
    flex: 1;
    margin-right: 10px;
}

.form-field-row .field-input input,
.form-field-row .field-input select {
    width: 100%;
    margin-bottom: 3px;
}

.form-field-row .field-actions {
    white-space: nowrap;
}

.field-handle {
    cursor: move;
    padding: 5px;
    color: #666;
    margin-right: 10px;
}

.field-required-checkbox {
    margin-left: 10px;
}

#add-field-btn {
    margin-top: 10px;
}

.a1ai-contact-forms-list .a1ai-contact-form-item {
    transition: all 0.2s ease;
}

.a1ai-contact-forms-list .a1ai-contact-form-item:hover {
    box-shadow: 0 2px 8px rgba(0,0,0,0.15);
}
</style>

<script type="text/javascript">
jQuery(document).ready(function($) {
    let fieldCounter = 0;

    // Handle chatbot selection for contact forms
    $('#contact_forms_chatbot_select').on('change', function() {
        if ($(this).val()) {
            // Get the nonce from the form
            const nonce = $('input[name="a1ai_select_nonce"]').val();

            // Build URL with all required parameters
            const currentUrl = new URL(window.location.href);
            currentUrl.searchParams.set('page', 'a1ai-chatbots');
            currentUrl.searchParams.set('chatbot_id', $(this).val());
            currentUrl.searchParams.set('a1ai_select_nonce', nonce);
            currentUrl.hash = '#tab-contact-forms';

            console.log('A1AI Contact Forms: Navigating to:', currentUrl.toString());

            // Navigate to URL with tab hash
            window.location.href = currentUrl.toString();
        }
    });

    // Add default fields on page load (only if no fields exist)
    if ($('#form-fields-container .form-field-row').length === 0) {
        addFormField('name', 'Name', 'text', true, 'Your full name');
        addFormField('email', 'Email', 'email', true, 'your@email.com');
        addFormField('message', 'Message', 'textarea', true, 'How can we help you?');
    }

    // Add new field button
    $('#add-field-btn').on('click', function() {
        addFormField('', '', 'text', false, '');
    });

    // Function to add a form field
    function addFormField(name, label, type, required, placeholder) {
        const fieldHtml = `
            <div class="form-field-row" data-field-id="${fieldCounter}">
                <span class="field-handle">⋮⋮</span>
                <div class="field-input" style="width: 20%;">
                    <input type="text" name="field_name[]" value="${name}" placeholder="Field Name" required>
                    <small>Field name (e.g., name, email)</small>
                </div>
                <div class="field-input" style="width: 20%;">
                    <input type="text" name="field_label[]" value="${label}" placeholder="Display Label" required>
                    <small>Label shown to users</small>
                </div>
                <div class="field-input" style="width: 15%;">
                    <select name="field_type[]">
                        <option value="text" ${type === 'text' ? 'selected' : ''}>Text</option>
                        <option value="email" ${type === 'email' ? 'selected' : ''}>Email</option>
                        <option value="tel" ${type === 'tel' ? 'selected' : ''}>Phone</option>
                        <option value="textarea" ${type === 'textarea' ? 'selected' : ''}>Textarea</option>
                        <option value="select" ${type === 'select' ? 'selected' : ''}>Select</option>
                        <option value="checkbox" ${type === 'checkbox' ? 'selected' : ''}>Checkbox</option>
                    </select>
                    <small>Field type</small>
                </div>
                <div class="field-input" style="width: 25%;">
                    <input type="text" name="field_placeholder[]" value="${placeholder}" placeholder="Placeholder text">
                    <small>Placeholder or help text</small>
                </div>
                <div class="field-required-checkbox">
                    <label>
                        <input type="checkbox" name="field_required[]" value="${fieldCounter}" ${required ? 'checked' : ''}>
                        Required
                    </label>
                </div>
                <div class="field-actions">
                    <button type="button" class="button button-small remove-field-btn">Remove</button>
                </div>
            </div>
        `;

        $('#form-fields-container').append(fieldHtml);
        fieldCounter++;
    }

    // Remove field
    $(document).on('click', '.remove-field-btn', function() {
        $(this).closest('.form-field-row').remove();
    });

    // Edit contact form
    $('.a1ai-edit-contact-form').on('click', function() {
        const formData = JSON.parse($(this).attr('data-form'));

        // Populate form fields
        $('#edit_form_id').val(formData.id);
        $('#form_name').val(formData.name);
        $('#form_type').val(formData.type);
        $('#notification_email').val(formData.notification_email || '');
        $('#save_submissions').prop('checked', formData.save_submissions);
        $('#email_notifications').prop('checked', formData.email_notifications);

        // Clear existing fields
        $('#form-fields-container').empty();
        fieldCounter = 0;

        // Add form fields
        if (formData.fields && formData.fields.length > 0) {
            formData.fields.forEach(function(field) {
                addFormField(field.name, field.label, field.type, field.required, field.placeholder || '');
            });
        }

        // Update button text and show cancel
        $('#save-form-btn').text('Update Contact Form');
        $('#cancel-edit-btn').show();

        // Scroll to form
        $('html, body').animate({
            scrollTop: $('#contact-form-editor').offset().top - 50
        }, 500);
    });

    // Cancel edit
    $('#cancel-edit-btn').on('click', function() {
        // Reset form
        $('#contact-form-editor')[0].reset();
        $('#edit_form_id').val('');
        $('#save-form-btn').text('Save Contact Form');
        $(this).hide();

        // Reset to default fields
        $('#form-fields-container').empty();
        fieldCounter = 0;
        addFormField('name', 'Name', 'text', true, 'Your full name');
        addFormField('email', 'Email', 'email', true, 'your@email.com');
        addFormField('message', 'Message', 'textarea', true, 'How can we help you?');
    });

    // Make fields sortable (only if jQuery UI is available)
    if ($.fn.sortable) {
        $('#form-fields-container').sortable({
            handle: '.field-handle',
            placeholder: 'ui-state-highlight',
            helper: 'clone'
        });
        console.log('A1AI: Form fields made sortable');
    } else {
        console.log('A1AI: jQuery UI sortable not available, drag/drop disabled');
        // Hide drag handles since they won't work
        $('.field-handle').hide();
    }
});
</script>
