<?php
/**
 * Admin chatbot edit template
 *
 * @link       https://a1aidigital.com
 * @since      1.0.0
 *
 * @package    A1AI
 * @subpackage A1AI/admin/partials
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly




// Get chatbot if in edit mode
$chatbot_id = isset($_GET['id']) ? intval($_GET['id']) : 0;
$is_edit = $chatbot_id > 0;

// Force database upgrade and clear caches if editing
if ($is_edit) {
    // Clear any cached data
    wp_cache_delete('a1ai_chatbot_' . $chatbot_id, 'a1ai');
    wp_cache_delete('a1ai_all_chatbots', 'a1ai');
    
    // Force database upgrade check
    $current_db_version = get_option('a1ai_db_version', '0.0.0');
    if (version_compare($current_db_version, A1AI_VERSION, '<')) {
        // Trigger upgrade process
        $a1ai = new A1AI();
        $a1ai->check_database_upgrade();
    }
}

// Verify nonce if in edit mode
if ($is_edit && (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_key($_GET['_wpnonce']), 'a1ai_chatbot_action'))) {
    wp_die(esc_html__('Security check failed.', 'a1ai-chatbot'));
}

$chatbot = $is_edit ? $data_manager->get_chatbot($chatbot_id) : null;

// If editing but chatbot doesn't exist, show error
if ($is_edit && !$chatbot) {
    wp_die(esc_html__('Chatbot not found.', 'a1ai-chatbot'));
}

// Set default values
$chatbot_name = $is_edit ? (isset($chatbot['name']) ? $chatbot['name'] : '') : '';
$settings = $is_edit ? (isset($chatbot['settings']) && is_array($chatbot['settings']) ? $chatbot['settings'] : array()) : array();

// Ensure settings is always an array and handle potential unserialize issues
if (!is_array($settings)) {
    $settings = array();
}

// Load system prompt and custom instructions from dedicated tables if editing
if ($is_edit && $chatbot_id > 0) {
    // Get system prompt from custom instructions table
    $system_instructions = $data_manager->get_custom_instructions($chatbot_id, 'system');
    if (!empty($system_instructions) && is_array($system_instructions)) {
        foreach ($system_instructions as $instruction) {
            if ($instruction['instruction_name'] === 'System Prompt' && !empty($instruction['instruction_content'])) {
                $settings['system_prompt'] = $instruction['instruction_content'];
                break;
            }
        }
    }
    
    // Get general custom instructions from dedicated table
    $custom_instructions = $data_manager->get_custom_instructions($chatbot_id, 'general');
    if (!empty($custom_instructions) && is_array($custom_instructions)) {
        // Combine all custom instructions into one text
        $instructions_text = '';
        foreach ($custom_instructions as $instruction) {
            if (!empty($instruction['instruction_content'])) {
                $instructions_text .= $instruction['instruction_content'] . "\n\n";
            }
        }
        if (!empty($instructions_text)) {
            $settings['custom_instructions'] = trim($instructions_text);
        }
    }
}



// Ensure all settings have default values
$default_settings = array(
    'custom_instructions' => '',
    'system_prompt' => '',
    'welcome_message' => __('Hello! How can I help you today?', 'a1ai-chatbot'),
    'model' => get_option('a1ai_default_model', 'gpt-4o'),
    'temperature' => get_option('a1ai_default_temperature', 0.7),
    'max_tokens' => get_option('a1ai_default_max_tokens', 1000),
    'chat_bubble_text' => __('Chat with us!', 'a1ai-chatbot'),
    'chatbot_title' => __('Chat Support', 'a1ai-chatbot'),
    'placeholder' => __('Type your message here...', 'a1ai-chatbot'),
    'notification_email' => '',
    'gdpr_enabled' => false,
    'gdpr_consent_text' => __('By clicking "I Agree", you consent to the use of AI-powered chat services and acknowledge that your conversations may be processed to provide you with assistance. You can withdraw your consent at any time by closing this chat.', 'a1ai-chatbot'),
    'primary_color' => '#2271b1',
    'font_size' => 'medium',
    'chat_width' => '350px',
    'chat_height' => '500px',
    // Conversation Settings
    'primary_action' => '',
    'secondary_actions' => '',
    'conversation_stages' => '',
    'action_guidance_tone' => 'moderate',
    'repetition_threshold' => '6',
    'variation_strategies' => '',
    'conversation_tracking' => false,
    'action_tracking' => false,
);

// Merge existing settings with defaults to ensure all keys exist
$settings = array_merge($default_settings, $settings);



// Helper function to safely get array values
function a1ai_get_setting($settings, $key, $default = '') {
    if (!is_array($settings)) {
        return $default;
    }
    return isset($settings[$key]) ? $settings[$key] : $default;
}

$title = $is_edit 
    ? sprintf(
        /* translators: %s: Name of the chatbot */
        __('Edit Chatbot: %s', 'a1ai-chatbot'), 
        $chatbot_name
    ) 
    : __('Add New Chatbot', 'a1ai-chatbot');
?>

<div class="a1ai-wrap">
    <div class="a1ai-header">
        <h1><?php echo esc_html($title); ?></h1>
    </div>

    <div class="a1ai-notices"></div>
    
    <?php
    // Check if API key is set
    $api_key = get_option('a1ai_openai_api_key', '');
    if (empty($api_key)) {
        ?>
        <div class="a1ai-notice a1ai-notice-warning">
            <p>
                <?php echo esc_html__('The OpenAI API key is not set. Please configure it in the', 'a1ai-chatbot'); ?>
                <a href="<?php echo esc_url(admin_url('admin.php?page=a1ai-settings')); ?>"><?php echo esc_html__('Settings', 'a1ai-chatbot'); ?></a>
                <?php echo esc_html__('page before creating chatbots.', 'a1ai-chatbot'); ?>
            </p>
        </div>
        <?php
    }
    ?>

    <div class="a1ai-container">
        <form id="a1ai-chatbot-form" class="a1ai-chatbot-form">
            <input type="hidden" name="chatbot_id" value="<?php echo esc_attr($chatbot_id); ?>" />
            <input type="hidden" name="nonce" value="<?php echo esc_attr(wp_create_nonce('a1ai_admin_nonce')); ?>" />
            
            <div class="a1ai-tabs-nav">
                <a href="#tab-general" class="active"><?php echo esc_html__('General', 'a1ai-chatbot'); ?></a>
                <a href="#tab-messages"><?php echo esc_html__('Messages', 'a1ai-chatbot'); ?></a>
                <a href="#tab-ai"><?php echo esc_html__('AI Settings', 'a1ai-chatbot'); ?></a>
                <a href="#tab-appearance"><?php echo esc_html__('Appearance', 'a1ai-chatbot'); ?></a>
                <a href="#tab-conversation"><?php echo esc_html__('Conversation Settings', 'a1ai-chatbot'); ?></a>
            </div>
            
            <div class="a1ai-tab-content" id="tab-general">
                <h2><?php echo esc_html__('General Settings', 'a1ai-chatbot'); ?></h2>
                
                <div class="a1ai-form-field">
                    <label for="chatbot_name"><?php echo esc_html__('Chatbot Name', 'a1ai-chatbot'); ?> <span class="required">*</span></label>
                    <input type="text" 
                        id="chatbot_name" 
                        name="chatbot_name" 
                        value="<?php echo esc_attr($chatbot_name); ?>" 
                        required
                    />
                    <p class="description">
                        <?php echo esc_html__('Enter a name for your chatbot for identification in the admin area.', 'a1ai-chatbot'); ?>
                    </p>
                </div>
                
                <div class="a1ai-form-field">
                    <label for="chatbot_title"><?php echo esc_html__('Chatbot Title', 'a1ai-chatbot'); ?></label>
                    <input type="text" 
                        id="chatbot_title" 
                        name="settings[chatbot_title]" 
                        value="<?php echo esc_attr(a1ai_get_setting($settings, 'chatbot_title')); ?>" 
                    />
                    <p class="description">
                        <?php echo esc_html__('Title displayed in the chatbot header (class="a1ai-chatbot-title").', 'a1ai-chatbot'); ?>
                    </p>
                </div>
                
                <div class="a1ai-form-field">
                    <label for="chat_bubble_text"><?php echo esc_html__('Chat Bubble Text', 'a1ai-chatbot'); ?></label>
                    <input type="text" 
                        id="chat_bubble_text" 
                        name="settings[chat_bubble_text]" 
                        value="<?php echo esc_attr(a1ai_get_setting($settings, 'chat_bubble_text')); ?>" 
                    />
                    <p class="description">
                        <?php echo esc_html__('Text displayed on the chat bubble when minimized.', 'a1ai-chatbot'); ?>
                    </p>
                </div>
                
                <div class="a1ai-form-field">
                    <label for="placeholder"><?php echo esc_html__('Input Placeholder', 'a1ai-chatbot'); ?></label>
                    <input type="text" 
                        id="placeholder" 
                        name="settings[placeholder]" 
                        value="<?php echo esc_attr(a1ai_get_setting($settings, 'placeholder')); ?>" 
                    />
                    <p class="description">
                        <?php echo esc_html__('Placeholder text shown in the message input field.', 'a1ai-chatbot'); ?>
                    </p>
                </div>
                
                <div class="a1ai-form-field">
                    <label for="notification_email"><?php echo esc_html__('New Chat Alert Email', 'a1ai-chatbot'); ?></label>
                    <input type="email" 
                        id="notification_email" 
                        name="settings[notification_email]" 
                        value="<?php echo esc_attr(a1ai_get_setting($settings, 'notification_email')); ?>" 
                        placeholder="<?php echo esc_attr__('admin@example.com', 'a1ai-chatbot'); ?>"
                        pattern="[a-z0-9._%+-]+@[a-z0-9.-]+\.[a-z]{2,}$"
                    />
                    <p class="description">
                        <?php echo esc_html__('Email address to receive notifications when a new chat starts. Leave empty to disable notifications.', 'a1ai-chatbot'); ?>
                    </p>
                </div>

                <?php if (class_exists('A1AI_Upgrade_Helper') && A1AI_Upgrade_Helper::is_upgrade_active()) : ?>
                    <div class="a1ai-form-field">
                        <label for="conversation_inactivity_timeout">
                            <?php echo esc_html__('Conversation Inactivity Timeout (minutes)', 'a1ai-chatbot'); ?>
                        </label>
                        <input type="number" 
                            id="conversation_inactivity_timeout" 
                            name="settings[conversation_inactivity_timeout]" 
                            value="<?php echo esc_attr(a1ai_get_setting($settings, 'conversation_inactivity_timeout', 20)); ?>" 
                            min="5" 
                            max="120" 
                            step="5"
                        />
                        <p class="description">
                            <?php echo esc_html__('After this many minutes of inactivity, a full conversation summary email will be sent. Default: 20 minutes. (Premium Feature)', 'a1ai-chatbot'); ?>
                        </p>
                    </div>
                <?php endif; ?>
                
                <?php if ($is_edit) : ?>
                    <div class="a1ai-form-field">
                        <label><?php echo esc_html__('Shortcode', 'a1ai-chatbot'); ?></label>
                        <div class="a1ai-shortcode-display">
                            <code>[a1ai_chatbot id="<?php echo esc_attr($chatbot_id); ?>"]</code>
                            <button type="button" class="button a1ai-copy-shortcode" data-shortcode='[a1ai_chatbot id="<?php echo esc_attr($chatbot_id); ?>"]'>
                                <?php echo esc_html__('Copy', 'a1ai-chatbot'); ?>
                            </button>
                        </div>
                        <p class="description">
                            <?php echo esc_html__('Use this shortcode to add this chatbot to your posts, pages, or widgets.', 'a1ai-chatbot'); ?>
                        </p>
                    </div>
                <?php endif; ?>
            </div>
            
            <div class="a1ai-tab-content" id="tab-messages" style="display: none;">
                <h2><?php echo esc_html__('Message Configuration', 'a1ai-chatbot'); ?></h2>
                
                <div class="a1ai-form-field">
                    <label for="welcome_message"><?php echo esc_html__('Welcome Message', 'a1ai-chatbot'); ?></label>
                    <input type="text" 
                        id="welcome_message" 
                        name="settings[welcome_message]" 
                        value="<?php echo esc_attr(a1ai_get_setting($settings, 'welcome_message')); ?>" 
                    />
                    <p class="description">
                        <?php echo esc_html__('Initial message displayed when the chat starts.', 'a1ai-chatbot'); ?>
                    </p>
                </div>
                
                <div class="a1ai-form-field">
                    <label for="system_prompt"><?php echo esc_html__('System Prompt', 'a1ai-chatbot'); ?></label>
                    <textarea 
                        id="system_prompt" 
                        name="settings[system_prompt]" 
                        rows="4"
                    ><?php 
                        $system_prompt = a1ai_get_setting($settings, 'system_prompt', '');
                        // Clean up any warning text that might have been saved
                        if (strpos($system_prompt, 'Warning:') !== false || strpos($system_prompt, 'Undefined array key') !== false) {
                            $system_prompt = '';
                        }
                        echo esc_textarea($system_prompt); 
                    ?></textarea>
                    <p class="description">
                        <?php echo esc_html__('Instructions to set the behavior of the AI. Leave empty for default behavior.', 'a1ai-chatbot'); ?>
                    </p>
                </div>
                
                <div class="a1ai-form-field">
                    <label for="custom_instructions"><?php echo esc_html__('Custom Instructions', 'a1ai-chatbot'); ?></label>
                    <textarea 
                        id="custom_instructions" 
                        name="settings[custom_instructions]" 
                        rows="6"
                    ><?php 
                        $custom_instructions = a1ai_get_setting($settings, 'custom_instructions', '');
                        // Clean up any warning text that might have been saved
                        if (strpos($custom_instructions, 'Warning:') !== false || strpos($custom_instructions, 'Undefined array key') !== false) {
                            $custom_instructions = '';
                        }
                        echo esc_textarea($custom_instructions); 
                    ?></textarea>
                    <p class="description">
                        <?php echo esc_html__('Additional instructions to guide the AI in generating responses. Include knowledge specific to your website, business rules, or constraints.', 'a1ai-chatbot'); ?>
                    </p>
                </div>
                
                <div class="a1ai-form-field">
                    <label for="gdpr_enabled">
                        <input type="checkbox" 
                            id="gdpr_enabled" 
                            name="settings[gdpr_enabled]" 
                            value="1" 
                            <?php checked(isset($settings['gdpr_enabled']) && $settings['gdpr_enabled'], true); ?>
                        />
                        <?php echo esc_html__('Enable GDPR Compliance Consent Mechanism', 'a1ai-chatbot'); ?>
                    </label>
                    <p class="description">
                        <?php echo esc_html__('When enabled, users must provide explicit consent before using the chatbot. This helps ensure GDPR compliance.', 'a1ai-chatbot'); ?>
                    </p>
                </div>
                
                <div class="a1ai-form-field" id="gdpr-consent-text-field" style="<?php echo (isset($settings['gdpr_enabled']) && $settings['gdpr_enabled']) ? '' : 'display: none;'; ?>">
                    <label for="gdpr_consent_text"><?php echo esc_html__('GDPR Consent Text', 'a1ai-chatbot'); ?></label>
                    <textarea 
                        id="gdpr_consent_text" 
                        name="settings[gdpr_consent_text]" 
                        rows="4"
                    ><?php 
                        $gdpr_consent_text = a1ai_get_setting($settings, 'gdpr_consent_text', '');
                        // Clean up any warning text that might have been saved
                        if (strpos($gdpr_consent_text, 'Warning:') !== false || strpos($gdpr_consent_text, 'Undefined array key') !== false) {
                            $gdpr_consent_text = __('By clicking "I Agree", you consent to the use of AI-powered chat services and acknowledge that your conversations may be processed to provide you with assistance. You can withdraw your consent at any time by closing this chat.', 'a1ai-chatbot');
                        }
                        echo esc_textarea($gdpr_consent_text); 
                    ?></textarea>
                    <p class="description">
                        <?php echo esc_html__('Text displayed to users requesting their consent to use the AI chatbot. This should clearly explain how their data will be used.', 'a1ai-chatbot'); ?>
                    </p>
                </div>
            </div>
            
            <div class="a1ai-tab-content" id="tab-ai" style="display: none;">
                <h2><?php echo esc_html__('AI Configuration', 'a1ai-chatbot'); ?></h2>
                
                <div class="a1ai-form-field">
                    <label for="model"><?php echo esc_html__('AI Model', 'a1ai-chatbot'); ?></label>
                    <select id="model" name="settings[model]">
                        <option value="gpt-4o" <?php selected(a1ai_get_setting($settings, 'model'), 'gpt-4o'); ?>>GPT-4o</option>
                        <option value="gpt-4-turbo" <?php selected(a1ai_get_setting($settings, 'model'), 'gpt-4-turbo'); ?>>GPT-4 Turbo</option>
                        <option value="gpt-3.5-turbo" <?php selected(a1ai_get_setting($settings, 'model'), 'gpt-3.5-turbo'); ?>>GPT-3.5 Turbo</option>
                    </select>
                    <p class="description">
                        <?php echo esc_html__('Select the AI model to use for this chatbot.', 'a1ai-chatbot'); ?>
                    </p>
                </div>
                
                <div class="a1ai-form-field">
                    <label for="temperature"><?php echo esc_html__('Temperature', 'a1ai-chatbot'); ?></label>
                    <input type="number"
                        id="temperature"
                        name="settings[temperature]"
                        value="<?php echo esc_attr(a1ai_get_setting($settings, 'temperature')); ?>"
                        min="0"
                        max="2"
                        step="0.1"
                    />
                    <p class="description">
                        <?php echo esc_html__('Controls randomness: Lower values (0.1-0.5) are more focused and deterministic, higher values (0.7-1.0) are more creative and diverse.', 'a1ai-chatbot'); ?>
                    </p>
                </div>
                
                <div class="a1ai-form-field">
                    <label for="max_tokens"><?php echo esc_html__('Max Tokens', 'a1ai-chatbot'); ?></label>
                    <input type="number"
                        id="max_tokens"
                        name="settings[max_tokens]"
                        value="<?php echo esc_attr(a1ai_get_setting($settings, 'max_tokens')); ?>"
                        min="100"
                        max="4000"
                        step="100"
                    />
                    <p class="description">
                        <?php echo esc_html__('Maximum length of the response. Higher values may result in more comprehensive answers but can increase costs.', 'a1ai-chatbot'); ?>
                    </p>
                </div>
            </div>
            
            <div class="a1ai-tab-content" id="tab-appearance" style="display: none;">
                <h2><?php echo esc_html__('Appearance Settings', 'a1ai-chatbot'); ?></h2>
                
                <div class="a1ai-form-field">
                    <label for="primary_color"><?php echo esc_html__('Primary Color', 'a1ai-chatbot'); ?></label>
                    <input type="color"
                        id="primary_color"
                        name="settings[primary_color]"
                        value="<?php echo esc_attr(a1ai_get_setting($settings, 'primary_color')); ?>"
                    />
                    <p class="description">
                        <?php echo esc_html__('The main color used for the chatbot header and buttons.', 'a1ai-chatbot'); ?>
                    </p>
                </div>
                
                <div class="a1ai-form-field">
                    <label for="font_size"><?php echo esc_html__('Font Size', 'a1ai-chatbot'); ?></label>
                    <select id="font_size" name="settings[font_size]">
                        <option value="small" <?php selected(a1ai_get_setting($settings, 'font_size'), 'small'); ?>><?php echo esc_html__('Small', 'a1ai-chatbot'); ?></option>
                        <option value="medium" <?php selected(a1ai_get_setting($settings, 'font_size'), 'medium'); ?>><?php echo esc_html__('Medium', 'a1ai-chatbot'); ?></option>
                        <option value="large" <?php selected(a1ai_get_setting($settings, 'font_size'), 'large'); ?>><?php echo esc_html__('Large', 'a1ai-chatbot'); ?></option>
                    </select>
                    <p class="description">
                        <?php echo esc_html__('Font size used in the chatbot messages.', 'a1ai-chatbot'); ?>
                    </p>
                </div>
                
                <div class="a1ai-form-field">
                    <label for="chat_width"><?php echo esc_html__('Chat Width', 'a1ai-chatbot'); ?></label>
                    <input type="text"
                        id="chat_width"
                        name="settings[chat_width]"
                        value="<?php 
                            $chat_width = a1ai_get_setting($settings, 'chat_width', '350px');
                            // Clean up any warning text that might have been saved
                            if (strpos($chat_width, 'Warning:') !== false || strpos($chat_width, 'Undefined array key') !== false) {
                                $chat_width = '350px';
                            }
                            echo esc_attr($chat_width); 
                        ?>"
                    />
                    <p class="description">
                        <?php 
                        /* translators: %s: CSS values like 350px, 100%, etc. */
                        echo esc_html__('Width of the chat window. Use CSS values like 350px, 100%, etc.', 'a1ai-chatbot'); 
                        ?>
                    </p>
                </div>
                
                <div class="a1ai-form-field">
                    <label for="chat_height"><?php echo esc_html__('Chat Height', 'a1ai-chatbot'); ?></label>
                    <input type="text"
                        id="chat_height"
                        name="settings[chat_height]"
                        value="<?php 
                            $chat_height = a1ai_get_setting($settings, 'chat_height', '500px');
                            // Clean up any warning text that might have been saved
                            if (strpos($chat_height, 'Warning:') !== false || strpos($chat_height, 'Undefined array key') !== false) {
                                $chat_height = '500px';
                            }
                            echo esc_attr($chat_height); 
                        ?>"
                    />
                    <p class="description">
                        <?php echo esc_html__('Height of the chat window. Use CSS values like 500px, 80vh, etc.', 'a1ai-chatbot'); ?>
                    </p>
                </div>
            </div>
            
            <div class="a1ai-tab-content" id="tab-conversation" style="display: none;">
                <h2><?php echo esc_html__('Conversation Settings & Action Guidance', 'a1ai-chatbot'); ?></h2>
                
                <?php if ($is_edit && $chatbot_id > 0): ?>
                    <!-- Action Guidance Settings -->
                    <div class="a1ai-section">
                        <h3><?php echo esc_html__('Action Guidance Settings', 'a1ai-chatbot'); ?></h3>
                        <p><?php echo esc_html__('Configure how the chatbot guides users toward specific actions and goals.', 'a1ai-chatbot'); ?></p>
                        
                        <div class="a1ai-form-field">
                            <label for="primary_action"><?php echo esc_html__('Primary Action Goal', 'a1ai-chatbot'); ?></label>
                            <textarea id="primary_action" name="settings[primary_action]" rows="3" class="large-text"><?php echo esc_textarea(a1ai_get_setting($settings, 'primary_action')); ?></textarea>
                            <p class="description"><?php echo esc_html__('The main action you want users to take (e.g., "Contact us for a consultation", "Purchase our product", "Sign up for our newsletter")', 'a1ai-chatbot'); ?></p>
                        </div>
                        
                        <div class="a1ai-form-field">
                            <label for="secondary_actions"><?php echo esc_html__('Secondary Actions', 'a1ai-chatbot'); ?></label>
                            <textarea id="secondary_actions" name="settings[secondary_actions]" rows="3" class="large-text"><?php echo esc_textarea(a1ai_get_setting($settings, 'secondary_actions')); ?></textarea>
                            <p class="description"><?php echo esc_html__('Additional actions to suggest when primary action isn\'t appropriate (one per line)', 'a1ai-chatbot'); ?></p>
                        </div>
                        
                        <div class="a1ai-form-field">
                            <label for="conversation_stages"><?php echo esc_html__('Conversation Stage Triggers', 'a1ai-chatbot'); ?></label>
                            <textarea id="conversation_stages" name="settings[conversation_stages]" rows="3" class="large-text"><?php echo esc_textarea(a1ai_get_setting($settings, 'conversation_stages')); ?></textarea>
                            <p class="description"><?php echo esc_html__('Define conversation stages and when to trigger action guidance (e.g., "After 3 messages: Suggest consultation", "When user shows interest: Offer demo")', 'a1ai-chatbot'); ?></p>
                        </div>
                        
                        <div class="a1ai-form-field">
                            <label for="action_guidance_tone"><?php echo esc_html__('Action Guidance Tone', 'a1ai-chatbot'); ?></label>
                            <select id="action_guidance_tone" name="settings[action_guidance_tone]">
                                <option value="subtle" <?php selected(a1ai_get_setting($settings, 'action_guidance_tone'), 'subtle'); ?>><?php echo esc_html__('Subtle - Gentle suggestions', 'a1ai-chatbot'); ?></option>
                                <option value="moderate" <?php selected(a1ai_get_setting($settings, 'action_guidance_tone'), 'moderate'); ?>><?php echo esc_html__('Moderate - Clear recommendations', 'a1ai-chatbot'); ?></option>
                                <option value="direct" <?php selected(a1ai_get_setting($settings, 'action_guidance_tone'), 'direct'); ?>><?php echo esc_html__('Direct - Strong calls to action', 'a1ai-chatbot'); ?></option>
                            </select>
                            <p class="description"><?php echo esc_html__('How strongly the chatbot should guide users toward actions.', 'a1ai-chatbot'); ?></p>
                        </div>
                    </div>
                    
                    <!-- Repetition Prevention Settings -->
                    <div class="a1ai-section">
                        <h3><?php echo esc_html__('Repetition Prevention', 'a1ai-chatbot'); ?></h3>
                        <p><?php echo esc_html__('Configure how the chatbot handles repetitive conversations and maintains engagement.', 'a1ai-chatbot'); ?></p>
                        
                        <div class="a1ai-form-field">
                            <label for="repetition_threshold"><?php echo esc_html__('Repetition Threshold', 'a1ai-chatbot'); ?></label>
                            <input type="number" id="repetition_threshold" name="settings[repetition_threshold]" value="<?php echo esc_attr(a1ai_get_setting($settings, 'repetition_threshold', '3')); ?>" min="1" max="10" />
                            <p class="description"><?php echo esc_html__('Number of similar messages before suggesting a different approach.', 'a1ai-chatbot'); ?></p>
                        </div>
                        
                        <div class="a1ai-form-field">
                            <label for="variation_strategies"><?php echo esc_html__('Variation Strategies', 'a1ai-chatbot'); ?></label>
                            <textarea id="variation_strategies" name="settings[variation_strategies]" rows="3" class="large-text"><?php echo esc_textarea(a1ai_get_setting($settings, 'variation_strategies')); ?></textarea>
                            <p class="description"><?php echo esc_html__('Strategies to use when conversations become repetitive (e.g., "Ask clarifying questions", "Offer alternative solutions", "Suggest contacting support")', 'a1ai-chatbot'); ?></p>
                        </div>
                    </div>
                    
                    <!-- Conversation Analytics -->
                    <div class="a1ai-section">
                        <h3><?php echo esc_html__('Conversation Analytics', 'a1ai-chatbot'); ?></h3>
                        <p><?php echo esc_html__('Monitor conversation patterns and action guidance effectiveness.', 'a1ai-chatbot'); ?></p>
                        
                        <div class="a1ai-form-field">
                            <label for="conversation_tracking"><?php echo esc_html__('Enable Conversation Tracking', 'a1ai-chatbot'); ?></label>
                            <input type="checkbox" id="conversation_tracking" name="settings[conversation_tracking]" value="1" <?php checked(a1ai_get_setting($settings, 'conversation_tracking'), '1'); ?> />
                            <p class="description"><?php echo esc_html__('Track conversation patterns and user engagement metrics.', 'a1ai-chatbot'); ?></p>
                        </div>
                        
                        <div class="a1ai-form-field">
                            <label for="action_tracking"><?php echo esc_html__('Enable Action Tracking', 'a1ai-chatbot'); ?></label>
                            <input type="checkbox" id="action_tracking" name="settings[action_tracking]" value="1" <?php checked(a1ai_get_setting($settings, 'action_tracking'), '1'); ?> />
                            <p class="description"><?php echo esc_html__('Track when users take suggested actions or show interest.', 'a1ai-chatbot'); ?></p>
                        </div>
                    </div>
                <?php else: ?>
                    <div class="a1ai-section">
                        <p><?php echo esc_html__('Conversation settings will be available after creating the chatbot.', 'a1ai-chatbot'); ?></p>
                    </div>
                <?php endif; ?>
            </div>
            
            <div class="submit">
                <input type="submit" name="submit" id="submit" class="button button-primary" value="<?php echo $is_edit ? esc_attr__('Update Chatbot', 'a1ai-chatbot') : esc_attr__('Create Chatbot', 'a1ai-chatbot'); ?>" />
                <span class="spinner"></span>
                <a href="<?php echo esc_url(admin_url('admin.php?page=a1ai-chatbots')); ?>" class="button">
                    <?php echo esc_html__('Cancel', 'a1ai-chatbot'); ?>
                </a>
            </div>
        </form>
    </div>
</div>

 
