<?php
/**
 * Admin adjustments management template
 *
 * @link       https://a1aidigital.com
 * @since      1.0.0
 *
 * @package    A1AI
 * @subpackage A1AI/admin/partials
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

// Load plugin functions if not already loaded
if (!function_exists('is_plugin_active')) {
    include_once(ABSPATH . 'wp-admin/includes/plugin.php');
}

// Get all chatbots for selection
$chatbots = $data_manager->get_chatbots();

// Get adjustments for the selected chatbot (secured with nonce)
$selected_chatbot_id = 0;
if ( isset($_GET['chatbot_id'], $_GET['a1ai_select_nonce']) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['a1ai_select_nonce'] ) ), 'a1ai_select_chatbot' ) ) {
    $selected_chatbot_id = intval($_GET['chatbot_id']);
}
$adjustments = array();

if ($selected_chatbot_id > 0) {
    $adjustments = $data_manager->get_adjustments($selected_chatbot_id);
}

// Handle deletion via GET with nonce verification
if ( isset($_GET['delete_adjustment']) ) {
    if ( current_user_can('manage_options') ) {
        if ( isset($_GET['_wpnonce']) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'a1ai_delete_adjustment' ) ) {
            $adjustment_id = intval( $_GET['delete_adjustment'] );
            if ( $adjustment_id > 0 ) {
                $result = $data_manager->delete_adjustment( $adjustment_id );
                if ( $result ) {
                    echo '<div class="notice notice-success"><p>' . esc_html__('Adjustment deleted successfully!', 'a1ai-chatbot') . '</p></div>';
                }
                if ( $selected_chatbot_id > 0 ) {
                    $adjustments = $data_manager->get_adjustments( $selected_chatbot_id );
                }
            }
        } else {
            echo '<div class="notice notice-error"><p>' . esc_html__('Security check failed for deletion.', 'a1ai-chatbot') . '</p></div>';
        }
    } else {
        echo '<div class="notice notice-error"><p>' . esc_html__('You do not have permission to perform this action.', 'a1ai-chatbot') . '</p></div>';
    }
}

// Handle form submissions
if ( isset($_SERVER['REQUEST_METHOD']) && 'POST' === $_SERVER['REQUEST_METHOD'] ) {
    if ( isset($_POST['a1ai_action']) ) {
        $a1ai_action = sanitize_key( wp_unslash( $_POST['a1ai_action'] ) );
        switch ( $a1ai_action ) {
            case 'save_adjustment':
            case 'add_adjustment':
                if ( ! current_user_can('manage_options') ) {
                    echo '<div class="notice notice-error"><p>' . esc_html__('You do not have permission to perform this action.', 'a1ai-chatbot') . '</p></div>';
                    break;
                }
                if ( ! isset($_POST['a1ai_adjustment_nonce']) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['a1ai_adjustment_nonce'] ) ), 'a1ai_save_adjustment' ) ) {
                    echo '<div class="notice notice-error"><p>' . esc_html__('Security check failed. Please try again.', 'a1ai-chatbot') . '</p></div>';
                    break;
                }
                if ( isset($_POST['chatbot_id'], $_POST['user_input'], $_POST['ai_output']) ) {
                    $chatbot_id = intval( $_POST['chatbot_id'] );
                    $user_input = sanitize_textarea_field( wp_unslash( $_POST['user_input'] ) );
                    $ai_output  = sanitize_textarea_field( wp_unslash( $_POST['ai_output'] ) );

                    // Get match types and temperature
                    $ui_match_type = isset($_POST['ui_match_type']) ? sanitize_text_field( wp_unslash( $_POST['ui_match_type'] ) ) : 'Broad';
                    $ai_match_type = isset($_POST['ai_match_type']) ? sanitize_text_field( wp_unslash( $_POST['ai_match_type'] ) ) : 'Broad';
                    $ai_temperature = isset($_POST['ai_temperature']) ? floatval( $_POST['ai_temperature'] ) : 0.7;

                    // Validate match types - restrict premium options to upgrade users only
                    $valid_match_types = array('Default');
                    if (class_exists('A1AI_Upgrade_Helper') && A1AI_Upgrade_Helper::is_upgrade_active()) {
                        $valid_match_types = array('Default', 'Broad', 'Phrase', 'Exact');
                    }

                    if (!in_array($ui_match_type, $valid_match_types)) {
                        $ui_match_type = 'Default';
                    }
                    if (!in_array($ai_match_type, $valid_match_types)) {
                        $ai_match_type = 'Default';
                    }

                    // Validate temperature - restrict to default for free users
                    if (!class_exists('A1AI_Upgrade_Helper') || !A1AI_Upgrade_Helper::is_upgrade_active()) {
                        $ai_temperature = 0.7; // Force default for free users
                    } else {
                        // Validate temperature (0.1 to 1.0) for premium users
                        if ($ai_temperature < 0.1 || $ai_temperature > 1.0) {
                            $ai_temperature = 0.7;
                        }
                    }

                    $result = $data_manager->add_adjustment( $chatbot_id, $user_input, $ai_output, $ui_match_type, $ai_match_type, 0, $ai_temperature );
                    if ( $result ) {
                        echo '<div class="notice notice-success"><p>' . esc_html__('Adjustment added successfully!', 'a1ai-chatbot') . '</p></div>';
                    }

                    // Refresh adjustments list
                    $adjustments = $selected_chatbot_id > 0 ? $data_manager->get_adjustments( $selected_chatbot_id ) : array();
                }
                break;
                
            case 'update_adjustment':
            case 'edit_adjustment':
                if ( ! current_user_can('manage_options') ) {
                    echo '<div class="notice notice-error"><p>' . esc_html__('You do not have permission to perform this action.', 'a1ai-chatbot') . '</p></div>';
                    break;
                }
                if ( ! isset($_POST['a1ai_edit_adjustment_nonce']) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['a1ai_edit_adjustment_nonce'] ) ), 'a1ai_edit_adjustment' ) ) {
                    echo '<div class="notice notice-error"><p>' . esc_html__('Security check failed. Please try again.', 'a1ai-chatbot') . '</p></div>';
                    break;
                }
                if ( isset($_POST['adjustment_id'], $_POST['user_input'], $_POST['ai_output']) ) {
                    $adjustment_id = intval( $_POST['adjustment_id'] );
                    $user_input = sanitize_textarea_field( wp_unslash( $_POST['user_input'] ) );
                    $ai_output  = sanitize_textarea_field( wp_unslash( $_POST['ai_output'] ) );

                    // Get match types and temperature
                    $ui_match_type = isset($_POST['ui_match_type']) ? sanitize_text_field( wp_unslash( $_POST['ui_match_type'] ) ) : 'Broad';
                    $ai_match_type = isset($_POST['ai_match_type']) ? sanitize_text_field( wp_unslash( $_POST['ai_match_type'] ) ) : 'Broad';
                    $ai_temperature = isset($_POST['ai_temperature']) ? floatval( $_POST['ai_temperature'] ) : 0.7;

                    // Validate match types - restrict premium options to upgrade users only
                    $valid_match_types = array('Default');
                    if (class_exists('A1AI_Upgrade_Helper') && A1AI_Upgrade_Helper::is_upgrade_active()) {
                        $valid_match_types = array('Default', 'Broad', 'Phrase', 'Exact');
                    }

                    if (!in_array($ui_match_type, $valid_match_types)) {
                        $ui_match_type = 'Default';
                    }
                    if (!in_array($ai_match_type, $valid_match_types)) {
                        $ai_match_type = 'Default';
                    }

                    // Validate temperature - restrict to default for free users
                    if (!class_exists('A1AI_Upgrade_Helper') || !A1AI_Upgrade_Helper::is_upgrade_active()) {
                        $ai_temperature = 0.7; // Force default for free users
                    } else {
                        // Validate temperature (0.1 to 1.0) for premium users
                        if ($ai_temperature < 0.1 || $ai_temperature > 1.0) {
                            $ai_temperature = 0.7;
                        }
                    }

                    $update_data = array(
                        'user_input' => $user_input,
                        'ai_output'  => $ai_output,
                        'ui_match_type' => $ui_match_type,
                        'ai_match_type' => $ai_match_type,
                        'ai_temperature' => $ai_temperature
                    );

                    $result = $data_manager->update_adjustment( $adjustment_id, $update_data );
                    if ( $result ) {
                        echo '<div class="notice notice-success"><p>' . esc_html__('Adjustment updated successfully!', 'a1ai-chatbot') . '</p></div>';
                    }

                    // Refresh adjustments list
                    $adjustments = $selected_chatbot_id > 0 ? $data_manager->get_adjustments( $selected_chatbot_id ) : array();
                }
                break;
                
            case 'delete_adjustment':
                if ( ! current_user_can('manage_options') ) {
                    echo '<div class="notice notice-error"><p>' . esc_html__('You do not have permission to perform this action.', 'a1ai-chatbot') . '</p></div>';
                    break;
                }
                if ( ! isset($_POST['_wpnonce']) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ) ), 'a1ai_delete_adjustment' ) ) {
                    echo '<div class="notice notice-error"><p>' . esc_html__('Security check failed. Please try again.', 'a1ai-chatbot') . '</p></div>';
                    break;
                }
                if ( isset($_POST['adjustment_id']) ) {
                    $adjustment_id = intval( $_POST['adjustment_id'] );
                    $result = $data_manager->delete_adjustment( $adjustment_id );
                    if ( $result ) {
                        echo '<div class="notice notice-success"><p>' . esc_html__('Adjustment deleted successfully!', 'a1ai-chatbot') . '</p></div>';
                    }
                    
                    // Refresh adjustments list
                    $adjustments = $selected_chatbot_id > 0 ? $data_manager->get_adjustments( $selected_chatbot_id ) : array();
                }
                break;
        }
    }
}
?>

<div class="a1ai-adjustments-container">
    <h2><?php echo esc_html__('Manage Adjustments', 'a1ai-chatbot'); ?></h2>
    
    <div class="a1ai-adjustments-description">
        <p><?php echo esc_html__('Adjustments allow you to fine-tune how the AI responds to specific user inputs. Create patterns that match user messages and define how the AI should respond.', 'a1ai-chatbot'); ?></p>
        <?php if (!class_exists('A1AI_Upgrade_Helper') || !A1AI_Upgrade_Helper::is_upgrade_active()): ?>
        <div class="a1ai-upgrade-notice" style="background: #f0f8ff; border: 1px solid #0073aa; border-radius: 6px; padding: 12px; margin: 15px 0;">
            <p style="margin: 0; color: #0073aa;">
                <strong>ℹ️ Free Version:</strong> Basic adjustments with Default matching are available.
                <a href="#tab-upgrade" onclick="document.querySelector('a[href=&quot;#tab-upgrade&quot;]').click(); return false;" style="color: #0073aa; text-decoration: none; font-weight: bold;">View the Upgrade tab</a>
                to see all premium adjustment features.
            </p>
        </div>
        <?php endif; ?>
    </div>

    <!-- Chatbot Selection -->
    <div class="a1ai-chatbot-selector">
        <form method="get" action="<?php echo esc_url(admin_url('admin.php')); ?>">
            <input type="hidden" name="page" value="a1ai-chatbots">
            <input type="hidden" name="active_tab" value="adjustments">
            <?php wp_nonce_field( 'a1ai_select_chatbot', 'a1ai_select_nonce' ); ?>

            <?php
            // Preserve adjust mode parameters during form submission (upgrade feature)
            if (isset($_GET['adjust_mode']) && $_GET['adjust_mode'] == '1') {
                echo '<input type="hidden" name="adjust_mode" value="1">';
                if (isset($_GET['user_input'])) {
                    echo '<input type="hidden" name="user_input" value="' . esc_attr(sanitize_text_field(wp_unslash($_GET['user_input']))) . '">';
                }
                if (isset($_GET['ai_response'])) {
                    echo '<input type="hidden" name="ai_response" value="' . esc_attr(sanitize_textarea_field(wp_unslash($_GET['ai_response']))) . '">';
                }
                if (isset($_GET['chatbot_name'])) {
                    echo '<input type="hidden" name="chatbot_name" value="' . esc_attr(sanitize_text_field(wp_unslash($_GET['chatbot_name']))) . '">';
                }
            }
            ?>

            <label for="chatbot_id"><?php echo esc_html__('Select Chatbot:', 'a1ai-chatbot'); ?></label>
            <select name="chatbot_id" id="chatbot_id" onchange="this.form.submit()">
                <option value=""><?php echo esc_html__('-- Select a Chatbot --', 'a1ai-chatbot'); ?></option>
                <?php foreach ($chatbots as $chatbot) : ?>
                    <option value="<?php echo esc_attr($chatbot['id']); ?>" <?php selected($selected_chatbot_id, $chatbot['id']); ?>>
                        <?php echo esc_html($chatbot['name']); ?>
                    </option>
                <?php endforeach; ?>
            </select>
        </form>
    </div>

    <?php if ($selected_chatbot_id > 0) : ?>
        <!-- Add New Adjustment Form -->
        <div class="a1ai-add-adjustment">
            <h3><?php echo esc_html__('Add New Adjustment', 'a1ai-chatbot'); ?></h3>
            <form method="post" action="<?php echo esc_url(admin_url('admin.php?page=a1ai-chatbots&active_tab=adjustments&chatbot_id=' . $selected_chatbot_id)); ?>">
                <?php wp_nonce_field('a1ai_save_adjustment', 'a1ai_adjustment_nonce'); ?>
                <input type="hidden" name="a1ai_action" value="save_adjustment">
                <input type="hidden" name="chatbot_id" value="<?php echo esc_attr($selected_chatbot_id); ?>">
                
                <div class="a1ai-form-row">
                    <div class="a1ai-form-field" style="width: 70%; display: inline-block; vertical-align: top;">
                        <label for="user_input"><?php echo esc_html__('User Input Pattern:', 'a1ai-chatbot'); ?></label>
                        <textarea name="user_input" id="user_input" rows="3" placeholder="<?php echo esc_attr__('Enter a pattern that matches what users might type...', 'a1ai-chatbot'); ?>" required></textarea>
                        <p class="description">
                            <?php echo esc_html__('Use +word for required words. Examples:', 'a1ai-chatbot'); ?><br>
                            • <strong><?php echo esc_html__('Default:', 'a1ai-chatbot'); ?></strong> <?php echo esc_html__('Basic free plugin matching', 'a1ai-chatbot'); ?><br>
                            • <strong><?php echo esc_html__('Broad:', 'a1ai-chatbot'); ?></strong> <?php echo esc_html__('Loose matching, +pricing (must include "pricing")', 'a1ai-chatbot'); ?><br>
                            • <strong><?php echo esc_html__('Phrase:', 'a1ai-chatbot'); ?></strong> <?php echo esc_html__('Core phrase must be present', 'a1ai-chatbot'); ?><br>
                            • <strong><?php echo esc_html__('Exact:', 'a1ai-chatbot'); ?></strong> <?php echo esc_html__('Nearly verbatim matching required', 'a1ai-chatbot'); ?>
                        </p>
                    </div>
                    <div class="a1ai-form-field" style="width: 25%; display: inline-block; vertical-align: top; margin-left: 3%;">
                        <label for="ui_match_type"><?php echo esc_html__('Match Type:', 'a1ai-chatbot'); ?></label>
                        <select name="ui_match_type" id="ui_match_type">
                            <option value="Default" selected><?php echo esc_html__('Default', 'a1ai-chatbot'); ?></option>
                            <?php if (class_exists('A1AI_Upgrade_Helper') && A1AI_Upgrade_Helper::is_upgrade_active()): ?>
                            <option value="Broad"><?php echo esc_html__('Broad', 'a1ai-chatbot'); ?></option>
                            <option value="Phrase"><?php echo esc_html__('Phrase', 'a1ai-chatbot'); ?></option>
                            <option value="Exact"><?php echo esc_html__('Exact', 'a1ai-chatbot'); ?></option>
                            <?php else: ?>
                            <option value="Broad" disabled><?php echo esc_html__('Broad (Premium)', 'a1ai-chatbot'); ?></option>
                            <option value="Phrase" disabled><?php echo esc_html__('Phrase (Premium)', 'a1ai-chatbot'); ?></option>
                            <option value="Exact" disabled><?php echo esc_html__('Exact (Premium)', 'a1ai-chatbot'); ?></option>
                            <?php endif; ?>
                        </select>
                    </div>
                </div>

                <div class="a1ai-form-row">
                    <div class="a1ai-form-field" style="width: 45%; display: inline-block; vertical-align: top;">
                        <label for="ai_output"><?php echo esc_html__('AI Assistant Response:', 'a1ai-chatbot'); ?></label>
                        <textarea name="ai_output" id="ai_output" rows="3" placeholder="<?php echo esc_attr__('Enter how the AI should respond...', 'a1ai-chatbot'); ?>" required></textarea>
                        <p class="description">
                            <?php echo esc_html__('Use +word for required elements, {instructions} for AI guidance:', 'a1ai-chatbot'); ?><br>
                            • <strong><?php echo esc_html__('Direct:', 'a1ai-chatbot'); ?></strong> <?php echo esc_html__('"Our pricing starts at $99"', 'a1ai-chatbot'); ?><br>
                            • <strong><?php echo esc_html__('Instructions:', 'a1ai-chatbot'); ?></strong> <?php echo esc_html__('"{be friendly and provide pricing details}"', 'a1ai-chatbot'); ?>
                        </p>
                    </div>
                    <div class="a1ai-form-field" style="width: 25%; display: inline-block; vertical-align: top; margin-left: 3%;">
                        <label for="ai_match_type"><?php echo esc_html__('Response Type:', 'a1ai-chatbot'); ?></label>
                        <select name="ai_match_type" id="ai_match_type">
                            <option value="Default" selected><?php echo esc_html__('Default', 'a1ai-chatbot'); ?></option>
                            <?php if (class_exists('A1AI_Upgrade_Helper') && A1AI_Upgrade_Helper::is_upgrade_active()): ?>
                            <option value="Broad"><?php echo esc_html__('Broad', 'a1ai-chatbot'); ?></option>
                            <option value="Phrase"><?php echo esc_html__('Phrase', 'a1ai-chatbot'); ?></option>
                            <option value="Exact"><?php echo esc_html__('Exact', 'a1ai-chatbot'); ?></option>
                            <?php else: ?>
                            <option value="Broad" disabled><?php echo esc_html__('Broad (Premium)', 'a1ai-chatbot'); ?></option>
                            <option value="Phrase" disabled><?php echo esc_html__('Phrase (Premium)', 'a1ai-chatbot'); ?></option>
                            <option value="Exact" disabled><?php echo esc_html__('Exact (Premium)', 'a1ai-chatbot'); ?></option>
                            <?php endif; ?>
                        </select>
                    </div>
                    <div class="a1ai-form-field" style="width: 20%; display: inline-block; vertical-align: top; margin-left: 3%;">
                        <label for="ai_temperature"><?php echo esc_html__('Temperature:', 'a1ai-chatbot'); ?></label>
                        <select name="ai_temperature" id="ai_temperature" <?php if (!class_exists('A1AI_Upgrade_Helper') || !A1AI_Upgrade_Helper::is_upgrade_active()) echo 'disabled'; ?>>
                            <option value="0.7" selected><?php echo esc_html__('0.7 - Creative', 'a1ai-chatbot'); ?></option>
                            <?php if (class_exists('A1AI_Upgrade_Helper') && A1AI_Upgrade_Helper::is_upgrade_active()): ?>
                            <option value="0.1"><?php echo esc_html__('0.1 - Very Focused', 'a1ai-chatbot'); ?></option>
                            <option value="0.3"><?php echo esc_html__('0.3 - Focused', 'a1ai-chatbot'); ?></option>
                            <option value="0.5"><?php echo esc_html__('0.5 - Balanced', 'a1ai-chatbot'); ?></option>
                            <option value="0.9"><?php echo esc_html__('0.9 - Very Creative', 'a1ai-chatbot'); ?></option>
                            <option value="1.0"><?php echo esc_html__('1.0 - Maximum', 'a1ai-chatbot'); ?></option>
                            <?php endif; ?>
                        </select>
                        <p class="description"><?php echo esc_html__('Controls AI creativity', 'a1ai-chatbot'); ?><?php if (!class_exists('A1AI_Upgrade_Helper') || !A1AI_Upgrade_Helper::is_upgrade_active()) echo ' (Premium Feature)'; ?></p>
                    </div>
                </div>
                
                <div class="a1ai-form-row">
                    <button type="submit" class="button button-primary"><?php echo esc_html__('Save Adjustment', 'a1ai-chatbot'); ?></button>
                </div>
            </form>
        </div>

        <!-- Existing Adjustments -->
        <div class="a1ai-existing-adjustments">
            <h3><?php echo esc_html__('Existing Adjustments', 'a1ai-chatbot'); ?></h3>
            
            <?php if (empty($adjustments)) : ?>
                <div class="a1ai-empty-state">
                    <p><?php echo esc_html__('No adjustments found for this chatbot. Create your first adjustment above.', 'a1ai-chatbot'); ?></p>
                </div>
            <?php else : ?>
                <div class="a1ai-adjustments-list">
                    <?php foreach ($adjustments as $adjustment) : 
                            // Get chatbot name
                            $chatbot = $data_manager->get_chatbot($adjustment['chatbot_id']);
                            $chatbot_name = $chatbot ? $chatbot['name'] : __('Unknown Chatbot', 'a1ai-chatbot');
                        ?>
                        <div class="a1ai-adjustment-item">
                            <div class="a1ai-adjustment-content">
                                <div class="a1ai-adjustment-user">
                                    <strong><?php echo esc_html__('User Input:', 'a1ai-chatbot'); ?></strong>
                                    <p><?php echo esc_html($adjustment['user_input']); ?></p>
                                </div>
                                <div class="a1ai-adjustment-ai">
                                    <strong><?php echo esc_html__('AI Response:', 'a1ai-chatbot'); ?></strong>
                                    <p><?php echo esc_html($adjustment['ai_output']); ?></p>
                                </div>
                                <div class="a1ai-adjustment-meta">
                                    <span class="a1ai-chatbot">
                                        <strong><?php echo esc_html__('Chatbot:', 'a1ai-chatbot'); ?></strong> 
                                        <?php echo esc_html($chatbot_name); ?>
                                    </span>
                                    <span class="a1ai-match-type">
                                        <strong><?php echo esc_html__('UI Match:', 'a1ai-chatbot'); ?></strong> 
                                        <?php echo esc_html($adjustment['ui_match_type']); ?>
                                    </span>
                                    <span class="a1ai-match-type">
                                        <strong><?php echo esc_html__('AI Match:', 'a1ai-chatbot'); ?></strong>
                                        <?php echo esc_html($adjustment['ai_match_type']); ?>
                                    </span>
                                    <span class="a1ai-temperature">
                                        <strong><?php echo esc_html__('Temperature:', 'a1ai-chatbot'); ?></strong>
                                        <?php echo esc_html($adjustment['ai_temperature'] ?? '0.7'); ?>
                                    </span>
                                    <span class="a1ai-date">
                                        <?php echo esc_html__('Created:', 'a1ai-chatbot'); ?>
                                        <?php echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($adjustment['created_at']))); ?>
                                    </span>
                                </div>
                            </div>
                            <div class="a1ai-adjustment-actions">
                                <button type="button" class="button button-small a1ai-edit-adjustment"
                                        data-adjustment-id="<?php echo esc_attr($adjustment['id']); ?>"
                                        data-user-input="<?php echo esc_attr($adjustment['user_input']); ?>"
                                        data-ai-output="<?php echo esc_attr($adjustment['ai_output']); ?>"
                                        data-ui-match="<?php echo esc_attr($adjustment['ui_match_type']); ?>"
                                        data-ai-match="<?php echo esc_attr($adjustment['ai_match_type']); ?>"
                                        data-ai-temperature="<?php echo esc_attr($adjustment['ai_temperature'] ?? 0.7); ?>">
                                    <?php echo esc_html__('Edit', 'a1ai-chatbot'); ?>
                                </button>
                                <a href="<?php echo esc_url(wp_nonce_url(admin_url('admin.php?page=a1ai-chatbots&delete_adjustment=' . $adjustment['id'] . '&chatbot_id=' . $selected_chatbot_id), 'a1ai_delete_adjustment')); ?>" 
                                   class="button button-small a1ai-delete-adjustment" 
                                   onclick="return confirm('<?php echo esc_js(__('Are you sure you want to delete this adjustment?', 'a1ai-chatbot')); ?>')">
                                    <?php echo esc_html__('Delete', 'a1ai-chatbot'); ?>
                                </a>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
    <?php else : ?>
        <div class="a1ai-notice a1ai-notice-info">
            <p><?php echo esc_html__('Please select a chatbot to manage its adjustments.', 'a1ai-chatbot'); ?></p>
        </div>
    <?php endif; ?>
    
    <!-- Edit Adjustment Modal -->
    <div id="a1ai-edit-modal" class="a1ai-modal" style="display: none;">
        <div class="a1ai-modal-content">
            <div class="a1ai-modal-header">
                <h3><?php echo esc_html__('Edit Adjustment', 'a1ai-chatbot'); ?></h3>
                <span class="a1ai-modal-close">&times;</span>
            </div>
            <form id="a1ai-edit-form" method="post" action="<?php echo esc_url(admin_url('admin.php?page=a1ai-chatbots&active_tab=adjustments&chatbot_id=' . $selected_chatbot_id)); ?>">
                <?php wp_nonce_field('a1ai_edit_adjustment', 'a1ai_edit_adjustment_nonce'); ?>
                <input type="hidden" name="a1ai_action" value="edit_adjustment">
                <input type="hidden" name="adjustment_id" id="edit_adjustment_id">
                <input type="hidden" name="chatbot_id" value="<?php echo esc_attr($selected_chatbot_id); ?>">
                
                <div class="a1ai-form-row">
                    <div class="a1ai-form-field" style="width: 70%; display: inline-block; vertical-align: top;">
                        <label for="edit_user_input"><?php echo esc_html__('User Input Pattern:', 'a1ai-chatbot'); ?></label>
                        <textarea name="user_input" id="edit_user_input" rows="3" required></textarea>
                        <p class="description">
                            <?php echo esc_html__('Use +word for required words. Examples:', 'a1ai-chatbot'); ?><br>
                            • <strong><?php echo esc_html__('Default:', 'a1ai-chatbot'); ?></strong> <?php echo esc_html__('Basic free plugin matching', 'a1ai-chatbot'); ?><br>
                            • <strong><?php echo esc_html__('Broad:', 'a1ai-chatbot'); ?></strong> <?php echo esc_html__('Loose matching, +pricing (must include "pricing")', 'a1ai-chatbot'); ?><br>
                            • <strong><?php echo esc_html__('Phrase:', 'a1ai-chatbot'); ?></strong> <?php echo esc_html__('Core phrase must be present', 'a1ai-chatbot'); ?><br>
                            • <strong><?php echo esc_html__('Exact:', 'a1ai-chatbot'); ?></strong> <?php echo esc_html__('Nearly verbatim matching required', 'a1ai-chatbot'); ?>
                        </p>
                    </div>
                    <div class="a1ai-form-field" style="width: 25%; display: inline-block; vertical-align: top; margin-left: 3%;">
                        <label for="edit_ui_match_type"><?php echo esc_html__('Match Type:', 'a1ai-chatbot'); ?></label>
                        <select name="ui_match_type" id="edit_ui_match_type">
                            <option value="Default"><?php echo esc_html__('Default', 'a1ai-chatbot'); ?></option>
                            <?php if (class_exists('A1AI_Upgrade_Helper') && A1AI_Upgrade_Helper::is_upgrade_active()): ?>
                            <option value="Broad"><?php echo esc_html__('Broad', 'a1ai-chatbot'); ?></option>
                            <option value="Phrase"><?php echo esc_html__('Phrase', 'a1ai-chatbot'); ?></option>
                            <option value="Exact"><?php echo esc_html__('Exact', 'a1ai-chatbot'); ?></option>
                            <?php else: ?>
                            <option value="Broad" disabled><?php echo esc_html__('Broad (Premium)', 'a1ai-chatbot'); ?></option>
                            <option value="Phrase" disabled><?php echo esc_html__('Phrase (Premium)', 'a1ai-chatbot'); ?></option>
                            <option value="Exact" disabled><?php echo esc_html__('Exact (Premium)', 'a1ai-chatbot'); ?></option>
                            <?php endif; ?>
                        </select>
                    </div>
                </div>

                <div class="a1ai-form-row">
                    <div class="a1ai-form-field" style="width: 45%; display: inline-block; vertical-align: top;">
                        <label for="edit_ai_output"><?php echo esc_html__('AI Assistant Response:', 'a1ai-chatbot'); ?></label>
                        <textarea name="ai_output" id="edit_ai_output" rows="3" required></textarea>
                        <p class="description">
                            <?php echo esc_html__('Use +word for required elements, {instructions} for AI guidance:', 'a1ai-chatbot'); ?><br>
                            • <strong><?php echo esc_html__('Direct:', 'a1ai-chatbot'); ?></strong> <?php echo esc_html__('"Our pricing starts at $99"', 'a1ai-chatbot'); ?><br>
                            • <strong><?php echo esc_html__('Instructions:', 'a1ai-chatbot'); ?></strong> <?php echo esc_html__('"{be friendly and provide pricing details}"', 'a1ai-chatbot'); ?>
                        </p>
                    </div>
                    <div class="a1ai-form-field" style="width: 25%; display: inline-block; vertical-align: top; margin-left: 3%;">
                        <label for="edit_ai_match_type"><?php echo esc_html__('Response Type:', 'a1ai-chatbot'); ?></label>
                        <select name="ai_match_type" id="edit_ai_match_type">
                            <option value="Default"><?php echo esc_html__('Default', 'a1ai-chatbot'); ?></option>
                            <?php if (class_exists('A1AI_Upgrade_Helper') && A1AI_Upgrade_Helper::is_upgrade_active()): ?>
                            <option value="Broad"><?php echo esc_html__('Broad', 'a1ai-chatbot'); ?></option>
                            <option value="Phrase"><?php echo esc_html__('Phrase', 'a1ai-chatbot'); ?></option>
                            <option value="Exact"><?php echo esc_html__('Exact', 'a1ai-chatbot'); ?></option>
                            <?php else: ?>
                            <option value="Broad" disabled><?php echo esc_html__('Broad (Premium)', 'a1ai-chatbot'); ?></option>
                            <option value="Phrase" disabled><?php echo esc_html__('Phrase (Premium)', 'a1ai-chatbot'); ?></option>
                            <option value="Exact" disabled><?php echo esc_html__('Exact (Premium)', 'a1ai-chatbot'); ?></option>
                            <?php endif; ?>
                        </select>
                    </div>
                    <div class="a1ai-form-field" style="width: 20%; display: inline-block; vertical-align: top; margin-left: 3%;">
                        <label for="edit_ai_temperature"><?php echo esc_html__('Temperature:', 'a1ai-chatbot'); ?></label>
                        <select name="ai_temperature" id="edit_ai_temperature" <?php if (!class_exists('A1AI_Upgrade_Helper') || !A1AI_Upgrade_Helper::is_upgrade_active()) echo 'disabled'; ?>>
                            <option value="0.7"><?php echo esc_html__('0.7 - Creative', 'a1ai-chatbot'); ?></option>
                            <?php if (class_exists('A1AI_Upgrade_Helper') && A1AI_Upgrade_Helper::is_upgrade_active()): ?>
                            <option value="0.1"><?php echo esc_html__('0.1 - Very Focused', 'a1ai-chatbot'); ?></option>
                            <option value="0.3"><?php echo esc_html__('0.3 - Focused', 'a1ai-chatbot'); ?></option>
                            <option value="0.5"><?php echo esc_html__('0.5 - Balanced', 'a1ai-chatbot'); ?></option>
                            <option value="0.9"><?php echo esc_html__('0.9 - Very Creative', 'a1ai-chatbot'); ?></option>
                            <option value="1.0"><?php echo esc_html__('1.0 - Maximum', 'a1ai-chatbot'); ?></option>
                            <?php endif; ?>
                        </select>
                        <p class="description"><?php echo esc_html__('Controls AI creativity', 'a1ai-chatbot'); ?><?php if (!class_exists('A1AI_Upgrade_Helper') || !A1AI_Upgrade_Helper::is_upgrade_active()) echo ' (Premium Feature)'; ?></p>
                    </div>
                </div>
                
                <div class="a1ai-form-row">
                    <button type="submit" class="button button-primary"><?php echo esc_html__('Update Adjustment', 'a1ai-chatbot'); ?></button>
                    <button type="button" class="button a1ai-modal-cancel"><?php echo esc_html__('Cancel', 'a1ai-chatbot'); ?></button>
                </div>
            </form>
        </div>
    </div>
</div>

<script type="text/javascript">
jQuery(document).ready(function($) {
    // Check if we're in adjust mode first
    const urlParams = new URLSearchParams(window.location.search);
    const adjustMode = urlParams.get('adjust_mode');

    // Only check for upgrade plugin if we're in adjust mode
    if (adjustMode === '1') {
        // Only run if upgrade plugin is active (paid feature)
        if (typeof window.a1aiUpgradeAjax !== 'undefined' && window.a1aiUpgradeAjax.upgradePluginActive) {
            console.log('A1AI: Upgrade plugin detected, processing adjust mode');

            // Get the data from URL parameters
            const chatbotId = urlParams.get('chatbot_id');
            const userInput = decodeURIComponent(urlParams.get('user_input') || '');
            const aiResponse = decodeURIComponent(urlParams.get('ai_response') || '');

            console.log('A1AI: Data to populate:', {
                chatbotId: chatbotId,
                userInput: userInput,
                aiResponse: aiResponse
            });

            // Switch to adjustments tab if not already active
            if (window.location.hash === '#tab-adjustments' || window.location.hash === '') {
                // Click the adjustments tab to ensure it's active
                const adjustmentsTab = $('a[href="#tab-adjustments"]');
                if (adjustmentsTab.length && !adjustmentsTab.hasClass('active')) {
                    adjustmentsTab.click();
                    console.log('A1AI: Switched to adjustments tab');
                }

                // Auto-select chatbot and populate form
                setTimeout(function() {
                    if (chatbotId) {
                        // Check if chatbot is already selected (after page reload)
                        const currentChatbotId = $('#chatbot_id').val();

                        if (currentChatbotId != chatbotId) {
                            // Select the chatbot - this will trigger form submission with preserved parameters
                            $('#chatbot_id').val(chatbotId);
                            console.log('A1AI: Selecting chatbot:', chatbotId);
                            $('#chatbot_id').trigger('change');
                        } else {
                            // Chatbot already selected, proceed with form population
                            console.log('A1AI: Chatbot already selected:', chatbotId);

                            // Populate the form fields immediately
                            if (userInput) {
                                $('#user_input').val(userInput);
                                console.log('A1AI: Populated user input');
                            }
                            if (aiResponse) {
                                $('#ai_output').val(aiResponse);
                                console.log('A1AI: Populated AI response');
                            }

                            // Scroll to the form
                            if ($('#user_input').length) {
                                $('html, body').animate({
                                    scrollTop: $('#user_input').offset().top - 100
                                }, 500);

                                // Focus on the user input field
                                $('#user_input').focus();
                            }

                            console.log('A1AI: Auto-population complete');
                        }
                    }
                }, 500);
            }
        } else {
            console.log('A1AI: Upgrade plugin not available for adjust mode');
        }
    } else {
        // Not in adjust mode, no need to check for upgrade plugin
        console.log('A1AI: Not in adjust mode, skipping auto-population');
    }

    // Handle edit button clicks
    $('.a1ai-edit-adjustment').on('click', function() {
        const $this = $(this);
        const adjustmentId = $this.data('adjustment-id');
        const userInput = $this.data('user-input');
        const aiOutput = $this.data('ai-output');
        const uiMatch = $this.data('ui-match') || 'Broad';
        const aiMatch = $this.data('ai-match') || 'Broad';
        const aiTemperature = $this.data('ai-temperature') || 0.7;

        // Populate the edit modal
        $('#edit_adjustment_id').val(adjustmentId);
        $('#edit_user_input').val(userInput);
        $('#edit_ai_output').val(aiOutput);
        $('#edit_ui_match_type').val(uiMatch);
        $('#edit_ai_match_type').val(aiMatch);
        $('#edit_ai_temperature').val(aiTemperature);

        // Show the modal
        $('#a1ai-edit-modal').show();
    });

    // Handle modal close
    $('.a1ai-modal-close, .a1ai-modal-cancel').on('click', function() {
        $('#a1ai-edit-modal').hide();
    });

    // Close modal when clicking outside
    $('#a1ai-edit-modal').on('click', function(e) {
        if (e.target === this) {
            $(this).hide();
        }
    });
});
</script>

