/**
 * Admin JavaScript for A1AI Chatbot
 *
 * @package    A1AI
 * @subpackage A1AI/admin/js
 */

(function($) {
    'use strict';

    /**
     * Admin JS functionality
     */
    const A1AIAdmin = {
        
        /**
         * Convert URLs in text to clickable links
         * @param {string} text - The text to process
         * @returns {string} - Text with URLs converted to clickable links
         */
        makeLinksClickable: function(text) {
            // Regular expression to match URLs
            const urlRegex = /(https?:\/\/[^\s<>"{}|\\^`[\]]+)/gi;
            
            return text.replace(urlRegex, function(url) {
                // Check if it's an internal link (same domain)
                const currentDomain = window.location.hostname;
                try {
                    const urlObj = new URL(url);
                    const isInternal = urlObj.hostname === currentDomain;
                    
                    // Create clickable link
                    const linkClass = isInternal ? 'a1ai-link-internal' : 'a1ai-link-external';
                    const target = isInternal ? '_self' : '_blank';
                    const rel = isInternal ? '' : 'noopener noreferrer';
                    
                    return `<a href="${url}" class="${linkClass}" target="${target}" rel="${rel}">${url}</a>`;
                } catch (e) {
                    return url;
                }
            });
        },
        /**
         * Initialize the admin functionality
         */
        init: function() {
            try {
                this.setupFormHandlers();
                this.setupTabs();
                this.setupTemperaturePresets();
                this.setupBusinessAutoDetect();
                this.setupChartInitialization();
                this.setupChatbotManagement();
            } catch (error) {
                // Error initializing A1AI Admin
            }
        },

        /**
         * Setup form submission handlers
         */
        setupFormHandlers: function() {
            // Settings form submission
            $('#a1ai-settings-form').on('submit', function(e) {
                e.preventDefault();

                const $form = $(this);
                const $submitButton = $form.find('.button-primary');
                const $spinner = $form.find('.spinner');

                // Show spinner, disable submit button
                $spinner.addClass('is-active');
                $submitButton.prop('disabled', true);

                // Collect form data
                const formData = new FormData($form[0]);
                formData.append('action', 'a1ai_admin_action');
                formData.append('action_type', 'save_settings');
                formData.append('nonce', a1aichAdmin.nonce);

                // Explicitly set password field values (browsers sometimes don't include them in FormData)
                const openaiKey = $('#a1ai_openai_api_key').val();
                const eddKey = $('#a1ai_edd_api_key').val();
                formData.set('a1ai_openai_api_key', openaiKey);
                formData.set('a1ai_edd_api_key', eddKey);

                // Send AJAX request
                $.ajax({
                    url: a1aichAdmin.ajaxUrl,
                    type: 'POST',
                    data: formData,
                    processData: false,
                    contentType: false,
                    success: function(response) {
                        $spinner.removeClass('is-active');
                        $submitButton.prop('disabled', false);

                        if (response.success) {
                            A1AIAdmin.showNotice('success', response.data.message);
                            // Reload page after short delay to show updated values from database
                            setTimeout(function() {
                                window.location.reload();
                            }, 1500);
                        } else {
                            A1AIAdmin.showNotice('error', response.data.message);
                        }
                    },
                    error: function(xhr, status, error) {
                        $spinner.removeClass('is-active');
                        $submitButton.prop('disabled', false);
                        A1AIAdmin.showNotice('error', a1aichAdmin.strings.saveError);
                    }
                });
            });
            
            // Chatbot form submission
            $('#a1ai-chatbot-form').on('submit', function(e) {
                e.preventDefault();
                
                const $form = $(this);
                const $submitButton = $form.find('.button-primary');
                const $spinner = $form.find('.spinner');
                
                // Show spinner, disable submit button
                $spinner.addClass('is-active');
                $submitButton.prop('disabled', true);
                
                // Collect form data
                const formData = new FormData($form[0]);
                formData.append('action', 'a1ai_admin_action');
                formData.append('action_type', 'save_chatbot');
                formData.append('nonce', a1aichAdmin.nonce);
                
                // Explicitly handle checkbox values
                const gdprEnabled = $('#gdpr_enabled').is(':checked') ? '1' : '0';
                formData.set('settings[gdpr_enabled]', gdprEnabled);

                // Send AJAX request
                $.ajax({
                    url: a1aichAdmin.ajaxUrl,
                    type: 'POST',
                    data: formData,
                    processData: false,
                    contentType: false,
                    success: function(response) {
                        $spinner.removeClass('is-active');
                        $submitButton.prop('disabled', false);
                        
                        if (response.success) {
                            A1AIAdmin.showNotice('success', response.data.message);
                            
                            // If creating a new chatbot, redirect to edit page
                            if (response.data.redirect) {
                                window.location.href = response.data.redirect;
                            }
                        } else {
                            A1AIAdmin.showNotice('error', response.data.message);
                        }
                    },
                    error: function() {
                        $spinner.removeClass('is-active');
                        $submitButton.prop('disabled', false);
                        A1AIAdmin.showNotice('error', a1aichAdmin.strings.saveError);
                    }
                });
            });

            // GDPR toggle functionality
            $('#gdpr_enabled').on('change', function() {
                const $consentTextField = $('#gdpr-consent-text-field');
                if ($(this).is(':checked')) {
                    $consentTextField.show();
                } else {
                    $consentTextField.hide();
                }
            });

            // Copy shortcode functionality
            $('.a1ai-copy-shortcode').on('click', function(e) {
                e.preventDefault();
                const shortcode = $(this).data('shortcode');

                // Try to copy to clipboard
                if (navigator.clipboard) {
                    navigator.clipboard.writeText(shortcode).then(function() {
                        $(e.target).text('Copied!');
                        setTimeout(function() {
                            $(e.target).text('Copy');
                        }, 2000);
                    });
                } else {
                    // Fallback for older browsers
                    const textArea = document.createElement('textarea');
                    textArea.value = shortcode;
                    document.body.appendChild(textArea);
                    textArea.select();
                    document.execCommand('copy');
                    document.body.removeChild(textArea);

                    $(e.target).text('Copied!');
                    setTimeout(function() {
                        $(e.target).text('Copy');
                    }, 2000);
                }
            });
        },
        
        /**
         * Setup tabs functionality
         */
        setupTabs: function() {
            // Skip tab setup if we're on the chatbots page (it has its own tab handler)
            if (window.location.href.indexOf('page=a1ai-chatbots') !== -1) {
                return;
            }

            $('.a1ai-tabs-nav a').on('click', function(e) {
                e.preventDefault();

                const $this = $(this);
                const target = $this.attr('href');

                // Update active tab
                $('.a1ai-tabs-nav a').removeClass('active');
                $this.addClass('active');

                // Show target tab content
                $('.a1ai-tab-content').hide();
                $(target).show();

                // Update URL hash
                if (history.pushState) {
                    history.pushState(null, null, target);
                } else {
                    location.hash = target;
                }
            });

            // Check for hash in URL
            const hash = window.location.hash;
            if (hash) {
                $('.a1ai-tabs-nav a[href="' + hash + '"]').click();
            } else {
                // Show first tab by default
                $('.a1ai-tabs-nav a').first().click();
            }
        },
        
        /**
         * Setup analytics chart initialization
         */
        setupChartInitialization: function() {
            const $chartCanvas = $('#a1ai-analytics-chart');
            const $fallback = $('#a1ai-chart-fallback');
            
            if (!$chartCanvas.length) {
                return;
            }
            
            // Show fallback message initially
            $fallback.show();
            
            // Check if Chart.js is available
            if (typeof Chart === 'undefined') {
                $fallback.html('<p>Chart.js is not available. Please refresh the page.</p>');
                return;
            }

            // Load chart data
            $.ajax({
                url: a1aichAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'a1ai_admin_action',
                    action_type: 'get_analytics',
                    period: 'month',
                    nonce: a1aichAdmin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        try {
                            A1AIAdmin.initChart($chartCanvas, response.data);
                            $fallback.hide(); // Hide fallback on success
                        } catch (error) {
                            $fallback.html('<p>Error loading chart. Please refresh the page.</p>');
                        }
                    } else {
                        $fallback.html('<p>Failed to load chart data. Please refresh the page.</p>');
                    }
                },
                error: function(xhr, status, error) {
                    $fallback.html('<p>Error loading chart data. Please refresh the page.</p>');
                }
            });
        },
        
        /**
         * Initialize analytics chart
         * 
         * @param {jQuery} $canvas The chart canvas element
         * @param {Object} data The chart data
         */
        initChart: function($canvas, data) {
            try {
                const ctx = $canvas[0].getContext('2d');

                // Validate data structure
                if (!data || !data.labels || !data.messages || !data.conversations) {
                    return;
                }

                new Chart(ctx, {
                    type: 'line',
                    data: {
                        labels: data.labels,
                        datasets: [
                            {
                                label: 'Messages',
                                data: data.messages,
                                borderColor: '#2271b1',
                                backgroundColor: 'rgba(34, 113, 177, 0.1)',
                                borderWidth: 2,
                                tension: 0.1
                            },
                            {
                                label: 'Conversations',
                                data: data.conversations,
                                borderColor: '#46b450',
                                backgroundColor: 'rgba(70, 180, 80, 0.1)',
                                borderWidth: 2,
                                tension: 0.1
                            }
                        ]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        scales: {
                            y: {
                                beginAtZero: true
                            }
                        }
                    }
                });
            } catch (error) {
                // Error creating chart
            }
        },

        /**
         * Setup temperature preset functionality
         */
        setupTemperaturePresets: function() {
            const $presetRadios = $('input[name="a1ai_temperature_preset"]');
            const $customWrapper = $('#a1ai_custom_temperature_wrapper');
            const $customInput = $('#a1ai_custom_temperature_input');
            const $hiddenField = $('#a1ai_temperature_value');

            if (!$presetRadios.length) {
                return; // Not on settings page
            }

            // Function to update temperature value
            function updateTemperatureValue() {
                const selectedPreset = $('input[name="a1ai_temperature_preset"]:checked').val();

                if (selectedPreset === 'custom') {
                    // Show custom input and use its value
                    $customWrapper.slideDown(200);
                    $hiddenField.val($customInput.val());
                } else {
                    // Hide custom input and use preset value
                    $customWrapper.slideUp(200);
                    $hiddenField.val(selectedPreset);
                }
            }

            // Handle preset radio change
            $presetRadios.on('change', updateTemperatureValue);

            // Handle custom input change
            $customInput.on('input change', function() {
                if ($('input[name="a1ai_temperature_preset"]:checked').val() === 'custom') {
                    $hiddenField.val($(this).val());
                }
            });

            // Initialize on page load
            updateTemperatureValue();
        },

        /**
         * Setup business info auto-detection (Premium feature)
         */
        setupBusinessAutoDetect: function() {
            const $autoDetectBtn = $('#a1ai-auto-detect-business');
            const $statusDiv = $('#a1ai-auto-detect-status');
            const $statusText = $statusDiv.find('.status-text');
            const $spinner = $statusDiv.find('.spinner');

            if (!$autoDetectBtn.length) {
                return; // Premium feature not available
            }

            $autoDetectBtn.on('click', function(e) {
                e.preventDefault();

                // Show status
                $statusDiv.slideDown(200);
                $spinner.addClass('is-active');
                $statusText.text('Scanning your website for business information...');
                $autoDetectBtn.prop('disabled', true);

                // Make AJAX call to upgrade plugin
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'a1ai_upgrade_auto_detect_business',
                        nonce: a1aichAdmin.nonce
                    },
                    success: function(response) {
                        $spinner.removeClass('is-active');

                        if (response.success && response.data) {
                            $statusText.html('<span style="color: #46b450;">✓ Detection complete! Found business information.</span>');

                            // Fill in the form fields with detected data
                            const data = response.data;

                            if (data.business_name) {
                                $('#business_name').val(data.business_name);
                            }
                            if (data.tagline) {
                                $('#tagline').val(data.tagline);
                            }
                            if (data.business_type) {
                                $('#business_type').val(data.business_type);
                            }
                            if (data.services && data.services.length > 0) {
                                $('#services').val(data.services.join('\n'));
                            }
                            if (data.products && data.products.length > 0) {
                                $('#products').val(data.products.join('\n'));
                            }
                            if (data.hours) {
                                $.each(data.hours, function(day, time) {
                                    $('input[name="business_info[hours][' + day + ']"]').val(time);
                                });
                            }
                            if (data.address) {
                                $('#address').val(data.address);
                            }
                            if (data.phone) {
                                $('#phone').val(data.phone);
                            }
                            if (data.email) {
                                $('#business_email').val(data.email);
                            }
                            if (data.policies) {
                                $('#policies').val(data.policies);
                            }

                            // Show success message
                            A1AIAdmin.showNotice('success', 'Business information detected! Please review and adjust as needed, then save.');

                            // Hide status after 3 seconds
                            setTimeout(function() {
                                $statusDiv.slideUp(200);
                            }, 3000);

                        } else {
                            $statusText.html('<span style="color: #ffcc00;">⚠ Detection completed but limited information found. Please fill in manually.</span>');
                            setTimeout(function() {
                                $statusDiv.slideUp(200);
                            }, 5000);
                        }

                        $autoDetectBtn.prop('disabled', false);
                    },
                    error: function(xhr, status, error) {
                        $spinner.removeClass('is-active');
                        $statusText.html('<span style="color: #dc3232;">✗ Error during detection. Please fill in manually.</span>');
                        $autoDetectBtn.prop('disabled', false);

                        setTimeout(function() {
                            $statusDiv.slideUp(200);
                        }, 5000);
                    }
                });
            });
        },

        /**
         * Setup chatbot management functionality
         */
        setupChatbotManagement: function() {
            // Delete chatbot confirmation
            $('.a1ai-delete-chatbot').on('click', function(e) {
                e.preventDefault();
                
                if (!confirm(a1aichAdmin.strings.confirmDelete)) {
                    return;
                }
                
                const $button = $(this);
                const chatbotId = $button.data('id');
                
                // Send delete request
                $.ajax({
                    url: a1aichAdmin.ajaxUrl,
                    type: 'POST',
                    data: {
                        action: 'a1ai_admin_action',
                        action_type: 'delete_chatbot',
                        chatbot_id: chatbotId,
                        nonce: a1aichAdmin.nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            // Remove chatbot row
                            $button.closest('.a1ai-chatbot-item').fadeOut(300, function() {
                                $(this).remove();
                            });
                            
                            A1AIAdmin.showNotice('success', response.data.message);
                        } else {
                            A1AIAdmin.showNotice('error', response.data.message);
                        }
                    },
                    error: function() {
                        A1AIAdmin.showNotice('error', a1aichAdmin.strings.deleteError);
                    }
                });
            });
            
            // View conversation functionality
            $('.a1ai-view-conversation').on('click', function(e) {
                e.preventDefault();
                e.stopPropagation();
                const conversationId = $(this).data('conversation-id');
                A1AIAdmin.loadConversation(conversationId);
            });
            
            // Row click functionality
            $('.a1ai-conversation-row').on('click', function() {
                const conversationId = $(this).data('conversation-id');
                A1AIAdmin.loadConversation(conversationId);
            });
            
            // Delete conversation functionality
            $('.a1ai-delete-conversation').on('click', function(e) {
                e.preventDefault();
                e.stopPropagation();
                const conversationId = $(this).data('conversation-id');
                
                if (!confirm('Are you sure you want to delete this conversation? This action cannot be undone.')) {
                    return;
                }
                
                $.ajax({
                    url: a1aichAdmin.ajaxUrl,
                    type: 'POST',
                    data: {
                        action: 'a1ai_admin_action',
                        action_type: 'delete_conversation',
                        conversation_id: conversationId,
                        nonce: a1aichAdmin.nonce
                    },
                    success: function(response) {
                        if (response.success) {
                            A1AIAdmin.showNotice('success', 'Conversation deleted successfully.');
                            // Reload page to update list
                            location.reload();
                        } else {
                            A1AIAdmin.showNotice('error', response.data.message || 'Error deleting conversation.');
                        }
                    },
                    error: function() {
                        A1AIAdmin.showNotice('error', 'Error deleting conversation.');
                    }
                });
            });
            
            // Preview close functionality
            $('.a1ai-preview-close').on('click', function() {
                A1AIAdmin.hideConversationPreview();
            });
        },
        
        /**
         * Load conversation details into preview panel
         */
        loadConversation: function(conversationId) {
            // Update selected row
            $('.a1ai-conversation-row').removeClass('selected');
            $(`.a1ai-conversation-row[data-conversation-id="${conversationId}"]`).addClass('selected');

            $.ajax({
                url: a1aichAdmin.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'a1ai_admin_action',
                    action_type: 'get_conversation',
                    conversation_id: conversationId,
                    nonce: a1aichAdmin.nonce
                },
                success: function(response) {
                    if (response.success) {
                        A1AIAdmin.displayConversation(response.data);
                    } else {
                        A1AIAdmin.showNotice('error', response.data.message || 'Error loading conversation.');
                    }
                },
                error: function(xhr, status, error) {
                    A1AIAdmin.showNotice('error', 'Error loading conversation.');
                }
            });
        },
        
        /**
         * Display conversation in preview panel
         */
        displayConversation: function(data) {
            const $preview = $('#a1ai-conversation-preview');
            const $placeholder = $preview.find('.a1ai-preview-placeholder');
            const $content = $preview.find('.a1ai-preview-content');
            const $messagesContainer = $preview.find('.a1ai-preview-messages');

            // Hide placeholder and show content
            $placeholder.hide();
            $content.show();

            // Update conversation info
            const conversation = data.conversation;
            const messages = data.messages;

            // Update meta information
            $preview.find('.a1ai-preview-chatbot').text('Chatbot: ' + (conversation.chatbot_name || 'Unknown'));
            $preview.find('.a1ai-preview-user').text('User: ' + (conversation.user_name || 'Guest'));
            $preview.find('.a1ai-preview-time').text('Started: ' + new Date(conversation.created_at).toLocaleString());

            // Check if upgrade plugin is available
            const upgradePluginAvailable = typeof a1aiUpgradeAjax !== 'undefined';

            // Build messages HTML with conversation pairs and adjust buttons
            let messagesHtml = '';
            let currentPair = null;
            let pairIndex = 0;

            messages.forEach(function(message, index) {
                if (message.role === 'user') {
                    // Start new pair
                    if (currentPair) {
                        // Close previous pair if it exists
                        messagesHtml += '</div>';
                    }

                    pairIndex++;
                    currentPair = {
                        user: message,
                        assistant: null
                    };

                    const time = new Date(message.created_at).toLocaleString();
                    messagesHtml += `
                        <div class="a1ai-message-pair" data-pair-index="${pairIndex}">
                            <div class="a1ai-message-pair-header">
                                <div class="a1ai-message-pair-title">Conversation Pair #${pairIndex}</div>
                            </div>
                            <div class="a1ai-conversation-message ${message.role}">
                                <div class="role">${message.role}</div>
                                <div class="content">${A1AIAdmin.makeLinksClickable(message.content)}</div>
                                <div class="time">${time}</div>
                            </div>
                    `;
                } else if (message.role === 'assistant' && currentPair) {
                    // Add assistant response to current pair
                    const time = new Date(message.created_at).toLocaleString();
                    const adjustButtonHtml = upgradePluginAvailable ? `
                        <div class="a1ai-adjust-actions" style="text-align: center; margin-top: 15px;">
                            <button type="button" class="a1ai-adjust-button"
                                    data-user-input="${encodeURIComponent(currentPair.user.content)}"
                                    data-ai-response="${encodeURIComponent(message.content)}"
                                    data-chatbot-id="${conversation.chatbot_id}"
                                    data-chatbot-name="${encodeURIComponent(conversation.chatbot_name || 'Unknown')}">
                                Adjust This Conversation
                            </button>
                        </div>
                    ` : '';

                    messagesHtml += `
                            <div class="a1ai-conversation-message ${message.role}">
                                <div class="role">${message.role}</div>
                                <div class="content">${A1AIAdmin.makeLinksClickable(message.content)}</div>
                                <div class="time">${time}</div>
                            </div>
                            ${adjustButtonHtml}
                    `;
                    currentPair.assistant = message;
                } else {
                    // Handle standalone messages
                    const time = new Date(message.created_at).toLocaleString();
                    messagesHtml += `
                        <div class="a1ai-conversation-message ${message.role}">
                            <div class="role">${message.role}</div>
                            <div class="content">${A1AIAdmin.makeLinksClickable(message.content)}</div>
                            <div class="time">${time}</div>
                        </div>
                    `;
                }
            });

            // Close last pair if it exists
            if (currentPair) {
                messagesHtml += '</div>';
            }

            $messagesContainer.html(messagesHtml);

            // Bind adjust button events if upgrade plugin is available
            if (upgradePluginAvailable) {
                A1AIAdmin.bindAdjustButtons();
            }
            
            // Scroll to top of messages
            $messagesContainer.scrollTop(0);
        },
        
        /**
         * Hide conversation preview
         */
        hideConversationPreview: function() {
            const $preview = $('#a1ai-conversation-preview');
            const $placeholder = $preview.find('.a1ai-preview-placeholder');
            const $content = $preview.find('.a1ai-preview-content');
            
            $content.hide();
            $placeholder.show();
            
            // Remove selected row
            $('.a1ai-conversation-row').removeClass('selected');
        },

        /**
         * Bind adjust button functionality
         */
        bindAdjustButtons: function() {
            $('.a1ai-adjust-button').off('click').on('click', function() {
                const userInput = decodeURIComponent($(this).data('user-input'));
                const aiResponse = decodeURIComponent($(this).data('ai-response'));
                const chatbotId = $(this).data('chatbot-id');
                const chatbotName = decodeURIComponent($(this).data('chatbot-name'));

                // Create URL for adjustments tab
                const adjustUrl = new URL(window.location.origin + window.location.pathname);
                adjustUrl.searchParams.set('page', 'a1ai-chatbots');
                adjustUrl.searchParams.set('chatbot_id', chatbotId);
                adjustUrl.searchParams.set('chatbot_name', chatbotName);
                adjustUrl.searchParams.set('user_input', userInput);
                adjustUrl.searchParams.set('ai_response', aiResponse);
                adjustUrl.searchParams.set('adjust_mode', '1');
                adjustUrl.hash = 'tab-adjustments';

                // Redirect to adjustments tab
                window.location.href = adjustUrl.toString();
            });
        },

        /**
         * Show admin notice
         * 
         * @param {string} type The notice type (success, error, warning)
         * @param {string} message The notice message
         */
        showNotice: function(type, message) {
            const $noticeContainer = $('.a1ai-notices');
            
            if (!$noticeContainer.length) {
                return;
            }
            
            const $notice = $('<div class="a1ai-notice a1ai-notice-' + type + '">' + message + '</div>');
            
            $noticeContainer.html($notice);
            
            // Auto-hide notice after 5 seconds
            setTimeout(function() {
                $notice.fadeOut(300, function() {
                    $(this).remove();
                });
            }, 5000);
        }
    };
    
    // Initialize on document ready
    $(document).ready(function() {
        // Check if admin variables are available
        if (typeof a1aichAdmin === 'undefined') {
            return;
        }

        A1AIAdmin.init();

        // Initialize content awareness functionality
        A1AIAdmin.initContentAwareness();
    });

    // Content awareness functionality
    A1AIAdmin.initContentAwareness = function() {
        // Check if upgrade plugin is available (same way as other functions)
        const upgradePluginAvailable = typeof a1aiUpgradeAjax !== 'undefined';

        if (!upgradePluginAvailable) {
            return;
        }

        // Load initial statistics for both tiers
        this.loadAnalysisStatistics(1); // Tier 1
        this.loadAnalysisStatistics(2); // Tier 2

        // Use delegated event handlers to work even if tab isn't active yet
        // Tier 1 analyze all content
        $(document).on('click', '#a1ai-analyze-all-content-tier1', function() {
            if (confirm('This will analyze all content with Tier 1. This may take a while. Continue?')) {
                A1AIAdmin.startAnalysis('all', 1);
            }
        });

        // Tier 1 analyze recent content
        $(document).on('click', '#a1ai-analyze-recent-content-tier1', function() {
            A1AIAdmin.startAnalysis('recent', 1);
        });

        // Tier 1 clear analysis data
        $(document).on('click', '#a1ai-clear-analysis-data-tier1', function() {
            if (confirm('This will clear all Tier 1 analysis data. Are you sure?')) {
                A1AIAdmin.clearAnalysisData(1);
            }
        });

        // Tier 1 view analysis results
        $(document).on('click', '#a1ai-view-analysis-results-tier1', function() {
            A1AIAdmin.viewAnalysisResults(1);
        });

        // Tier 2 analyze all content
        $(document).on('click', '#a1ai-analyze-all-content-tier2', function() {
            if (confirm('This will analyze all content with Tier 2. This may take a while. Continue?')) {
                A1AIAdmin.startAnalysis('all', 2);
            }
        });

        // Tier 2 analyze recent content
        $(document).on('click', '#a1ai-analyze-recent-content-tier2', function() {
            A1AIAdmin.startAnalysis('recent', 2);
        });

        // Tier 2 clear analysis data
        $(document).on('click', '#a1ai-clear-analysis-data-tier2', function() {
            if (confirm('This will clear all Tier 2 analysis data. Are you sure?')) {
                A1AIAdmin.clearAnalysisData(2);
            }
        });

        // Tier 2 view analysis results
        $(document).on('click', '#a1ai-view-analysis-results-tier2', function() {
            A1AIAdmin.viewAnalysisResults(2);
        });

        // Tier 2 enable/disable controls
        $('#a1ai-tier2-enabled').on('change', function() {
            const isEnabled = $(this).is(':checked');

            // Enable/disable strategy dropdown
            $('#a1ai-analysis-strategy').prop('disabled', !isEnabled);
            if (isEnabled) {
                $('#a1ai-analysis-strategy').css('opacity', '1');
                $('.a1ai-tier2-controls').removeClass('tier2-disabled');
            } else {
                $('#a1ai-analysis-strategy').css('opacity', '0.5');
                $('.a1ai-tier2-controls').addClass('tier2-disabled');
            }

            // Enable/disable Tier 2 buttons
            $('#a1ai-analyze-all-content-tier2, #a1ai-analyze-recent-content-tier2, #a1ai-clear-analysis-data-tier2, #a1ai-view-analysis-results-tier2').prop('disabled', !isEnabled);

            // Save state to localStorage
            localStorage.setItem('a1ai_tier2_enabled', isEnabled);
        });

        // Strategy dropdown change handler
        $('#a1ai-analysis-strategy').on('change', function() {
            localStorage.setItem('a1ai_analysis_strategy', $(this).val());
        });

        // Save content settings button
        $('#a1ai-save-content-settings').on('click', function() {
            const $button = $(this);
            const originalText = $button.text();

            // Get nonce from button data attribute or upgrade plugin
            const upgradeNonce = $button.data('nonce') || (typeof a1aiUpgradeAjax !== 'undefined' && a1aiUpgradeAjax.nonce) || '';

            $button.prop('disabled', true).text('Saving...');

            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'a1ai_upgrade_action',
                    action_type: 'save_content_settings',
                    nonce: upgradeNonce,
                    auto_analyze: $('#a1ai_auto_analyze').is(':checked') ? 1 : 0,
                    generate_sitemap: $('#a1ai_generate_sitemap').is(':checked') ? 1 : 0,
                    analysis_frequency: $('#a1ai_analysis_frequency').val(),
                    min_content_length: $('#a1ai_min_content_length').val(),
                    tier2_enabled: $('#a1ai-tier2-enabled').is(':checked') ? 1 : 0,
                    analysis_strategy: $('#a1ai-analysis-strategy').val()
                },
                success: function(response) {
                    if (response.success) {
                        $button.text('Saved!');
                        setTimeout(function() {
                            $button.prop('disabled', false).text(originalText);
                        }, 2000);
                    } else {
                        alert('Error saving settings: ' + (response.data || 'Unknown error'));
                        $button.prop('disabled', false).text(originalText);
                    }
                },
                error: function(xhr, status, error) {
                    alert('Error saving settings.');
                    $button.prop('disabled', false).text(originalText);
                }
            });
        });

        // Restore Tier 2 settings from localStorage
        A1AIAdmin.restoreTier2Settings();
    };

    // Start analysis
    A1AIAdmin.startAnalysis = function(type, tier) {
        tier = tier || 1; // Default to tier 1

        const $progress = $('#a1ai-analysis-progress-tier' + tier);
        const $buttons = tier === 1 ?
            $('#a1ai-analyze-all-content-tier1, #a1ai-analyze-recent-content-tier1') :
            $('#a1ai-analyze-all-content-tier2, #a1ai-analyze-recent-content-tier2');

        // Get Tier 2 settings (for strategy)
        const tier2Enabled = $('#a1ai-tier2-enabled').is(':checked');
        const analysisStrategy = $('#a1ai-analysis-strategy').val() || 'tier1_fallback';

        // Save current settings to localStorage before analysis
        localStorage.setItem('a1ai_tier2_enabled', tier2Enabled);
        localStorage.setItem('a1ai_analysis_strategy', analysisStrategy);

        $progress.show();
        $buttons.prop('disabled', true);

        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'a1ai_upgrade_action',
                action_type: 'start_content_analysis',
                nonce: (typeof a1aiUpgradeAjax !== 'undefined' && a1aiUpgradeAjax.nonce) || '',
                analysis_type: type,
                tier: tier || 1,
                analysis_strategy: analysisStrategy
            },
            success: function(response) {
                if (response.success) {
                    alert('Analysis completed! Processed ' + response.data.processed + ' posts.');

                    // Refresh statistics and restore controls
                    A1AIAdmin.loadAnalysisStatistics(tier);
                    A1AIAdmin.resetAnalysisControls(tier);
                } else {
                    alert('Error: ' + response.data);
                    A1AIAdmin.resetAnalysisControls(tier);
                }
            },
            error: function(xhr, status, error) {
                alert('An error occurred while starting analysis.');
                A1AIAdmin.resetAnalysisControls();
            }
        });
    };

    // Monitor analysis progress
    A1AIAdmin.monitorAnalysisProgress = function(taskId) {
        const checkProgress = function() {
            $.ajax({
                url: ajaxurl,
                type: 'POST',
                data: {
                    action: 'a1ai_upgrade_action',
                    action_type: 'check_analysis_progress',
                    nonce: (typeof a1aiUpgradeAjax !== 'undefined' && a1aiUpgradeAjax.nonce) || '',
                    task_id: taskId
                },
                success: function(response) {
                    if (response.success) {
                        const progress = response.data.progress;
                        $('.a1ai-progress-fill').css('width', progress + '%');
                        $('.a1ai-progress-text').text(response.data.message);

                        if (progress >= 100) {
                            A1AIAdmin.resetAnalysisControls();
                            A1AIAdmin.loadAnalysisStatistics();
                        } else {
                            setTimeout(checkProgress, 1000);
                        }
                    } else {
                        A1AIAdmin.resetAnalysisControls();
                    }
                },
                error: function() {
                    A1AIAdmin.resetAnalysisControls();
                }
            });
        };

        checkProgress();
    };

    // Reset analysis controls
    A1AIAdmin.resetAnalysisControls = function(tier) {
        tier = tier || 1;
        $('#a1ai-analysis-progress-tier' + tier).hide();
        const $buttons = tier === 1 ?
            $('#a1ai-analyze-all-content-tier1, #a1ai-analyze-recent-content-tier1') :
            $('#a1ai-analyze-all-content-tier2, #a1ai-analyze-recent-content-tier2');
        $buttons.prop('disabled', false);
        $('#a1ai-analysis-progress-tier' + tier + ' .a1ai-progress-fill').css('width', '0%');

        // Restore Tier 2 settings after reset
        A1AIAdmin.restoreTier2Settings();
    };

    // Restore Tier 2 settings from localStorage
    A1AIAdmin.restoreTier2Settings = function() {
        // Restore Tier 2 enabled state
        const tier2Enabled = localStorage.getItem('a1ai_tier2_enabled');
        if (tier2Enabled === 'true') {
            $('#a1ai-tier2-enabled').prop('checked', true);
            $('#a1ai-analysis-strategy').prop('disabled', false).css('opacity', '1');
            $('.a1ai-tier2-controls').removeClass('tier2-disabled');
            $('#a1ai-analyze-all-content-tier2, #a1ai-analyze-recent-content-tier2, #a1ai-clear-analysis-data-tier2, #a1ai-view-analysis-results-tier2').prop('disabled', false);
        } else {
            $('#a1ai-tier2-enabled').prop('checked', false);
            $('#a1ai-analysis-strategy').prop('disabled', true).css('opacity', '0.5');
            $('.a1ai-tier2-controls').addClass('tier2-disabled');
            $('#a1ai-analyze-all-content-tier2, #a1ai-analyze-recent-content-tier2, #a1ai-clear-analysis-data-tier2, #a1ai-view-analysis-results-tier2').prop('disabled', true);
        }

        // Restore analysis strategy
        const analysisStrategy = localStorage.getItem('a1ai_analysis_strategy');
        if (analysisStrategy) {
            $('#a1ai-analysis-strategy').val(analysisStrategy);
        }
    };

    // Clear analysis data
    A1AIAdmin.clearAnalysisData = function(tier) {
        tier = tier || 1;
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'a1ai_upgrade_action',
                action_type: 'clear_analysis_data',
                nonce: (typeof a1aiUpgradeAjax !== 'undefined' && a1aiUpgradeAjax.nonce) || '',
                tier: tier
            },
            success: function(response) {
                if (response.success) {
                    A1AIAdmin.loadAnalysisStatistics(tier);
                    alert('Tier ' + tier + ' analysis data cleared successfully.');
                } else {
                    alert('Error: ' + response.data);
                }
            }
        });
    };

    // Load analysis statistics
    A1AIAdmin.loadAnalysisStatistics = function(tier) {
        tier = tier || 1;
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'a1ai_upgrade_action',
                action_type: 'get_analysis_statistics',
                nonce: (typeof a1aiUpgradeAjax !== 'undefined' && a1aiUpgradeAjax.nonce) || '',
                tier: tier
            },
            success: function(response) {
                if (response.success) {
                    const suffix = '-tier' + tier;
                    $('#a1ai-total-analyzed' + suffix).text(response.data.total_analyzed || 0);
                    $('#a1ai-pending-analysis' + suffix).text(response.data.pending_analysis || 0);
                    $('#a1ai-coverage' + suffix).text((response.data.coverage || 0) + '%');
                    $('#a1ai-last-analysis' + suffix).text(response.data.last_analysis || '-');
                }
            }
        });
    };

    // View analysis results
    A1AIAdmin.viewAnalysisResults = function(tier) {
        tier = tier || 1;
        $.ajax({
            url: ajaxurl,
            type: 'POST',
            data: {
                action: 'a1ai_upgrade_action',
                action_type: 'get_analysis_results',
                nonce: (typeof a1aiUpgradeAjax !== 'undefined' && a1aiUpgradeAjax.nonce) || '',
                tier: tier
            },
            success: function(response) {
                if (response.success) {
                    A1AIAdmin.displayAnalysisResults(response.data);
                    $('#a1ai-analysis-modal').show();
                } else {
                    alert('Error loading analysis results: ' + (response.data || 'Unknown error'));
                }
            },
            error: function(xhr, status, error) {
                alert('AJAX error loading analysis results.');
            }
        });
    };

    // Display analysis results
    A1AIAdmin.displayAnalysisResults = function(results) {
        let html = '';

        if (results.length === 0) {
            html = '<p>No analysis results found. Click "Analyze All Content" to analyze your website content.</p>';
        } else {
            html = '<style>';
            html += '.a1ai-analysis-table { width: 100%; border-collapse: collapse; margin-top: 10px; }';
            html += '.a1ai-analysis-table th, .a1ai-analysis-table td { border: 1px solid #ddd; padding: 8px; text-align: left; vertical-align: top; }';
            html += '.a1ai-analysis-table th { background-color: #f2f2f2; font-weight: bold; }';
            html += '.a1ai-analysis-table .col-title { width: 20%; }';
            html += '.a1ai-analysis-table .col-type { width: 8%; }';
            html += '.a1ai-analysis-table .col-keywords { width: 30%; }';
            html += '.a1ai-analysis-table .col-topics { width: 25%; }';
            html += '.a1ai-analysis-table .col-sentiment { width: 12%; }';
            html += '.a1ai-analysis-table .col-date { width: 15%; }';
            html += '.a1ai-analysis-table .keywords-cell { font-size: 12px; word-wrap: break-word; }';
            html += '.a1ai-analysis-table .topics-cell { font-size: 12px; }';
            html += '.a1ai-sentiment-positive { color: #0073aa; font-weight: bold; }';
            html += '.a1ai-sentiment-negative { color: #d63638; font-weight: bold; }';
            html += '.a1ai-sentiment-professional { color: #135e96; }';
            html += '</style>';

            html += '<table class="a1ai-analysis-table"><thead><tr>';
            html += '<th class="col-title">Post Title</th>';
            html += '<th class="col-type">Type</th>';
            html += '<th class="col-tier">Tier</th>';
            html += '<th class="col-keywords">Keywords</th>';
            html += '<th class="col-topics">Topics/Categories</th>';
            html += '<th class="col-sentiment">Sentiment</th>';
            html += '<th class="col-date">Analyzed</th>';
            html += '</tr></thead><tbody>';

            results.forEach(function(result) {
                html += '<tr>';
                html += '<td class="col-title"><a href="' + result.edit_url + '" target="_blank" title="Edit post">' + result.title + '</a></td>';
                html += '<td class="col-type">' + result.post_type + '</td>';

                // Tier column with badge styling
                var tierBadge = result.analysis_tier === 'tier2' ?
                    '<span style="background: #0073aa; color: white; padding: 2px 6px; border-radius: 3px; font-size: 11px;">Tier 2</span>' :
                    '<span style="background: #666; color: white; padding: 2px 6px; border-radius: 3px; font-size: 11px;">Tier 1</span>';
                html += '<td class="col-tier">' + tierBadge + '</td>';

                html += '<td class="col-keywords keywords-cell">' + (result.keywords || '-') + '</td>';
                html += '<td class="col-topics topics-cell">' + (result.topics || '-') + '</td>';

                // Add color coding for sentiment
                var sentimentClass = '';
                if (result.sentiment_label && result.sentiment_label.includes('positive')) {
                    sentimentClass = 'a1ai-sentiment-positive';
                } else if (result.sentiment_label && result.sentiment_label.includes('negative')) {
                    sentimentClass = 'a1ai-sentiment-negative';
                } else if (result.sentiment_label === 'professional') {
                    sentimentClass = 'a1ai-sentiment-professional';
                }

                html += '<td class="col-sentiment ' + sentimentClass + '">' + (result.sentiment_label || 'neutral') + '</td>';
                html += '<td class="col-date">' + result.last_analyzed + '</td>';
                html += '</tr>';
            });

            html += '</tbody></table>';
            html += '<p style="margin-top: 15px; font-size: 12px; color: #666;"><strong>Analysis Summary:</strong> ' + results.length + ' pieces of content analyzed. Keywords are ranked by importance and frequency.</p>';
        }

        $('#a1ai-analysis-results-content').html(html);
    };

})(jQuery); 