<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://a1aidigital.com
 * @since      1.0.0
 *
 * @package    A1AI
 * @subpackage A1AI/admin
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and hooks for the admin area.
 *
 * @package    A1AI
 * @subpackage A1AI/admin
 * @author     A1AI <info@a1ai.com>
 */
class A1AI_Admin {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * The data manager instance.
     *
     * @since    1.0.0
     * @access   private
     * @var      A1AI_Data_Manager    $data_manager    Handles database operations.
     */
    private $data_manager;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0.0
     * @param    string                $plugin_name       The name of this plugin.
     * @param    string                $version           The version of this plugin.
     * @param    A1AI_Data_Manager     $data_manager      The data manager instance.
     */
    public function __construct($plugin_name = 'a1ai', $version = '1.3.0', $data_manager = null) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
        $this->data_manager = $data_manager;
        
        // If data manager wasn't passed, create one
        if (null === $this->data_manager) {
            require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-a1ai-data-manager.php';
            $this->data_manager = new A1AI_Data_Manager();
        }

        // Register AJAX handlers
        add_action('wp_ajax_a1ai_get_contact_conversation', array($this, 'get_contact_conversation'));
    }

    /**
     * Register the stylesheets for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_styles($hook) {
        
        // Only load on our plugin pages
        if (!$this->is_plugin_page($hook)) {
            return;
        }
        
        wp_enqueue_style(
            'a1ai-admin',
            plugin_dir_url(__FILE__) . 'css/a1ai-admin.css',
            array(),
            $this->version,
            'all'
        );
        
        // Enqueue adjustments-specific styles
        // phpcs:disable WordPress.Security.NonceVerification.Recommended -- Used only for style enqueuing, not form processing
        if (strpos($hook, 'a1ai') !== false && isset($_GET['page']) && sanitize_text_field(wp_unslash($_GET['page'])) === 'a1ai-chatbots') {
            wp_enqueue_style(
                'a1ai-admin-adjustments',
                plugin_dir_url(__FILE__) . 'css/a1ai-admin-adjustments.css',
                array('a1ai-admin'),
            $this->version,
            'all'
        );
        }
        // phpcs:enable WordPress.Security.NonceVerification.Recommended
    }

    /**
     * Register the JavaScript for the admin area.
     *
     * @since    1.0.0
     */
    public function enqueue_scripts($hook) {
        
        // Only load on our plugin pages
        if (!$this->is_plugin_page($hook)) {
            return;
        }
        
        // Enqueue Chart.js if needed on dashboard
        $dependencies = array('jquery');
        if (strpos($hook, 'page_a1ai') !== false) {
                wp_enqueue_script(
                    'a1ai-chartjs',
                    plugin_dir_url(dirname(__FILE__)) . 'assets/js/chart.min.js',
                    array(),
                '4.4.2',
                    true
                );
                $dependencies[] = 'a1ai-chartjs';
        }

        wp_enqueue_script(
            'a1ai-admin',
            plugin_dir_url(__FILE__) . 'js/a1ai-admin.js',
            $dependencies,
            $this->version,
            true
        );
        
        // Enqueue page-specific scripts
        if (strpos($hook, 'a1ai') !== false) {
            // phpcs:disable WordPress.Security.NonceVerification.Recommended -- Used only for script enqueuing, not form processing
            // Chatbots page (includes adjustments tab)
            if (isset($_GET['page']) && sanitize_text_field(wp_unslash($_GET['page'])) === 'a1ai-chatbots') {
                wp_enqueue_script(
                    'a1ai-admin-chatbots',
                    plugin_dir_url(__FILE__) . 'js/a1ai-admin-chatbots.js',
                    array('jquery', 'a1ai-admin'),
                    $this->version,
                    true
                );

                wp_enqueue_script(
                    'a1ai-admin-adjustments',
                    plugin_dir_url(__FILE__) . 'js/a1ai-admin-adjustments.js',
                    array('jquery', 'a1ai-admin'),
                    $this->version,
                    true
                );
            }
            
            // Chatbot edit page
            if (isset($_GET['page']) && sanitize_text_field(wp_unslash($_GET['page'])) === 'a1ai-chatbots' && isset($_GET['action']) && sanitize_text_field(wp_unslash($_GET['action'])) === 'edit') {
                wp_enqueue_script(
                    'a1ai-admin-chatbot-edit',
                    plugin_dir_url(__FILE__) . 'js/a1ai-admin-chatbot-edit.js',
                    array('jquery', 'a1ai-admin'),
                    $this->version,
                    true
                );
            }
            // phpcs:enable WordPress.Security.NonceVerification.Recommended
        }
        
        // Localize script with necessary data and nonce
        wp_localize_script(
            'a1ai-admin',
            'a1aichAdmin',
            array(
                'ajaxUrl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('a1ai_admin_nonce'),
                'copiedText' => esc_html__('Copied!', 'a1ai-chatbot'),
                'defaultGdprText' => esc_html__('By clicking "I Agree", you consent to the use of AI-powered chat services and acknowledge that your conversations may be processed to provide you with assistance. You can withdraw your consent at any time by closing this chat.', 'a1ai-chatbot'),
                'strings' => array(
                    'saveSuccess' => esc_html__('Settings saved successfully.', 'a1ai-chatbot'),
                    'saveError' => esc_html__('Error saving settings.', 'a1ai-chatbot'),
                    'deleteSuccess' => esc_html__('Chatbot deleted successfully.', 'a1ai-chatbot'),
                    'deleteError' => esc_html__('Error deleting chatbot.', 'a1ai-chatbot'),
                    'confirmDelete' => esc_html__('Are you sure you want to delete this chatbot? This action cannot be undone.', 'a1ai-chatbot'),
                )
            )
        );
    }

    /**
     * Check if current page is a plugin page.
     *
     * @since    1.0.0
     * @param    string    $hook    Current admin page.
     * @return   bool                True if it's a plugin page.
     */
    private function is_plugin_page($hook) {
        // Always load admin scripts on a1ai pages
        if (strpos($hook, 'a1ai') !== false) {
            return true;
        }
        
        // Also check if we're on a plugin page by checking the page parameter
        // phpcs:disable WordPress.Security.NonceVerification.Recommended -- Admin page protected by capabilities
        if (isset($_GET['page'])) {
            $page = sanitize_text_field(wp_unslash($_GET['page']));
            if (strpos($page, 'a1ai') === 0) {
                return true;
            }
        }
        // phpcs:enable WordPress.Security.NonceVerification.Recommended
        
        return false;
    }

    /**
     * Register the admin menu.
     *
     * @since    1.0.0
     */
    public function register_admin_menu() {
        // Include plugin.php to use is_plugin_active()
        if (!function_exists('is_plugin_active')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        // Main menu
        add_menu_page(
            __('A1AI Chatbot', 'a1ai-chatbot'),
            __('A1AI Chatbot', 'a1ai-chatbot'),
            'manage_options',
            'a1ai',
            array($this, 'render_dashboard_page'),
            'dashicons-format-chat',
            30
        );
        
        // Submenu - Dashboard
        add_submenu_page(
            'a1ai',
            __('Dashboard', 'a1ai-chatbot'),
            __('Dashboard', 'a1ai-chatbot'),
            'manage_options',
            'a1ai',
            array($this, 'render_dashboard_page')
        );
        
        // Submenu - Chatbots
        add_submenu_page(
            'a1ai',
            __('Chatbots', 'a1ai-chatbot'),
            __('Chatbots', 'a1ai-chatbot'),
            'manage_options',
            'a1ai-chatbots',
            array($this, 'render_chatbots_page')
        );
        

        
        // Submenu - Settings
        add_submenu_page(
            'a1ai',
            __('Settings', 'a1ai-chatbot'),
            __('Settings', 'a1ai-chatbot'),
            'manage_options',
            'a1ai-settings',
            array($this, 'render_settings_page')
        );

        // Submenu - Contact Requests (Premium Feature)
        // Only show if upgrade plugin is active with valid license
        if (class_exists('A1AI_Upgrade_Helper') && A1AI_Upgrade_Helper::is_upgrade_active()) {
            add_submenu_page(
                'a1ai',
                __('Contact Requests', 'a1ai-chatbot'),
                __('Contact Requests', 'a1ai-chatbot'),
                'manage_options',
                'a1ai-contact-requests',
                array($this, 'display_contact_requests_page')
            );
        }
    }

    /**
     * Add settings link to plugin list.
     *
     * @since    1.0.0
     * @param    array    $links    Existing plugin action links.
     * @return   array              Modified plugin action links.
     */
    public function add_settings_link($links) {
        $settings_link = '<a href="' . admin_url('admin.php?page=a1ai-settings') . '">' . __('Settings', 'a1ai-chatbot') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }

    /**
     * Render the dashboard page.
     *
     * @since    1.0.0
     */
    public function render_dashboard_page() {
        // Pass necessary data to the template
        $data_manager = $this->data_manager;
        require_once plugin_dir_path(__FILE__) . 'partials/a1ai-admin-dashboard.php';
    }

    /**
     * Render the chatbots page.
     *
     * @since    1.0.0
     */
    public function render_chatbots_page() {
        // Verify nonce if action is present
        if (isset($_GET['action']) && in_array($_GET['action'], array('edit', 'delete'))) {
            if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_key($_GET['_wpnonce']), 'a1ai_chatbot_action')) {
                wp_die(esc_html__('Security check failed.', 'a1ai-chatbot'));
            }
        }
        
        $action = isset($_GET['action']) ? sanitize_text_field(wp_unslash($_GET['action'])) : '';
        
        if ('add' === $action || 'edit' === $action) {
            // Pass necessary data to the template
            $data_manager = $this->data_manager;
            require_once plugin_dir_path(__FILE__) . 'partials/a1ai-admin-chatbot-edit.php';
        } else {
            // Pass necessary data to the template
            $data_manager = $this->data_manager;
            require_once plugin_dir_path(__FILE__) . 'partials/a1ai-admin-chatbots.php';
        }
    }

    /**
     * Render the settings page.
     *
     * @since    1.0.0
     */
    public function render_settings_page() {
        // Pass necessary data to the template
        $data_manager = $this->data_manager;
        require_once plugin_dir_path(__FILE__) . 'partials/a1ai-admin-settings.php';
    }


    
    /**
     * Handle admin AJAX requests.
     *
     * @since    1.0.0
     */
    public function handle_admin_ajax() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_key($_POST['nonce']), 'a1ai_admin_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'a1ai-chatbot')));
        }
        
        // Check capabilities
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('You do not have sufficient permissions.', 'a1ai-chatbot')));
        }

        // Handle different actions
        $action_type = isset($_POST['action_type']) ? sanitize_text_field(wp_unslash($_POST['action_type'])) : '';

        // Rate limiting for AI assistant chat to prevent API abuse
        if ('assistant_chat' === $action_type) {
            $user_id = get_current_user_id();
            $rate_limit_key = 'a1ai_admin_assistant_limit_' . $user_id;
            $request_count = get_transient($rate_limit_key);
            $rate_limit = apply_filters('a1ai_rate_limit_assistant_per_minute', 20); // Lower limit for admin assistant

            if ($request_count && $request_count >= $rate_limit) {
                wp_send_json_error(array('message' => __('Rate limit exceeded. Please wait before sending more messages.', 'a1ai-chatbot')));
            }

            // Increment request count
            set_transient($rate_limit_key, ($request_count ? $request_count + 1 : 1), MINUTE_IN_SECONDS);
        }
        
        switch ($action_type) {
            case 'save_settings':
                $this->save_settings();
                break;
                
            case 'save_chatbot':
                $this->save_chatbot();
                break;
                
            case 'delete_chatbot':
                $this->delete_chatbot();
                break;
                
            case 'get_analytics':
                $this->get_analytics();
                break;
                
            case 'get_conversation':
                $this->get_conversation();
                break;
                
            case 'delete_conversation':
                $this->delete_conversation();
                break;
                
            case 'save_adjustment':
                $this->save_adjustment();
                break;
                
            case 'delete_adjustment':
                $this->delete_adjustment();
                break;
                
            case 'save_action_guidance':
                $this->save_action_guidance();
                break;
                
            case 'save_repetition_prevention':
                $this->save_repetition_prevention();
                break;
                
            case 'submit_contact_request':
                $this->submit_contact_request();
                break;
                
            case 'test_email':
                $this->test_email();
                break;
                
            case 'assistant_chat':
                $this->handle_assistant_chat();
                break;
                
            case 'get_export_data':
                $this->get_export_data();
                break;

            case 'get_trends_data':
                $this->get_trends_data();
                break;

            default:
                wp_send_json_error(array('message' => __('Invalid action.', 'a1ai-chatbot')));
                break;
        }
    }
    
    /**
     * Save plugin settings.
     *
     * @since    1.0.0
     */
    private function save_settings() {
        // Nonce and capabilities already verified in handle_admin_ajax()
        
        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in handle_admin_ajax()
        
        // General settings
        if (isset($_POST['a1ai_openai_api_key'])) {
            update_option('a1ai_openai_api_key', sanitize_text_field(wp_unslash($_POST['a1ai_openai_api_key'])));
        }
        
        if (isset($_POST['a1ai_edd_api_key'])) {
            $license_key = sanitize_text_field(wp_unslash($_POST['a1ai_edd_api_key']));
            update_option('a1ai_edd_api_key', $license_key);

            // Also save to upgrade plugin license option and activate
            if (!empty($license_key) && class_exists('A1AI_Upgrade_Helper') && A1AI_Upgrade_Helper::is_plugin_installed()) {
                // Update the upgrade plugin's license key
                update_option('a1ai_upgrade_license', $license_key);

                // Attempt to activate the license automatically
                $license_manager_path = WP_PLUGIN_DIR . '/a1ai-chatbot-upgrade/includes/class-a1ai-upgrade-edd-license-manager.php';
                if (file_exists($license_manager_path)) {
                    require_once $license_manager_path;
                    $license_manager = new A1AI_Upgrade_EDD_License_Manager();
                    $license_manager->activate_license($license_key);
                }
            }
        }
        
        // AI Settings
        if (isset($_POST['a1ai_default_model'])) {
            update_option('a1ai_default_model', sanitize_text_field(wp_unslash($_POST['a1ai_default_model'])));
        }
        
        if (isset($_POST['a1ai_default_temperature'])) {
            $temperature = min(max(0, floatval($_POST['a1ai_default_temperature'])), 2);
            update_option('a1ai_default_temperature', $temperature);
        }
        
        if (isset($_POST['a1ai_default_max_tokens'])) {
            $max_tokens = min(max(100, intval($_POST['a1ai_default_max_tokens'])), 4000);
            update_option('a1ai_default_max_tokens', $max_tokens);
        }
        
        // Global Chatbot Settings
        $global_enabled = isset($_POST['a1ai_global_chatbot_enabled']) ? (bool) $_POST['a1ai_global_chatbot_enabled'] : false;
        update_option('a1ai_global_chatbot_enabled', $global_enabled);
        
        if (isset($_POST['a1ai_global_chatbot_id'])) {
            update_option('a1ai_global_chatbot_id', intval($_POST['a1ai_global_chatbot_id']));
        }
        
        if (isset($_POST['a1ai_global_chatbot_position'])) {
            update_option('a1ai_global_chatbot_position', sanitize_text_field(wp_unslash($_POST['a1ai_global_chatbot_position'])));
        }
        
        // Advanced Settings
        $keep_history = isset($_POST['a1ai_keep_conversation_history']) ? (bool) $_POST['a1ai_keep_conversation_history'] : false;
        update_option('a1ai_keep_conversation_history', $keep_history);
        
        if (isset($_POST['a1ai_conversation_retention_days'])) {
            $retention_days = min(max(1, intval($_POST['a1ai_conversation_retention_days'])), 365);
            update_option('a1ai_conversation_retention_days', $retention_days);
        }
        
        $keep_data = isset($_POST['a1ai_keep_data_on_uninstall']) ? (bool) $_POST['a1ai_keep_data_on_uninstall'] : false;
        update_option('a1ai_keep_data_on_uninstall', $keep_data);
        
        $show_powered_by = isset($_POST['a1ai_show_powered_by']) ? (bool) $_POST['a1ai_show_powered_by'] : false;
        update_option('a1ai_show_powered_by', $show_powered_by);

        // Content Awareness Settings (from upgrade plugin)
        if (isset($_POST['a1ai_auto_analyze']) || isset($_POST['a1ai_generate_sitemap']) ||
            isset($_POST['a1ai_analysis_frequency']) || isset($_POST['a1ai_min_content_length']) ||
            isset($_POST['a1ai_tier2_enabled']) || isset($_POST['a1ai_analysis_strategy'])) {

            $content_settings = get_option('a1ai_content_awareness_settings', array());

            if (isset($_POST['a1ai_auto_analyze'])) {
                $content_settings['auto_analyze'] = (bool) $_POST['a1ai_auto_analyze'] ? 1 : 0;
            }
            if (isset($_POST['a1ai_generate_sitemap'])) {
                $content_settings['generate_sitemap'] = (bool) $_POST['a1ai_generate_sitemap'] ? 1 : 0;
            }
            if (isset($_POST['a1ai_analysis_frequency'])) {
                $content_settings['analysis_frequency'] = sanitize_text_field(wp_unslash($_POST['a1ai_analysis_frequency']));
            }
            if (isset($_POST['a1ai_min_content_length'])) {
                $content_settings['min_content_length'] = intval($_POST['a1ai_min_content_length']);
            }
            if (isset($_POST['a1ai_tier2_enabled'])) {
                $content_settings['tier2_enabled'] = (bool) $_POST['a1ai_tier2_enabled'] ? 1 : 0;
            }
            if (isset($_POST['a1ai_analysis_strategy'])) {
                $content_settings['analysis_strategy'] = sanitize_text_field(wp_unslash($_POST['a1ai_analysis_strategy']));
            }

            update_option('a1ai_content_awareness_settings', $content_settings);
        }

        // Business Information Settings
        if (isset($_POST['business_info']) && is_array($_POST['business_info'])) {
            $chatbot_id = isset($_POST['business_info_chatbot_id']) ? intval($_POST['business_info_chatbot_id']) : 0;

            if ($chatbot_id > 0) {
                // Process business info data
                $business_data = array();

                if (isset($_POST['business_info']['business_name'])) {
                    $business_data['business_name'] = sanitize_text_field(wp_unslash($_POST['business_info']['business_name']));
                }

                if (isset($_POST['business_info']['tagline'])) {
                    $business_data['tagline'] = sanitize_textarea_field(wp_unslash($_POST['business_info']['tagline']));
                }

                if (isset($_POST['business_info']['business_type'])) {
                    $business_data['business_type'] = sanitize_text_field(wp_unslash($_POST['business_info']['business_type']));
                }

                // Process services (convert line-separated text to array)
                if (isset($_POST['business_info']['services'])) {
                    $services_text = sanitize_textarea_field(wp_unslash($_POST['business_info']['services']));
                    $services_array = array_filter(array_map('trim', explode("\n", $services_text)));
                    $business_data['services'] = $services_array;
                }

                // Process products (convert line-separated text to array)
                if (isset($_POST['business_info']['products'])) {
                    $products_text = sanitize_textarea_field(wp_unslash($_POST['business_info']['products']));
                    $products_array = array_filter(array_map('trim', explode("\n", $products_text)));
                    $business_data['products'] = $products_array;
                }

                // Process hours (already an array from form)
                if (isset($_POST['business_info']['hours']) && is_array($_POST['business_info']['hours'])) {
                    $hours_array = array();
                    // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Array will be sanitized in loop
                    $hours_input = wp_unslash($_POST['business_info']['hours']); // Unslash the array first
                    foreach ($hours_input as $day => $time) {
                        $hours_array[sanitize_key($day)] = sanitize_text_field($time);
                    }
                    $business_data['hours'] = $hours_array;
                }

                if (isset($_POST['business_info']['address'])) {
                    $business_data['address'] = sanitize_textarea_field(wp_unslash($_POST['business_info']['address']));
                }

                if (isset($_POST['business_info']['phone'])) {
                    $business_data['phone'] = sanitize_text_field(wp_unslash($_POST['business_info']['phone']));
                }

                if (isset($_POST['business_info']['email'])) {
                    $business_data['email'] = sanitize_email(wp_unslash($_POST['business_info']['email']));
                }

                if (isset($_POST['business_info']['website'])) {
                    $business_data['website'] = esc_url_raw(wp_unslash($_POST['business_info']['website']));
                }

                if (isset($_POST['business_info']['policies'])) {
                    $business_data['policies'] = sanitize_textarea_field(wp_unslash($_POST['business_info']['policies']));
                }

                // Save business info using data manager
                $this->data_manager->save_business_info($chatbot_id, $business_data);
            }
        }

        // phpcs:enable WordPress.Security.NonceVerification.Missing

        wp_send_json_success(array('message' => __('Settings saved successfully.', 'a1ai-chatbot')));
    }
    
    /**
     * Save chatbot data.
     *
     * @since    1.0.0
     */
    private function save_chatbot() {
        // Nonce and capabilities already verified in handle_admin_ajax()
        
        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in handle_admin_ajax()
        
        // Get chatbot ID and name
        $chatbot_id = isset($_POST['chatbot_id']) ? intval($_POST['chatbot_id']) : 0;
        $chatbot_name = isset($_POST['chatbot_name']) ? sanitize_text_field(wp_unslash($_POST['chatbot_name'])) : '';
        
        // Validate required fields
        if (empty($chatbot_name)) {
            wp_send_json_error(array('message' => __('Chatbot name is required.', 'a1ai-chatbot')));
        }
        
        // Get and sanitize settings array
        $raw_settings = array();
        if (isset($_POST['settings']) && is_array($_POST['settings'])) {
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
            $raw_settings = wp_kses_post_deep(wp_unslash($_POST['settings']));
        }
        $settings = $this->sanitize_settings($raw_settings);
        
        // Save chatbot
        $result = $this->data_manager->save_chatbot($chatbot_id, $chatbot_name, $settings);
        
        // Save system prompt and custom instructions to dedicated tables
        if ($result !== false) {
            $final_chatbot_id = $chatbot_id > 0 ? $chatbot_id : $result;
            
            // Handle system prompt - save if provided, deactivate if empty
            if (!empty($settings['system_prompt'])) {
                $this->data_manager->save_custom_instruction(
                    $final_chatbot_id,
                    'System Prompt',
                    $settings['system_prompt'],
                    'system',
                    0
                );
            } else {
                // Deactivate existing system prompt if none provided
                $existing_system_instructions = $this->data_manager->get_custom_instructions($final_chatbot_id, 'system');
                if (!empty($existing_system_instructions)) {
                    foreach ($existing_system_instructions as $instruction) {
                        if ($instruction['instruction_name'] === 'System Prompt') {
                            $this->data_manager->update_custom_instruction($instruction['id'], array('is_active' => 0));
                            break;
                        }
                    }
                }
            }
            
            // Save custom instructions if provided
            if (!empty($settings['custom_instructions'])) {
                $this->data_manager->save_custom_instruction(
                    $final_chatbot_id,
                    'Custom Instructions',
                    $settings['custom_instructions'],
                    'general',
                    1
                );
            } else {
                // Clear existing custom instructions if none provided
                $existing_instructions = $this->data_manager->get_custom_instructions($final_chatbot_id);
                if (!empty($existing_instructions)) {
                    foreach ($existing_instructions as $instruction) {
                        // Only deactivate general instructions, keep system prompt
                        if ($instruction['instruction_type'] !== 'system') {
                            $this->data_manager->update_custom_instruction($instruction['id'], array('is_active' => 0));
                        }
                    }
                }
            }
        }
        
        if ($result === false) {
            wp_send_json_error(array('message' => __('Error saving chatbot.', 'a1ai-chatbot')));
        }
        
        $success_message = $chatbot_id > 0 
            ? __('Chatbot updated successfully.', 'a1ai-chatbot') 
            : __('Chatbot created successfully.', 'a1ai-chatbot');
            
        $response_data = array('message' => $success_message);
        
        // If this is a new chatbot, add the redirect URL to the edit page
        if ($chatbot_id === 0) {
            $edit_url = add_query_arg(
                array(
                    'page' => 'a1ai-chatbots',
                    'action' => 'edit',
                    'id' => $result,
                    '_wpnonce' => wp_create_nonce('a1ai_chatbot_action')
                ),
                admin_url('admin.php')
            );
            $response_data['redirect'] = $edit_url;
        }
        
        // phpcs:enable WordPress.Security.NonceVerification.Missing
        
        wp_send_json_success($response_data);
    }
    
    /**
     * Delete a chatbot.
     *
     * @since    1.0.0
     */
    private function delete_chatbot() {
        // Nonce and capabilities already verified in handle_admin_ajax()
        
        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in handle_admin_ajax()
        
        $chatbot_id = isset($_POST['chatbot_id']) ? intval($_POST['chatbot_id']) : 0;
        
        if ($chatbot_id <= 0) {
            wp_send_json_error(array('message' => __('Invalid chatbot ID.', 'a1ai-chatbot')));
        }
        
        // Check if this is the global chatbot
        $global_chatbot_id = get_option('a1ai_global_chatbot_id', 0);
        if ($global_chatbot_id == $chatbot_id) {
            update_option('a1ai_global_chatbot_id', 0);
            update_option('a1ai_global_chatbot_enabled', false);
        }
        
        $result = $this->data_manager->delete_chatbot($chatbot_id);
        
        // phpcs:enable WordPress.Security.NonceVerification.Missing
        
        if ($result) {
            wp_send_json_success(array('message' => __('Chatbot deleted successfully.', 'a1ai-chatbot')));
        } else {
            wp_send_json_error(array('message' => __('Error deleting chatbot.', 'a1ai-chatbot')));
        }
    }
    
    /**
     * Get analytics data for charts.
     *
     * @since    1.0.0
     */
    private function get_analytics() {
        // Nonce and capabilities already verified in handle_admin_ajax()
        
        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in handle_admin_ajax()
        
        $period = isset($_POST['period']) ? sanitize_text_field(wp_unslash($_POST['period'])) : 'month';
        $valid_periods = array('day', 'week', 'month', 'year');
        
        if (!in_array($period, $valid_periods)) {
            $period = 'month';
        }
        
        $stats = $this->data_manager->get_usage_stats($period);
        
        // Format data for Chart.js
        $labels = array();
        $messages_data = array();
        $conversations_data = array();
        
        // Process message counts
        foreach ($stats['messages'] as $message) {
            $labels[] = $message['date'];
            $messages_data[] = intval($message['count']);
        }
        
        // Process conversation counts
        // Create conversation data array with same labels
        foreach ($labels as $label) {
            $count = 0;
            
            // Find matching conversation count
            foreach ($stats['conversations'] as $conversation) {
                if ($conversation['date'] === $label) {
                    $count = intval($conversation['count']);
                    break;
                }
            }
            
            $conversations_data[] = $count;
        }
        
        $chart_data = array(
            'labels' => $labels,
            'messages' => $messages_data,
            'conversations' => $conversations_data,
            'totals' => $stats['totals'],
        );
        
        // phpcs:enable WordPress.Security.NonceVerification.Missing
        
        wp_send_json_success($chart_data);
    }

    /**
     * Sanitize settings array.
     * 
     * @since    1.0.0
     * @param    array    $settings    The settings array to sanitize.
     * @return   array                 The sanitized settings array.
     */
    private function sanitize_settings($settings) {
        if (!is_array($settings)) {
            return array();
        }
        
        $sanitized_settings = array();
        
        // Text fields
        $text_fields = array('chat_bubble_text', 'chatbot_title', 'welcome_message', 'chat_width', 'chat_height', 'primary_color', 'font_size', 'model', 'placeholder');
        foreach ($text_fields as $field) {
            $sanitized_settings[$field] = isset($settings[$field]) ? sanitize_text_field(wp_unslash($settings[$field])) : '';
        }
        
        // Email field with validation
        if (isset($settings['notification_email']) && !empty($settings['notification_email'])) {
            $email = sanitize_email(wp_unslash($settings['notification_email']));
            if (is_email($email)) {
                $sanitized_settings['notification_email'] = $email;
            } else {
                $sanitized_settings['notification_email'] = '';
            }
        } else {
            $sanitized_settings['notification_email'] = '';
        }
        
        // Textarea fields
        $textarea_fields = array('system_prompt', 'custom_instructions', 'gdpr_consent_text', 'primary_action', 'secondary_actions', 'conversation_stages', 'variation_strategies');
        foreach ($textarea_fields as $field) {
            $sanitized_settings[$field] = isset($settings[$field]) ? sanitize_textarea_field(wp_unslash($settings[$field])) : '';
        }

        // Select/dropdown fields for conversation settings
        $sanitized_settings['action_guidance_tone'] = isset($settings['action_guidance_tone']) && in_array($settings['action_guidance_tone'], array('subtle', 'moderate', 'direct')) ? $settings['action_guidance_tone'] : 'moderate';

        // Boolean fields - explicitly set to true only if value is "1"
        $sanitized_settings['gdpr_enabled'] = (isset($settings['gdpr_enabled']) && $settings['gdpr_enabled'] === "1") ? true : false;
        $sanitized_settings['conversation_tracking'] = (isset($settings['conversation_tracking']) && $settings['conversation_tracking'] === "1") ? true : false;
        $sanitized_settings['action_tracking'] = (isset($settings['action_tracking']) && $settings['action_tracking'] === "1") ? true : false;

        // Numeric fields
        $sanitized_settings['temperature'] = isset($settings['temperature']) ? min(max(0, floatval($settings['temperature'])), 2) : 0.7;
        $sanitized_settings['max_tokens'] = isset($settings['max_tokens']) ? min(max(100, intval($settings['max_tokens'])), 4000) : 1000;
        $sanitized_settings['repetition_threshold'] = isset($settings['repetition_threshold']) ? min(max(1, intval($settings['repetition_threshold'])), 10) : 6;
        
        return $sanitized_settings;
    }

    /**
     * Get conversation details for admin view.
     *
     * @since    1.0.0
     */
    private function get_conversation() {
        // Nonce and capabilities already verified in handle_admin_ajax()
        
        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in handle_admin_ajax()
        
        $conversation_id = isset($_POST['conversation_id']) ? intval($_POST['conversation_id']) : 0;
        
        if ($conversation_id <= 0) {
            wp_send_json_error(array('message' => __('Invalid conversation ID.', 'a1ai-chatbot')));
        }
        
        // Get conversation details
        $conversation = $this->data_manager->get_conversation($conversation_id);
        
        if (empty($conversation)) {
            wp_send_json_error(array('message' => __('Conversation not found.', 'a1ai-chatbot')));
        }
        
        // Get messages for this conversation
        $messages = $this->data_manager->get_messages($conversation_id);
        
        // Get chatbot name
        $chatbot = $this->data_manager->get_chatbot($conversation['chatbot_id']);
        $chatbot_name = $chatbot ? $chatbot['name'] : __('Unknown Chatbot', 'a1ai-chatbot');
        
        // Get user name
        $user_name = __('Guest', 'a1ai-chatbot');
        if ($conversation['user_id'] > 0) {
            $user = get_user_by('id', $conversation['user_id']);
            if ($user) {
                $user_name = $user->display_name;
            }
        }
        
        // Add additional info to conversation
        $conversation['chatbot_name'] = $chatbot_name;
        $conversation['user_name'] = $user_name;
        
        $response_data = array(
            'conversation' => $conversation,
            'messages' => $messages,
        );
        
        // phpcs:enable WordPress.Security.NonceVerification.Missing
        
        wp_send_json_success($response_data);
    }

    /**
     * Delete a conversation.
     *
     * @since    1.0.0
     */
    private function delete_conversation() {
        // Nonce and capabilities already verified in handle_admin_ajax()
        
        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in handle_admin_ajax()
        
        $conversation_id = isset($_POST['conversation_id']) ? intval($_POST['conversation_id']) : 0;
        
        if ($conversation_id <= 0) {
            wp_send_json_error(array('message' => __('Invalid conversation ID.', 'a1ai-chatbot')));
        }
        
        // Get conversation to verify it exists
        $conversation = $this->data_manager->get_conversation($conversation_id);
        
        if (empty($conversation)) {
            wp_send_json_error(array('message' => __('Conversation not found.', 'a1ai-chatbot')));
        }
        
        // Delete messages first (foreign key constraint)
        $deleted_messages = $this->data_manager->delete_messages_by_conversation($conversation_id);
        
        // Delete conversation
        $deleted_conversation = $this->data_manager->delete_conversation($conversation_id);
        
        if (false === $deleted_conversation) {
            wp_send_json_error(array('message' => __('Failed to delete conversation.', 'a1ai-chatbot')));
        }
        
        // phpcs:enable WordPress.Security.NonceVerification.Missing
        
        wp_send_json_success(array('message' => __('Conversation deleted successfully.', 'a1ai-chatbot')));
    }

    /**
     * Save adjustment.
     *
     * @since    1.0.0
     */
    private function save_adjustment() {
        // Nonce and capabilities already verified in handle_admin_ajax()
        
        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in handle_admin_ajax()
        
        $chatbot_id = isset($_POST['chatbot_id']) ? intval($_POST['chatbot_id']) : 0;
        $user_input = isset($_POST['user_input']) ? sanitize_textarea_field(wp_unslash($_POST['user_input'])) : '';
        $ai_output = isset($_POST['ai_output']) ? sanitize_textarea_field(wp_unslash($_POST['ai_output'])) : '';
        $ui_match_type = isset($_POST['ui_match_type']) ? sanitize_text_field(wp_unslash($_POST['ui_match_type'])) : 'Broad';
        $ai_match_type = isset($_POST['ai_match_type']) ? sanitize_text_field(wp_unslash($_POST['ai_match_type'])) : 'Broad';
        
        if ($chatbot_id <= 0) {
            wp_send_json_error(array('message' => __('Invalid chatbot ID.', 'a1ai-chatbot')));
        }
        
        if (empty($user_input) || empty($ai_output)) {
            wp_send_json_error(array('message' => __('User input and AI output are required.', 'a1ai-chatbot')));
        }
        
        // Validate match types
        $valid_match_types = array('Flexible', 'Broad', 'Phrase', 'Exact');
        if (!in_array($ui_match_type, $valid_match_types, true)) {
            $ui_match_type = 'Broad';
        }
        if (!in_array($ai_match_type, $valid_match_types, true)) {
            $ai_match_type = 'Broad';
        }
        
        // Save adjustment
        $result = $this->data_manager->add_adjustment($chatbot_id, $user_input, $ai_output, $ui_match_type, $ai_match_type);
        
        if (false === $result) {
            wp_send_json_error(array('message' => __('Failed to save adjustment.', 'a1ai-chatbot')));
        }
        
        // phpcs:enable WordPress.Security.NonceVerification.Missing
        
        wp_send_json_success(array(
            'message' => __('Adjustment saved successfully.', 'a1ai-chatbot'),
            'adjustment_id' => $result
        ));
    }

    /**
     * Delete adjustment.
     *
     * @since    1.0.0
     */
    private function delete_adjustment() {
        // Nonce and capabilities already verified in handle_admin_ajax()
        
        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in handle_admin_ajax()
        
        $adjustment_id = isset($_POST['adjustment_id']) ? intval($_POST['adjustment_id']) : 0;
        
        if ($adjustment_id <= 0) {
            wp_send_json_error(array('message' => __('Invalid adjustment ID.', 'a1ai-chatbot')));
        }
        
        // Get adjustment to verify it exists
        $adjustment = $this->data_manager->get_adjustment($adjustment_id);
        
        if (empty($adjustment)) {
            wp_send_json_error(array('message' => __('Adjustment not found.', 'a1ai-chatbot')));
        }
        
        // Delete adjustment
        $result = $this->data_manager->delete_adjustment($adjustment_id);
        
        if (false === $result) {
            wp_send_json_error(array('message' => __('Failed to delete adjustment.', 'a1ai-chatbot')));
        }
        
        // phpcs:enable WordPress.Security.NonceVerification.Missing
        
        wp_send_json_success(array('message' => __('Adjustment deleted successfully.', 'a1ai-chatbot')));
    }

    /**
     * Save action guidance settings.
     *
     * @since    1.4.0
     */
    private function save_action_guidance() {
        // Nonce and capabilities already verified in handle_admin_ajax()
        
        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in handle_admin_ajax()
        
        $chatbot_id = isset($_POST['chatbot_id']) ? intval($_POST['chatbot_id']) : 0;
        
        if ($chatbot_id <= 0) {
            wp_send_json_error(array('message' => __('Invalid chatbot ID.', 'a1ai-chatbot')));
        }
        
        // Get current settings
        $action_guidance_settings = get_option('a1ai_action_guidance_settings', array());
        $conversation_settings = get_option('a1ai_conversation_settings', array());
        
        // Update action guidance settings
        $action_guidance_settings[$chatbot_id] = array(
            'primary_action' => isset($_POST['primary_action']) ? sanitize_textarea_field(wp_unslash($_POST['primary_action'])) : '',
            'secondary_actions' => isset($_POST['secondary_actions']) ? sanitize_textarea_field(wp_unslash($_POST['secondary_actions'])) : '',
        );
        
        // Update conversation settings
        $conversation_settings[$chatbot_id] = array_merge(
            $conversation_settings[$chatbot_id] ?? array(),
            array(
                'enable_stage_detection' => isset($_POST['enable_stage_detection']),
                'action_stage_threshold' => isset($_POST['action_stage_threshold']) ? intval($_POST['action_stage_threshold']) : 5,
            )
        );
        
        // Save settings
        $result1 = update_option('a1ai_action_guidance_settings', $action_guidance_settings);
        $result2 = update_option('a1ai_conversation_settings', $conversation_settings);
        
        if ($result1 && $result2) {
            wp_send_json_success(array('message' => __('Action guidance settings saved successfully.', 'a1ai-chatbot')));
        } else {
            wp_send_json_error(array('message' => __('Failed to save action guidance settings.', 'a1ai-chatbot')));
        }
        
        // phpcs:enable WordPress.Security.NonceVerification.Missing
    }

    /**
     * Save repetition prevention settings.
     *
     * @since    1.4.0
     */
    private function save_repetition_prevention() {
        // Nonce and capabilities already verified in handle_admin_ajax()
        
        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in handle_admin_ajax()
        
        $chatbot_id = isset($_POST['chatbot_id']) ? intval($_POST['chatbot_id']) : 0;
        
        if ($chatbot_id <= 0) {
            wp_send_json_error(array('message' => __('Invalid chatbot ID.', 'a1ai-chatbot')));
        }
        
        // Get current settings
        $conversation_settings = get_option('a1ai_conversation_settings', array());
        
        // Update repetition prevention settings
        $conversation_settings[$chatbot_id] = array_merge(
            $conversation_settings[$chatbot_id] ?? array(),
            array(
                'similarity_threshold' => isset($_POST['similarity_threshold']) ? floatval($_POST['similarity_threshold']) : 0.7,
                'repetition_action' => isset($_POST['repetition_action']) ? sanitize_text_field(wp_unslash($_POST['repetition_action'])) : 'suggest_alternatives',
                'enable_repetition_detection' => isset($_POST['enable_repetition_detection']),
            )
        );
        
        // Save settings
        $result = update_option('a1ai_conversation_settings', $conversation_settings);
        
        if ($result) {
            wp_send_json_success(array('message' => __('Repetition prevention settings saved successfully.', 'a1ai-chatbot')));
        } else {
            wp_send_json_error(array('message' => __('Failed to save repetition prevention settings.', 'a1ai-chatbot')));
        }
        
        // phpcs:enable WordPress.Security.NonceVerification.Missing
    }

    /**
     * Submit contact request from AI assistant.
     *
     * @since    1.4.0
     */
    private function submit_contact_request() {
        // Nonce and capabilities already verified in handle_admin_ajax()
        
        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in handle_admin_ajax()
        
        // Debug logging - only in development
        if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
            // Development logging only
        }
        
        $name = isset($_POST['name']) ? sanitize_text_field(wp_unslash($_POST['name'])) : '';
        $email = isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '';
        $phone = isset($_POST['phone']) ? sanitize_text_field(wp_unslash($_POST['phone'])) : '';
        $message = isset($_POST['message']) ? sanitize_textarea_field(wp_unslash($_POST['message'])) : '';
        $conversation_id = isset($_POST['conversation_id']) ? sanitize_text_field(wp_unslash($_POST['conversation_id'])) : '';

        // Debug logging - only in development
        if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
            // Development logging only
        }

        if (empty($name) || empty($email) || empty($phone)) {
            if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                // Development logging only
            }
            wp_send_json_error(array('message' => __('Please fill in all required fields (Name, Email, Phone).', 'a1ai-chatbot')));
        }

        // Store contact request in database as fallback
        $contact_data = array(
            'name' => $name,
            'email' => $email,
            'phone' => $phone,
            'message' => $message,
            'conversation_id' => $conversation_id,
            'timestamp' => current_time('mysql'),
            'status' => 'pending',
            'type' => 'contact_request' // Mark as contact request (not review)
        );
        
        // Try to store in options as a simple fallback
        $existing_requests = get_option('a1ai_contact_requests', array());
        $existing_requests[] = $contact_data;
        update_option('a1ai_contact_requests', $existing_requests);

        $subject = __('New Contact Request from A1AI Chatbot Assistant -', 'a1ai-chatbot') . ' ' . $name;
        $body = "New contact request from the A1AI Chatbot Assistant:\n\n";
        $body .= "Name: " . $name . "\n";
        $body .= "Email: " . $email . "\n";
        $body .= "Phone: " . $phone . "\n";
        $body .= "Message:\n" . $message . "\n\n";
        $body .= "This request was submitted through the A1AI Chatbot Assistant on " . get_site_url() . "\n";
        $body .= "Timestamp: " . current_time('mysql') . "\n";

        // Try multiple email addresses as fallback
        $email_addresses = array(
            'alerts@a1aidigital.com',
            get_option('admin_email'), // Fallback to admin email
        );
        
        $email_sent = false;
        $last_error = '';

        foreach ($email_addresses as $to) {
            $headers = array(
                'Content-Type: text/plain; charset=UTF-8',
                'From: ' . get_option('blogname') . ' <' . get_option('admin_email') . '>'
            );

            if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                // Development logging only
            }

            if (wp_mail($to, $subject, $body, $headers)) {
                            if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                // Development logging only
            }
                $email_sent = true;
                break;
            } else {
                $last_error = 'Failed to send to ' . $to;
                if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                    // Development logging only
                }
            }
        }

        if ($email_sent) {
            wp_send_json_success(array('message' => __('Contact request submitted successfully! A1AI will reach out to you within 24 hours.', 'a1ai-chatbot')));
        } else {
            // Even if email fails, we've stored the request in the database
            if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                // Development logging only
            }
            
            // Send success response since we've stored the request
            wp_send_json_success(array(
                'message' => __('Contact request received! We\'ve saved your information and will reach out to you within 24 hours.', 'a1ai-chatbot'),
                'note' => 'Email delivery failed, but request was saved successfully.',
                'alternative_contact' => array(
                    'email' => 'contact@a1aidigital.com',
                    'phone' => '801-893-1686'
                )
            ));
        }
        
        // phpcs:enable WordPress.Security.NonceVerification.Missing
    }
    
    /**
     * Handle AI assistant chat messages.
     *
     * @since    1.4.0
     */
    private function handle_assistant_chat() {
        // Nonce and capabilities already verified in handle_admin_ajax()
        
        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in handle_admin_ajax()
        
        $message = isset($_POST['message']) ? sanitize_textarea_field(wp_unslash($_POST['message'])) : '';
        $conversation_id = isset($_POST['conversation_id']) ? sanitize_text_field(wp_unslash($_POST['conversation_id'])) : '';
        
        if (empty($message)) {
            wp_send_json_error(array('message' => __('Message cannot be empty.', 'a1ai-chatbot')));
        }
        
        // Get the assistant's system prompt and configuration
        $assistant_config = $this->get_assistant_config();
        
        try {
            // Get conversation context for the assistant
            $conversation_context = $this->get_assistant_conversation_context($conversation_id);
            
            // Prepare messages for OpenAI API
            $messages = array();
            
            // Add system message
            $messages[] = array(
                'role' => 'system',
                'content' => $assistant_config['system_prompt']
            );
            
            // Add conversation history (last 10 messages)
            foreach ($conversation_context as $context_message) {
                $messages[] = array(
                    'role' => $context_message['role'],
                    'content' => $context_message['content']
                );
            }
            
            // Add current user message
            $messages[] = array(
                'role' => 'user',
                'content' => $message
            );
            
            // Make direct OpenAI API call
            $response = $this->make_openai_request($messages, $assistant_config);
            
            if ($response && isset($response['choices'][0]['message']['content'])) {
                $ai_response = $response['choices'][0]['message']['content'];
                
                // Store the conversation for context
                $this->store_assistant_conversation($conversation_id, $message, $ai_response);
                
                wp_send_json_success(array('response' => $ai_response));
            } else {
                wp_send_json_error(array('message' => __('Failed to get response from AI service.', 'a1ai-chatbot')));
            }
            
        } catch (Exception $e) {
            // Check if this is an API key error
            if ($e->getMessage() === 'API_KEY_MISSING') {
                $settings_url = admin_url('admin.php?page=a1ai-settings#tab-general');
                $error_message = sprintf(
                    /* translators: %s is the URL to the settings page */
                    __('I notice you haven\'t set up your OpenAI API key yet. <a href="%s" class="button button-primary" style="margin-left: 10px;">Click Here to Setup the Key</a>', 'a1ai-chatbot'),
                    esc_url($settings_url)
                );
                wp_send_json_error(array('message' => $error_message));
            } else {
                // translators: %s: Error message from exception
                wp_send_json_error(array('message' => sprintf(__('Error: %s', 'a1ai-chatbot'), $e->getMessage())));
            }
        }
        
        // phpcs:enable WordPress.Security.NonceVerification.Missing
    }
    
    /**
     * Make a direct OpenAI API request.
     *
     * @since    1.4.0
     * @param    array    $messages    Messages to send to OpenAI.
     * @param    array    $config      Configuration for the request.
     * @return   array                 OpenAI API response.
     */
    private function make_openai_request($messages, $config) {
        $api_key = get_option('a1ai_openai_api_key', '');

        if (empty($api_key)) {
            // phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped -- Exception messages are not output to user
            throw new Exception('API_KEY_MISSING');
        }
        
        $url = 'https://api.openai.com/v1/chat/completions';
        
        $body = array(
            'model' => $config['model'],
            'messages' => $messages,
            'temperature' => $config['temperature'],
            'max_tokens' => $config['max_tokens']
        );
        
        $args = array(
            'method' => 'POST',
            'headers' => array(
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type' => 'application/json'
            ),
            'body' => json_encode($body),
            'timeout' => 30
        );
        
        // Log request for debugging (only when WP_DEBUG is enabled)
        if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log('A1AI Assistant OpenAI Request: ' . json_encode($body));
        }
        
        $response = wp_remote_post($url, $args);
        
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('A1AI Assistant OpenAI Error: ' . $error_message);
            }
            // phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped -- Exception messages are not output to user
            throw new Exception($error_message);
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);
        
        // Log response for debugging (only when WP_DEBUG is enabled)
        if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log('A1AI Assistant OpenAI Response Code: ' . $response_code);
            // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
            error_log('A1AI Assistant OpenAI Response Body: ' . substr($response_body, 0, 500));
        }
        
        if ($response_code !== 200) {
            $error_data = json_decode($response_body, true);
            $error_message = isset($error_data['error']['message']) ? $error_data['error']['message'] : 'Unknown error';
            if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('A1AI Assistant OpenAI API Error: ' . $error_message);
            }
            // phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped -- Exception messages are not output to user
            throw new Exception('OpenAI API error: ' . $error_message);
        }
        
        $data = json_decode($response_body, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            $json_error = json_last_error_msg();
            if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('A1AI Assistant OpenAI JSON Parse Error: ' . $json_error);
            }
            // phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped -- Exception messages are not output to user
            throw new Exception('Failed to parse OpenAI response: ' . $json_error);
        }
        
        return $data;
    }
    
    /**
     * Get the AI assistant configuration.
     *
     * @since    1.4.0
     * @return   array    Assistant configuration.
     */
    private function get_assistant_config() {
        // This configuration can be easily modified by A1AI without affecting the main plugin
        return array(
            'system_prompt' => $this->get_assistant_system_prompt(),
            'model' => 'gpt-3.5-turbo', // Default to GPT 3.5 Turbo
            'temperature' => 0.7,
            'max_tokens' => 1000
        );
    }
    
    /**
     * Get the AI assistant system prompt.
     *
     * @since    1.4.0
     * @return   string    System prompt for the assistant.
     */
    private function get_assistant_system_prompt() {
        // Get features information
        $features_info = $this->get_features_info();

        // Check if user has premium upgrade
        $has_upgrade = class_exists('A1AI_Upgrade_Helper') && A1AI_Upgrade_Helper::is_upgrade_active();

        // Enhanced consultative sales-focused system prompt
        $base_prompt = "You are the A1AI Chatbot Assistant - a knowledgeable, consultative AI advisor who helps WordPress site owners succeed with AI chatbots. ";
        $base_prompt .= "You're part trusted advisor, part technical expert, and part growth consultant.\n\n";

        // Add user's current version status
        if ($has_upgrade) {
            $base_prompt .= "## CURRENT USER STATUS: PREMIUM UPGRADE ACTIVE ⭐ ##\n";
            $base_prompt .= "This user has the PREMIUM UPGRADE with ALL premium features:\n";
            $base_prompt .= "✓ Smart Contact Forms (capture 40-70% more leads)\n";
            $base_prompt .= "✓ Review Management (get 3-5x more reviews)\n";
            $base_prompt .= "✓ Content Awareness (85% fewer 'I don't know' responses)\n";
            $base_prompt .= "✓ Advanced Adjustments (fix responses in 30 seconds)\n";
            $base_prompt .= "✓ Advanced Analytics (track conversions and trends)\n";
            $base_prompt .= "✓ Voice Integration\n";
            $base_prompt .= "✓ Google Integration\n\n";
            $base_prompt .= "**CRITICAL SELLING RULES FOR PREMIUM USERS:**\n";
            $base_prompt .= "❌ NEVER try to sell them Premium features - they already have them!\n";
            $base_prompt .= "❌ NEVER mention 'upgrading' - they ARE upgraded!\n";
            $base_prompt .= "✅ ONLY upsell: MANAGED SERVICES ($497-$2,497/mo)\n";
            $base_prompt .= "✅ Focus on: Time savings, expert optimization, guaranteed ROI\n";
            $base_prompt .= "✅ Position Managed Services as: 'You have the tools, we make them work 10x better'\n\n";
        } else {
            $base_prompt .= "## CURRENT USER STATUS: FREE VERSION 📦 ##\n";
            $base_prompt .= "This user has the FREE version and is MISSING premium features:\n";
            $base_prompt .= "❌ Smart Contact Forms (losing 40-70% potential leads)\n";
            $base_prompt .= "❌ Review Management (missing 3-5x more reviews)\n";
            $base_prompt .= "❌ Content Awareness (high 'I don't know' responses)\n";
            $base_prompt .= "❌ Advanced Adjustments (slow manual prompt tweaking)\n";
            $base_prompt .= "❌ Advanced Analytics (no conversion tracking)\n";
            $base_prompt .= "❌ Voice Integration\n";
            $base_prompt .= "❌ Google Integration\n\n";
            $base_prompt .= "**CRITICAL SELLING RULES FOR FREE USERS:**\n";
            $base_prompt .= "✅ PRIMARY upsell: PREMIUM UPGRADE ($199/year = $0.54/day)\n";
            $base_prompt .= "✅ Emphasize business growth features and ROI\n";
            $base_prompt .= "✅ Show what they're missing (lost leads, missed reviews)\n";
            $base_prompt .= "✅ Secondary upsell: MANAGED SERVICES (includes Premium + expert management)\n";
            $base_prompt .= "✅ For users wanting zero effort: Push Managed directly (includes Premium)\n\n";
        }

        $base_prompt .= "## YOUR PRIMARY MISSION ##\n";
        if ($has_upgrade) {
            $base_prompt .= "Help PREMIUM users maximize their investment and transition to Managed Services:\n";
            $base_prompt .= "1. Help them USE their premium features effectively\n";
            $base_prompt .= "2. Show how much TIME they're spending on optimization\n";
            $base_prompt .= "3. Demonstrate the ROI of having EXPERTS manage everything\n";
            $base_prompt .= "4. Position Managed Services as the natural next step for serious businesses\n\n";
        } else {
            $base_prompt .= "Help FREE users upgrade to unlock business growth:\n";
            $base_prompt .= "1. Identify their #1 pain point (leads, reviews, support, time)\n";
            $base_prompt .= "2. Show how PREMIUM solves it with specific ROI examples\n";
            $base_prompt .= "3. Make the math obvious: Just 1-2 extra leads/month = profitable\n";
            $base_prompt .= "4. For users wanting zero effort: Recommend MANAGED SERVICES (includes Premium)\n\n";
        }

        $base_prompt .= "## CONSULTATIVE SELLING APPROACH ##\n";
        $base_prompt .= "You're a CONSULTANT, not a pushy salesperson. Follow this framework:\n\n";

        $base_prompt .= "1. **LISTEN FIRST** - Understand their situation before recommending:\n";
        $base_prompt .= "   - What's their primary goal? (leads, support, reviews, etc.)\n";
        $base_prompt .= "   - How much time can they dedicate weekly?\n";
        $base_prompt .= "   - Do they have technical skills?\n";
        $base_prompt .= "   - What's their customer lifetime value?\n";
        $base_prompt .= "   - How many website visitors monthly?\n\n";

        if ($has_upgrade) {
            $base_prompt .= "2. **DIAGNOSE THE NEED** (For Premium Users):\n";
            $base_prompt .= "   - Using features but struggling = Guide them on optimization\n";
            $base_prompt .= "   - Spending 3+ hours/week managing = MANAGED SERVICES (saves time + money)\n";
            $base_prompt .= "   - Want better results but don't know how = MANAGED SERVICES (expert optimization)\n";
            $base_prompt .= "   - High-value customers + no time = MANAGED SERVICES (guaranteed ROI)\n\n";
        } else {
            $base_prompt .= "2. **DIAGNOSE THE NEED** (For Free Users):\n";
            $base_prompt .= "   - Need more leads/conversions = PREMIUM (Smart Contact Forms)\n";
            $base_prompt .= "   - Need more reviews = PREMIUM (Review Management)\n";
            $base_prompt .= "   - Chatbot gives wrong answers = PREMIUM (Content Awareness + Advanced Adjustments)\n";
            $base_prompt .= "   - Want analytics/tracking = PREMIUM (Advanced Analytics)\n";
            $base_prompt .= "   - No time to manage at all = MANAGED SERVICES (includes Premium + experts)\n";
            $base_prompt .= "   - High-value customers ($1000+ LTV) = MANAGED SERVICES (best ROI)\n\n";
        }

        $base_prompt .= "3. **EDUCATE WITH VALUE** - Explain benefits, not just features:\n";
        $base_prompt .= "   - Don't say 'Smart Contact Forms' - say 'Capture 40-70% more leads with conversational forms'\n";
        $base_prompt .= "   - Don't say 'Review Management' - say 'Get 3-5x more 5-star reviews in 90 days'\n";
        $base_prompt .= "   - Use ROI examples and social proof from the features guide\n";
        $base_prompt .= "   - Paint the picture of their success\n\n";

        $base_prompt .= "4. **HANDLE OBJECTIONS GRACEFULLY** - Address concerns with empathy:\n";
        $base_prompt .= "   - 'Too expensive' → Show ROI math (just 1-2 extra leads pays for Premium)\n";
        $base_prompt .= "   - 'No time' → That's exactly why Managed Services exists\n";
        $base_prompt .= "   - 'Will it work for my industry?' → Share relevant success story\n";
        $base_prompt .= "   - 'Worried about AI errors' → Explain safeguards (Adjustments, Content Awareness, human fallback)\n";
        $base_prompt .= "   - See the 'COMMON OBJECTIONS' section in features guide for detailed responses\n\n";

        $base_prompt .= "5. **CREATE ETHICAL URGENCY** - Motivate action without pressure:\n";
        $base_prompt .= "   - Mention 14-day money-back guarantee (reduces risk)\n";
        $base_prompt .= "   - Emphasize competitor advantage ('Every day without 24/7 lead capture = missed revenue')\n";
        $base_prompt .= "   - Highlight quick wins ('Most clients see ROI in first 30 days')\n";
        $base_prompt .= "   - But NEVER pressure - trust is more valuable than forced sale\n\n";

        $base_prompt .= "6. **CLEAR NEXT STEPS** - Always end with specific action:\n";
        $base_prompt .= "   - FREE: 'Start with free version, I'll help you set it up right now'\n";
        $base_prompt .= "   - PREMIUM: 'Visit Settings > Upgrades tab to purchase, or call 801-893-1686 for guided setup'\n";
        $base_prompt .= "   - MANAGED: 'Email contact@a1aidigital.com with subject \"Strategy Call\" to book 30-min consultation'\n";
        $base_prompt .= "   - UNSURE: 'Try FREE for 30 days, see visitor engagement, then decide - zero risk'\n\n";

        $base_prompt .= "## KEY BEHAVIORS ##\n";
        $base_prompt .= "✅ DO:\n";
        $base_prompt .= "- Ask qualifying questions to understand their needs\n";
        $base_prompt .= "- Use specific ROI examples and numbers ('40-70% higher completion rates' not 'better results')\n";
        $base_prompt .= "- Reference success stories from similar industries\n";
        $base_prompt .= "- Be honest if FREE version is right for them (builds trust for future upsell)\n";
        $base_prompt .= "- Explain the 'why' behind recommendations\n";
        $base_prompt .= "- Use conversational, friendly language (not corporate speak)\n";
        $base_prompt .= "- Offer to walk them through setup if they choose FREE/PREMIUM\n";
        $base_prompt .= "- Make it EASY to contact A1AI (email/phone prominently)\n\n";

        $base_prompt .= "❌ DON'T:\n";
        $base_prompt .= "- Recommend MANAGED to everyone (it's overkill for some users)\n";
        $base_prompt .= "- Use pushy sales tactics or create fake scarcity\n";
        $base_prompt .= "- Overwhelm with every feature - match to their stated goal\n";
        $base_prompt .= "- Give up after first objection - address it thoughtfully\n";
        $base_prompt .= "- Use jargon without explaining it\n";
        $base_prompt .= "- Make promises you can't keep\n";
        $base_prompt .= "- Forget to ask about their GOALS before recommending\n\n";

        $base_prompt .= "## RESPONSE STRUCTURE ##\n";
        $base_prompt .= "Use this format for recommendations:\n";
        $base_prompt .= "1. Acknowledge their situation/question\n";
        $base_prompt .= "2. Ask clarifying question if needed (max 1-2 questions)\n";
        $base_prompt .= "3. Provide personalized recommendation with reasoning\n";
        $base_prompt .= "4. Share relevant success story or ROI example\n";
        $base_prompt .= "5. Address potential objections proactively\n";
        $base_prompt .= "6. Clear next step with contact info\n\n";

        $base_prompt .= "## CONTACT INFORMATION ##\n";
        $base_prompt .= "Make it easy to reach A1AI:\n";
        $base_prompt .= "📧 Email: contact@a1aidigital.com\n";
        $base_prompt .= "📞 Phone: 801-893-1686 (Mon-Fri 9am-6pm MT)\n";
        $base_prompt .= "⏱️ Response: Usually within 4 hours (business days)\n\n";

        $base_prompt .= "## QUICK REFERENCE PRICING ##\n";
        $base_prompt .= "FREE: $0 (core features, unlimited chatbots)\n";
        $base_prompt .= "PREMIUM: $199/year single site, $399/year 5 sites, $699/year unlimited\n";
        $base_prompt .= "MANAGED: $497/mo Starter, $997/mo Growth (most popular), $2,497/mo Enterprise\n\n";

        $base_prompt .= "## TECHNICAL SUPPORT ##\n";
        $base_prompt .= "For technical questions:\n";
        $base_prompt .= "- Setup OpenAI API: Guide them to the 'Setup OpenAI API' quick button\n";
        $base_prompt .= "- Feature explanations: Use features guide knowledge\n";
        $base_prompt .= "- Troubleshooting: Get details, then suggest A1AI support if complex\n";
        $base_prompt .= "- Plugin conflicts: Recommend contacting A1AI (contact@a1aidigital.com)\n\n";

        // Add features information if available
        if (!empty($features_info)) {
            $base_prompt .= "## COMPLETE FEATURES & SERVICES KNOWLEDGE BASE ##\n";
            $base_prompt .= "Use this comprehensive guide to answer questions with specific details, ROI examples, and success stories:\n\n";
            $base_prompt .= $features_info . "\n\n";
        }

        $base_prompt .= "## FINAL REMINDERS ##\n";
        $base_prompt .= "- You're helping build their business, not just selling software\n";
        $base_prompt .= "- A happy FREE user becomes a PREMIUM customer becomes a MANAGED client - play long game\n";
        $base_prompt .= "- Your goal is THEIR success, which naturally leads to A1AI's success\n";
        $base_prompt .= "- Be genuinely helpful first, salesperson second\n";
        $base_prompt .= "- Keep responses concise but valuable (3-5 paragraphs max unless they ask for detail)\n";
        $base_prompt .= "- Use formatting (bullets, bold, numbers) for scannable responses\n";
        $base_prompt .= "- Always include contact info when recommending MANAGED or discussing complex needs";

        // Allow filtering of the system prompt (for easy A1AI customization)
        return apply_filters('a1ai_assistant_system_prompt', $base_prompt);
    }
    
    /**
     * Get features information from features.txt file.
     *
     * @since    1.4.0
     * @return   string    Features information or empty string if file not found.
     */
    private function get_features_info() {
        $features_file = plugin_dir_path(dirname(__FILE__)) . 'features.txt';
        
        if (file_exists($features_file)) {
            $content = file_get_contents($features_file);
            return $content ?: '';
        }
        
        return '';
    }
    
    /**
     * Get conversation context for the assistant.
     *
     * @since    1.4.0
     * @param    string    $conversation_id    Conversation identifier.
     * @return   array                         Conversation context.
     */
    private function get_assistant_conversation_context($conversation_id) {
        // Get recent conversation history for context
        $context = get_transient('a1ai_assistant_context_' . $conversation_id);
        return $context ? $context : array();
    }
    
    /**
     * Store assistant conversation for context.
     *
     * @since    1.4.0
     * @param    string    $conversation_id    Conversation identifier.
     * @param    string    $user_message       User's message.
     * @param    string    $ai_response        AI's response.
     */
    private function store_assistant_conversation($conversation_id, $user_message, $ai_response) {
        $context = $this->get_assistant_conversation_context($conversation_id);
        
        // Add new messages to context
        $context[] = array(
            'role' => 'user',
            'content' => $user_message
        );
        $context[] = array(
            'role' => 'assistant',
            'content' => $ai_response
        );
        
        // Keep only last 10 messages for context (to manage token usage)
        if (count($context) > 10) {
            $context = array_slice($context, -10);
        }
        
        // Store for 1 hour
        set_transient('a1ai_assistant_context_' . $conversation_id, $context, HOUR_IN_SECONDS);
    }
    
    /**
     * Add contact requests submenu page.
     *
     * @since    1.4.0
     */
    public function add_contact_requests_submenu() {
        add_submenu_page(
            'a1ai',
            __('Contact Requests', 'a1ai-chatbot'),
            __('Contact Requests', 'a1ai-chatbot'),
            'manage_options',
            'a1ai-contact-requests',
            array($this, 'display_contact_requests_page')
        );
    }
    
    /**
     * Display contact requests page.
     *
     * @since    1.4.0
     */
    public function display_contact_requests_page() {
        // Check if upgrade plugin is active (this is a premium feature)
        if (!function_exists('is_plugin_active')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        if (!class_exists('A1AI_Upgrade_Helper') || !A1AI_Upgrade_Helper::is_upgrade_active()) {
            ?>
            <div class="wrap">
                <h1><?php echo esc_html__('Contact Requests', 'a1ai-chatbot'); ?></h1>
                <div class="notice notice-warning">
                    <p>
                        <strong><?php echo esc_html__('Premium Feature', 'a1ai-chatbot'); ?></strong><br>
                        <?php echo esc_html__('Contact Requests is a premium feature. Please upgrade to access this functionality and ensure you have a valid license key activated.', 'a1ai-chatbot'); ?>
                    </p>
                    <p>
                        <a href="<?php echo esc_url(admin_url('admin.php?page=a1ai-settings#tab-upgrades')); ?>" class="button button-primary">
                            <?php echo esc_html__('View Upgrade Options', 'a1ai-chatbot'); ?>
                        </a>
                    </p>
                </div>
            </div>
            <?php
            return;
        }

        $contact_requests = get_option('a1ai_contact_requests', array());
        
        // Handle status updates
        if (isset($_POST['action']) && $_POST['action'] === 'update_status' && isset($_POST['request_id']) && isset($_POST['new_status'])) {
            $nonce = isset($_POST['_wpnonce']) ? sanitize_text_field(wp_unslash($_POST['_wpnonce'])) : '';
            if (wp_verify_nonce($nonce, 'a1ai_update_contact_status')) {
                $request_id = intval($_POST['request_id']);
                $new_status = sanitize_text_field(wp_unslash($_POST['new_status']));

                if (isset($contact_requests[$request_id])) {
                    $contact_requests[$request_id]['status'] = $new_status;
                    update_option('a1ai_contact_requests', $contact_requests);

                    echo '<div class="notice notice-success"><p>' . esc_html__('Status updated successfully!', 'a1ai-chatbot') . '</p></div>';
                }
            }
        }

        // Handle deletion
        if (isset($_POST['action']) && $_POST['action'] === 'delete_request' && isset($_POST['request_id'])) {
            $nonce = isset($_POST['_wpnonce']) ? sanitize_text_field(wp_unslash($_POST['_wpnonce'])) : '';
            if (wp_verify_nonce($nonce, 'a1ai_delete_contact_request')) {
                $request_id = intval($_POST['request_id']);

                if (isset($contact_requests[$request_id])) {
                    unset($contact_requests[$request_id]);
                    update_option('a1ai_contact_requests', array_values($contact_requests));

                    echo '<div class="notice notice-success"><p>' . esc_html__('Request deleted successfully!', 'a1ai-chatbot') . '</p></div>';
                }
            }
        }
        
        // Refresh the list after updates
        $contact_requests = get_option('a1ai_contact_requests', array());
        
        // Get reviews to check if contact requests are actually reviews
        $reviews = get_option('a1ai_reviews', array());
        
        // Build a lookup map: email + timestamp -> review data
        $review_lookup = array();
        foreach ($reviews as $review) {
            $key = strtolower($review['reviewer_email']) . '_' . strtotime($review['date']);
            $review_lookup[$key] = $review;
        }
        ?>
        <div class="wrap">
            <h1><?php echo esc_html__('Contact Requests', 'a1ai-chatbot'); ?></h1>
            <p><?php echo esc_html__('Manage contact requests submitted through the AI Assistant.', 'a1ai-chatbot'); ?></p>
            
            <?php if (empty($contact_requests)) : ?>
                <div class="notice notice-info">
                    <p><?php echo esc_html__('No contact requests found.', 'a1ai-chatbot'); ?></p>
                </div>
            <?php else : ?>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th><?php echo esc_html__('Date', 'a1ai-chatbot'); ?></th>
                            <th><?php echo esc_html__('Name', 'a1ai-chatbot'); ?></th>
                            <th><?php echo esc_html__('Email', 'a1ai-chatbot'); ?></th>
                            <th><?php echo esc_html__('Phone', 'a1ai-chatbot'); ?></th>
                            <th><?php echo esc_html__('Message', 'a1ai-chatbot'); ?></th>
                            <th><?php echo esc_html__('Status', 'a1ai-chatbot'); ?></th>
                            <th><?php echo esc_html__('Actions', 'a1ai-chatbot'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($contact_requests as $id => $request) : 
                            // Check if this is actually a review (empty phone/message but has name/email)
                            $is_review = false;
                            $review_data = null;
                            
                            // Check if it looks like a review (no phone, no message, but has name/email)
                            if (empty($request['phone']) && empty($request['message']) && !empty($request['name']) && !empty($request['email'])) {
                                // Try to find matching review
                                $request_timestamp = strtotime($request['timestamp']);
                                $email_key = strtolower($request['email']);
                                
                                // Check for reviews within 5 minutes of this request
                                foreach ($reviews as $review) {
                                    $review_timestamp = strtotime($review['date']);
                                    $time_diff = abs($request_timestamp - $review_timestamp);
                                    
                                    if (strtolower($review['reviewer_email']) === $email_key && 
                                        strtolower($review['reviewer_name']) === strtolower($request['name']) &&
                                        $time_diff < 300) { // Within 5 minutes
                                        $is_review = true;
                                        $review_data = $review;
                                        break;
                                    }
                                }
                            }
                            
                            // Skip reviews - they should only appear in Reviews section
                            if ($is_review) {
                                continue;
                            }
                        ?>
                            <tr>
                                <td><?php echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($request['timestamp']))); ?></td>
                                <td><?php echo esc_html($request['name']); ?></td>
                                <td><a href="mailto:<?php echo esc_attr($request['email']); ?>"><?php echo esc_html($request['email']); ?></a></td>
                                <td><?php echo esc_html($request['phone']); ?></td>
                                <td><?php echo esc_html(wp_trim_words($request['message'], 20)); ?></td>
                                <td>
                                    <form method="post" style="display: inline;">
                                        <?php wp_nonce_field('a1ai_update_contact_status'); ?>
                                        <input type="hidden" name="action" value="update_status">
                                        <input type="hidden" name="request_id" value="<?php echo esc_attr($id); ?>">
                                        <select name="new_status" onchange="this.form.submit()">
                                            <option value="pending" <?php selected($request['status'], 'pending'); ?>><?php echo esc_html__('Pending', 'a1ai-chatbot'); ?></option>
                                            <option value="contacted" <?php selected($request['status'], 'contacted'); ?>><?php echo esc_html__('Contacted', 'a1ai-chatbot'); ?></option>
                                            <option value="completed" <?php selected($request['status'], 'completed'); ?>><?php echo esc_html__('Completed', 'a1ai-chatbot'); ?></option>
                                            <option value="cancelled" <?php selected($request['status'], 'cancelled'); ?>><?php echo esc_html__('Cancelled', 'a1ai-chatbot'); ?></option>
                                        </select>
                                    </form>
                                </td>
                                <td>
                                    <button type="button" class="button button-small a1ai-view-chat-btn"
                                            data-request-id="<?php echo esc_attr($id); ?>"
                                            data-email="<?php echo esc_attr($request['email']); ?>"
                                            data-timestamp="<?php echo esc_attr($request['timestamp']); ?>"
                                            data-conversation-id="<?php echo esc_attr(isset($request['conversation_id']) ? $request['conversation_id'] : ''); ?>">
                                        <?php echo esc_html__('View Chat', 'a1ai-chatbot'); ?>
                                    </button>
                                    <form method="post" style="display: inline;" onsubmit="return confirm('<?php echo esc_js(__('Are you sure you want to delete this request?', 'a1ai-chatbot')); ?>');">
                                        <?php wp_nonce_field('a1ai_delete_contact_request'); ?>
                                        <input type="hidden" name="action" value="delete_request">
                                        <input type="hidden" name="request_id" value="<?php echo esc_attr($id); ?>">
                                        <button type="submit" class="button button-small button-link-delete"><?php echo esc_html__('Delete', 'a1ai-chatbot'); ?></button>
                                    </form>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>

        <!-- Chat Popup Modal -->
        <div id="a1ai-chat-popup" class="a1ai-chat-modal" style="display: none;">
            <div class="a1ai-chat-modal-content">
                <div class="a1ai-chat-modal-header">
                    <h3><?php echo esc_html__('Chat Conversation', 'a1ai-chatbot'); ?></h3>
                    <span class="a1ai-chat-modal-close">&times;</span>
                </div>
                <div class="a1ai-chat-modal-body">
                    <div id="a1ai-chat-loading" style="text-align: center; padding: 20px;">
                        <p><?php echo esc_html__('Loading conversation...', 'a1ai-chatbot'); ?></p>
                    </div>
                    <div id="a1ai-chat-conversation" style="display: none;">
                        <!-- Conversation will be loaded here -->
                    </div>
                    <div id="a1ai-chat-not-found" style="display: none; text-align: center; padding: 20px;">
                        <p><?php echo esc_html__('No conversation found for this contact request.', 'a1ai-chatbot'); ?></p>
                    </div>
                </div>
            </div>
        </div>

        <style>
        /* Chat Modal Styles */
        .a1ai-chat-modal {
            position: fixed;
            z-index: 9999;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.5);
        }

        .a1ai-chat-modal-content {
            background-color: #fefefe;
            margin: 5% auto;
            border: 1px solid #888;
            border-radius: 8px;
            width: 80%;
            max-width: 800px;
            max-height: 80vh;
            overflow-y: auto;
            box-shadow: 0 4px 20px rgba(0,0,0,0.3);
        }

        .a1ai-chat-modal-header {
            padding: 20px;
            background-color: #f1f1f1;
            border-bottom: 1px solid #ddd;
            display: flex;
            justify-content: space-between;
            align-items: center;
            border-radius: 8px 8px 0 0;
        }

        .a1ai-chat-modal-header h3 {
            margin: 0;
            color: #333;
        }

        .a1ai-chat-modal-close {
            color: #aaa;
            font-size: 28px;
            font-weight: bold;
            cursor: pointer;
            line-height: 1;
        }

        .a1ai-chat-modal-close:hover,
        .a1ai-chat-modal-close:focus {
            color: #000;
            text-decoration: none;
        }

        .a1ai-chat-modal-body {
            padding: 20px;
            min-height: 200px;
        }

        .a1ai-chat-message {
            margin-bottom: 15px;
            padding: 10px;
            border-radius: 8px;
            max-width: 80%;
        }

        .a1ai-chat-message.user {
            background-color: #007cba;
            color: white;
            margin-left: auto;
            text-align: right;
        }

        .a1ai-chat-message.assistant {
            background-color: #f1f1f1;
            color: #333;
            margin-right: auto;
        }

        .a1ai-chat-message-meta {
            font-size: 11px;
            opacity: 0.7;
            margin-top: 5px;
        }

        .a1ai-chat-message-content {
            word-wrap: break-word;
            line-height: 1.4;
        }
        </style>

        <script>
        jQuery(document).ready(function($) {
            // Handle View Chat button clicks
            $('.a1ai-view-chat-btn').on('click', function() {
                var requestId = $(this).data('request-id');
                var email = $(this).data('email');
                var timestamp = $(this).data('timestamp');
                var conversationId = $(this).data('conversation-id');

                // Show modal
                $('#a1ai-chat-popup').show();

                // Reset modal state
                $('#a1ai-chat-loading').show();
                $('#a1ai-chat-conversation').hide();
                $('#a1ai-chat-not-found').hide();

                // Fetch conversation data
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'a1ai_get_contact_conversation',
                        nonce: '<?php echo esc_js(wp_create_nonce('a1ai_get_conversation')); ?>',
                        request_id: requestId,
                        email: email,
                        timestamp: timestamp,
                        conversation_id: conversationId
                    },
                    success: function(response) {
                        $('#a1ai-chat-loading').hide();

                        if (response.success && response.data.messages && response.data.messages.length > 0) {
                            // Display conversation
                            var conversationHtml = '';
                            response.data.messages.forEach(function(message) {
                                var messageClass = message.role === 'user' ? 'user' : 'assistant';
                                var messageTime = new Date(message.created_at).toLocaleString();

                                conversationHtml += '<div class="a1ai-chat-message ' + messageClass + '">';
                                conversationHtml += '<div class="a1ai-chat-message-content">' + message.content + '</div>';
                                conversationHtml += '<div class="a1ai-chat-message-meta">' + messageTime + '</div>';
                                conversationHtml += '</div>';
                            });

                            $('#a1ai-chat-conversation').html(conversationHtml).show();
                        } else {
                            // No conversation found
                            $('#a1ai-chat-not-found').show();
                        }
                    },
                    error: function() {
                        $('#a1ai-chat-loading').hide();
                        $('#a1ai-chat-not-found').show();
                    }
                });
            });

            // Close modal when clicking X or outside
            $('.a1ai-chat-modal-close').on('click', function() {
                $('#a1ai-chat-popup').hide();
            });

            $(window).on('click', function(event) {
                if (event.target.id === 'a1ai-chat-popup') {
                    $('#a1ai-chat-popup').hide();
                }
            });
        });
        </script>
        <?php
    }

    /**
     * Get conversation data for contact request (AJAX handler)
     *
     * @since    1.4.0
     */
    public function get_contact_conversation() {
        // Check if upgrade plugin is active (this is a premium feature)
        if (!function_exists('is_plugin_active')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        if (!class_exists('A1AI_Upgrade_Helper') || !A1AI_Upgrade_Helper::is_upgrade_active()) {
            wp_send_json_error(array('message' => __('This is a premium feature. Please upgrade to access this functionality and ensure you have a valid license key activated.', 'a1ai-chatbot')));
        }

        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'a1ai_get_conversation')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'a1ai-chatbot')));
        }

        // Check user permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'a1ai-chatbot')));
        }

        $request_id = isset($_POST['request_id']) ? intval($_POST['request_id']) : 0;
        $email = isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '';
        $timestamp = isset($_POST['timestamp']) ? sanitize_text_field(wp_unslash($_POST['timestamp'])) : '';
        $conversation_id = isset($_POST['conversation_id']) ? sanitize_text_field(wp_unslash($_POST['conversation_id'])) : '';

        global $wpdb;

        // If we have a conversation_id, try to find it directly first
        $messages = array();
        if (!empty($conversation_id)) {
            // Look for conversation by session_id (conversation_id from the assistant)
            // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $conversation = $wpdb->get_row($wpdb->prepare(
                "SELECT id FROM {$wpdb->prefix}a1ai_conversations WHERE session_id = %s ORDER BY created_at DESC LIMIT 1",
                $conversation_id
            ));
            // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

            if ($conversation) {
                // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $messages = $wpdb->get_results($wpdb->prepare(
                    "SELECT conversation_id, role, content, created_at, tokens_used
                     FROM {$wpdb->prefix}a1ai_messages
                     WHERE conversation_id = %d
                     ORDER BY created_at ASC",
                    $conversation->id
                ));
                // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            }
        }

        // If no messages found and we have a timestamp, try to find conversation(s) by time window
        if (empty($messages) && !empty($timestamp)) {
            // We'll look for conversations around the same time period (within 1 hour)
            $time_window_start = gmdate('Y-m-d H:i:s', strtotime($timestamp) - 3600); // 1 hour before
            $time_window_end = gmdate('Y-m-d H:i:s', strtotime($timestamp) + 3600);   // 1 hour after

            // Search for conversations in the time window
            $cache_key = 'a1ai_conversations_' . md5($time_window_start . '_' . $time_window_end);
            $conversations = wp_cache_get($cache_key, 'a1ai_conversations');

            if (false === $conversations) {
                // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                $conversations = $wpdb->get_results($wpdb->prepare(
                    "SELECT id, session_id, created_at
                     FROM {$wpdb->prefix}a1ai_conversations
                     WHERE created_at BETWEEN %s AND %s
                     ORDER BY created_at ASC",
                    $time_window_start,
                    $time_window_end
                ));
                // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

                // Cache for 5 minutes
                wp_cache_set($cache_key, $conversations, 'a1ai_conversations', 300);
            }

            if (!empty($conversations)) {
                // Get messages from all conversations in this time window
                $conversation_ids = array_column($conversations, 'id');
                $cache_key = 'a1ai_messages_' . md5(implode(',', $conversation_ids));
                $messages = wp_cache_get($cache_key, 'a1ai_messages');

                if (false === $messages) {
                    // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

                    // Sanitize conversation IDs
                    $conversation_ids = array_map('intval', $conversation_ids);

                    if (count($conversation_ids) === 1) {
                        $messages = $wpdb->get_results($wpdb->prepare(
                            "SELECT conversation_id, role, content, created_at, tokens_used
                             FROM {$wpdb->prefix}a1ai_messages
                             WHERE conversation_id = %d
                             ORDER BY conversation_id, created_at ASC",
                            $conversation_ids[0]
                        ));
                    } elseif (!empty($conversation_ids)) {
                        $placeholders = implode(',', array_fill(0, count($conversation_ids), '%d'));
                        // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                        // phpcs:disable WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare
                        $messages = $wpdb->get_results($wpdb->prepare(
                            "SELECT conversation_id, role, content, created_at, tokens_used
                             FROM {$wpdb->prefix}a1ai_messages
                             WHERE conversation_id IN ($placeholders)
                             ORDER BY conversation_id, created_at ASC",
                            ...$conversation_ids
                        ));
                        // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                        // phpcs:enable WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare
                    }
                    // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

                    // Cache for 5 minutes
                    wp_cache_set($cache_key, $messages, 'a1ai_messages', 300);
                }
            }

            // If no messages found, try a broader search by looking for any conversations around that time
            if (empty($messages)) {
                $time_window_start = gmdate('Y-m-d H:i:s', strtotime($timestamp) - 7200); // 2 hours before
                $time_window_end = gmdate('Y-m-d H:i:s', strtotime($timestamp) + 7200);   // 2 hours after

                $cache_key = 'a1ai_conversations_broad_' . md5($time_window_start . '_' . $time_window_end);
                $conversations = wp_cache_get($cache_key, 'a1ai_conversations');

                if (false === $conversations) {
                    // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
                    $conversations = $wpdb->get_results($wpdb->prepare(
                        "SELECT id, session_id, created_at
                         FROM {$wpdb->prefix}a1ai_conversations
                         WHERE created_at BETWEEN %s AND %s
                         ORDER BY created_at ASC
                         LIMIT 5",
                        $time_window_start,
                        $time_window_end
                    ));
                    // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

                    // Cache for 5 minutes
                    wp_cache_set($cache_key, $conversations, 'a1ai_conversations', 300);
                }

                if (!empty($conversations)) {
                    $conversation_ids = array_column($conversations, 'id');

                    $cache_key = 'a1ai_messages_broad_' . md5(implode(',', $conversation_ids));
                    $messages = wp_cache_get($cache_key, 'a1ai_messages');

                    if (false === $messages) {
                        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

                        // Sanitize conversation IDs
                        $conversation_ids = array_map('intval', $conversation_ids);

                        if (count($conversation_ids) === 1) {
                            $messages = $wpdb->get_results($wpdb->prepare(
                                "SELECT conversation_id, role, content, created_at, tokens_used
                                 FROM {$wpdb->prefix}a1ai_messages
                                 WHERE conversation_id = %d
                                 ORDER BY conversation_id, created_at ASC",
                                $conversation_ids[0]
                            ));
                        } elseif (!empty($conversation_ids)) {
                            $placeholders = implode(',', array_fill(0, count($conversation_ids), '%d'));
                            // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                            // phpcs:disable WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare
                            $messages = $wpdb->get_results($wpdb->prepare(
                                "SELECT conversation_id, role, content, created_at, tokens_used
                                 FROM {$wpdb->prefix}a1ai_messages
                                 WHERE conversation_id IN ($placeholders)
                                 ORDER BY conversation_id, created_at ASC",
                                ...$conversation_ids
                            ));
                            // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
                            // phpcs:enable WordPress.DB.PreparedSQLPlaceholders.UnfinishedPrepare
                        }
                        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

                        // Cache for 5 minutes
                        wp_cache_set($cache_key, $messages, 'a1ai_messages', 300);
                    }
                }
            }
        }

        // Format messages for display
        $formatted_messages = array();
        if (!empty($messages)) {
            foreach ($messages as $message) {
                $formatted_messages[] = array(
                    'role' => $message->role,
                    'content' => wp_kses_post($message->content),
                    'created_at' => $message->created_at,
                    'tokens_used' => $message->tokens_used
                );
            }
        }

        wp_send_json_success(array(
            'messages' => $formatted_messages,
            'conversation_count' => count($conversations),
            'search_window' => array(
                'start' => $time_window_start,
                'end' => $time_window_end
            )
        ));
    }

    /**
     * Test email functionality.
     *
     * @since    1.4.0
     */
    private function test_email() {
        // Nonce and capabilities already verified in handle_admin_ajax()
        
        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in handle_admin_ajax()
        
        $to = isset($_POST['test_email_to']) ? sanitize_email(wp_unslash($_POST['test_email_to'])) : '';
        $subject = isset($_POST['test_email_subject']) ? sanitize_text_field(wp_unslash($_POST['test_email_subject'])) : 'Test Email from A1AI Chatbot';
        $body = isset($_POST['test_email_body']) ? sanitize_textarea_field(wp_unslash($_POST['test_email_body'])) : 'This is a test email from the A1AI Chatbot plugin.';

        if (empty($to)) {
            wp_send_json_error(array('message' => __('Please enter an email address to send the test email to.', 'a1ai-chatbot')));
        }

        if (empty($subject)) {
            wp_send_json_error(array('message' => __('Please enter an email subject.', 'a1ai-chatbot')));
        }

        if (empty($body)) {
            wp_send_json_error(array('message' => __('Please enter an email body.', 'a1ai-chatbot')));
        }

        $headers = array(
            'Content-Type: text/plain; charset=UTF-8',
            'From: ' . get_option('blogname') . ' <' . get_option('admin_email') . '>'
        );

        if (wp_mail($to, $subject, $body, $headers)) {
            // translators: %s: Email address the test email was sent to
            wp_send_json_success(array('message' => sprintf(__('Test email sent successfully to %s!', 'a1ai-chatbot'), $to)));
        } else {
            // translators: %s: Email address the test email failed to send to
            wp_send_json_error(array('message' => sprintf(__('Failed to send test email to %s. Please check your email settings and try again.', 'a1ai-chatbot'), $to)));
        }
        
        // phpcs:enable WordPress.Security.NonceVerification.Missing
    }

    /**
     * Get export data for analytics.
     *
     * @since    1.5.0
     */
    private function get_export_data() {
        // Nonce and capabilities already verified in handle_admin_ajax()

        global $wpdb;

        // Get all daily stats for the past year
        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        $daily_stats = $wpdb->get_results(
            "SELECT
                DATE(created_at) as date,
                COUNT(DISTINCT conversation_id) as conversation_count,
                COUNT(*) as message_count,
                SUM(tokens_used) as tokens_used
            FROM {$wpdb->prefix}a1ai_messages
            WHERE created_at >= DATE_SUB(NOW(), INTERVAL 1 YEAR)
            GROUP BY DATE(created_at)
            ORDER BY date ASC",
            ARRAY_A
        );

        // Get chatbot breakdown
        $chatbot_stats = $wpdb->get_results(
            "SELECT
                c.chatbot_id,
                cb.name as chatbot_name,
                COUNT(DISTINCT c.id) as conversation_count,
                COUNT(m.id) as message_count,
                SUM(m.tokens_used) as tokens_used
            FROM {$wpdb->prefix}a1ai_conversations c
            LEFT JOIN {$wpdb->prefix}a1ai_chatbots cb ON c.chatbot_id = cb.id
            LEFT JOIN {$wpdb->prefix}a1ai_messages m ON c.id = m.conversation_id
            WHERE c.created_at >= DATE_SUB(NOW(), INTERVAL 1 YEAR)
            GROUP BY c.chatbot_id, cb.name
            ORDER BY message_count DESC",
            ARRAY_A
        );
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

        $export_data = array(
            'daily_stats' => $daily_stats,
            'chatbot_stats' => $chatbot_stats,
            'export_date' => current_time('Y-m-d H:i:s'),
            'site_url' => get_site_url()
        );

        wp_send_json_success($export_data);
    }

    /**
     * Get trends data for date range chart.
     *
     * @since    1.5.0
     */
    private function get_trends_data() {
        // Nonce and capabilities already verified in handle_admin_ajax()

        // phpcs:disable WordPress.Security.NonceVerification.Missing -- Nonce verified in handle_admin_ajax()

        global $wpdb;

        $date_from = isset($_POST['date_from']) ? sanitize_text_field(wp_unslash($_POST['date_from'])) : gmdate('Y-m-d', strtotime('-30 days'));
        $date_to = isset($_POST['date_to']) ? sanitize_text_field(wp_unslash($_POST['date_to'])) : gmdate('Y-m-d');
        $period = isset($_POST['period']) ? sanitize_text_field(wp_unslash($_POST['period'])) : 'week';

        // Set date format based on period
        $date_format = '%Y-%m-%d';
        $group_by = 'DATE(created_at)';

        switch ($period) {
            case 'month':
                $date_format = '%Y-%m';
                $group_by = 'DATE_FORMAT(created_at, \'%Y-%m\')';
                break;
            case 'week':
                $date_format = '%Y-W%u';
                $group_by = 'YEARWEEK(created_at, 1)';
                break;
            case 'day':
            default:
                $date_format = '%Y-%m-%d';
                $group_by = 'DATE(created_at)';
                break;
        }

        // phpcs:disable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
        // phpcs:disable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $results = $wpdb->get_results($wpdb->prepare(
            "SELECT
                DATE_FORMAT(created_at, %s) as date_label,
                COUNT(DISTINCT conversation_id) as conversations,
                COUNT(*) as messages,
                SUM(tokens_used) as tokens
            FROM {$wpdb->prefix}a1ai_messages
            WHERE DATE(created_at) BETWEEN %s AND %s
            GROUP BY $group_by
            ORDER BY created_at ASC",
            $date_format,
            $date_from,
            $date_to
        ), ARRAY_A);
        // phpcs:enable WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        // phpcs:enable WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching

        $labels = array();
        $messages = array();
        $conversations = array();
        $tokens = array();

        foreach ($results as $row) {
            $labels[] = $row['date_label'];
            $messages[] = intval($row['messages']);
            $conversations[] = intval($row['conversations']);
            $tokens[] = intval($row['tokens']);
        }

        // phpcs:enable WordPress.Security.NonceVerification.Missing

        wp_send_json_success(array(
            'labels' => $labels,
            'messages' => $messages,
            'conversations' => $conversations,
            'tokens' => $tokens
        ));
    }
} 
