<?php
if (!defined('ABSPATH')) {
    exit;
}

add_action('rest_api_init', function () {
    register_rest_route('99digital/v1', '/create-order', [
        'methods' => 'POST',
        'callback' => 'digital99_create_order',
        'permission_callback' => function( $request ) {
            // Allow only if a valid shared token matches the site option
            $provided = $request->get_header('X-99Digital-Token');
            if (empty($provided)) {
                $provided = isset($_GET['token']) ? sanitize_text_field(wp_unslash($_GET['token'])) : '';
            }
            $expected = get_option('digital99_store_token');
            return !empty($expected) && hash_equals((string)$expected, (string)$provided);
        },
        'args' => [
            'billing' => [
                'required' => true,
                'validate_callback' => function($value) {
                    return is_array($value) && !empty($value['first_name']) && !empty($value['phone']);
                }
            ],
            'line_items' => [
                'required' => true,
                'validate_callback' => function($value) {
                    return is_array($value) && !empty($value);
                }
            ]
        ]
    ]);
});
function digital99_create_order($request) {
    try {
        $params = $request->get_json_params();
        if (!is_array($params)) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'Invalid JSON body'
            ], 400);
        }
        if (empty($params['billing']) || empty($params['line_items'])) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'Missing billing or line_items data'
            ], 400);
        }
        
        $billing = array_map('sanitize_text_field', wp_unslash($params['billing']));
        
        // Additional validation for email
        if (!empty($billing['email'])) {
            $billing['email'] = sanitize_email($billing['email']);
            if (!is_email($billing['email'])) {
                return new WP_REST_Response([
                    'success' => false,
                    'error' => 'Invalid email address'
                ], 400);
            }
        }
        
        $line_items = $params['line_items'];
        $shipping = isset($params['shipping']) && is_array($params['shipping']) ? array_map('sanitize_text_field', wp_unslash($params['shipping'])) : $billing; 
        if (empty($billing['first_name']) || empty($billing['phone'])) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'Missing required billing fields (first_name, phone)'
            ], 400);
        }
        // Validate and sanitize line items before processing
        foreach ($line_items as $item) {
            $item = is_array($item) ? $item : [];
            $product_id = isset($item['product_id']) ? absint($item['product_id']) : 0;
            if ($product_id <= 0 || !wc_get_product($product_id)) {
                return new WP_REST_Response([
                    'success' => false,
                    'error' => 'One or more products not found in WooCommerce.'
                ], 400);
            }
        }
        $order = wc_create_order(['customer_id' => 0]);
        if (is_wp_error($order)) {
            return new WP_REST_Response([
                'success' => false,
                'error' => 'Failed to create order: ' . $order->get_error_message()
            ], 500);
        }        
        foreach ($line_items as $item) {
            if (!is_array($item)) {
                continue; // Skip invalid items
            }
            
            // Sanitize item data
            $name = isset($item['name']) ? sanitize_text_field($item['name']) : '';
            $quantity = isset($item['quantity']) ? absint($item['quantity']) : 0;
            $price = isset($item['price']) ? floatval($item['price']) : 0;
            
            if (empty($name) || $quantity <= 0 || $price < 0) {
                continue;
            }
            
            $product_id = 0;
            $product = null;            
            if (!empty($item['product_id'])) {
                $product_id = absint($item['product_id']);
                if ($product_id > 0) {
                    $product = wc_get_product($product_id);
                }
            }
            if ($product) {
                $requested_qty = $quantity;

                if ($product->managing_stock()) {
                    $stock_qty = $product->get_stock_quantity();
                    if ($stock_qty !== null && $requested_qty > $stock_qty) {
                        $requested_qty = $stock_qty;
                    }
                }

                if ($requested_qty > 0) {
                    $order->add_product($product, $requested_qty);
                } 
            }
        }            
        $order->set_address([
            'first_name' => $billing['first_name'],
            'last_name' => $billing['last_name'] ?? '',
            'company' => $billing['company'] ?? '',
            'email' => $billing['email'] ?? '',
            'phone' => $billing['phone'],
            'address_1' => $billing['address_1'] ?? '',
            'address_2' => $billing['address_2'] ?? '',
            'city' => $billing['city'] ?? '',
            'state' => $billing['state'] ?? '',
            'postcode' => $billing['postcode'] ?? '',
            'country' => $billing['country'] ?? 'IL'
        ], 'billing');        
        $order->set_address([
            'first_name' => $shipping['first_name'] ?? $billing['first_name'],
            'last_name' => $shipping['last_name'] ?? $billing['last_name'] ?? '',
            'company' => $shipping['company'] ?? '',
            'address_1' => $shipping['address_1'] ?? $billing['address_1'] ?? '',
            'address_2' => $shipping['address_2'] ?? '',
            'city' => $shipping['city'] ?? $billing['city'] ?? '',
            'state' => $shipping['state'] ?? $billing['state'] ?? '',
            'postcode' => $shipping['postcode'] ?? $billing['postcode'] ?? '',
            'country' => $shipping['country'] ?? $billing['country'] ?? 'IL'
        ], 'shipping');        
        $order->set_payment_method(isset($params['payment_method']) ? sanitize_text_field($params['payment_method']) : 'tranzila');
        $order->set_payment_method_title(isset($params['payment_method_title']) ? sanitize_text_field($params['payment_method_title']) : 'תשלום במזומן');
        $order->set_created_via('WhatsApp');        
        if (!empty($params['customer_note'])) {
            $order->set_customer_note(sanitize_text_field($params['customer_note']));
        }        
        $order->add_meta_data('_order_source', 'WhatsApp');
        $order->add_meta_data('_customer_phone', $billing['phone']);

        // Set WooCommerce Order Attribution meta for the Origin column
        $order->add_meta_data('_wc_order_attribution_source_type', 'utm');
        $order->add_meta_data('_wc_order_attribution_utm_source', 'WhatsApp');
        
        if (!empty($params['meta_data'])) {
            foreach ($params['meta_data'] as $meta) {
                if (isset($meta['key']) && isset($meta['value'])) {
                    $order->add_meta_data(sanitize_text_field($meta['key']), sanitize_text_field((string) $meta['value']));
                }
            }
        }  
        $shipping_total = isset($params['shipping_total']) ? floatval($params['shipping_total']) : 0;
        if ($shipping_total > 0) {
            $shipping_item = new WC_Order_Item_Shipping();
            $shipping_item->set_method_title('משלוח');
            $shipping_item->set_method_id('custom_shipping');
            $shipping_item->set_total($shipping_total);
            $order->add_item($shipping_item);
        }
      
        $order->calculate_totals();
        
        // Sanitize and validate order status
        $order_status = isset($params['status']) ? sanitize_text_field($params['status']) : 'pending';
        $valid_statuses = array_keys(wc_get_order_statuses());
        if (!in_array('wc-' . $order_status, $valid_statuses, true)) {
            $order_status = 'pending'; // Default to pending if invalid status
        }
        
        $order->update_status($order_status, 'הזמנה נוצרה מ-Meta');  
        $order->save();
        $order_id = $order->get_id();
        $real_order_key = $order->get_order_key();

        $existing_key = get_post_meta($order_id, '_order_key', true);

        if (empty($existing_key)) {
            update_post_meta($order_id, '_order_key', $real_order_key);
        }
        
        // Generate payment URL for customer checkout
        $payment_url = $order->get_checkout_payment_url();
        
        return new WP_REST_Response([
            'success' => true,
            'order_id' => $order_id,
            'order_key' => $real_order_key,
            'payment_url' => $payment_url,
            'message' => 'Order created successfully'
        ], 200);
        
    } catch (Exception $e) {
        return new WP_REST_Response([
            'success' => false,
            'error' => 'Internal server error: ' . $e->getMessage()
        ], 500);
    }
}
function digital99_create_virtual_product($name, $price) {
    $product = new WC_Product_Simple();
    $product->set_name($name);
    $product->set_regular_price($price);
    $product->set_virtual(true);
    $product->set_downloadable(false);
    $product->set_status('publish');
    $product->set_catalog_visibility('hidden');
    $product->set_manage_stock(false);
    $product->set_stock_status('instock');
    $term = wp_insert_term('Meta Orders', 'product_cat', [
        'description' => 'מוצרים מהזמנות Meta',
        'slug' => 'meta-orders'
    ]);
    
    if (!is_wp_error($term)) {
        $product->set_category_ids([$term['term_id']]);
    }    
    $product_id = $product->save();    
    return $product_id;
}
function digital99_cleanup_virtual_products() {
    $args = [
        'post_type' => 'product',
        'post_status' => 'publish',
        'meta_query' => [
            [
                'key' => '_virtual',
                'value' => 'yes'
            ]
        ],
        'tax_query' => [
            [
                'taxonomy' => 'product_cat',
                'field' => 'slug',
                'terms' => 'meta-orders'
            ]
        ],
        'date_query' => [
            [
                'before' => '30 days ago'
            ]
        ]
    ];
    $products = get_posts($args);
    foreach ($products as $product) {
        wp_delete_post($product->ID, true);
    }
}
add_action('wp_loaded', function() {
    if (!wp_next_scheduled('digital99_cleanup_virtual_products')) {
        wp_schedule_event(time(), 'daily', 'digital99_cleanup_virtual_products');
    }
});

add_action('digital99_cleanup_virtual_products', 'digital99_cleanup_virtual_products');
?>