<?php
if (!defined('ABSPATH')) {
    exit;
}

class Digital99_API_Handler {

    private $server_url = 'https://api.99digital.co.il/whatsapp/v2/store';
    private $processed_deletions = [];
    private $product_action_queue = [];
    private $action_lock = [];

    public function __construct() {
        // Product created
        add_action('woocommerce_new_product', array($this, 'handle_new_product'), 10, 1);
        // Product deleted
        add_action('before_delete_post', array($this, 'handle_before_delete_product'), 10, 1);
        // Post updated
        add_action('post_updated', array($this, 'handle_post_updated'), 10, 3);
        // Product props updated
        add_action('woocommerce_product_object_updated_props', array($this, 'handle_product_props_update'), 10, 2);
        // Stock changes
        add_action('woocommerce_product_set_stock', array($this, 'handle_stock_change'), 10, 1);
        add_action('woocommerce_variation_set_stock', array($this, 'handle_stock_change'), 10, 1);
        // Meta data changes
        add_action('updated_post_meta', array($this, 'handle_updated_meta'), 10, 4);
        add_action('added_post_meta', array($this, 'handle_updated_meta'), 10, 4);
        add_action('deleted_post_meta', array($this, 'handle_updated_meta'), 10, 4);
        // Term changes (categories and tags)
        add_action('set_object_terms', array($this, 'handle_set_terms'), 10, 6);
        // Process queued product updates on shutdown
        add_action('shutdown', array($this, 'process_product_queue'));
        // Order status webhooks
        add_action('woocommerce_order_status_processing', array($this, 'handle_order_status_processing'));
        add_action('woocommerce_order_status_completed', array($this, 'handle_order_status_processing'));
    }

    // =========================================================================
    // Order status webhook
    // =========================================================================

    public function handle_order_status_processing($order_id) {
        $order = wc_get_order($order_id);
        if (!$order) {
            return;
        }

        $token = get_option('digital99_store_token');
        if (empty($token)) {
            return;
        }

        $payload = array(
            'id'       => $order->get_id(),
            'order_id' => $order->get_id(),
            'token'    => $token,
            'site_url' => get_home_url(),
            'status'   => $order->get_status(),
            'billing'  => array(
                'phone'      => $order->get_billing_phone(),
                'first_name' => $order->get_billing_first_name(),
                'last_name'  => $order->get_billing_last_name(),
            ),
        );

        $webhook_url = $this->server_url . '/tranzila-webhook';
        $response = wp_remote_post($webhook_url, array(
            'headers'   => array('Content-Type' => 'application/json'),
            'body'      => wp_json_encode($payload),
            'timeout'   => 15,
            'sslverify' => true,
        ));

        if (is_wp_error($response)) {
            error_log('Digital99: Order webhook error - ' . $response->get_error_message());
        }
    }

    // =========================================================================
    // Product change handlers
    // =========================================================================

    public function handle_new_product($product_id) {
        $this->add_to_queue($product_id, 'product_created', 1);
    }

    public function handle_before_delete_product($post_id) {
        $post = get_post($post_id);
        if (!$post || $post->post_type !== 'product') {
            return;
        }
        if (in_array($post_id, $this->processed_deletions)) {
            return;
        }
        $this->processed_deletions[] = $post_id;
        $this->send_product_data($post_id, 'deleted');
    }

    public function handle_stock_change($product) {
        if (is_object($product)) {
            $product_id = method_exists($product, 'get_id') ? $product->get_id() : 0;
            if ($product_id) {
                $this->add_to_queue($product_id, 'stock_changed', 3);
            }
        }
    }

    public function handle_updated_meta($meta_id, $post_id, $meta_key, $meta_value) {
        $post = get_post($post_id);
        if (!$post || $post->post_type !== 'product' || $post->post_status === 'auto-draft') {
            return;
        }
        $meta_actions = [
            '_regular_price'         => ['action' => 'price_changed',  'priority' => 1],
            '_sale_price'            => ['action' => 'sale_price_changed', 'priority' => 1],
            '_price'                 => ['action' => 'price_changed',  'priority' => 1],
            '_stock'                 => ['action' => 'stock_changed',  'priority' => 3],
            '_stock_status'          => ['action' => 'stock_changed',  'priority' => 3],
            '_thumbnail_id'          => ['action' => 'images_changed', 'priority' => 4],
            '_product_image_gallery' => ['action' => 'images_changed', 'priority' => 4],
            '_sku'                   => ['action' => 'sku_changed',    'priority' => 3],
        ];
        if (isset($meta_actions[$meta_key])) {
            $config = $meta_actions[$meta_key];
            $this->add_to_queue($post_id, $config['action'], $config['priority']);
        }
    }

    public function handle_post_updated($post_id, $post_after, $post_before) {
        if (!$post_after || $post_after->post_type !== 'product') {
            return;
        }
        if ($post_after->post_status === 'auto-draft') {
            return;
        }
        if ($post_before->post_title !== $post_after->post_title) {
            $this->add_to_queue($post_id, 'name_changed', 2);
            return;
        }
        if ($post_before->post_content !== $post_after->post_content) {
            $this->add_to_queue($post_id, 'description_changed', 3);
            return;
        }
        if ($post_before->post_excerpt !== $post_after->post_excerpt) {
            $this->add_to_queue($post_id, 'description_changed', 3);
            return;
        }
        if ($post_before->post_status !== $post_after->post_status) {
            $this->add_to_queue($post_id, 'status_changed', 2);
            return;
        }
    }

    public function handle_product_props_update($product, $updated_props) {
        if (!is_object($product)) {
            return;
        }
        $product_id = method_exists($product, 'get_id') ? $product->get_id() : 0;
        if (!$product_id) {
            return;
        }
        $product_status = method_exists($product, 'get_status') ? $product->get_status() : '';
        if ($product_status === 'auto-draft') {
            return;
        }
        $action = $this->determine_action_priority($updated_props);
        if ($action['name']) {
            $this->add_to_queue($product_id, $action['name'], $action['priority']);
        }
    }

    public function handle_set_terms($object_id, $terms, $tt_ids, $taxonomy, $append, $old_tt_ids) {
        $post = get_post($object_id);
        if (!$post || $post->post_type !== 'product' || $post->post_status === 'auto-draft') {
            return;
        }
        sort($tt_ids);
        sort($old_tt_ids);
        if ($tt_ids === $old_tt_ids) {
            return;
        }
        if ($taxonomy === 'product_cat') {
            $this->add_to_queue($object_id, 'categories_changed', 4);
        } elseif ($taxonomy === 'product_tag') {
            $this->add_to_queue($object_id, 'tags_changed', 5);
        }
    }

    // =========================================================================
    // Product action queue
    // =========================================================================

    private function is_action_locked($product_id, $action) {
        $key = $product_id . '_' . $action;
        return isset($this->action_lock[$key]) && (time() - $this->action_lock[$key]) < 3;
    }

    private function lock_action($product_id, $action) {
        $key = $product_id . '_' . $action;
        $this->action_lock[$key] = time();
    }

    private function add_to_queue($product_id, $action, $priority = 3) {
        if ($this->is_action_locked($product_id, $action)) {
            return;
        }
        $key = $product_id . '_' . $action;
        if (isset($this->product_action_queue[$key]) &&
            $this->product_action_queue[$key]['priority'] <= $priority) {
            return;
        }
        $this->product_action_queue[$key] = [
            'product_id' => $product_id,
            'action'     => $action,
            'priority'   => $priority,
            'timestamp'  => time(),
        ];
        $this->lock_action($product_id, $action);
    }

    public function process_product_queue() {
        if (empty($this->product_action_queue)) {
            return;
        }
        $products_actions = [];
        foreach ($this->product_action_queue as $item) {
            $product_id = $item['product_id'];
            $products_actions[$product_id][] = $item;
        }
        foreach ($products_actions as $product_id => $actions) {
            usort($actions, function ($a, $b) {
                if ($a['priority'] === $b['priority']) {
                    return $b['timestamp'] - $a['timestamp'];
                }
                return $a['priority'] - $b['priority'];
            });
            $primary_action = $actions[0];
            $this->send_product_data($primary_action['product_id'], $primary_action['action']);
        }
        $this->product_action_queue = [];
    }

    // =========================================================================
    // Send product data to server
    // =========================================================================

    private function send_product_data($product_id, $action) {
        $transient_key = 'digital99_sent_' . $product_id . '_' . $action;
        if (get_transient($transient_key)) {
            return false;
        }
        set_transient($transient_key, true, 10);

        $token = get_option('digital99_store_token');
        if (empty($token)) {
            return false;
        }

        $product_data = $this->build_product_data($product_id, $action, $token);
        if (!$product_data) {
            return false;
        }

        $api_url  = $this->server_url . '/updateProduct';
        $response = wp_remote_post($api_url, [
            'body'      => wp_json_encode($product_data),
            'headers'   => ['Content-Type' => 'application/json'],
            'timeout'   => 30,
            'sslverify' => true,
        ]);

        if (is_wp_error($response)) {
            return false;
        }

        $code = wp_remote_retrieve_response_code($response);
        return $code === 200;
    }

    // =========================================================================
    // Build product data payload
    // =========================================================================

    private function build_product_data($product_id, $action, $token) {
        if ($action === 'deleted') {
            return [
                'action'     => $action,
                'product_id' => $product_id,
                'site_url'   => home_url(),
                'token'      => $token,
                'timestamp'  => gmdate('c'),
            ];
        }

        $product = wc_get_product($product_id);
        if (!$product) {
            return false;
        }

        $product_status = method_exists($product, 'get_status') ? $product->get_status() : '';
        if ($product_status === 'auto-draft') {
            return false;
        }

        return [
            'action'              => $action,
            'product_id'          => $product_id,
            'name'                => $product->get_name(),
            'slug'                => $product->get_slug(),
            'description'         => $product->get_description(),
            'short_description'   => $product->get_short_description(),
            'sku'                 => $product->get_sku(),
            'regular_price'       => $product->get_regular_price(),
            'sale_price'          => $product->get_sale_price() ? $product->get_sale_price() : null,
            'price'               => $product->get_price(),
            'stock_quantity'      => $product->get_stock_quantity(),
            'stock_status'        => $product->get_stock_status(),
            'manage_stock'        => $product->get_manage_stock(),
            'status'              => $product->get_status(),
            'type'                => $product->get_type(),
            'featured'            => $product->get_featured(),
            'catalog_visibility'  => $product->get_catalog_visibility(),
            'weight'              => $product->get_weight(),
            'dimensions'          => $this->get_product_dimensions($product),
            'categories'          => $this->get_product_categories($product),
            'tags'                => $this->get_product_tags($product),
            'images'              => $this->get_product_images($product),
            'attributes'          => $this->get_product_attributes($product),
            'variations'          => $this->get_product_variations($product),
            'date_created'        => $this->get_product_date_safe($product, 'get_date_created'),
            'date_modified'       => $this->get_product_date_safe($product, 'get_date_modified'),
            'permalink'           => $product->get_permalink(),
            'site_url'            => home_url(),
            'token'               => $token,
            'timestamp'           => gmdate('c'),
        ];
    }

    private function determine_action_priority($updated_props) {
        if (in_array('regular_price', $updated_props)) {
            return ['name' => 'price_changed', 'priority' => 1];
        }
        if (in_array('sale_price', $updated_props)) {
            return ['name' => 'sale_price_changed', 'priority' => 1];
        }
        if (in_array('name', $updated_props)) {
            return ['name' => 'name_changed', 'priority' => 2];
        }
        if (in_array('status', $updated_props)) {
            return ['name' => 'status_changed', 'priority' => 2];
        }
        if (in_array('description', $updated_props) || in_array('short_description', $updated_props)) {
            return ['name' => 'description_changed', 'priority' => 3];
        }
        if (in_array('sku', $updated_props)) {
            return ['name' => 'sku_changed', 'priority' => 3];
        }
        if (in_array('category_ids', $updated_props)) {
            return ['name' => 'categories_changed', 'priority' => 4];
        }
        if (in_array('image_id', $updated_props) || in_array('gallery_image_ids', $updated_props)) {
            return ['name' => 'images_changed', 'priority' => 4];
        }
        if (!empty($updated_props)) {
            return ['name' => 'product_updated', 'priority' => 5];
        }
        return ['name' => null, 'priority' => 0];
    }

    // =========================================================================
    // Product data helpers
    // =========================================================================

    private function get_product_dimensions($product) {
        return [
            'length' => $product->get_length(),
            'width'  => $product->get_width(),
            'height' => $product->get_height(),
        ];
    }

    private function get_product_categories($product) {
        $categories = [];
        foreach ($product->get_category_ids() as $cat_id) {
            $term = get_term($cat_id, 'product_cat');
            if ($term && !is_wp_error($term)) {
                $categories[] = [
                    'id'   => $cat_id,
                    'name' => $term->name,
                    'slug' => $term->slug,
                ];
            }
        }
        return $categories;
    }

    private function get_product_tags($product) {
        $tags = [];
        foreach ($product->get_tag_ids() as $tag_id) {
            $term = get_term($tag_id, 'product_tag');
            if ($term && !is_wp_error($term)) {
                $tags[] = [
                    'id'   => $tag_id,
                    'name' => $term->name,
                    'slug' => $term->slug,
                ];
            }
        }
        return $tags;
    }

    private function get_product_images($product) {
        $images = [];
        if ($image_id = $product->get_image_id()) {
            $image_url = wp_get_attachment_image_url($image_id, 'full');
            if ($image_url) {
                $images[] = [
                    'id'   => $image_id,
                    'url'  => $image_url,
                    'alt'  => get_post_meta($image_id, '_wp_attachment_image_alt', true),
                    'main' => true,
                ];
            }
        }
        foreach ($product->get_gallery_image_ids() as $image_id) {
            $image_url = wp_get_attachment_image_url($image_id, 'full');
            if ($image_url) {
                $images[] = [
                    'id'   => $image_id,
                    'url'  => $image_url,
                    'alt'  => get_post_meta($image_id, '_wp_attachment_image_alt', true),
                    'main' => false,
                ];
            }
        }
        return $images;
    }

    private function get_product_attributes($product) {
        $attributes = [];
        foreach ($product->get_attributes() as $attribute) {
            $attributes[] = [
                'name'      => $attribute->get_name(),
                'options'   => $attribute->get_options(),
                'visible'   => $attribute->get_visible(),
                'variation' => $attribute->get_variation(),
            ];
        }
        return $attributes;
    }

    private function get_product_variations($product) {
        $variations = [];
        if ($product->get_type() === 'variable') {
            foreach ($product->get_available_variations() as $variation_data) {
                $variation_id = $variation_data['variation_id'];
                $variation    = wc_get_product($variation_id);
                if ($variation) {
                    $variations[] = [
                        'id'             => $variation_id,
                        'sku'            => $variation->get_sku(),
                        'price'          => $variation->get_price(),
                        'regular_price'  => $variation->get_regular_price(),
                        'sale_price'     => $variation->get_sale_price(),
                        'stock_quantity' => $variation->get_stock_quantity(),
                        'stock_status'   => $variation->get_stock_status(),
                        'attributes'     => $variation_data['attributes'] ?? [],
                    ];
                }
            }
        }
        return $variations;
    }

    private function get_product_date_safe($product, $method) {
        if (!method_exists($product, $method)) {
            return '';
        }
        $date = $product->$method();
        if (!$date || !method_exists($date, 'date')) {
            return '';
        }
        return $date->date('Y-m-d H:i:s');
    }

    // =========================================================================
    // WooCommerce API key management
    // =========================================================================

    public function cleanup_api_keys() {
        global $wpdb;
        $wpdb->delete(
            $wpdb->prefix . 'woocommerce_api_keys',
            array('description' => '99דיגיטל - מפתח אוטומטי'),
            array('%s')
        );
        delete_option('digital99_consumer_key');
        delete_option('digital99_consumer_secret');
    }

    // =========================================================================
    // Uninstall cleanup
    // =========================================================================

    public static function on_uninstall() {
        $token = get_option('digital99_store_token');
        $data  = [
            'site_url'  => home_url(),
            'token'     => $token,
            'event'     => 'plugin_uninstalled',
            'timestamp' => gmdate('c'),
        ];
        wp_remote_post('https://api.99digital.co.il/whatsapp/v2/store/uninstall', [
            'body'      => wp_json_encode($data),
            'headers'   => ['Content-Type' => 'application/json'],
            'timeout'   => 30,
            'sslverify' => true,
        ]);
    }
}
