<?php
if (!defined('ABSPATH')) {
    exit;
}
class Digital99_Admin_Settings {
    
    private $api_base_url = 'https://api.99digital.co.il/whatsapp/v2/store';

    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'settings_init'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));

        add_action('wp_ajax_digital99_save_messages', array($this, 'save_messages_ajax_handler'));
        add_action('wp_ajax_digital99_load_messages', array($this, 'load_messages_ajax_handler'));
        add_action('wp_ajax_digital99_verify_token', array($this, 'verify_token_ajax_handler'));
        add_action('wp_ajax_digital99_disconnect', array($this, 'disconnect_ajax_handler'));
        add_action('wp_ajax_digital99_sync_products', array($this, 'sync_products_ajax_handler'));
    }

    public function verify_token_ajax_handler() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'digital99_meta_nonce')) {
            wp_send_json_error(array('message' => 'בדיקת אבטחה נכשלה'));
            return;
        }
        
        $token = isset($_POST['token']) ? sanitize_text_field(wp_unslash($_POST['token'])) : '';
        
        if (empty($token)) {
            wp_send_json_error(array('message' => 'נא להזין טוקן'));
            return;
        }
        
        // Get or create WooCommerce API keys
        $consumer_key = get_option('digital99_consumer_key', '');
        $consumer_secret = get_option('digital99_consumer_secret', '');
        
        // If keys don't exist, create them
        if (empty($consumer_key) || empty($consumer_secret)) {
            $this->create_woocommerce_api_keys();
            $consumer_key = get_option('digital99_consumer_key', '');
            $consumer_secret = get_option('digital99_consumer_secret', '');
        }
        
        $response = wp_remote_post($this->api_base_url . '/verifyToken', array(
            'method' => 'POST',
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/json; charset=utf-8',
            ),
            'body' => wp_json_encode(array(
                'token' => $token,
                'site_url' => get_site_url(),
                'consumer_key' => $consumer_key,
                'consumer_secret' => $consumer_secret
            )),
            'sslverify' => true
        ));
        
        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => 'שגיאה בתקשורת עם השרת: ' . $response->get_error_message()));
            return;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (!is_array($data)) {
            wp_send_json_error(array('message' => 'טוקן לא תקין'));
            return;
        }
        
        if (isset($data['success']) && $data['success'] && isset($data['data'])) {
            // Save token locally
            update_option('digital99_store_token', $token);
            
            // Return the verification data including messages
            // Product sync will be triggered via JS after successful verification
            wp_send_json_success(array(
                'onboarding_link' => isset($data['data']['onboarding_link']) ? esc_url($data['data']['onboarding_link']) : '',
                'whatsapp_number' => isset($data['data']['whatsapp_number']) ? sanitize_text_field($data['data']['whatsapp_number']) : '',
                'status' => isset($data['data']['status']) ? absint($data['data']['status']) : 0,
                'message_1_welcome' => isset($data['data']['message_1_welcome']) ? sanitize_textarea_field($data['data']['message_1_welcome']) : '',
                'message_2_cart_sent' => isset($data['data']['message_2_cart_sent']) ? sanitize_textarea_field($data['data']['message_2_cart_sent']) : '',
                'message_3_cart_not_sent' => isset($data['data']['message_3_cart_not_sent']) ? sanitize_textarea_field($data['data']['message_3_cart_not_sent']) : '',
                'message_4_summary_and_payment' => isset($data['data']['message_4_summary_and_payment']) ? sanitize_textarea_field($data['data']['message_4_summary_and_payment']) : '',
                'message_5_payment_completed' => isset($data['data']['message_5_payment_completed']) ? sanitize_textarea_field($data['data']['message_5_payment_completed']) : '',
                'message_6_no_payment' => isset($data['data']['message_6_no_payment']) ? sanitize_textarea_field($data['data']['message_6_no_payment']) : '',
                'message_7_pickup' => isset($data['data']['message_7_pickup']) ? sanitize_textarea_field($data['data']['message_7_pickup']) : '',
                'message_8_shipping' => isset($data['data']['message_8_shipping']) ? sanitize_textarea_field($data['data']['message_8_shipping']) : '',
                'message_9_shipping_price' => isset($data['data']['message_9_shipping_price']) ? sanitize_text_field($data['data']['message_9_shipping_price']) : '',
                'shipping_method' => isset($data['data']['shipping_method']) ? absint($data['data']['shipping_method']) : 1
            ));
        } else {
            $error_message = isset($data['message']) ? sanitize_text_field($data['message']) : 'הטוקן שהזנת אינו תקין';
            wp_send_json_error(array('message' => $error_message));
        }
    }
    
    /**
     * Create WooCommerce API keys for remote access
     */
    private function create_woocommerce_api_keys() {
        global $wpdb;
        
        // Check if WooCommerce is active
        if (!class_exists('WooCommerce')) {
            return false;
        }
        
        $table_name = $wpdb->prefix . 'woocommerce_api_keys';
        $table_exists = $wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name)) == $table_name;
        if (!$table_exists) {
            return false;
        }
        
        // Get an admin user
        $admin_users = get_users(array('role' => 'administrator', 'number' => 1));
        if (empty($admin_users)) {
            return false;
        }
        $user_id = $admin_users[0]->ID;
        
        // Clean up existing keys first
        $wpdb->delete(
            $wpdb->prefix . 'woocommerce_api_keys',
            array('description' => '99דיגיטל - WhatsApp Store'),
            array('%s')
        );
        delete_option('digital99_consumer_key');
        delete_option('digital99_consumer_secret');
        
        // Generate new keys
        $consumer_key = 'ck_' . wp_generate_password(32, false);
        $consumer_secret = 'cs_' . wp_generate_password(40, false);
        
        // Save to options (plain text for sending to server)
        update_option('digital99_consumer_key', $consumer_key);
        update_option('digital99_consumer_secret', $consumer_secret);
        
        // Insert into WooCommerce API keys table (hashed)
        $inserted = $wpdb->insert(
            $wpdb->prefix . 'woocommerce_api_keys',
            array(
                'user_id' => $user_id,
                'description' => '99דיגיטל - WhatsApp Store',
                'permissions' => 'read_write',
                'consumer_key' => wp_hash($consumer_key),
                'consumer_secret' => $consumer_secret,
                'truncated_key' => substr($consumer_key, -7),
                'last_access' => null
            ),
            array('%d', '%s', '%s', '%s', '%s', '%s', '%s')
        );
        
        return $inserted !== false;
    }
    
    /**
     * Sync all existing products to the server
     */
    private function sync_all_products_to_server() {
        $token = get_option('digital99_store_token');
        if (empty($token)) {
            return 0;
        }
        
        // Check if WooCommerce is active
        if (!function_exists('wc_get_products')) {
            return 0;
        }
        
        // Get all published products using WooCommerce function
        $products = wc_get_products(array(
            'status' => 'publish',
            'limit' => -1,
            'return' => 'objects'
        ));
        
        if (empty($products)) {
            return 0;
        }
        
        $synced_count = 0;
        
        foreach ($products as $product) {
            if (!$product || !is_object($product)) {
                continue;
            }
            
            $product_id = $product->get_id();
            
            // Build product data
            $product_data = array(
                'action' => 'update',
                'product_id' => $product_id,
                'name' => $product->get_name(),
                'slug' => $product->get_slug(),
                'description' => $product->get_description(),
                'short_description' => $product->get_short_description(),
                'sku' => $product->get_sku(),
                'regular_price' => $product->get_regular_price(),
                'sale_price' => $product->get_sale_price() ? $product->get_sale_price() : null,
                'price' => $product->get_price(),
                'stock_quantity' => $product->get_stock_quantity(),
                'stock_status' => $product->get_stock_status(),
                'manage_stock' => $product->get_manage_stock(),
                'status' => $product->get_status(),
                'type' => $product->get_type(),
                'featured' => $product->get_featured(),
                'catalog_visibility' => $product->get_catalog_visibility(),
                'weight' => $product->get_weight(),
                'dimensions' => array(
                    'length' => $product->get_length(),
                    'width' => $product->get_width(),
                    'height' => $product->get_height()
                ),
                'categories' => $this->get_product_categories_for_sync($product),
                'tags' => $this->get_product_tags_for_sync($product),
                'images' => $this->get_product_images_for_sync($product),
                'permalink' => $product->get_permalink(),
                'site_url' => home_url(),
                'token' => $token,
                'timestamp' => gmdate('c')
            );
            
            // Send to server
            $api_url = $this->api_base_url . '/updateProduct';
            $response = wp_remote_post($api_url, array(
                'body' => wp_json_encode($product_data),
                'headers' => array('Content-Type' => 'application/json'),
                'timeout' => 30,
                'sslverify' => true
            ));
            
            if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
                $synced_count++;
            }
            
            // Small delay between requests
            usleep(100000); // 0.1 second
        }
        
        return $synced_count;
    }
    
    private function get_product_categories_for_sync($product) {
        $categories = array();
        $term_ids = $product->get_category_ids();
        foreach ($term_ids as $term_id) {
            $term = get_term($term_id, 'product_cat');
            if ($term && !is_wp_error($term)) {
                $categories[] = array(
                    'id' => $term->term_id,
                    'name' => $term->name,
                    'slug' => $term->slug
                );
            }
        }
        return $categories;
    }
    
    private function get_product_tags_for_sync($product) {
        $tags = array();
        $term_ids = $product->get_tag_ids();
        foreach ($term_ids as $term_id) {
            $term = get_term($term_id, 'product_tag');
            if ($term && !is_wp_error($term)) {
                $tags[] = array(
                    'id' => $term->term_id,
                    'name' => $term->name,
                    'slug' => $term->slug
                );
            }
        }
        return $tags;
    }
    
    private function get_product_images_for_sync($product) {
        $images = array();
        $image_id = $product->get_image_id();
        if ($image_id) {
            $images[] = array(
                'id' => $image_id,
                'url' => wp_get_attachment_url($image_id),
                'alt' => get_post_meta($image_id, '_wp_attachment_image_alt', true),
                'main' => true
            );
        }
        $gallery_ids = $product->get_gallery_image_ids();
        foreach ($gallery_ids as $gallery_id) {
            $images[] = array(
                'id' => $gallery_id,
                'url' => wp_get_attachment_url($gallery_id),
                'alt' => get_post_meta($gallery_id, '_wp_attachment_image_alt', true),
                'main' => false
            );
        }
        return $images;
    }
    
    public function disconnect_ajax_handler() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'digital99_meta_nonce')) {
            wp_send_json_error(array('message' => 'בדיקת אבטחה נכשלה'));
            return;
        }
        
        // Remove the stored token
        delete_option('digital99_store_token');
        
        wp_send_json_success(array('message' => 'החיבור נותק בהצלחה'));
    }
    
    public function sync_products_ajax_handler() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'digital99_meta_nonce')) {
            wp_send_json_error(array('message' => 'בדיקת אבטחה נכשלה'));
            return;
        }
        
        // Check if WooCommerce is active
        if (!function_exists('wc_get_products')) {
            wp_send_json_error(array('message' => 'WooCommerce לא פעיל'));
            return;
        }
        
        // Check token
        $token = get_option('digital99_store_token');
        if (empty($token)) {
            wp_send_json_error(array('message' => 'טוקן לא נמצא'));
            return;
        }
        
        // Count products
        $products = wc_get_products(array(
            'status' => 'publish',
            'limit' => -1,
            'return' => 'ids'
        ));
        
        $total_products = is_array($products) ? count($products) : 0;
        
        if ($total_products === 0) {
            wp_send_json_success(array(
                'message' => 'אין מוצרים מפורסמים לסנכרון',
                'synced_count' => 0,
                'total_products' => 0
            ));
            return;
        }
        
        $synced_count = $this->sync_all_products_to_server();
        
        wp_send_json_success(array(
            'message' => 'סנכרון הושלם בהצלחה',
            'synced_count' => $synced_count,
            'total_products' => $total_products
        ));
    }

    public function add_admin_menu() {
        add_options_page(
            'WhatsApp Store Settings',
            'WhatsApp Store',
            'manage_options',
            'digital99-settings',
            array($this, 'settings_page')
        );
    }

    public function settings_init() {
        register_setting(
            'digital99_settings',
            'digital99_store_token',
            array(
                'sanitize_callback' => 'sanitize_text_field'
            )
        );

        add_settings_section(
            'digital99_section',
            'הגדרות חנות WhatsApp',
            null,
            'digital99-settings'
        );

        add_settings_field(
            'store_token',
            'תוקן חנות',
            array($this, 'token_field_callback'),
            'digital99-settings',
            'digital99_section'
        );
    }

    public function enqueue_admin_scripts($hook) {
        if ($hook !== 'settings_page_digital99-settings') {
            return;
        }

        wp_enqueue_style('digital99-admin-settings', plugin_dir_url(__FILE__) . '../assets/css/admin-settings.css', array(), '2.0.1');

        wp_enqueue_script('digital99-admin-settings', plugin_dir_url(__FILE__) . '../assets/js/admin-settings.js', array('jquery'), '2.0.1', true);

        wp_localize_script('digital99-admin-settings', 'digital99_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('digital99_meta_nonce'),
            'stored_token' => get_option('digital99_store_token', '')
        ));
    }

    public function save_messages_ajax_handler() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'digital99_meta_nonce')) {
            wp_die('Unauthorized');
        }
        
        // Define all message fields with their new names
        $message_fields = array(
            'message_1_welcome',
            'message_2_cart_sent',
            'message_3_cart_not_sent',
            'message_4_summary_and_payment',
            'message_5_payment_completed',
            'message_6_no_payment',
            'message_7_pickup',
            'message_8_shipping',
            'message_9_shipping_price'
        );
        
        // Validate all required fields are not empty
        $required_fields = array(
            'message_1_welcome',
            'message_2_cart_sent',
            'message_3_cart_not_sent',
            'message_4_summary_and_payment',
            'message_5_payment_completed',
            'message_6_no_payment',
            'message_8_shipping',
            'message_9_shipping_price'
        );
        $empty_fields = array();
        
        foreach ($required_fields as $field) {
            $value = isset($_POST[$field]) ? trim(sanitize_textarea_field(wp_unslash($_POST[$field]))) : '';
            if (empty($value)) {
                $empty_fields[] = $field;
            }
        }
        
        if (!empty($empty_fields)) {
            wp_send_json_error(array(
                'message' => 'כל השדות חייבים להיות ממולאים'
            ));
            return;
        }
        
        // Build messages array with new field names
        $messages = array();
        foreach ($message_fields as $field) {
            $messages[$field] = isset($_POST[$field]) ? sanitize_textarea_field(wp_unslash($_POST[$field])) : '';
        }
        
        // Add shipping_method as integer
        $shipping_method = isset($_POST['shipping_method']) ? absint($_POST['shipping_method']) : 1;
        if ($shipping_method < 1 || $shipping_method > 3) {
            $shipping_method = 1;
        }
        $messages['shipping_method'] = $shipping_method;
        
        // Build request data according to API spec
        $data_to_send = array_merge($messages, array(
            'site_url' => get_site_url()
        ));
        
        $response = $this->send_to_external_server($data_to_send);
        if ($response && isset($response['success']) && $response['success']) {
            wp_send_json_success(array(
                'message' => 'ההודעות נשמרו בהצלחה'
            ));
        } else {
            $error_msg = isset($response['message']) ? $response['message'] : 'שגיאה בשמירת הנתונים';
            wp_send_json_error(array(
                'message' => $error_msg
            ));
        }
    }

    public function load_messages_ajax_handler() {
        if (!isset($_POST['nonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'digital99_meta_nonce')) {
            wp_die('Unauthorized');
        }
        
        $token = get_option('digital99_store_token');
        if (empty($token)) {
            wp_send_json_error(array('message' => 'לא נמצא טוקן'));
            return;
        }
        
        // Use GET request to /siteData endpoint with query params
        $url = add_query_arg(array(
            'token' => $token,
            'site_url' => get_site_url()
        ), $this->api_base_url . '/siteData');
        
        $response = wp_remote_get($url, array(
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/json; charset=utf-8',
            ),
            'sslverify' => true
        ));
        
        if (is_wp_error($response)) {
            wp_send_json_error(array('message' => 'שגיאה בתקשורת עם השרת'));
            return;
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (isset($data['success']) && $data['success'] && isset($data['data'])) {
            $site_data = $data['data'];
            $messages = array();
            
            // New message field names
            $message_fields = array(
                'message_1_welcome',
                'message_2_cart_sent',
                'message_3_cart_not_sent',
                'message_4_summary_and_payment',
                'message_5_payment_completed',
                'message_6_no_payment',
                'message_7_pickup',
                'message_8_shipping',
                'message_9_shipping_price'
            );
            
            foreach ($message_fields as $field) {
                $messages[$field] = $site_data[$field] ?? '';
            }
            $messages['shipping_method'] = $site_data['shipping_method'] ?? 1;
            $messages['phone'] = $site_data['phone'] ?? '';
            
            wp_send_json_success($messages);
        } else {
            wp_send_json_error(array(
                'message' => 'לא ניתן לטעון הודעות מהשרת'
            ));
        }
    }

    private function send_to_external_server($data) {
        $url = $this->api_base_url . '/siteData';
        
        // SECURITY: Add token for authentication
        $token = get_option('digital99_store_token');
        if (empty($token)) {
            return false;
        }
        $data['token'] = $token;
        
        $response = wp_remote_post($url, array(
            'method' => 'POST',
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/json; charset=utf-8',
            ),
            'body' => wp_json_encode($data, JSON_UNESCAPED_UNICODE),
            'sslverify' => true
        ));

        if (is_wp_error($response)) {
            return false;
        }

        $body = wp_remote_retrieve_body($response);
        $decoded = json_decode($body, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            return false;
        }

        // Validate and sanitize the response data
        if (!is_array($decoded)) {
            return false;
        }

        return $decoded;
    }

    public function token_field_callback() {
        $token = get_option('digital99_store_token');
        echo '<input type="text" name="digital99_store_token" value="' . esc_attr($token) . '" size="50" style="direction: ltr; text-align: left;" />';
        echo '<p class="description">הזן את התוקן שקיבלת </p>';
    }
    
    private function get_connection_status() {
        $token = get_option('digital99_store_token');
        
        if (empty($token)) {
            return array(
                'connected' => false,
                'status' => 0,
                'whatsapp_number' => '',
                'onboarding_link' => '',
                'messages' => array()
            );
        }
        
        // Get consumer credentials
        $consumer_key = get_option('digital99_consumer_key', '');
        $consumer_secret = get_option('digital99_consumer_secret', '');
        
        // Check with server
        $response = wp_remote_post($this->api_base_url . '/verifyToken', array(
            'method' => 'POST',
            'timeout' => 15,
            'headers' => array(
                'Content-Type' => 'application/json; charset=utf-8',
            ),
            'body' => wp_json_encode(array(
                'token' => $token,
                'site_url' => get_site_url(),
                'consumer_key' => $consumer_key,
                'consumer_secret' => $consumer_secret
            )),
            'sslverify' => true
        ));
        
        if (is_wp_error($response)) {
            return array(
                'connected' => false,
                'status' => 0,
                'whatsapp_number' => '',
                'onboarding_link' => '',
                'messages' => array(),
                'error' => $response->get_error_message()
            );
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (isset($data['success']) && $data['success'] && isset($data['data'])) {
            return array(
                'connected' => true,
                'status' => isset($data['data']['status']) ? absint($data['data']['status']) : 0,
                'whatsapp_number' => isset($data['data']['whatsapp_number']) ? sanitize_text_field($data['data']['whatsapp_number']) : '',
                'onboarding_link' => isset($data['data']['onboarding_link']) ? esc_url($data['data']['onboarding_link']) : '',
                'messages' => array(
                    'message_1_welcome' => isset($data['data']['message_1_welcome']) ? $data['data']['message_1_welcome'] : '',
                    'message_2_cart_sent' => isset($data['data']['message_2_cart_sent']) ? $data['data']['message_2_cart_sent'] : '',
                    'message_3_cart_not_sent' => isset($data['data']['message_3_cart_not_sent']) ? $data['data']['message_3_cart_not_sent'] : '',
                    'message_4_summary_and_payment' => isset($data['data']['message_4_summary_and_payment']) ? $data['data']['message_4_summary_and_payment'] : '',
                    'message_5_payment_completed' => isset($data['data']['message_5_payment_completed']) ? $data['data']['message_5_payment_completed'] : '',
                    'message_6_no_payment' => isset($data['data']['message_6_no_payment']) ? $data['data']['message_6_no_payment'] : '',
                    'message_7_pickup' => isset($data['data']['message_7_pickup']) ? $data['data']['message_7_pickup'] : '',
                    'message_8_shipping' => isset($data['data']['message_8_shipping']) ? $data['data']['message_8_shipping'] : '',
                    'message_9_shipping_price' => isset($data['data']['message_9_shipping_price']) ? $data['data']['message_9_shipping_price'] : ''
                ),
                'shipping_method' => isset($data['data']['shipping_method']) ? absint($data['data']['shipping_method']) : 1
            );
        }
        
        return array(
            'connected' => false,
            'status' => 0,
            'whatsapp_number' => '',
            'onboarding_link' => '',
            'messages' => array()
        );
    }

    public function settings_page() {
        $connection = $this->get_connection_status();
        $stored_token = get_option('digital99_store_token');
        ?>
        <div class="wrap digital99-wrap">
            <h1 class="wp-heading-inline">חנות WhatsApp - 99דיגיטל</h1>
            <hr class="wp-header-end">
            
            <?php if (!$connection['connected'] || $connection['status'] != 1): ?>
                <!-- Not Connected / Pending State -->
                <div class="digital99-card">
                    <h2>חיבור לחנות WhatsApp</h2>
                    
                    <?php if ($connection['connected'] && $connection['status'] == 0 && !empty($connection['onboarding_link'])): ?>
                        <!-- Pending Onboarding State -->
                        <div class="notice notice-warning inline">
                            <p>⏳ <strong>נדרשת השלמת תהליך החיבור</strong></p>
                        </div>
                        
                        <table class="form-table">
                            <tr>
                                <th scope="row">מספר WhatsApp</th>
                                <td>
                                    <code dir="ltr"><?php echo esc_html($connection['whatsapp_number']); ?></code>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">קישור להשלמת החיבור</th>
                                <td>
                                    <a href="<?php echo esc_url($connection['onboarding_link']); ?>" target="_blank" class="button button-primary">
                                        🔗 פתח את דף החיבור
                                    </a>
                                    <p class="description">לחץ להשלמת תהליך ההתחברות ב-Meta</p>
                                </td>
                            </tr>
                        </table>
                        
                        <p class="submit">
                            <button id="check-onboarding-btn" class="button button-secondary" type="button">
                                בדוק סטטוס חיבור
                            </button>
                            <button id="disconnect-btn" class="button" type="button">התחל מחדש</button>
                        </p>
                        <div id="onboarding-status"></div>
                        
                    <?php else: ?>
                        <!-- Not Connected State - Token Input -->
                        <table class="form-table">
                            <tr>
                                <th scope="row"><label for="store-token">טוקן חנות</label></th>
                                <td>
                                    <input type="text" id="store-token" name="store_token" class="regular-text" placeholder="הזן טוקן כאן..." value="" dir="ltr" />
                                    <p class="description">הטוקן יסופק לך על ידי צוות טרנזילה</p>
                                </td>
                            </tr>
                        </table>
                        
                        <p class="submit">
                            <button id="verify-token-btn" class="button button-primary" type="button">
                                אמת טוקן והתחבר
                            </button>
                        </p>
                        <div id="token-status"></div>
                    <?php endif; ?>
                </div>
                
            <?php else: ?>
                <!-- Connected State - Show Tabs -->
                <nav class="nav-tab-wrapper wp-clearfix">
                    <a href="#settings" class="nav-tab nav-tab-active" data-tab="settings">⚙️ הגדרות</a>
                    <a href="#messages" class="nav-tab" data-tab="messages">💬 הודעות צ'אטבוט</a>
                </nav>
                
                <!-- Settings Tab -->
                <div id="tab-settings" class="digital99-tab-content active">
                    <div class="digital99-card">
                        <h2>סטטוס חיבור</h2>
                        
                        <div class="notice notice-success inline">
                            <p>✅ <strong>מחובר בהצלחה!</strong></p>
                        </div>
                        
                        <table class="form-table">
                            <tr>
                                <th scope="row">מספר WhatsApp</th>
                                <td>
                                    <code dir="ltr"><?php echo esc_html($connection['whatsapp_number']); ?></code>
                                </td>
                            </tr>
                            <tr>
                                <th scope="row">טוקן חנות</th>
                                <td>
                                    <input type="text" value="<?php echo esc_attr($stored_token); ?>" class="regular-text" readonly disabled dir="ltr" />
                                    <span class="dashicons dashicons-lock" style="color: #666; vertical-align: middle;"></span>
                                </td>
                            </tr>
                        </table>
                    </div>
                    
                    <div class="digital99-card">
                        <h2>סנכרון מוצרים</h2>
                        <p>סנכרן את כל המוצרים מהחנות לשרת WhatsApp</p>
                        
                        <p class="submit">
                            <button id="sync-products-btn" class="button button-primary" type="button">
                                📦 סנכרן כל המוצרים
                            </button>
                        </p>
                        <div id="sync-products-status"></div>
                    </div>
                    
                    <div class="digital99-card">
                        <h2>ניתוק חיבור</h2>
                        <p>ניתוק החיבור ימחק את הטוקן השמור. תצטרך להזין טוקן חדש כדי להתחבר מחדש.</p>
                        <p class="submit">
                            <button id="disconnect-btn" class="button button-secondary" type="button">🔓 נתק חיבור</button>
                        </p>
                    </div>
                </div>
                
                <!-- Messages Tab -->
                <div id="tab-messages" class="digital99-tab-content" style="display: none;">
                    <div class="digital99-card">
                        <h2>הודעות צ'אטבוט</h2>
                        <p>התאם אישית את ההודעות שיישלחו ללקוחות שלך בוואטסאפ</p>
                        
                        <div id="messages-loading" style="display: none;">
                            <span class="spinner is-active" style="float: right;"></span>
                            <span>טוען הודעות מהשרת...</span>
                        </div>
                        
                        <form id="digital99-messages-form">
                            <table class="form-table">
                                <tr>
                                    <th scope="row"><label for="message_1_welcome">👋 הודעת פתיחה</label></th>
                                    <td>
                                        <textarea id="message_1_welcome" name="message_1_welcome" rows="3" class="large-text"></textarea>
                                        <p class="description">הודעה כאשר הלקוח שולח הודעה ראשונית</p>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><label for="message_2_cart_sent">🛒 הודעה לעגלת קניות</label></th>
                                    <td>
                                        <textarea id="message_2_cart_sent" name="message_2_cart_sent" rows="3" class="large-text"></textarea>
                                        <p class="description">הודעה כאשר הלקוח שולח עגלה עם מוצרים</p>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><label for="message_3_cart_not_sent">❓ הודעה ללא עגלה</label></th>
                                    <td>
                                        <textarea id="message_3_cart_not_sent" name="message_3_cart_not_sent" rows="3" class="large-text"></textarea>
                                        <p class="description">הלקוח שלח הודעה רגילה במקום עגלת קניות</p>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><label for="message_4_summary_and_payment">💳 סיכום ותשלום</label></th>
                                    <td>
                                        <textarea id="message_4_summary_and_payment" name="message_4_summary_and_payment" rows="3" class="large-text"></textarea>
                                        <p class="description">הצגת סכום סופי ולינק לתשלום</p>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><label for="message_5_payment_completed">✅ תשלום הושלם</label></th>
                                    <td>
                                        <textarea id="message_5_payment_completed" name="message_5_payment_completed" rows="3" class="large-text"></textarea>
                                        <p class="description">הודעה כאשר התשלום בוצע בהצלחה</p>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><label for="message_6_no_payment">⏳ תשלום ממתין</label></th>
                                    <td>
                                        <textarea id="message_6_no_payment" name="message_6_no_payment" rows="3" class="large-text"></textarea>
                                        <p class="description">הלקוח שלח הודעה במקום לבצע תשלום</p>
                                    </td>
                                </tr>
                            </table>
                        </form>
                    </div>
                    
                    <div class="digital99-card">
                        <h2>הגדרות משלוח</h2>
                        
                        <form id="shipping-settings-form">
                            <table class="form-table">
                                <tr>
                                    <th scope="row"><label for="shipping_method">סוג משלוח</label></th>
                                    <td>
                                        <select id="shipping_method" name="shipping_method" class="regular-text">
                                            <option value="1">איסוף עצמי בלבד</option>
                                            <option value="2">משלוח בלבד</option>
                                            <option value="3">משלוח או איסוף עצמי</option>
                                        </select>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><label for="message_9_shipping_price">מחיר משלוח</label></th>
                                    <td>
                                        <input type="text" id="message_9_shipping_price" name="message_9_shipping_price" class="small-text" placeholder="0" dir="ltr" />
                                        <span>₪</span>
                                        <p class="description">המחיר למשלוח בש"ח</p>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><label for="message_8_shipping">הודעת בקשת כתובת</label></th>
                                    <td>
                                        <textarea id="message_8_shipping" name="message_8_shipping" rows="3" class="large-text"></textarea>
                                        <p class="description">בקשה מהלקוח להזין כתובת מלאה למשלוח</p>
                                    </td>
                                </tr>
                                <tr>
                                    <th scope="row"><label for="message_7_pickup">כתובת לאיסוף עצמי</label></th>
                                    <td>
                                        <textarea id="message_7_pickup" name="message_7_pickup" rows="3" class="large-text"></textarea>
                                        <p class="description">כתובת לאיסוף עצמי (אופציונלי)</p>
                                    </td>
                                </tr>
                            </table>
                        </form>
                    </div>
                    
                    <!-- Save Button for Messages Tab -->
                    <div id="digital99-save-bar" class="digital99-save-bar">
                        <button type="button" id="save-messages-btn" class="button button-primary button-large">
                            💾 שמור שינויים
                        </button>
                        <span id="save-messages-status"></span>
                    </div>
                </div>
                
            <?php endif; ?>
        </div>

        <?php
    }
}
?>