<?php
/*
Plugin Name: 103Graphics Media Inspector
Description: Fast, accurate media library analytics and export for WordPress. View, filter, and report on images, PDFs, and videos.
Version: 1.0.2
Author: Mamdouh Kaldas
Text Domain: 103graphics-media-inspector
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
*/

if (!defined('ABSPATH')) {
    exit;
}

class Graphics_Media_Inspector_Plugin {
    const ORIGINAL_META    = '_mfs_orig_filesize';
    const TOTAL_META       = '_mfs_filesize';
    const VARIANTS_META    = '_mfs_variant_sizes';
    const BATCH_PER_PAGE   = 200;
    const CRON_HOOK        = 'gmi_batch_index';

    public function __construct() {
        add_filter('manage_upload_columns', [$this, 'add_filesize_column']);
        add_action('manage_media_custom_column', [$this, 'render_filesize_column'], 10, 2);
        add_filter('manage_upload_sortable_columns', [$this, 'make_filesize_sortable']);
        add_action('pre_get_posts', [$this, 'enable_filesize_sorting']);
        add_action('add_attachment', [$this, 'save_attachment_filesize']);
        add_action('edit_attachment', [$this, 'save_attachment_filesize']);
        add_action('delete_attachment', [$this, 'cleanup_attachment_meta']);
        add_action('admin_menu', [$this, 'add_options_page']);
        add_action('admin_init', [$this, 'handle_csv_export']);
        add_action(self::CRON_HOOK, [$this, 'batch_index_attachments'], 10, 1);
        add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_assets']);
    }

    public function enqueue_admin_assets($hook) {
        if ($hook !== 'media_page_103graphics-media-inspector') {
            return;
        }
        wp_register_style('103graphics-media-inspector-admin', false, [], null);
        wp_enqueue_style('103graphics-media-inspector-admin');
        $css_normal = '
.mfs-options-bar { display:flex; flex-wrap:wrap; gap:18px; align-items:flex-end; margin-bottom:16px; }
.mfs-options-bar label { font-weight:600; color:#333; display:flex; align-items:center; gap:6px; }
.mfs-options-bar input[type="number"], .mfs-options-bar select {
    height:34px; padding:6px 8px; border-radius:4px; border:1px solid #ccd0d4; background:#fff;
}
.mfs-adv-controls { display:flex; gap:12px; align-items:center; flex-wrap:wrap; }
.mfs-largest-table img { max-width:80px; height:auto; display:block; border-radius:4px; }
.mfs-variants { margin-top:8px; display:none; }
.mfs-toggle { margin-left:8px; }
@media (max-width:980px) {
    .mfs-options-bar { flex-direction:column; align-items:stretch; }
    .mfs-adv-controls { flex-direction:column; align-items:stretch; }
}
';
        wp_add_inline_style('103graphics-media-inspector-admin', $css_normal);

        if (isset($_GET['mfs_print_mode']) && $_GET['mfs_print_mode'] == '1') {
            $css_print = '
#adminmenumain, #adminmenuwrap, #adminmenu, #wpadminbar, #wpfooter, #screen-meta, .mfs-options-bar, .mfs-group-card form, .mfs-options-bar .button { display:none !important; }
body, .wrap { background:#fff !important; }
.mfs-group-card { border:none !important; box-shadow:none !important; padding:0 !important; margin:0 !important; max-width:100% !important; }
.mfs-largest-table { font-size:15px; margin-left:auto !important; margin-right:auto !important; float:none !important; }
.mfs-print-meta { margin-bottom:28px;text-align:center; }
@media print {
    .mfs-largest-table th, .mfs-largest-table td { font-size:13px; }
}
';
            wp_add_inline_style('103graphics-media-inspector-admin', $css_print);
        }

        wp_register_script('103graphics-media-inspector-admin', '', [], null, true);
        wp_enqueue_script('103graphics-media-inspector-admin');
        $js  = 'document.addEventListener("DOMContentLoaded", function(){' . "\n";
        $js .= '    document.querySelectorAll(\'.mfs-toggle\').forEach(function(btn){' . "\n";
        $js .= '        btn.addEventListener(\'click\', function(e){' . "\n";
        $js .= '            e.preventDefault();' . "\n";
        $js .= '            var target = this.getAttribute(\'data-target\');' . "\n";
        $js .= '            var el = document.getElementById(target);' . "\n";
        $js .= '            if (!el) return;' . "\n";
        $js .= '            if (el.style.display === \'block\') {' . "\n";
        $js .= '                el.style.display = \'none\';' . "\n";
        $js .= '                this.textContent = \'Show Variants\';' . "\n";
        $js .= '            } else {' . "\n";
        $js .= '                el.style.display = \'block\';' . "\n";
        $js .= '                this.textContent = \'Hide Variants\';' . "\n";
        $js .= '            }' . "\n";
        $js .= '        });' . "\n";
        $js .= '    });' . "\n";
        $js .= '    if (window.location.search.indexOf("mfs_print_mode=1") !== -1) {' . "\n";
        $js .= '        window.onload = function(){window.print();};' . "\n";
        $js .= '    }' . "\n";
        $js .= '    var adv = document.getElementById(\'mfs-adv\');' . "\n";
        $js .= '    if (adv && location.search.match(/[&?](from|to|search)=/)) {' . "\n";
        $js .= '        adv.style.display = \'flex\';' . "\n";
        $js .= '    }' . "\n";
        $js .= '});' . "\n";
        wp_add_inline_script('103graphics-media-inspector-admin', $js);
    }

    private function get_mime_types_for($media_type) {
        switch ($media_type) {
            case 'image':
                return ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
            case 'pdf':
                return ['application/pdf'];
            case 'video':
                return ['video/mp4', 'video/quicktime', 'video/webm', 'video/mpeg', 'video/x-msvideo', 'video/x-matroska'];
            default:
                return [];
        }
    }

    private function build_media_query_args($count, $search, $mime_types, $date_query, $min_bytes) {
        $args = [
            'post_type'      => 'attachment',
            'post_status'    => 'inherit',
            'meta_key'       => self::ORIGINAL_META,
            'orderby'        => 'meta_value_num',
            'order'          => 'DESC',
            'posts_per_page' => $count,
            'fields'         => 'all',
        ];
        if (!empty($mime_types)) {
            $args['post_mime_type'] = $mime_types;
        }
        $meta_query = [];
        if ($min_bytes > 0) {
            $meta_query[] = [
                'key'     => self::ORIGINAL_META,
                'value'   => $min_bytes,
                'type'    => 'NUMERIC',
                'compare' => '>=',
            ];
        }
        if ($meta_query) {
            $args['meta_query'] = $meta_query;
        }
        return $args;
    }

    public function add_filesize_column($columns) {
        $columns['filesize'] = __('Original Size', '103graphics-media-inspector');
        return $columns;
    }

    public function render_filesize_column($column_name, $post_id) {
        if ('filesize' !== $column_name) {
            return;
        }
        $orig_size = get_post_meta($post_id, self::ORIGINAL_META, true);
        if ($orig_size === '') {
            $file = get_attached_file($post_id);
            if ($file && is_file($file)) {
                $orig_size = filesize($file);
                if (false !== $orig_size) {
                    update_post_meta($post_id, self::ORIGINAL_META, (int) $orig_size);
                }
            } else {
                $orig_size = false;
            }
        }
        if ($orig_size && is_numeric($orig_size)) {
            echo esc_html(size_format((int) $orig_size, 2));
        } else {
            echo '&mdash;';
        }
    }

    public function make_filesize_sortable($columns) {
        $columns['filesize'] = 'orig_filesize';
        return $columns;
    }

    public function enable_filesize_sorting($query) {
        global $pagenow;
        if (!is_admin()) return;
        if ($pagenow !== 'upload.php') return;
        if (!$query->is_main_query()) return;
        if (isset($_GET['orderby']) && in_array($_GET['orderby'], ['orig_filesize', 'filesize'], true)) {
            $query->set('meta_key', self::ORIGINAL_META);
            $query->set('orderby', 'meta_value_num');
        }
    }

    public function save_attachment_filesize($post_id) {
        if (!$post_id || get_post_type($post_id) !== 'attachment') {
            return;
        }
        $file = get_attached_file($post_id);
        if (!$file || !is_file($file)) {
            update_post_meta($post_id, self::ORIGINAL_META, '');
            return;
        }
        $orig_size = filesize($file);
        update_post_meta($post_id, self::ORIGINAL_META, (int) $orig_size);

        $total = $this->get_total_filesize($post_id);
        if ($total !== false) {
            update_post_meta($post_id, self::TOTAL_META, (int) $total);
        } else {
            update_post_meta($post_id, self::TOTAL_META, (int) $orig_size);
        }
    }

    public function get_total_filesize($post_id) {
        $file = get_attached_file($post_id);
        if (!$file || !is_file($file)) {
            return false;
        }
        $total = (int) filesize($file);
        $meta  = wp_get_attachment_metadata($post_id);
        if (!empty($meta['sizes']) && is_array($meta['sizes'])) {
            foreach ($meta['sizes'] as $variant) {
                if (empty($variant['file'])) {
                    continue;
                }
                $variant_file = path_join(dirname($file), $variant['file']);
                if (is_file($variant_file)) {
                    $total += (int) filesize($variant_file);
                }
            }
        }
        return $total;
    }

    public static function activation() {
        if (!wp_next_scheduled(self::CRON_HOOK)) {
            wp_schedule_single_event(time() + 30, self::CRON_HOOK, [0]);
        }
    }
    public static function deactivation() {
        wp_clear_scheduled_hook(self::CRON_HOOK);
    }

    public function batch_index_attachments($offset = 0) {
        $offset = (int) $offset;
        $per_page = self::BATCH_PER_PAGE;
        $attachments = get_posts([
            'post_type'      => 'attachment',
            'post_status'    => 'inherit',
            'posts_per_page' => $per_page,
            'offset'         => $offset,
            'fields'         => 'ids',
        ]);
        if (empty($attachments)) return;
        foreach ($attachments as $id) {
            $this->save_attachment_filesize($id);
        }
        if (count($attachments) === $per_page) {
            wp_schedule_single_event(time() + 5, self::CRON_HOOK, [$offset + $per_page]);
        }
    }

    public function cleanup_attachment_meta($post_id) {
        delete_post_meta($post_id, self::ORIGINAL_META);
        delete_post_meta($post_id, self::TOTAL_META);
        delete_post_meta($post_id, self::VARIANTS_META);
    }

    public function add_options_page() {
        add_media_page(
            __('103Graphics Media Inspector', '103graphics-media-inspector'),
            __('103Graphics Media Inspector', '103graphics-media-inspector'),
            'manage_options',
            '103graphics-media-inspector',
            [$this, 'render_options_page']
        );
    }

    public function handle_csv_export() {
        if (empty($_GET['mfs_export_csv']) || empty($_GET['page']) || $_GET['page'] !== '103graphics-media-inspector') {
            return;
        }
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Unauthorized', '103graphics-media-inspector'), '', ['response' => 403]);
        }
        if (empty($_REQUEST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_REQUEST['_wpnonce'])), 'gmi_export_csv')) {
            wp_die(esc_html__('Security check failed', '103graphics-media-inspector'), '', ['response' => 403]);
        }
        while (ob_get_level()) {
            ob_end_clean();
        }
        $count         = isset($_GET['count']) && is_numeric($_GET['count']) && intval($_GET['count']) > 0 ? intval($_GET['count']) : 20;
        $min_kb        = isset($_GET['min_kb']) && is_numeric($_GET['min_kb']) ? intval($_GET['min_kb']) : 0;
        $media_type    = isset($_GET['media_type']) ? sanitize_text_field(wp_unslash($_GET['media_type'])) : 'all';
        $show_variants = isset($_GET['show_variants']) && $_GET['show_variants'] === '1';
        $min_bytes     = $min_kb * 1024;
        $mime_types = $this->get_mime_types_for($media_type);
        $args = $this->build_media_query_args($count, '', $mime_types, [], $min_bytes);
        $filename = sanitize_file_name('media-inspector-export-' . date_i18n('Y-m-d') . '.csv');
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . esc_attr($filename) . '"');
        $out = fopen('php://output', 'w');
        if (!$out) {
            wp_die(esc_html__('Could not open output stream', '103graphics-media-inspector'));
        }
        $columns = ['ID', 'Title', 'Author', 'Date', 'File Name', 'Original Size', 'Total Size'];
        if ($show_variants) {
            $columns[] = 'Variant Details';
        }
        fputcsv($out, $columns);
        $query = new WP_Query($args);
        if ($query->have_posts()) {
            foreach ($query->posts as $attachment) {
                $meta       = wp_get_attachment_metadata($attachment->ID);
                $file       = get_attached_file($attachment->ID);
                $orig_size  = get_post_meta($attachment->ID, self::ORIGINAL_META, true);
                $total_size = get_post_meta($attachment->ID, self::TOTAL_META, true);
                $author     = get_the_author_meta('display_name', $attachment->post_author);
                $row = [
                    $attachment->ID,
                    $attachment->post_title,
                    $author,
                    get_the_date('Y-m-d', $attachment->ID),
                    $file ? basename($file) : '',
                    $orig_size ? size_format((int)$orig_size, 2) : '',
                    $total_size ? size_format((int)$total_size, 2) : '',
                ];
                if ($show_variants) {
                    $variant_arr = [];
                    if (!empty($meta['sizes']) && is_array($meta['sizes'])) {
                        $variant_list = [];
                        foreach ($meta['sizes'] as $size_name => $variant) {
                            if (empty($variant['file'])) continue;
                            $variant_file = $file ? path_join(dirname($file), $variant['file']) : '';
                            if ($variant_file && is_file($variant_file)) {
                                $variant_list[] = [
                                    'size_name' => $size_name,
                                    'file'      => $variant['file'],
                                    'size'      => filesize($variant_file),
                                ];
                            }
                        }
                        usort($variant_list, function($a, $b) {
                            return $b['size'] - $a['size'];
                        });
                        foreach ($variant_list as $v) {
                            $variant_arr[] = "{$v['size_name']} ({$v['file']}): " . size_format((int)$v['size'], 2);
                        }
                    }
                    $row[] = implode(' | ', $variant_arr);
                }
                fputcsv($out, $row);
            }
        }
        // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- fclose() on php://output allowed
        fclose($out);
        exit;
    }

    public function render_options_page() {
        if (!current_user_can('manage_options')) {
            wp_die(esc_html__('Unauthorized', '103graphics-media-inspector'), '', ['response' => 403]);
        }
        $is_print = !empty($_GET['mfs_print_mode']);

        // STRICT NONCE: ONLY process filters if valid nonce present, else show default view
        $media_inspector_nonce = isset($_GET['media_inspector_nonce']) ? sanitize_text_field(wp_unslash($_GET['media_inspector_nonce'])) : '';
        $filter_nonce_ok = $media_inspector_nonce && wp_verify_nonce($media_inspector_nonce, 'gmi_filter_search');

        if ($filter_nonce_ok) {
            $count        = isset($_GET['count']) && is_numeric($_GET['count']) && intval($_GET['count']) > 0 ? intval($_GET['count']) : 20;
            $min_kb       = isset($_GET['min_kb']) && is_numeric($_GET['min_kb']) ? intval($_GET['min_kb']) : 0;
            $media_type   = isset($_GET['media_type']) ? sanitize_text_field(wp_unslash($_GET['media_type'])) : 'all';
            $show_variants = isset($_GET['show_variants']) && $_GET['show_variants'] === '1';
        } else {
            $count = 20; $min_kb = 0; $media_type = 'all'; $show_variants = false;
        }
        $min_bytes    = $min_kb * 1024;
        $mime_types = $this->get_mime_types_for($media_type);
        $args = $this->build_media_query_args($count, '', $mime_types, [], $min_bytes);
        $query = new WP_Query($args);

        $current_query = wp_unslash($_GET);
        if (isset($current_query['page'])) {
            $current_query['page'] = '103graphics-media-inspector';
        } else {
            $current_query = array_merge($current_query, ['page' => '103graphics-media-inspector']);
        }
        $base_admin_url = add_query_arg($current_query, admin_url('upload.php'));
        $csv_url   = add_query_arg('mfs_export_csv', '1', $base_admin_url);
        $csv_url   = wp_nonce_url($csv_url, 'gmi_export_csv');
        $print_url = add_query_arg('mfs_print_mode', '1', $base_admin_url);
        if ($show_variants) {
            $csv_url = add_query_arg('show_variants', '1', $csv_url);
            $print_url = add_query_arg('show_variants', '1', $print_url);
        }
        $reset_url = add_query_arg(['page' => '103graphics-media-inspector'], admin_url('upload.php'));
        ?>
        <div class="wrap">
            <?php if ( $is_print ) : ?>
                <div class="mfs-print-meta">
                    <h1 style="margin-bottom:7px;font-size:2em;"><?php esc_html_e('103Graphics Media Inspector', '103graphics-media-inspector'); ?></h1>
                    <?php
                    if ( function_exists('get_custom_logo') && has_custom_logo() ) {
                        $logo_id = get_theme_mod('custom_logo');
                        $logo = wp_get_attachment_image($logo_id, 'medium', false, array(
                            'style' => 'max-height:65px;max-width:280px;width:auto;height:auto;display:block;margin:0 auto 15px;'
                        ));
                        echo $logo ? wp_kses_post($logo) : '';
                    }
                    ?>
                    <div style="font-size:1.4em;font-weight:600;"><?php echo esc_html( get_bloginfo('name') ); ?></div>
                    <div style="font-size:1.1em; margin-top:5px;color:#333;"><?php echo esc_html( date_i18n(get_option('date_format') . ' ' . get_option('time_format')) ); ?></div>
                </div>
            <?php else: ?>
            <h1><?php esc_html_e( '103Graphics Media Inspector', '103graphics-media-inspector' ); ?></h1>
            <section class="mfs-group-card" style="margin-bottom:18px;">
                <h2 style="margin:8px 0;"><?php esc_html_e( 'Advanced File Search', '103graphics-media-inspector' ); ?></h2>
                <form method="get" action="<?php echo esc_url( admin_url( 'upload.php' ) ); ?>" class="mfs-options-bar">
                    <input type="hidden" name="page" value="103graphics-media-inspector" />
                    <?php wp_nonce_field('gmi_filter_search', 'media_inspector_nonce'); ?>
                    <label>
                        <?php esc_html_e( 'Type', '103graphics-media-inspector' ); ?>
                        <select name="media_type">
                            <option value="image" <?php selected( $media_type, 'image' ); ?>><?php esc_html_e( 'Images', '103graphics-media-inspector' ); ?></option>
                            <option value="pdf" <?php selected( $media_type, 'pdf' ); ?>><?php esc_html_e( 'PDFs', '103graphics-media-inspector' ); ?></option>
                            <option value="video" <?php selected( $media_type, 'video' ); ?>><?php esc_html_e( 'Videos', '103graphics-media-inspector' ); ?></option>
                            <option value="all" <?php selected( $media_type, 'all' ); ?>><?php esc_html_e( 'All', '103graphics-media-inspector' ); ?></option>
                        </select>
                    </label>
                    <label>
                        <?php esc_html_e( 'Show', '103graphics-media-inspector' ); ?>
                        <input type="number" name="count" min="1" max="9999" step="1" value="<?php echo esc_attr( $count ); ?>" style="width:86px;" />
                    </label>
                    <label>
                        <?php esc_html_e( 'Min KB', '103graphics-media-inspector' ); ?>
                        <input type="number" name="min_kb" min="0" step="1" value="<?php echo esc_attr( $min_kb ); ?>" style="width:86px;" />
                    </label>
                    <label style="align-items:center;">
                        <input type="checkbox" name="show_variants" value="1" <?php checked( $show_variants ); ?> />
                        <?php esc_html_e( 'Show Variants', '103graphics-media-inspector' ); ?>
                    </label>
                    <a href="<?php echo esc_url( $csv_url ); ?>" class="button"><?php esc_html_e( 'Export CSV', '103graphics-media-inspector' ); ?></a>
                    <a href="<?php echo esc_url( $print_url ); ?>" class="button" target="_blank"><?php esc_html_e( 'Print View', '103graphics-media-inspector' ); ?></a>
                    <button type="submit" class="button button-primary"><?php esc_html_e( 'Apply', '103graphics-media-inspector' ); ?></button>
                    <a href="<?php echo esc_url( $reset_url ); ?>" class="button" style="background:#f6f7f7;color:#222;border:1px solid #ccd0d4;"><?php esc_html_e( 'Reset Filters', '103graphics-media-inspector' ); ?></a>
                </form>
            </section>
            <?php endif; ?>
            <section class="mfs-group-card">
                <h2 style="margin:8px 0;"><?php esc_html_e( 'Results', '103graphics-media-inspector' ); ?></h2>
                <?php
                if ( $query->have_posts() ) :
                    $grand_total = 0;
                    ?>
                    <table class="mfs-largest-table widefat" style="width:100%; border-collapse:collapse;">
                        <thead>
                            <tr>
                                <th><?php esc_html_e( 'Preview', '103graphics-media-inspector' ); ?></th>
                                <th><?php esc_html_e( 'File Details', '103graphics-media-inspector' ); ?></th>
                                <th><?php esc_html_e( 'Original Size', '103graphics-media-inspector' ); ?></th>
                                <th><?php esc_html_e( 'Total Size', '103graphics-media-inspector' ); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                        <?php
                        foreach ( $query->posts as $attachment ) {
                            $meta       = wp_get_attachment_metadata( $attachment->ID );
                            $file       = get_attached_file( $attachment->ID );
                            $orig_size  = get_post_meta( $attachment->ID, self::ORIGINAL_META, true );
                            $total_size = get_post_meta( $attachment->ID, self::TOTAL_META, true );
                            $full_url = wp_get_attachment_url( $attachment->ID );
                            $mime     = get_post_mime_type( $attachment->ID );
                            $thumb_html = '';
                            if ( wp_attachment_is_image( $attachment->ID ) ) {
                                $thumb_html = wp_get_attachment_image( $attachment->ID, [ 80, 80 ] );
                            } elseif ( false !== strpos( $mime, 'pdf' ) ) {
                                $thumb_html = wp_get_attachment_image( $attachment->ID, [ 80, 80 ] );
                                if ( empty( $thumb_html ) ) {
                                    $icon_url = includes_url( 'images/media/document.png' );
                                    $thumb_html = '<img src="' . esc_url( $icon_url ) . '" alt="PDF" style="width:80px;height:80px;border-radius:3px;background:#f6f7f7;object-fit:contain;" />';
                                }
                            } elseif ( false !== strpos( $mime, 'video' ) ) {
                                if ( has_post_thumbnail( $attachment->ID ) ) {
                                    $thumb_html = get_the_post_thumbnail( $attachment->ID, [ 80, 80 ] );
                                } else {
                                    $thumb_html = '<div style="width:80px;height:80px;border-radius:3px;background:#f6f7f7;display:flex;align-items:center;justify-content:center;">' .
                                        '<span style="font-size:28px;color:#888;">▶️</span></div>';
                                }
                            } else {
                                $thumb_html = '<div style="width:80px;height:80px;border-radius:3px;background:#f6f7f7;display:flex;align-items:center;justify-content:center;">' .
                                    '<span style="font-size:26px;color:#888;">📄</span></div>';
                            }
                            $variants_output = '';
                            $has_variants = false;
                            $variant_list = [];
                            if ( ! empty( $meta['sizes'] ) && is_array( $meta['sizes'] ) && $file && is_file( $file ) ) {
                                foreach ( $meta['sizes'] as $size_name => $variant ) {
                                    if ( empty( $variant['file'] ) ) {
                                        continue;
                                    }
                                    $variant_file = path_join( dirname( $file ), $variant['file'] );
                                    if ( is_file( $variant_file ) ) {
                                        $variant_list[] = [
                                            'size_name' => $size_name,
                                            'file'      => $variant['file'],
                                            'size'      => filesize( $variant_file ),
                                        ];
                                    }
                                }
                            }
                            if ( $variant_list ) {
                                $has_variants = true;
                                usort( $variant_list, function( $a, $b ) {
                                    return $b['size'] - $a['size'];
                                } );
                                if ( $show_variants ) {
                                    $variants_output .= '<ul style="margin:6px 0 0 16px;padding-left:16px;">';
                                    foreach ( $variant_list as $v ) {
                                        $variants_output .= '<li><em>' . esc_html( $v['size_name'] ) . '</em> (' . esc_html( $v['file'] ) . '): ' . esc_html( size_format( (int) $v['size'], 2 ) ) . '</li>';
                                    }
                                    $variants_output .= '</ul>';
                                }
                            }
                            if ( $min_bytes > 0 && ( ! $orig_size || (int) $orig_size < $min_bytes ) ) {
                                continue;
                            }
                            $grand_total += (int) $total_size;
                            $toggle_id = 'mfs-var-' . $attachment->ID;
                            $toggle_html = ( $has_variants && $show_variants ) ? '<button class="mfs-toggle button" data-target="' . esc_attr( $toggle_id ) . '">' . esc_html__( 'Show Variants', '103graphics-media-inspector' ) . '</button>' : '';
                            echo '<tr>';
                            echo '<td style="width:110px;text-align:center;">';
                            if ( $full_url ) {
                                echo '<a href="' . esc_url( $full_url ) . '" target="_blank" rel="noopener">' . ( $thumb_html ? wp_kses_post($thumb_html) : '' ) . '</a>';
                            } else {
                                echo $thumb_html ? wp_kses_post($thumb_html) : '';
                            }
                            echo '</td>';
                            echo '<td>';
                            if ( $file && is_file( $file ) ) {
                                echo '<strong>' . esc_html( basename( $file ) ) . '</strong>: ' . esc_html( size_format( (int) filesize( $file ), 2 ) );
                            } else {
                                echo '<strong>' . esc_html( $attachment->post_title ) . '</strong>';
                            }
                            if ( $has_variants && $show_variants ) {
                                echo ' ' . ( $toggle_html ? wp_kses_post($toggle_html) : '' );
                                echo '<div id="' . esc_attr( $toggle_id ) . '" class="mfs-variants">' . ( $variants_output ? wp_kses_post($variants_output) : '' ) . '</div>';
                            }
                            echo '</td>';
                            echo '<td>' . ( $orig_size ? esc_html( size_format( (int) $orig_size, 2 ) ) : '&mdash;' ) . '</td>';
                            echo '<td>' . ( $total_size ? esc_html( size_format( (int) $total_size, 2 ) ) : '&mdash;' ) . '</td>';
                            echo '</tr>';
                        }
                        ?>
                        </tbody>
                        <tfoot>
                            <tr>
                                <th colspan="3" style="text-align:right;"><?php esc_html_e( 'Total of Displayed Files (Total Size):', '103graphics-media-inspector' ); ?></th>
                                <th><?php echo esc_html( size_format( (int) $grand_total, 2 ) ); ?></th>
                            </tr>
                        </tfoot>
                    </table>
                <?php
                else :
                    echo '<p>' . esc_html__( 'No files found.', '103graphics-media-inspector' ) . '</p>';
                endif;
                ?>
            </section>
        </div>
        <?php
    }
}

register_activation_hook(__FILE__, ['Graphics_Media_Inspector_Plugin', 'activation']);
register_deactivation_hook(__FILE__, ['Graphics_Media_Inspector_Plugin', 'deactivation']);
new Graphics_Media_Inspector_Plugin();
